/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.



#include <mrpt/config.h>

#include <mrpt/gui/CDisplayWindow.h>

#include <mrpt/system/os.h>
#include <mrpt/utils/CMRPTImage.h>
#include <mrpt/utils/CMRPTImageFloat.h>

#include <mrpt/gui/WxSubsystem.h>
#include <mrpt/gui/WxUtils.h>

using namespace mrpt;
using namespace mrpt::gui;
using namespace mrpt::utils;
using namespace mrpt::system;
using namespace std;


IMPLEMENTS_SERIALIZABLE(CDisplayWindow,CSerializable,mrpt::gui)


#if MRPT_HAS_WXWIDGETS


BEGIN_EVENT_TABLE(CWindowDialog,wxFrame)

END_EVENT_TABLE()



const long CWindowDialog::ID_IMAGE_BITMAP = wxNewId();
const long ID_MENUITEM1 = wxNewId();
const long ID_MENUITEM2 = wxNewId();
const long ID_MENUITEM3 = wxNewId();


CWindowDialog::CWindowDialog(
    CDisplayWindow *win2D,
    WxSubsystem::CWXMainFrame* parent,
    wxWindowID id,
    const std::string &caption) :
    m_win2D( win2D ),
    m_mainFrame(parent)
{
	// Load my custom icons:
//    wxArtProvider::Push(new MyArtProvider);

    Create(
        parent,
        id,
        _U(caption.c_str()),
        wxDefaultPosition,
        wxDefaultSize,
        wxDEFAULT_FRAME_STYLE,
        _T("id"));
    SetClientSize(wxSize(672,539));

/*    {
    	wxIcon FrameIcon;
    	FrameIcon.CopyFromBitmap(wxArtProvider::GetBitmap(wxART_MAKE_ART_ID_FROM_STR(_T("MAIN_ICON")),wxART_FRAME_ICON));
    	SetIcon(FrameIcon);
    }
*/

    // Create the image object:
    wxBitmap dummyBmp;
    m_image = new wxStaticBitmap( this, ID_IMAGE_BITMAP, dummyBmp );

    wxFlexGridSizer  *flex1 = new wxFlexGridSizer(1,1);
    flex1->Add(m_image,wxLEFT | wxTOP | wxEXPAND );
    SetSizer(flex1);

    //m_canvas = new CMyGLCanvas( win2D, this, wxID_ANY, wxDefaultPosition, wxDefaultSize );


    // Menu:
    wxMenuBar *MenuBar1 = new wxMenuBar();

    wxMenu *Menu1 = new wxMenu();
    wxMenuItem *MenuItem3 = new wxMenuItem(Menu1, ID_MENUITEM3, _("Save to file..."), _(""), wxITEM_NORMAL);
    Menu1->Append(MenuItem3);
    wxMenuItem *MenuItem1 = new wxMenuItem(Menu1, ID_MENUITEM1, _("Close"), _(""), wxITEM_NORMAL);
    Menu1->Append(MenuItem1);
    MenuBar1->Append(Menu1, _("&File"));

    wxMenu *Menu2 = new wxMenu();
    wxMenuItem *MenuItem2 = new wxMenuItem(Menu2, ID_MENUITEM2, _("About..."), _(""), wxITEM_NORMAL);
    Menu2->Append(MenuItem2);
    MenuBar1->Append(Menu2, _("&Help"));

    SetMenuBar(MenuBar1);


    // Events:
    Connect(wxID_ANY,wxEVT_CLOSE_WINDOW,(wxObjectEventFunction)&CWindowDialog::OnClose);
    Connect(ID_MENUITEM1,wxEVT_COMMAND_MENU_SELECTED,(wxObjectEventFunction)&CWindowDialog::OnMenuClose);
    Connect(ID_MENUITEM2,wxEVT_COMMAND_MENU_SELECTED,(wxObjectEventFunction)&CWindowDialog::OnMenuAbout);
    Connect(ID_MENUITEM3,wxEVT_COMMAND_MENU_SELECTED,(wxObjectEventFunction)&CWindowDialog::OnMenuSave);

	Connect(wxID_ANY,wxEVT_KEY_DOWN,(wxObjectEventFunction)&CWindowDialog::OnChar);

    // Register ourselves into the list of windows:
    //WxSubsystem::m_csWindowsList.enter();
	//WxSubsystem::m_windowsList[ this ] = m_win2D;
	//WxSubsystem::m_csWindowsList.leave();

    // Increment number of windows:
    //int winCount =
    WxSubsystem::CWXMainFrame::notifyWindowCreation();
    //cout << "[CWindowDialog] Notifying new window: " << winCount << endl;

}

// Destructor
CWindowDialog::~CWindowDialog()
{
}

// OnClose event:
void CWindowDialog::OnClose(wxCloseEvent& event)
{
    // Set the m_hwnd=NULL in our parent object.
    m_win2D->notifyChildWindowDestruction();

    // Delete ourselves from the list of open windows:
    //WxSubsystem::m_csWindowsList.enter();
    //std::map<wxFrame*,CDisplayWindow*>::iterator	dumm;
    //dumm = WxSubsystem::m_windowsList.find( this );
    //if (dumm!=WxSubsystem::m_windowsList.end())
    //{
    //    WxSubsystem::m_windowsList.erase( dumm );
    //}
	//WxSubsystem::m_csWindowsList.leave();


    // Decrement number of windows:
    int winCount = WxSubsystem::CWXMainFrame::notifyWindowDestruction();
    //cout << "[~CWindowDialog] Notifying window destruction: " << winCount << endl;

    // Close main frame if we were the last window. If new windows are created, a new main frame will be created again.
    if (!winCount)
    {
        //cout << "[~CWindowDialog] Closing main frame." << endl;
        m_mainFrame->Close();
    }

	// Signal we are destroyed:
	//cout << "[~CWindowDialog] Signaling semaphore meaning window is actually closed." << endl;
    m_win2D->m_semWindowDestroyed.release();

    event.Skip();
}

void CWindowDialog::OnChar(wxKeyEvent& event)
{
	//event.m_keyCode;
	cout << "char" << endl;
}

// Menu: Close
void CWindowDialog::OnMenuClose(wxCommandEvent& event)
{
    Close();
}
// Menu: About
void CWindowDialog::OnMenuAbout(wxCommandEvent& event)
{
    ::wxMessageBox(_("Image viewer\n Class gui::CDisplayWindow\n MRPT C++ library"),_("About..."));
}

// Menu: Save to file
void CWindowDialog::OnMenuSave(wxCommandEvent& event)
{
    wxFileDialog dialog(
        this,
        wxT("Save image as..."),
        wxT("."),
        wxT("image.png"),
        wxT("PNG image files (*.png)|*.png"),
#if wxCHECK_VERSION(2, 8, 0)
        wxFD_SAVE | wxFD_OVERWRITE_PROMPT );
#else
        wxSAVE | wxOVERWRITE_PROMPT );
#endif

    if (wxID_OK == dialog.ShowModal())
    {
        try
        {
            wxBitmap  bmp ( m_image->GetBitmap() );
            bmp.SaveFile( dialog.GetPath(), wxBITMAP_TYPE_PNG );
        }
        catch(...)
        {
        }
    }
}


#endif


/*---------------------------------------------------------------
					Constructor
 ---------------------------------------------------------------*/
CDisplayWindow::CDisplayWindow( const std::string &windowCaption )
	: m_semThreadReady(0,1),
	  m_semWindowDestroyed(0,1),
	  m_caption(windowCaption)
{
	MRPT_TRY_START

    m_hwnd                      = 0;
    m_keyPushed                 = false;
    m_enableCursorCoordinates   = true;

#if MRPT_HAS_WXWIDGETS
	// Create the main wxThread:
	// -------------------------------
	if (!WxSubsystem::createOneInstanceMainThread() )
        return; // Error!

    // Create window:
    WxSubsystem::TRequestToWxMainThread  *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->source2D = this;
    REQ->str      = m_caption;
    REQ->OPCODE   = 200;
    REQ->voidPtr  = m_hwnd.getPtrToPtr();
    WxSubsystem::pushPendingWxRequest( REQ );

    // Wait for the window to realize and signal it's alive:
    if (!WxSubsystem::isConsoleApp)
    {
    	mrpt::system::sleep(20);	// Force at least 1-2 timer ticks for processing the event:
    	wxApp::GetInstance()->Yield(true);
    }
	if(!m_semThreadReady.waitForSignal(6000))  // 2 secs should be enough...
		cerr << "[CDisplayWindow::CDisplayWindow] Timeout waiting window creation." << endl;

#else
	THROW_EXCEPTION("MRPT compiled without wxWidgets!")
#endif

	MRPT_TRY_END
}

/*---------------------------------------------------------------
					Destructor
 ---------------------------------------------------------------*/
CDisplayWindow::~CDisplayWindow( )
{
#if MRPT_HAS_WXWIDGETS
    // Send close request:
    WxSubsystem::TRequestToWxMainThread  *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->source2D = this;
    REQ->OPCODE   = 299;
    WxSubsystem::pushPendingWxRequest( REQ );

	// Wait until the thread ends:
    if (!WxSubsystem::isConsoleApp)
    {
    	mrpt::system::sleep(20);	// Force at least 1-2 timer ticks for processing the event:
    	wxApp::GetInstance()->Yield(true);
    }
	if(!m_semWindowDestroyed.waitForSignal(6000))  // 2 secs should be enough...
	{
	    cerr << "[CDisplayWindow::~CDisplayWindow] Timeout waiting window destruction." << endl;
	}

	WxSubsystem::waitWxShutdownsIfNoWindows();
#endif
}


/*---------------------------------------------------------------
					showImage
 ---------------------------------------------------------------*/
void  CDisplayWindow::showImage( const	CMRPTImage& img )
{
#if MRPT_HAS_WXWIDGETS
	MRPT_TRY_START;

    // Send message of new image:
	wxImage *newImg = mrpt::gui::MRPTImage2wxImage( img );

    // Send a request to destroy this object:
    WxSubsystem::TRequestToWxMainThread *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->source2D = this;
    REQ->OPCODE   = 201;
    REQ->voidPtr  = m_hwnd.get();
    REQ->voidPtr2 = (void*) newImg;
    WxSubsystem::pushPendingWxRequest( REQ );


	MRPT_TRY_END;
#endif
}

/*---------------------------------------------------------------
					showImageAndPoints
 ---------------------------------------------------------------*/
void  CDisplayWindow::showImageAndPoints( const CMRPTImage &img, const vector_float &x, const vector_float &y, unsigned int color )
{
#if MRPT_HAS_WXWIDGETS
	MRPT_TRY_START;
	ASSERT_( x.size() == y.size() );

	CMRPTImage imgColor(1,1,3);
	imgColor = img;	// Make a copy of the image
	for(unsigned int i = 0; i < x.size(); i++)
		imgColor.cross(x[i],y[i],color,'+');

	showImage(imgColor);
	MRPT_TRY_END;
#endif
}

/*---------------------------------------------------------------
					showImageAndPoints
 ---------------------------------------------------------------*/
void  CDisplayWindow::showImageAndPoints( const CMRPTImage &img, const mrpt::vision::CFeatureList &list, unsigned int color )
{
#if MRPT_HAS_WXWIDGETS
	MRPT_TRY_START;

	CMRPTImage imgColor(1,1,3);
	imgColor = img;	// Make a copy of the image
	for( mrpt::vision::CFeatureList::const_iterator i = list.begin(); i != list.end(); i++ )
		imgColor.cross( (*i)->x, (*i)->y, color, '+' );

	showImage(imgColor);
	MRPT_TRY_END;
#endif
}


/*---------------------------------------------------------------
					showImageAndPoints
 ---------------------------------------------------------------*/
void  CDisplayWindow::showImageAndPoints( const	CMRPTImageFloat &img, const vector_float &x, const vector_float &y, unsigned int color )
{
	MRPT_TRY_START;
	ASSERT_( x.size() == y.size() );

	// Suboptimal but...
	CMRPTImage imgColor(1,1,3);
	imgColor = img;  // Force to be a RGB image!
	for(unsigned int i = 0; i < x.size(); i++)
		imgColor.cross(x[i],y[i],color,'+');
	showImage( imgColor );

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					showImage
 ---------------------------------------------------------------*/
void  CDisplayWindow::showImage( const CMRPTImageFloat& img )
{
	MRPT_TRY_START;

	// Suboptimal but...
	CMRPTImage imgColor(1,1,3);
	imgColor = img ;  // Force to be a RGB image!
	showImage( imgColor );

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					isOpen
 ---------------------------------------------------------------*/
bool CDisplayWindow::isOpen()
{
    return m_hwnd!=NULL;
}

/*---------------------------------------------------------------
					waitForKey
 ---------------------------------------------------------------*/
void  CDisplayWindow::waitForKey()
{
    m_keyPushed = false;
	while (!m_keyPushed && !os::kbhit())
	{
		system::sleep(10);
		// Are we still alive?
		if (!isOpen()) break;
	}
	if (os::kbhit())	os::getch();
}


/*---------------------------------------------------------------
					resize
 ---------------------------------------------------------------*/
void  CDisplayWindow::resize(
	unsigned int width,
	unsigned int height )
{
#if MRPT_HAS_WXWIDGETS
	if (!isOpen())
	{
		cerr << "[CDisplayWindow::resize] Window closed!: " << m_caption << endl;
		return;
	}

    // Send a request to destroy this object:
    WxSubsystem::TRequestToWxMainThread *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->source2D = this;
    REQ->OPCODE   = 203;
    REQ->x        = width;
    REQ->y        = height;
    WxSubsystem::pushPendingWxRequest( REQ );
#endif
}

/*---------------------------------------------------------------
					setPos
 ---------------------------------------------------------------*/
void  CDisplayWindow::setPos( int x, int y )
{
#if MRPT_HAS_WXWIDGETS
	if (!isOpen())
	{
		cerr << "[CDisplayWindow::setPos] Window closed!: " << m_caption << endl;
		return;
	}

    // Send a request to destroy this object:
    WxSubsystem::TRequestToWxMainThread *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->source2D = this;
    REQ->OPCODE   = 202;
    REQ->x        = x;
    REQ->y        = y;
    WxSubsystem::pushPendingWxRequest( REQ );
#endif
}

/*---------------------------------------------------------------
					setWindowTitle
 ---------------------------------------------------------------*/
void  CDisplayWindow::setWindowTitle( const std::string &str )
{
#if MRPT_HAS_WXWIDGETS
	if (!isOpen())
	{
		cerr << "[CDisplayWindow::setWindowTitle] Window closed!: " << m_caption << endl;
		return;
	}

    // Send a request to destroy this object:
    WxSubsystem::TRequestToWxMainThread *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->source2D = this;
    REQ->OPCODE   = 204;
    REQ->str      = str;
    WxSubsystem::pushPendingWxRequest( REQ );
#endif
}

/*---------------------------------------------------------------
					notifyChildWindowDestruction
 ---------------------------------------------------------------*/
void CDisplayWindow::notifyChildWindowDestruction()
{
    //cout << "[CDisplayWindow::notifyChildWindowDestruction] Called." << endl;
    m_hwnd = NULL;
}

/*---------------------------------------------------------------
					writeToStream
 ---------------------------------------------------------------*/
void  CDisplayWindow::writeToStream(mrpt::utils::CStream &out, int *getVersion) const
{
	THROW_EXCEPTION("Class not serializable");
}

/*---------------------------------------------------------------
					readFromStream
 ---------------------------------------------------------------*/
void  CDisplayWindow::readFromStream(mrpt::utils::CStream &in, int version)
{
	THROW_EXCEPTION("Class not serializable");
}

