/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/opengl/CSetOfObjects.h>
#include <mrpt/opengl/CTexturedPlane.h>
#include <mrpt/math/CVectorTemplate.h>

#include <mrpt/utils/CStringList.h>

#include "opengl_internals.h"

using namespace mrpt;
using namespace mrpt::opengl;
using namespace mrpt::utils;
using namespace mrpt::math;
using namespace std;

IMPLEMENTS_SERIALIZABLE( CSetOfObjects, CRenderizable, mrpt::opengl )

/*---------------------------------------------------------------
							render
  ---------------------------------------------------------------*/
void   CSetOfObjects::clear()
{
	m_objects.clear(); // clear the list and delete objects (if there are no more copies out there!)
}

/*---------------------------------------------------------------
							render
  ---------------------------------------------------------------*/
void   CSetOfObjects::render()
{
#if MRPT_HAS_OPENGL_GLUT
	CListOpenGLObjects::iterator	it;
	try
	{
		for (it=m_objects.begin();it!=m_objects.end();it++)
		{
			// 3D coordinates transformation:
			glMatrixMode(GL_MODELVIEW);
			glPushMatrix();

			glTranslated((*it)->m_x, (*it)->m_y, (*it)->m_z);
			glRotatef((*it)->m_roll, 1.0, 0.0, 0.0);
			glRotatef((*it)->m_pitch, 0.0, 1.0, 0.0);
			glRotatef((*it)->m_yaw, 0.0, 0.0, 1.0);

			// Set color:
			glColor4f( (*it)->m_color_R,(*it)->m_color_G,(*it)->m_color_B,(*it)->m_color_A);

			(*it)->render();

			if ((*it)->m_show_name)
			{
				glDisable(GL_DEPTH_TEST);
				glRasterPos3f(0.0f,0.0f,0.0f);
				glColor3f(1.0f,1.0f,1.0f);

				GLfloat		raster_pos[4];
				glGetFloatv( GL_CURRENT_RASTER_POSITION, raster_pos);
				float eye_distance= raster_pos[3];

				void *font=NULL;
				if (eye_distance<2)
						font = GLUT_BITMAP_TIMES_ROMAN_24;
				else if(eye_distance<200)
					font = GLUT_BITMAP_TIMES_ROMAN_10;

				if (font)
					CRenderizable::renderTextBitmap( (*it)->m_name.c_str(), font);

				glEnable(GL_DEPTH_TEST);
			}

			glPopMatrix();
			checkOpenGLError();
		}
	}
	catch(exception &e)
	{
		char	str[1000];
		os::sprintf(str,1000,"Exception while rendering a class '%s'\n%s",
			(*it)->GetRuntimeClass()->className,
			e.what() );
		THROW_EXCEPTION(str);
	}
	catch(...)
	{
		THROW_EXCEPTION("Runtime error!");
	}
#endif
}

/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  CSetOfObjects::writeToStream(CStream &out,int *version) const
{
	if (version)
		*version = 0;
	else
	{
		writeToStreamRender(out);

		uint32_t	n;
		n = (uint32_t)m_objects.size();
		out << n;
		for (CListOpenGLObjects::const_iterator	it=m_objects.begin();it!=m_objects.end();it++)
			out << **it;
	}
}

/*---------------------------------------------------------------
	Implements the reading from a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void  CSetOfObjects::readFromStream(CStream &in,int version)
{

	switch(version)
	{
	case 0:
		{
			readFromStreamRender(in);

			uint32_t	n;
			in >> n;
			clear();
			m_objects.resize(n);

			for_each(m_objects.begin(),m_objects.end(), ObjectReadFromStream(&in) );

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}

/*---------------------------------------------------------------
					initializeAllTextures
  ---------------------------------------------------------------*/
void  CSetOfObjects::initializeAllTextures()
{
#if MRPT_HAS_OPENGL_GLUT
	CListOpenGLObjects::iterator it;
	//deque<mrpt::opengl::CRenderizable*>::iterator it;
	for (it=m_objects.begin();it!=m_objects.end();it++)
	{
		if ( (*it)->GetRuntimeClass() == CLASS_ID_NAMESPACE(CTexturedPlane,opengl))
		{
			getAs<CTexturedPlane>(*it)->loadTextureInOpenGL();
		}
		else
		if ( (*it)->GetRuntimeClass() == CLASS_ID_NAMESPACE(CSetOfObjects,opengl))
		{
			getAs<CSetOfObjects>(*it)->initializeAllTextures();
		}
	}
#endif
}


CSetOfObjects::CSetOfObjects( )
{
}


CSetOfObjects::~CSetOfObjects()
{
	clear();
}


void CSetOfObjects::insert( const CRenderizablePtr &newObject )
{
	m_objects.push_back(newObject);
}


/*---------------------------------------------------------------
					renderTextBitmap
  ---------------------------------------------------------------*/
void	CRenderizable::renderTextBitmap( const char *str, void *fontStyle )
{
#if MRPT_HAS_OPENGL_GLUT
	MRPT_TRY_START;
	while ( *str ) glutBitmapCharacter( fontStyle ,*(str++) );
	MRPT_TRY_END;
#endif
}


/*--------------------------------------------------------------
					dumpListOfObjects
  ---------------------------------------------------------------*/
void CSetOfObjects::dumpListOfObjects( utils::CStringList  &lst )
{
	for (CListOpenGLObjects::iterator	it=m_objects.begin();it!=m_objects.end();it++)
	{
		// Single obj:
		string  s( (*it)->GetRuntimeClass()->className );
		if ((*it)->m_name.size())
			s+= string(" (") +(*it)->m_name + string(")");
		lst.add( s );

		if ((*it)->GetRuntimeClass() == CLASS_ID_NAMESPACE(CSetOfObjects,mrpt::opengl))
		{
			CSetOfObjects *objs = getAs<CSetOfObjects>(*it);

			utils::CStringList  auxLst;
			objs->dumpListOfObjects(auxLst);
			for (size_t i=0;i<auxLst.size();i++)
				lst.add( string(" ")+auxLst(i) );
		}
	}
}

/*--------------------------------------------------------------
					removeObject
  ---------------------------------------------------------------*/
void CSetOfObjects::removeObject( const CRenderizablePtr &obj )
{
	for (CListOpenGLObjects::iterator it=m_objects.begin();it!=m_objects.end();it++)
		if (it->pointer() == obj.pointer())
		{
			m_objects.erase(it);
			return;
		}
		else if ( (*it)->GetRuntimeClass()==CLASS_ID_NAMESPACE(CSetOfObjects,opengl) )
			getAs<CSetOfObjects>(*it)->removeObject(obj);
}

