/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 


#include <mrpt/poses/CPose3DPDFSOG.h>

using namespace mrpt::poses;
using namespace mrpt::utils;
using namespace std;

IMPLEMENTS_SERIALIZABLE( CPose3DPDFSOG, CPose3DPDF, mrpt::poses )

/*---------------------------------------------------------------
	Constructor
  ---------------------------------------------------------------*/
CPose3DPDFSOG::CPose3DPDFSOG( size_t nModes ) :
	m_modes(nModes)
{

}

/*---------------------------------------------------------------
						getEstimatedPoint
  Returns an estimate of the pose, (the mean, or mathematical expectation of the PDF)
 ---------------------------------------------------------------*/
CPose3D  CPose3DPDFSOG::getEstimatedPose() const
{
	size_t		N = m_modes.size();

	if (N)
	{
		double			X=0,Y=0,Z=0,YAW=0,PITCH=0,ROLL=0;
		double			ang,sumW=0;

		double			W_yaw_R=0,W_yaw_L=0;
		double			yaw_R=0,yaw_L=0;
		double			W_roll_R=0,W_roll_L=0;
		double			roll_R=0,roll_L=0;


		std::deque<TGaussianMode>::const_iterator	it;
		double w;

		for (it=m_modes.begin();it!=m_modes.end();it++)
		{
			sumW += w = exp(it->log_w);

			X += it->val.mean.x * w;
			Y += it->val.mean.y * w;
			Z += it->val.mean.z * w;
			PITCH	+= w * it->val.mean.pitch;

			// Angles Yaw and Roll are especials!:
			ang = it->val.mean.yaw;
			if (fabs( ang )>1.5707963267948966192313216916398f)
			{
				// LEFT HALF: 0,2pi
				if (ang<0) ang = M_2PI + ang;
				yaw_L += ang * w;
				W_yaw_L += w;
			}
			else
			{
				// RIGHT HALF: -pi,pi
				yaw_R += ang * w;
				W_yaw_R += w;
			}

			// Angles Yaw and Roll are especials!:
			ang = it->val.mean.roll;
			if (fabs( ang )>1.5707963267948966192313216916398f)
			{
				// LEFT HALF: 0,2pi
				if (ang<0) ang = M_2PI + ang;
				roll_L += ang * w;
				W_roll_L += w;
			}
			else
			{
				// RIGHT HALF: -pi,pi
				roll_R += ang * w;
				W_roll_R += w;
			}
		}

		if (sumW==0) return CPose3D(0,0,0,0,0,0);

		X /= sumW;
		Y /= sumW;
		Z /= sumW;
		PITCH /= sumW;

		// Next: Yaw and Roll:
		// -----------------------------------
		// The mean value from each side:
		if (W_yaw_L>0)	yaw_L /= W_yaw_L;  // [0,2pi]
		if (W_yaw_R>0)	yaw_R /= W_yaw_R;  // [-pi,pi]
		// Left side to [-pi,pi] again:
		if (yaw_L>M_PI) yaw_L = yaw_L - M_2PI;

		// The mean value from each side:
		if (W_roll_L>0)	roll_L /= W_roll_L;  // [0,2pi]
		if (W_roll_R>0)	roll_R /= W_roll_R;  // [-pi,pi]
		// Left side to [-pi,pi] again:
		if (roll_L>M_PI) roll_L = roll_L - M_2PI;

		// The total means:
		YAW = ((yaw_L * W_yaw_L + yaw_R * W_yaw_R )/(W_yaw_L+W_yaw_R));
		ROLL = ((roll_L * W_roll_L + roll_R * W_roll_R )/(W_roll_L+W_roll_R));

		return CPose3D(X,Y,Z,YAW,PITCH,ROLL);
	}
	else
	{
		return CPose3D(0,0,0);
	}
}

/*---------------------------------------------------------------
						getEstimatedCovariance
 ---------------------------------------------------------------*/
CMatrixD  CPose3DPDFSOG::getEstimatedCovariance() const
{
	size_t		N = m_modes.size();

	if (N)
	{
		// 1) Get the mean:
		double		w,sumW = 0;
		CMatrixD		estCov(6,6);
		CMatrixD		estMean( getEstimatedPose() );

		estCov.zeros();

		std::deque<TGaussianMode>::const_iterator	it;

		for (it=m_modes.begin();it!=m_modes.end();it++)
		{
			sumW += w = exp(it->log_w);
			CMatrixD		estMean_i(it->val.mean);
			estCov += w * ( it->val.cov + ((estMean_i-estMean)*(~(estMean_i-estMean))) );
		}

		if (sumW==0)
				return CMatrixD(6,6);		// Zeros!
		else	return estCov * (1.0/sumW);
	}
	else
	{
		return CMatrixD(3,3);
	}
}

/*---------------------------------------------------------------
						writeToStream
  ---------------------------------------------------------------*/
void  CPose3DPDFSOG::writeToStream(CStream &out,int *version) const
{
	if (version)
		*version = 2;
	else
	{
		uint32_t	N = m_modes.size();
		std::deque<TGaussianMode>::const_iterator		it;

		out << N;

		for (it=m_modes.begin();it!=m_modes.end();it++)
		{
			out << it->log_w;
			out << it->val.mean;
			out << it->val.cov(0,0) << it->val.cov(1,1) << it->val.cov(2,2);
			out << it->val.cov(0,1) << it->val.cov(0,2) << it->val.cov(1,2);
		}
	}
}

/*---------------------------------------------------------------
						readFromStream
  ---------------------------------------------------------------*/
void  CPose3DPDFSOG::readFromStream(CStream &in,int version)
{
	switch(version)
	{
	case 0:
	case 1:
	case 2:
		{
			uint32_t								N;
			std::deque<TGaussianMode>::iterator		it;
			float									x0;
			double									x;

			in >> N;

			m_modes.resize(N);

			for (it=m_modes.begin();it!=m_modes.end();it++)
			{
				in >> it->log_w;

				// In version 0, weights were linear!!
				if (version==0) it->log_w = log(max(1e-300,it->log_w));

				in >> it->val.mean;

				it->val.cov.setSize(3,3);

				if (version==1) // were floats
				{
					in >> x0; it->val.cov(0,0) = x0;
					in >> x0; it->val.cov(1,1) = x0;
					in >> x0; it->val.cov(2,2) = x0;

					in >> x0; it->val.cov(1,0) = x0; it->val.cov(0,1) = x0;
					in >> x0; it->val.cov(2,0) = x0; it->val.cov(0,2) = x0;
					in >> x0; it->val.cov(1,2) = x0; it->val.cov(2,1) = x0;
				}
				else
				{
					in >> x; it->val.cov(0,0) = x;
					in >> x; it->val.cov(1,1) = x;
					in >> x; it->val.cov(2,2) = x;

					in >> x; it->val.cov(1,0) = x; it->val.cov(0,1) = x;
					in >> x; it->val.cov(2,0) = x; it->val.cov(0,2) = x;
					in >> x; it->val.cov(1,2) = x; it->val.cov(2,1) = x;
				}

			}

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}



/*---------------------------------------------------------------
						operator =
  ---------------------------------------------------------------*/
void  CPose3DPDFSOG::copyFrom(const CPose3DPDF &o)
{
	MRPT_TRY_START;

	if (this == &o) return;		// It may be used sometimes

	if (o.GetRuntimeClass()==CLASS_ID(CPose3DPDFSOG))
	{
		m_modes = static_cast<const CPose3DPDFSOG*>(&o)->m_modes;
	}
	else
	{
		// Approximate as a mono-modal gaussian pdf:
		/** \todo: Something more elaborate?
		  */
		m_modes.resize(1);
		m_modes[0].log_w = 0;
		m_modes[0].val.mean = o.getEstimatedPose();
		m_modes[0].val.cov = o.getEstimatedCovariance();
	}

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						saveToTextFile
  ---------------------------------------------------------------*/
void  CPose3DPDFSOG::saveToTextFile(const std::string &file) const
{
	FILE	*f=os::fopen(file.c_str(),"wt");
	if (!f) return;


	for (std::deque<TGaussianMode>::const_iterator it=m_modes.begin();it!=m_modes.end();it++)
		os::fprintf(f,"%e %e %e %e %e %e %e %e %e %e\n",
			exp(it->log_w),
			it->val.mean.x, it->val.mean.y, it->val.mean.z,
			it->val.cov(0,0),it->val.cov(1,1),it->val.cov(2,2),
			it->val.cov(0,1),it->val.cov(0,2),it->val.cov(1,2) );
	os::fclose(f);
}

/*---------------------------------------------------------------
						changeCoordinatesReference
 ---------------------------------------------------------------*/
void  CPose3DPDFSOG::changeCoordinatesReference(const CPose3D &newReferenceBase )
{
	CMatrixD		M = newReferenceBase.getHomogeneousMatrix();
	M.setSize(3,3);	// Clip the 4x4 matrix

	// The variance in phi is unmodified:
	M(0,2) = 0; M(1,2) = 0;
	M(2,0) = 0; M(2,1) = 0;
	M(2,2) = 1;

	for (std::deque<TGaussianMode>::iterator it=m_modes.begin();it!=m_modes.end();it++)
		it->val.changeCoordinatesReference( newReferenceBase );

	assureSymmetry();
}

/*---------------------------------------------------------------
					bayesianFusion
 ---------------------------------------------------------------*/
void  CPose3DPDFSOG::bayesianFusion( CPose3DPDF &p1_, CPose3DPDF &p2_ )
{
	MRPT_TRY_START;

	// p1: CPose3DPDFSOG, p2: CPosePDFGaussian:

	ASSERT_( p1_.GetRuntimeClass() == CLASS_ID(CPose3DPDFSOG) );
	ASSERT_( p2_.GetRuntimeClass() == CLASS_ID(CPose3DPDFSOG) );

	THROW_EXCEPTION("TODO!!!");

/*
	CPose3DPDFSOG		*p1 = (CPose3DPDFSOG*)&p1_;
	CPose3DPDFSOG		*p2 = (CPose3DPDFSOG*)&p2_;

	// Compute the new kernel means, covariances, and weights after multiplying to the Gaussian "p2":
	CPosePDFGaussian	auxGaussianProduct,auxSOG_Kernel_i;
	TGaussianMode		newKernel;



	CMatrixD				covInv( p2->cov.inv() );
	CMatrixD				eta(3,1);
	eta(0,0) = p2->mean.x;
	eta(1,0) = p2->mean.y;
	eta(2,0) = p2->mean.phi;
	eta = covInv * eta;

	// Normal distribution canonical form constant:
	// See: http://www-static.cc.gatech.edu/~wujx/paper/Gaussian.pdf
	double				a = -0.5*( 3*log(M_2PI) - log( covInv.det() ) + (~eta * p2->cov * eta)(0,0) );

	this->m_modes.clear();
	for (std::deque<TGaussianMode>::iterator it =p1->m_modes.begin();it!=p1->m_modes.end();it++)
	{
		auxSOG_Kernel_i.mean = it->mean;
		auxSOG_Kernel_i.cov  = it->cov;
		auxGaussianProduct.bayesianFusion( auxSOG_Kernel_i, *p2 );

		// ----------------------------------------------------------------------
		// The new weight is given by:
		//
		//   w'_i = w_i * exp( a + a_i - a' )
		//
		//      a = -1/2 ( dimensionality * log(2pi) - log(det(Cov^-1)) + (Cov^-1 * mu)^t * Cov^-1 * (Cov^-1 * mu) )
		//
		// ----------------------------------------------------------------------
		newKernel.mean = auxGaussianProduct.mean;
		newKernel.cov  = auxGaussianProduct.cov;

		CMatrixD		covInv_i( auxSOG_Kernel_i.cov.inv() );
		CMatrixD		eta_i(3,1);
		eta_i(0,0) = auxSOG_Kernel_i.mean.x;
		eta_i(1,0) = auxSOG_Kernel_i.mean.y;
		eta_i(2,0) = auxSOG_Kernel_i.mean.phi;
		eta_i = covInv_i * eta_i;

		CMatrixD		new_covInv_i( newKernel.cov.inv() );
		CMatrixD		new_eta_i(3,1);
		new_eta_i(0,0) = newKernel.mean.x;
		new_eta_i(1,0) = newKernel.mean.y;
		new_eta_i(2,0) = newKernel.mean.phi;
		new_eta_i = new_covInv_i * new_eta_i;

		double		a_i	    = -0.5*( 3*log(M_2PI) - log( new_covInv_i.det() ) + (~eta_i * auxSOG_Kernel_i.cov * eta_i)(0,0) );
		double		new_a_i = -0.5*( 3*log(M_2PI) - log( new_covInv_i.det() ) + (~new_eta_i * newKernel.cov * new_eta_i)(0,0) );

		newKernel.w	   = it->w * exp( a + a_i - new_a_i );

		// Add to the results (in "this") the new kernel:
		this->m_modes.push_back( newKernel );
	}
*/
	normalizeWeights();

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						assureSymmetry
 ---------------------------------------------------------------*/
void  CPose3DPDFSOG::assureSymmetry()
{
	MRPT_TRY_START;
	// Differences, when they exist, appear in the ~15'th significant
	//  digit, so... just take one of them arbitrarily!
	for (std::deque<TGaussianMode>::iterator it=m_modes.begin();it!=m_modes.end();it++)
	{
		it->val.cov(0,1) = it->val.cov(1,0);
		it->val.cov(0,2) = it->val.cov(2,0);
		it->val.cov(1,2) = it->val.cov(2,1);
	}

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						normalizeWeights
 ---------------------------------------------------------------*/
void  CPose3DPDFSOG::normalizeWeights()
{
	MRPT_TRY_START;

	if (!m_modes.size()) return;

	std::deque<TGaussianMode>::iterator		it;
	double									maxW = m_modes[0].log_w;
	for (it=m_modes.begin();it!=m_modes.end();it++)
		maxW = max(maxW,it->log_w);

	for (it=m_modes.begin();it!=m_modes.end();it++)
		it->log_w -= maxW;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						drawSingleSample
 ---------------------------------------------------------------*/
void CPose3DPDFSOG::drawSingleSample( CPose3D &outPart ) const
{
	THROW_EXCEPTION("TO DO!");
}

/*---------------------------------------------------------------
						drawManySamples
 ---------------------------------------------------------------*/
void CPose3DPDFSOG::drawManySamples( size_t N, std::vector<vector_double> & outSamples ) const
{
	THROW_EXCEPTION("TO DO!");
}

/*---------------------------------------------------------------
						inverse
 ---------------------------------------------------------------*/
void  CPose3DPDFSOG::inverse(CPose3DPDF &o) const
{
	MRPT_TRY_START;
	ASSERT_( o.GetRuntimeClass() == CLASS_ID(CPose3DPDFSOG) );
	CPose3DPDFSOG	*out = static_cast<CPose3DPDFSOG*>( &o );

	// Prepare the output SOG:
	out->m_modes.resize(m_modes.size());

	std::deque<TGaussianMode>::const_iterator	it;
	std::deque<TGaussianMode>::iterator	outIt;

	for (it=m_modes.begin(),outIt=out->m_modes.begin();it!=m_modes.end();it++,outIt++)
	{
		it->val.inverse( outIt->val );
		outIt->log_w = it->log_w;
	}


	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						appendFrom
 ---------------------------------------------------------------*/
void  CPose3DPDFSOG::appendFrom( const CPose3DPDFSOG &o )
{
	MRPT_TRY_START;

	ASSERT_( &o != this );  // Don't be bad...
	if (o.m_modes.empty()) return;

	// Make copies:
	for (deque<TGaussianMode>::const_iterator it = o.m_modes.begin();it!=o.m_modes.end();it++)
		m_modes.push_back( TGaussianMode( *it ) );

	normalizeWeights();
	MRPT_TRY_END;
}
