/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.

#include <mrpt/slam/CMultiMetricMap.h>
#include <mrpt/slam/CColouredPointsMap.h>
#include <mrpt/slam/CSimplePointsMap.h>

#include <mrpt/math/CPolygon.h>

#include <mrpt/opengl/CPointCloudColoured.h>

#include <mrpt/vision/utils.h>

using namespace mrpt::slam;
using namespace mrpt::utils;
using namespace mrpt::poses;
using namespace mrpt::math;

IMPLEMENTS_SERIALIZABLE(CColouredPointsMap, CPointsMap,mrpt::slam)

/*---------------------------------------------------------------
						Constructor
  ---------------------------------------------------------------*/
CColouredPointsMap::CColouredPointsMap()
{
	reserve( 400 );
}

/*---------------------------------------------------------------
						Destructor
  ---------------------------------------------------------------*/
CColouredPointsMap::~CColouredPointsMap()
{
}

/*---------------------------------------------------------------
						Copy constructor
  ---------------------------------------------------------------*/
void  CColouredPointsMap::copyFrom(const CPointsMap &obj)
{
	MRPT_TRY_START;

	if (this==&obj)
		return;

	x = obj.x;
	y = obj.y;
	z = obj.z;
	pointWeight = obj.pointWeight;

	m_largestDistanceFromOriginIsUpdated = obj.m_largestDistanceFromOriginIsUpdated;
	m_largestDistanceFromOrigin = obj.m_largestDistanceFromOrigin;

	m_KDTreeDataIsUpdated				 = false;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					LoadFromRangeScan
 Transform the range scan into a set of cartessian coordinated
   points, leaving a given min. distance between them.
  ---------------------------------------------------------------*/
void  CColouredPointsMap::loadFromRangeScan(
	const CObservation2DRangeScan		&rangeScan,
	const CPose3D						*robotPose )
{
	int			i;
	CPose3D		sensorPose3D;

	m_largestDistanceFromOriginIsUpdated  = false;
	m_KDTreeDataIsUpdated				 = false;

	// If robot pose is supplied, compute sensor pose relative to it.
	if (!robotPose)
			sensorPose3D = rangeScan.sensorPose;
	else	sensorPose3D = (*robotPose) + rangeScan.sensorPose;

	if (!insertionOptions.addToExistingPointsMap)
	{
		x.clear();
		y.clear();
		z.clear();

		pointWeight.clear();
	}

	int		sizeRangeScan = rangeScan.scan.size();

	// For a great gain in efficiency:
	if ( x.size()+2*sizeRangeScan > x.capacity() )
	{
		reserve( (size_t) (x.size() * 1.2f) + 3*sizeRangeScan );
	}

	// --------------------------------------------------------------------------
	//  SPECIAL CASE OF HORIZONTAL SCAN: QUICKER IMPLEMENTATION
	// --------------------------------------------------------------------------
	CMatrix		HM = sensorPose3D.getHomogeneousMatrix();

	// --------------------------------------------------------------------------
	//		GENERAL CASE OF SCAN WITH ARBITRARY 3D ORIENTATION
	// --------------------------------------------------------------------------
	{
		// For quicker access:
		float		m00 = HM(0,0);
		float		m01 = HM(0,1);
		//float		m02 = HM(0,2);
		float		m03 = HM(0,3);
		float		m10 = HM(1,0);
		float		m11 = HM(1,1);
		//float		m12 = HM(1,2);
		float		m13 = HM(1,3);
		float		m20 = HM(2,0);
		float		m21 = HM(2,1);
		//float		m22 = HM(2,2);
		float		m23 = HM(2,3);

		float		lx_1,ly_1,lz_1,lx,ly,lz;		// Punto anterior y actual:
		float		lx_2,ly_2,lz_2;				// Punto antes del anterior

		// Initial last point:
		lx_1 = -100; ly_1 = -100; lz_1 = -100;
		lx_2 = -100; ly_2 = -100; lz_2 = -100;

		// ------------------------------------------------------
		//		Pass range scan to a set of 2D points:
		// ------------------------------------------------------
		vector_float		scan_x,scan_y;
		double		Ang, dA;
		if (rangeScan.rightToLeft)
		{
			Ang = - 0.5 * rangeScan.aperture;
			dA  = rangeScan.aperture / sizeRangeScan;
		}
		else
		{
			Ang = + 0.5 * rangeScan.aperture;
			dA  = - rangeScan.aperture / sizeRangeScan;
		}

		scan_x.resize( sizeRangeScan );
		scan_y.resize( sizeRangeScan );

		vector_float::iterator		 scan_x_it, scan_y_it;
		vector_float::const_iterator scan_it;

		for ( scan_x_it=scan_x.begin(),
			  scan_y_it=scan_y.begin(),
			  scan_it=rangeScan.scan.begin();
				scan_it!=rangeScan.scan.end();
			  scan_x_it++, scan_y_it++,scan_it++)
		{
			*scan_x_it = *scan_it * cos(  Ang );
			*scan_y_it = *scan_it * sin(  Ang );
			Ang+=dA;
		}

		float  minDistSqrBetweenLaserPoints = square( insertionOptions.minDistBetweenLaserPoints );

		// If the user doesn't want a minimum distance:
		if (insertionOptions.minDistBetweenLaserPoints<0)
			minDistSqrBetweenLaserPoints = -1;

		// ----------------------------------------------------------------
		//   Transform these points into 3D using the pose transformation:
		// ----------------------------------------------------------------
		bool	lastPointWasValid = true;
		bool	thisIsTheFirst = true;
		bool  	lastPointWasInserted = false;
		float	changeInDirection = 0;

		ASSERT_(colorScheme.z_max!=colorScheme.z_min);
		const float Az_1_color = 1.0/(colorScheme.z_max-colorScheme.z_min);
		float	pR=1,pG=1,pB=1;

		for (i=0;i<sizeRangeScan;i++)
		{
			// Punto actual del scan:
			if ( rangeScan.validRange[i] ) //(rangeScan.scan[i]< rangeScan.maxRange )
			{
				float rel_z = m20*scan_x[i] + m21*scan_y[i];

				lx = m00*scan_x[i] + m01*scan_y[i] + m03;
				ly = m10*scan_x[i] + m11*scan_y[i] + m13;
				lz = rel_z + m23;

				// Compute color:
				switch( colorScheme.scheme )
				{
				case cmFromHeightRelativeToSensor:
					{
						float q = (rel_z-colorScheme.z_min)*Az_1_color;
						if (q<0) q=0; else if (q>1) q=1;

						mrpt::vision::jet2rgb(q, pR,pG,pB);
					}
					break;
				default:
					THROW_EXCEPTION("Unknown color scheme");
				}


				lastPointWasInserted = false;

				// Add if distance > minimum only:
				float d2 = (square(lx-lx_1) + square(ly-ly_1) + square(lz-lz_1) );
				if ( thisIsTheFirst || (lastPointWasValid && (d2 > minDistSqrBetweenLaserPoints)) )
				{
					thisIsTheFirst = false;
					// Si quieren que interpolemos tb. los puntos lejanos, hacerlo:
					if (insertionOptions.also_interpolate && i>1)
					{
						float d = sqrt( d2 );

						if ((lx!=lx_1 || ly!=ly_1) && (lx_1!=lx_2 || ly_1!=ly_2) )
								changeInDirection = atan2(ly-ly_1,lx-lx_1)-atan2(ly_1-ly_2,lx_1-lx_2);
						else	changeInDirection = 0;

						// Conditions to really interpolate the points:
						if (d>=2*insertionOptions.minDistBetweenLaserPoints &&
							d<insertionOptions.maxDistForInterpolatePoints &&
							fabs(changeInDirection)<DEG2RAD(5) )
						{
							int nInterpol = round(d / (2*sqrt(minDistSqrBetweenLaserPoints)));

							for (int q=1;q<nInterpol;q++)
							{
								float i_x = lx_1 + q*(lx-lx_1)/nInterpol;
								float i_y = ly_1 + q*(ly-ly_1)/nInterpol;
								float i_z = lz_1 + q*(lz-lz_1)/nInterpol;

								x.push_back( i_x );
								y.push_back( i_y );
								z.push_back( i_z );
								pointWeight.push_back( 1 );
								m_color_R.push_back(pR);
								m_color_G.push_back(pG);
								m_color_B.push_back(pB);
							}
						} // End of interpolate:
					}

					x.push_back( lx );
					y.push_back( ly );
					z.push_back( lz );
					pointWeight.push_back( 1 );
					m_color_R.push_back(pR);
					m_color_G.push_back(pG);
					m_color_B.push_back(pB);

					lastPointWasInserted = true;

					lx_2 = lx_1;
					ly_2 = ly_1;
					lz_2 = lz_1;

					lx_1 = lx;
					ly_1 = ly;
					lz_1 = lz;
				}

			}

			// Save for next iteration:
			lastPointWasValid = rangeScan.validRange[i] != 0;

		}

		// The last point
		if (lastPointWasValid && !lastPointWasInserted)
		{
			x.push_back( lx );
			y.push_back( ly );
			z.push_back( lz );
			pointWeight.push_back( 1 );
			m_color_R.push_back(pR);
			m_color_G.push_back(pG);
			m_color_B.push_back(pB);
		}
	}
}

/*---------------------------------------------------------------
					load2D_from_text_file
  Load from a text file. In each line there are a point coordinates.
    Returns false if any error occured, true elsewere.
  ---------------------------------------------------------------*/
bool  CColouredPointsMap::load2D_from_text_file(std::string file)
{
	MRPT_TRY_START;

	m_largestDistanceFromOriginIsUpdated  = false;
	m_KDTreeDataIsUpdated				 = false;

	FILE	*f=os::fopen(file.c_str(),"rt");
	if (!f) return false;

	char		str[500];
	char		*ptr,*ptr1,*ptr2;

	// Clear current map:
	x.clear();
	y.clear();
	z.clear();
	pointWeight.clear();

	while (!feof(f))
	{
		// Read one string line:
		str[0] = 0;
		if (!fgets(str,sizeof(str),f)) break;

		// Find the first digit:
		ptr=str;
		while (ptr[0] && (ptr[0]==' ' || ptr[0]=='\t' || ptr[0]=='\r' || ptr[0]=='\n'))
			ptr++;

		// And try to parse it:
		float	xx = strtod(ptr,&ptr1);
		if (ptr1!=ptr)
		{
			float	yy = strtod(ptr1,&ptr2);
			if (ptr2!=ptr1)
			{
				x.push_back(xx);
				y.push_back(yy);
				z.push_back(0);
				pointWeight.push_back(1);
			}
		}
	}

	os::fclose(f);
	return true;

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
					load3D_from_text_file
  Load from a text file. In each line there are a point coordinates.
    Returns false if any error occured, true elsewere.
  ---------------------------------------------------------------*/
bool  CColouredPointsMap::load3D_from_text_file(std::string file)
{
	MRPT_TRY_START;

	m_largestDistanceFromOriginIsUpdated  = false;
	m_KDTreeDataIsUpdated				 = false;

	FILE	*f=os::fopen(file.c_str(),"rt");
	if (!f) return false;

	char		str[100];
	char		*ptr,*ptr1,*ptr2,*ptr3;

	// Clear current map:
	x.clear();
	y.clear();
	z.clear();
	pointWeight.clear();

	while (!feof(f))
	{
		// Read one string line:
		str[0] = 0;
		if (!fgets(str,sizeof(str),f)) break;

		// Find the first digit:
		ptr=str;
		while (ptr[0] && (ptr[0]==' ' || ptr[0]=='\t' || ptr[0]=='\r' || ptr[0]=='\n'))
			ptr++;

		// And try to parse it:
		float	xx = strtod(ptr,&ptr1);
		if (ptr1!=str)
		{
			float	yy = strtod(ptr1,&ptr2);
			if (ptr2!=ptr1)
			{
				float	zz = strtod(ptr2,&ptr3);
				if (ptr3!=ptr2)
				{
					x.push_back(xx);
					y.push_back(yy);
					z.push_back(zz);
					pointWeight.push_back(1);
				}
			}
		}
	}

	os::fclose(f);
	return true;

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
					writeToStream
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  CColouredPointsMap::writeToStream(CStream &out, int *version) const
{
	if (version)
		*version = 4;
	else
	{
		uint32_t n = x.size();

		// First, write the number of points:
		out << n;

		if (n>0)
		{
			out.WriteBuffer(&x[0],sizeof(x[0])*n);
			out.WriteBuffer(&y[0],sizeof(x[0])*n);
			out.WriteBuffer(&z[0],sizeof(x[0])*n);
			out.WriteBuffer(&pointWeight[0],sizeof(pointWeight[0])*n);
		}

		// version 2: options saved too
		out	<< insertionOptions.minDistBetweenLaserPoints
			<< insertionOptions.addToExistingPointsMap
			<< insertionOptions.also_interpolate
			<< insertionOptions.disableDeletion
			<< insertionOptions.fuseWithExisting
			<< insertionOptions.isPlanarMap
			<< insertionOptions.matchStaticPointsOnly
			<< insertionOptions.maxDistForInterpolatePoints;

		// Insertion as 3D:
		out << m_disableSaveAs3DObject;

		// Added in version 3:
		out << insertionOptions.horizontalTolerance;
	}
}

/*---------------------------------------------------------------
					readFromStream
   Implements the reading from a CStream capability of
      CSerializable objects
  ---------------------------------------------------------------*/
void  CColouredPointsMap::readFromStream(CStream &in, int version)
{
	switch(version)
	{
	case 0:
	case 1:
	case 2:
	case 3:
		{
			m_largestDistanceFromOriginIsUpdated  = false;
			m_KDTreeDataIsUpdated				 = false;

			// Read the number of points:
			uint32_t n;
			in >> n;

			x.resize(n);
			y.resize(n);
			z.resize(n);
			pointWeight.resize(n,1);	// Default value=1

			if (n>0)
			{
				in.ReadBuffer(&x[0],sizeof(x[0])*n);
				in.ReadBuffer(&y[0],sizeof(x[0])*n);
				in.ReadBuffer(&z[0],sizeof(x[0])*n);

				// Version 1: weights are also stored:
				// Version 4: Type becomes long int -> uint32_t for portability!!
				if (version>=1)
				{
					if (version>=4)
							in.ReadBuffer(&pointWeight[0],sizeof(pointWeight[0])*n);
					else	in.ReadBuffer(&pointWeight[0],sizeof(unsigned long)*n);
				}
			}

			if (version>=2)
			{
				// version 2: options saved too
				in 	>> insertionOptions.minDistBetweenLaserPoints
					>> insertionOptions.addToExistingPointsMap
					>> insertionOptions.also_interpolate
					>> insertionOptions.disableDeletion
					>> insertionOptions.fuseWithExisting
					>> insertionOptions.isPlanarMap
					>> insertionOptions.matchStaticPointsOnly
					>> insertionOptions.maxDistForInterpolatePoints;

				in >> m_disableSaveAs3DObject;

				if (version>=3)
				{
					in >> insertionOptions.horizontalTolerance;
				}
			}

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};

}

/*---------------------------------------------------------------
					Clear
  ---------------------------------------------------------------*/
void  CColouredPointsMap::clear()
{
	x.clear();
	y.clear();
	z.clear();
	m_largestDistanceFromOriginIsUpdated  = false;
	m_KDTreeDataIsUpdated				 = false;
}

/*---------------------------------------------------------------
					setPoint
			Store a points coordinates:
  ---------------------------------------------------------------*/
void  CColouredPointsMap::setPoint(const size_t &index,CPoint2D &p)
{
	if (index>=this->x.size())
		THROW_EXCEPTION("Index out of bounds");

	this->x[index] = p.x;
	this->y[index] = p.y;
	this->z[index] = 0;

	m_largestDistanceFromOriginIsUpdated  = false;
	m_KDTreeDataIsUpdated				 = false;
}
void  CColouredPointsMap::setPoint(const size_t & index,CPoint3D &p)
{
	if (index>=this->x.size())
		THROW_EXCEPTION("Index out of bounds");

	this->x[index] = p.x;
	this->y[index] = p.y;
	this->z[index] = p.z;
	m_largestDistanceFromOriginIsUpdated  = false;
	m_KDTreeDataIsUpdated				 = false;
}
void  CColouredPointsMap::setPoint(const size_t & index,float x,float y)
{
	if (index>=this->x.size())
		THROW_EXCEPTION("Index out of bounds");

	this->x[index] = x;
	this->y[index] = y;
	this->z[index] = 0;
	m_largestDistanceFromOriginIsUpdated  = false;
	m_KDTreeDataIsUpdated				 = false;
}
void  CColouredPointsMap::setPoint(const size_t & index,float x,float y,float z)
{
	if (index>=this->x.size())
		THROW_EXCEPTION("Index out of bounds");

	this->x[index] = x;
	this->y[index] = y;
	this->z[index] = z;
	m_largestDistanceFromOriginIsUpdated  = false;
	m_KDTreeDataIsUpdated				 = false;
}

/*---------------------------------------------------------------
Insert the contents of another map into this one, fusing the previous content with the new one.
 This means that points very close to existing ones will be "fused", rather than "added". This prevents
 the unbounded increase in size of these class of maps.
 ---------------------------------------------------------------*/
void  CColouredPointsMap::fuseWith(
			CPointsMap			*otherMap,
			float				minDistForFuse,
			std::vector<bool>	*notFusedPoints)
{
	TMatchingPairList			correspondences;
	TMatchingPairList::iterator	corrsIt;
	size_t						nThis,nOther,i;
	CPoint3D					a,b;
	float						corrRatio;
	static CPose2D				nullPose(0,0,0);

	m_largestDistanceFromOriginIsUpdated  = false;
	m_KDTreeDataIsUpdated				 = false;

	nThis  =     this->getPointsCount();
	nOther = otherMap->getPointsCount();

	// Find correspondences between this map and the other one:
	// ------------------------------------------------------------
	computeMatchingWith2D( otherMap,			// The other map
						   nullPose,	// The other map's pose
						   minDistForFuse,	// Max. dist. for correspondence
						   0,
						   nullPose,
						   correspondences,
						   corrRatio );

	// Initially, all set to "true" -> "not fused".
	if (notFusedPoints)
	{
		notFusedPoints->clear();
		notFusedPoints->reserve( x.size() + nOther );
		notFusedPoints->resize( x.size(), true );
	}

	// Speeds-up possible memory reallocations:
	reserve( x.size() + nOther );

	// Merge matched points from both maps:
	//  AND add new points which have been not matched:
	// -------------------------------------------------
	for (i=0;i<nOther;i++)
	{
		unsigned long	w_a = otherMap->getPoint(i,a);	// Get "local" point into "a"

		// Find closest correspondence of "a":
		int				closestCorr = -1;
		float			minDist	= 1e6;
		for (corrsIt = correspondences.begin(); corrsIt!=correspondences.end(); corrsIt++)
		{
			if (corrsIt->other_idx==i)
			{
				float	dist = square( corrsIt->other_x - corrsIt->this_x ) +
							   square( corrsIt->other_y - corrsIt->this_y ) +
							   square( corrsIt->other_z - corrsIt->this_z );
				if (minDist<dist)
				{
					minDist = dist;
					closestCorr = corrsIt->this_idx;
				}
			}
		} // End of for each correspondence...


		if (closestCorr!=-1)
		{	// Merge:		FUSION
			unsigned long w_b = CPointsMap::getPoint(closestCorr,b);

			ASSERT_((w_a+w_b)>0);

			float	 F = 1.0f/(w_a+w_b);

			x[closestCorr]=F*(w_a*a.x+w_b*b.x);
			y[closestCorr]=F*(w_a*a.y+w_b*b.y);
			z[closestCorr]=F*(w_a*a.z+w_b*b.z);

			pointWeight[closestCorr] = w_a+w_b;

			// Append to fused points list
			if (notFusedPoints)
				(*notFusedPoints)[closestCorr] = false;
		}
		else
		{	// New point:	ADDITION
			x.push_back( a.x );
			y.push_back( a.y );
			z.push_back( a.z );
			pointWeight.push_back( 1 );
			if (notFusedPoints)
				(*notFusedPoints).push_back(false);
		}
	}


}

/*---------------------------------------------------------------
						insertPoint
 ---------------------------------------------------------------*/
void  CColouredPointsMap::insertPoint( float x, float y, float z )
{
	this->x.push_back(x);
	this->y.push_back(y);
	this->z.push_back(z);
	pointWeight.push_back(1);

	m_largestDistanceFromOriginIsUpdated  = false;
	m_KDTreeDataIsUpdated				 = false;
}

/*---------------------------------------------------------------
						insertPoint
 ---------------------------------------------------------------*/
void  CColouredPointsMap::insertPoint( CPoint3D p )
{
	x.push_back(p.x);
	y.push_back(p.y);
	z.push_back(p.z);
	pointWeight.push_back(1);

	m_largestDistanceFromOriginIsUpdated  = false;
	m_KDTreeDataIsUpdated				 = false;
}



/*---------------------------------------------------------------
						applyDeletionMask
 ---------------------------------------------------------------*/
void  CColouredPointsMap::applyDeletionMask( std::vector<bool> &mask )
{
	ASSERT_( getPointsCount()==mask.size() );

	size_t	i,j,n;

	// Remove marked points:
	n = mask.size();
	for (i=0,j=0;i<n;i++)
	{
		if (!mask[i])
		{
			x[j]				= x[i];
			y[j]				= y[i];
			z[j]				= z[i];
			pointWeight[j++]	= pointWeight[i];
		}
	}

	// Set new correct size:
	x.resize(j);
	y.resize(j);
	z.resize(j);
	pointWeight.resize(j);

	m_largestDistanceFromOriginIsUpdated  = false;
	m_KDTreeDataIsUpdated				 = false;
}

/*---------------------------------------------------------------
					insertObservation

Insert the observation information into this map.
 ---------------------------------------------------------------*/
bool  CColouredPointsMap::insertObservation(
	const CObservation	*obs,
    const CPose3D			*robotPose)
{
	MRPT_TRY_START;

	CPose2D		robotPose2D;
	CPose3D		robotPose3D;

	if (robotPose)
	{
		robotPose2D = (*robotPose);
		robotPose3D = (*robotPose);
	}
	else
	{
		// Default values are (0,0,0)
	}

	if ( CLASS_ID(CObservation2DRangeScan)==obs->GetRuntimeClass())
	{
		/********************************************************************

					OBSERVATION TYPE: CObservation2DRangeScan

			********************************************************************/
		m_largestDistanceFromOriginIsUpdated  = false;
		m_KDTreeDataIsUpdated				 = false;

		const CObservation2DRangeScan *o = static_cast<const CObservation2DRangeScan *>(obs);
		// Insert only HORIZONTAL scans??
		bool	reallyInsertIt;

		if (insertionOptions.isPlanarMap)
			 reallyInsertIt = o->isPlanarScan( insertionOptions.horizontalTolerance );
		else reallyInsertIt = true;

		if (reallyInsertIt)
		{
			CColouredPointsMap	auxMap;
			size_t					i,n;
			CPose3D				sensorPose3D = robotPose3D + o->sensorPose;
			CPose2D				sensorPose2D( sensorPose3D );
			CPolygon			pol( sensorPose2D.x, sensorPose2D.y );
			const float			*xs,*ys,*zs;
			float				x,y;
			vector_int			fusedPointIndices;
			std::vector<bool>	checkForDeletion;

			// 1) Fuse into the points map or add directly?
			// ----------------------------------------------
			if (insertionOptions.fuseWithExisting)
			{
				// Fuse:
				auxMap.insertionOptions = insertionOptions;
				auxMap.insertionOptions.addToExistingPointsMap = false;

				auxMap.loadFromRangeScan(
					*o,						// The laser range scan observation
					&robotPose3D			// The robot pose
					);

				fuseWith(	&auxMap,					// Fuse with this map
							insertionOptions.minDistBetweenLaserPoints,	// Min dist.
							&checkForDeletion		// Set to "false" if a point in "map" has been fused.
							);
			}
			else
			{
				// Don't fuse: Simply add
				insertionOptions.addToExistingPointsMap = true;
				loadFromRangeScan(
					*o,						// The laser range scan observation
					&robotPose3D				// The robot pose
					);

				// Don't build this vector if is not used later!
				if (!insertionOptions.disableDeletion)
				{
					n = getPointsCount();
					checkForDeletion.resize(n);
					for (i=0;i<n;i++) checkForDeletion[i] = true;
				}
			}

			if (! insertionOptions.disableDeletion )
			{
				// 2) Delete points in newly added free
				//      region, thus dynamic areas:
				// --------------------------------------
				// Load scan as a polygon:
				auxMap.getPointsBuffer( n, xs,ys,zs);
				pol.setAllVertices( n, xs, ys );

				// Check for deletion of points in "map"
				n = getPointsCount();
				for (i=0;i<n;i++)
				{
					if ( checkForDeletion[i] )		// Default to true, unless a fused point, which must be kept.
					{
						CPointsMap::getPoint(i,x,y);
						if ( !pol.PointIntoPolygon( x,y) )
							checkForDeletion[i] = false;	// Out of polygon, don't delete
					}
				}

				// Create a new points list just with non-deleted points.
				// ----------------------------------------------------------
				applyDeletionMask( checkForDeletion );
			}


			return true;
		}
		// A planar map and a non-horizontal scan.
		else return false;
	}
	else
	{
		return false;
	}

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
 Computes the likelihood that a given observation was taken from a given pose in the world being modeled with this map.
	takenFrom The robot's pose the observation is supposed to be taken from.
	obs The observation.
 This method returns a likelihood in the range [0,1].
 ---------------------------------------------------------------*/
double	 CColouredPointsMap::computeObservationLikelihood(
			const CObservation		*obs,
			const CPose3D				&takenFrom )
{
	THROW_EXCEPTION("Not implemented yet");
}


/*---------------------------------------------------------------
					auxParticleFilterCleanUp
 ---------------------------------------------------------------*/
void  CColouredPointsMap::auxParticleFilterCleanUp()
{
}

/*---------------------------------------------------------------
					reserve
 ---------------------------------------------------------------*/
void CColouredPointsMap::reserve(size_t newLength)
{
	x.reserve( newLength );
	y.reserve( newLength );
	z.reserve( newLength );
	m_color_R.reserve( newLength );
	m_color_G.reserve( newLength );
	m_color_B.reserve( newLength );
}

/*---------------------------------------------------------------
					getAs3DObject
 ---------------------------------------------------------------*/
void CColouredPointsMap::getAs3DObject( mrpt::opengl::CSetOfObjectsPtr	&outObj ) const
{
	if (m_disableSaveAs3DObject)
		return;

	opengl::CPointCloudColouredPtr  obj = opengl::CPointCloudColoured::Create();

	obj->loadFromPointsMap(this);
	obj->m_color_A = 1; //0.5;

	obj->setPointSize( 3.0f );

	outObj->insert(obj);
}

/*---------------------------------------------------------------
					TColourOptions
 ---------------------------------------------------------------*/
CColouredPointsMap::TColourOptions::TColourOptions() :
	scheme(cmFromHeightRelativeToSensor),
	z_min(-10),
	z_max(10)
{
}

/*---------------------------------------------------------------
					TColourOptions
 ---------------------------------------------------------------*/
void CColouredPointsMap::TColourOptions::loadFromConfigFile(
	const mrpt::utils::CConfigFileBase  &source,
	const std::string &section)
{
	MRPT_LOAD_CONFIG_VAR_CAST( scheme, int, TColouringMethod,   source,section )
	MRPT_LOAD_CONFIG_VAR(z_min, float,		source,section)
	MRPT_LOAD_CONFIG_VAR(z_max, float,		source,section)
}

/*---------------------------------------------------------------
					TColourOptions
 ---------------------------------------------------------------*/
void  CColouredPointsMap::TColourOptions::dumpToTextStream(utils::CStream		&out)
{
	out.printf("\n----------- [CColouredPointsMap::TColourOptions] ------------ \n\n");

	out.printf("scheme                                  = %u\n",	(unsigned)scheme);
	out.printf("z_min                                   = %f\n",	z_min );
	out.printf("z_max                                   = %f\n",	z_max );
}

/*---------------------------------------------------------------
					getPoint
 ---------------------------------------------------------------*/
void  CColouredPointsMap::getPoint( size_t index, float &x, float &y, float &z, float &R, float &G, float &B ) const
{
	if (index >= this->x.size())
		THROW_EXCEPTION("Index out of bounds");

	x = this->x[index];
	y = this->y[index];
	z = this->z[index];

	R = m_color_R[index];
	G = m_color_G[index];
	B = m_color_B[index];
}
