/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.



#include <mrpt/slam/CObservation2DRangeScan.h>
#include <mrpt/slam/CSimplePointsMap.h>
#include <mrpt/poses/CPosePDF.h>

#include <mrpt/math/utils.h>

using namespace mrpt::slam;
using namespace mrpt::utils;
using namespace mrpt::poses;
using namespace mrpt::math;

// This must be added to any CSerializable class implementation file.
IMPLEMENTS_SERIALIZABLE(CObservation2DRangeScan, CObservation,mrpt::slam)

/*---------------------------------------------------------------
							Constructor
 ---------------------------------------------------------------*/
CObservation2DRangeScan::CObservation2DRangeScan( ) :
	m_auxMap(),
	scan(),
	validRange(),
	aperture( M_PI ),
	rightToLeft( true ),
	maxRange( 80.0f ),
	sensorPose(),
	stdError( 0.01f ),
	beamAperture(0),
	deltaPitch(0)
{
}

/*---------------------------------------------------------------
							Destructor
 ---------------------------------------------------------------*/
CObservation2DRangeScan::~CObservation2DRangeScan()
{
}

/*---------------------------------------------------------------
  Implements the writing to a CStream capability of CSerializable objects
 ---------------------------------------------------------------*/
void  CObservation2DRangeScan::writeToStream(CStream &out, int *version) const
{
	if (version)
		*version = 6;
	else
	{
		// The data
		out << aperture << rightToLeft << maxRange << sensorPose;
		uint32_t	N = scan.size();
		out << N;
		ASSERT_(validRange.size() == scan.size() );
		out.WriteBuffer( &scan[0], sizeof(scan[0])*N );
		out.WriteBuffer( &validRange[0],sizeof(validRange[0])*N );
		out << stdError;
		out << timestamp;
		out << beamAperture;

		out << sensorLabel;

		out << deltaPitch;
	}
}

/*---------------------------------------------------------------
  Filter out invalid points by a minimum distance, a maximum angle and a certain distance at the end (z-coordinate of the lasers must be provided)
 ---------------------------------------------------------------*/
void CObservation2DRangeScan::truncateByDistanceAndAngle(float min_distance, float max_angle, float min_height, float max_height, float h )
{
	// FILTER OUT INVALID POINTS!!
	vector_float::iterator		itScan;
	std::vector<char>::iterator itValid;
	CPose3D						pose;
	unsigned int				k;
	unsigned int				nPts = scan.size();
	
	for( itScan = scan.begin(), itValid = validRange.begin(), k = 0;
		 itScan != scan.end();
		 itScan++, itValid++, k++ )
	{
		float ang	= fabs(k*aperture/nPts - aperture*0.5);
		float x		= (*itScan)*cos(ang);
		
		if( min_height != 0 || max_height != 0 )
		{
			ASSERT_( max_height > min_height );
			if( *itScan < min_distance || ang > max_angle || x > h - min_height || x < h - max_height )
				*itValid = false;
		} // end if
		else
			if( *itScan < min_distance || ang > max_angle )
				*itValid = false;
	}
}

/*---------------------------------------------------------------
  Implements the reading from a CStream capability of CSerializable objects
 ---------------------------------------------------------------*/
void  CObservation2DRangeScan::readFromStream(CStream &in, int version)
{
	switch(version)
	{
	case 0:
	case 1:
	case 2:
	case 3:
		{
			CMatrix covSensorPose;
			in >> aperture >> rightToLeft  >> maxRange >> sensorPose >> covSensorPose;
			uint32_t		N,i;

			in >> N;

			scan.resize(N);
			in.ReadBuffer( &scan[0], sizeof(scan[0])*N);

			if (version>=1)
			{
				// Load validRange:
				validRange.resize(N);
				in.ReadBuffer( &validRange[0], sizeof(validRange[0])*N );
			}
			else
			{
				// validRange: Default values: If distance is not maxRange
				validRange.resize(N);
				for (i=0;i<N;i++)
					validRange[i]= scan[i] < maxRange;
			}

			if (version>=2)
			{
				in >> stdError;
			}
			else
			{
				stdError = 0.01f;
			}

			if (version>=3)
			{
				in >> timestamp;
			}

			// Default values for newer versions:
			beamAperture = DEG2RAD(0.25f);

			deltaPitch  = 0;
			sensorLabel = "";

		} break;

	case 4:
	case 5:
	case 6:
		{
			uint32_t		N;

			CMatrix covSensorPose;

			in >> aperture >> rightToLeft  >> maxRange >> sensorPose;

			if (version<6) // covSensorPose was removed in version 6
				in  >> covSensorPose;

			in >> N;
			scan.resize(N);
			in.ReadBuffer( &scan[0], sizeof(scan[0])*N);
			validRange.resize(N);
			in.ReadBuffer( &validRange[0], sizeof(validRange[0])*N );
			in >> stdError;
			in.ReadBuffer( &timestamp, sizeof(timestamp));
			in >> beamAperture;

			if (version>=5)
			{
					in >> sensorLabel;
					in >> deltaPitch;
			}
			else
			{
				sensorLabel = "";
				deltaPitch  = 0;
			}

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};

}

/*---------------------------------------------------------------
 Inserts a pure virtual method for finding the likelihood between this
   and another observation, probably only of the same derived class. The operator
   may be asymmetric.

 \param anotherObs The other observation to compute likelihood with.
 \param anotherObsPose If known, the belief about the robot pose when the other observation was taken can be supplied here, or NULL if it is unknown.

 \return Returns a likelihood measurement, in the range [0,1].
 \exception std::exception On any error, as another observation being of an invalid class.
  ---------------------------------------------------------------*/
float  CObservation2DRangeScan::likelihoodWith(
	const CObservation		*anotherObs,
	const CPosePDF			*anotherObsPose ) const
{
	CSimplePointsMap						map1,map2;
	const CObservation2DRangeScan			*obs;
	float									maxDisForCorrespondence = 0.04f;
	CMetricMap::TMatchingPairList			correspondences;
	float									correspondencesRatio;
	CPose2D									otherObsPose(0,0,0);
	static CPose2D							nullPose(0,0,0);


	if ( this->GetRuntimeClass() != anotherObs->GetRuntimeClass() )
		THROW_EXCEPTION("anotherPose is not a CObservation2DRangeScan object")

	// Now we know it is safe to make the casting:
	obs = static_cast<const CObservation2DRangeScan*>(anotherObs);

	// Build a 2D points map for each scan:
	map1.insertionOptions.minDistBetweenLaserPoints = maxDisForCorrespondence;
	map2.insertionOptions.minDistBetweenLaserPoints = maxDisForCorrespondence;

	map1.loadFromRangeScan( *this );
	map2.loadFromRangeScan( *obs );

	// if PDF is available, get "mean" value as an estimation:
	if (anotherObsPose)	otherObsPose = anotherObsPose->getEstimatedPose();

	map1.computeMatchingWith2D(
			&map2,
			otherObsPose,
			maxDisForCorrespondence,
			0,
			nullPose,
			correspondences,
			correspondencesRatio);

	return correspondencesRatio;
}

/*---------------------------------------------------------------
						buildAuxPointsMap
 ---------------------------------------------------------------*/
const CPointsMap	*CObservation2DRangeScan::buildAuxPointsMap( CPointsMap::TInsertionOptions *options ) const
{
	if (!m_auxMap.get())
	{
		// Create:
		CPointsMap *m = new CSimplePointsMap();
		if (options)
			m->insertionOptions = *options;
		m->insertObservation( this );
		m_auxMap.set(m);
	}

	return m_auxMap.get();
}

/*---------------------------------------------------------------
						isPlanarScan
 ---------------------------------------------------------------*/
bool  CObservation2DRangeScan::isPlanarScan( const double &tolerance  ) const
{
	return (fabs(sensorPose.pitch)<=tolerance) &&
	       ( fabs(sensorPose.roll)<=tolerance || fabs(math::wrapToPi( sensorPose.roll-M_PI))<=tolerance );
}
