/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef  CLevenbergMarquardt_H
#define  CLevenbergMarquardt_H

#include <mrpt/utils/CDebugOutputCapable.h>
#include <mrpt/math/CMatrixD.h>

/*---------------------------------------------------------------
	Class
  ---------------------------------------------------------------*/
namespace mrpt
{
namespace math
{
	/** An implementation of the Levenberg-Marquardt algorithm for least-square minimization.
	 *
	 *  Refer to the <a href="http://babel.isa.uma.es/mrpt/index.php/Levenberg%E2%80%93Marquardt_algorithm">wiki page</a> for more details on the algorithm and usage.
	 *
	 */
	template <typename NUMTYPE = double>
	class MRPTDLLIMPEXP CLevenbergMarquardtTempl : public mrpt::utils::CDebugOutputCapable
	{
	public:

		/** The type of the function passed to execute */
		typedef void (*TFunctor)(const std::vector<NUMTYPE> &x,const std::vector<NUMTYPE> &y, std::vector<NUMTYPE> &out);

		struct TResultInfo
		{
			NUMTYPE		final_sqr_err;
			size_t		iterations_executed;
			CMatrixTemplateNumeric<NUMTYPE>	path;	//!< Each row is the optimized value at each iteration.
		};

		/** Executes the LM-method, with derivatives estimated from
		  *  "functor" Is a user-provided function which takes as input two vectors, in this order:
		  *		- x: The parameters to be optimized.
		  *		- userParam: The vector passed to the LM algorithm, unmodified.
		  *	  and must return the "error vector", or the error (not squared) in each measured dimension, so the sum of the square of that output is the overall square error.
		  */
		static void	execute(
			std::vector<NUMTYPE>		&out_optimal_x,
			const std::vector<NUMTYPE>	&x0,
			TFunctor					functor,
			const std::vector<NUMTYPE>	&increments,
			const std::vector<NUMTYPE>	&userParam,
			TResultInfo					&out_info,
			bool						verbose = false,
			const size_t				&maxIter = 200,
			const NUMTYPE				tau = 1e-3,
			const NUMTYPE				e1 = 1e-8,
			const NUMTYPE				e2 = 1e-8
			);

	}; // End of class def.


	typedef CLevenbergMarquardtTempl<double> CLevenbergMarquardt;  //!< The default name for the LM class is an instantiation for "double"

	} // End of namespace
} // End of namespace
#endif
