/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef  stl_extensions_H
#define  stl_extensions_H

#include <mrpt/utils/CSerializable.h>
#include <mrpt/utils/CStream.h>

#include <set>
#include <map>
#include <list>

namespace mrpt
{
namespace utils
{
	// Forward class declarations:
	template <class T>				class vector_serializable;
	template <class T>				class deque_serializable;
	template <class T>				class list_searchable;
	template <class T>				class list_searchable_serializable;
	template <class K,class V> 		class map_serializable;


	// Typedefs:
	typedef vector_serializable<int64_t> vector_long_serializable;
	typedef vector_serializable<int32_t> vector_int_serializable;
	typedef vector_serializable<float> vector_float_serializable;
	typedef vector_serializable<double> vector_double_serializable;
	typedef vector_serializable<uint16_t> vector_word_serializable;
	typedef vector_serializable<uint32_t> vector_uint_serializable;
	typedef vector_serializable<bool> vector_bool_serializable;
	typedef vector_serializable<uint8_t> vector_byte_serializable;
	typedef vector_serializable<size_t> vector_size_t_serializable;


	// Streaming function declarations:
#define 	DECLARE_STREAMING_FUNCTIONS(class_name) 		\
	template <class T> CStream& operator << (CStream& out, const class_name<T> &obj)	\
		{	obj.write(out);	return out;	}	\
	template <class T> CStream& operator >> (CStream& in, class_name<T> &obj)	\
		{	obj.read(in);	return in;		}

#define 	DECLARE_STREAMING_FUNCTIONS2(class_name) 		\
	template <class T,class V> CStream& operator << (CStream& out, const class_name<T,V> &obj)	\
		{	obj.write(out);	return out;	}	\
	template <class T,class V> CStream& operator >> (CStream& in, class_name<T,V> &obj)	\
		{	obj.read(in);	return in;		}

	DECLARE_STREAMING_FUNCTIONS(vector_serializable)
	DECLARE_STREAMING_FUNCTIONS(deque_serializable)
	DECLARE_STREAMING_FUNCTIONS(list_searchable_serializable)
	DECLARE_STREAMING_FUNCTIONS2(map_serializable)


	/** This class implements a STL container with features of both, a std::set and a std::list.
	  */
	template <class T>
	class list_searchable : public std::list<T>
	{
	public:
		void insert( const T &o ) { std::list<T>::push_back(o); }

		typename std::list<T>::iterator find( const T& i )
		{
			for (typename std::list<T>::iterator it=std::list<T>::begin();it!=std::list<T>::end();it++)
				if (*it==i)
					return it;
			return std::list<T>::end();
		}

		typename std::list<T>::const_iterator find( const T& i ) const
		{
			for (typename std::list<T>::const_iterator it=std::list<T>::begin();it!=std::list<T>::end();it++)
				if (*it==i)
					return it;
			return std::list<T>::end();
		}

	};

	/** A STL-vector derived class that implements basic read/write to ease implementing the mrpt::utils::CSerializable interface in classes having this type of data members.
	  */
	template <class T>
	class vector_serializable : public std::vector<T>
	{
	public:
		vector_serializable<T> & operator =(const std::vector<T> &o )
		{
			/*std::vector<T>::resize(o.size());
			typename std::vector<T>::iterator me;
			typename std::vector<T>::const_iterator he;


			for (me=std::vector<T>::begin(),he=o.begin();me!=std::vector<T>::end();me++,he++)
				*me = *he;*/
			std::vector<T>::clear();
			std::vector<T>::insert(std::vector<T>::begin(),o.begin(),o.end());
			return *this;
		}

		vector_serializable<T>(const std::vector<T> &o):std::vector<T>(o)	{}
		vector_serializable<T>():std::vector<T>()	{}

		void read( utils::CStream &in )
		{
			uint32_t n;
			in >> n;
			std::vector<T>::resize(n);
			for (typename std::vector<T>::iterator i=std::vector<T>::begin();i!=std::vector<T>::end();i++)
				in >> *i;
		}
		void write( utils::CStream &out ) const
		{
			out << static_cast<uint32_t>(std::vector<T>::size());
			for (typename std::vector<T>::const_iterator i=std::vector<T>::begin();i!=std::vector<T>::end();i++)
				out << *i;
		}
	};

	/** A STL-vector derived class that implements basic read/write to ease implementing the mrpt::utils::CSerializable interface in classes having this type of data members.
	  */
	template <class T>
	class deque_serializable : public std::deque<T>
	{
	public:
		void read( utils::CStream &in )
		{
			uint32_t i,n;
			in >> n;
			std::deque<T>::clear();
			for (i=0;i<n;i++)
			{
				T 	obj;	// New empty object
				std::deque<T>::push_back(obj);
				T & theObj = std::deque<T>::back(); // Get reference to the real object within the list
				in >> theObj;
			}
		}
		void write( utils::CStream &out ) const
		{
			out << static_cast<uint32_t>(std::deque<T>::size());
			for (typename std::deque<T>::const_iterator i=std::deque<T>::begin();i!=std::deque<T>::end();i++)
				out << *i;
		}
	};

	/** A list_searchable with basic read/write to ease implementing the mrpt::utils::CSerializable interface in classes having this type of data members.
	  */
	template <class T>
	class list_searchable_serializable : public list_searchable<T>
	{
	public:
		void read( utils::CStream &in )
		{
			uint32_t i,n;
			in >> n;
			list_searchable<T>::clear();
			for (i=0;i<n;i++)
			{
				//T 	obj;	// New empty object
				list_searchable<T>::push_back( T() );
				T & theObj = list_searchable<T>::back(); // Get reference to the real object within the list
				in >> theObj;
			}
		}
		void write( utils::CStream &out ) const
		{
			out << static_cast<uint32_t>(list_searchable<T>::size());
			for (typename list_searchable<T>::const_iterator i=list_searchable<T>::begin();i!=list_searchable<T>::end();i++)
				out << *i;
		}
	};

	/** A std::map with basic read/write to ease implementing the mrpt::utils::CSerializable interface in classes having this type of data members.
	  */
	template <class K,class V>
	class map_serializable: public std::map<K,V>
	{
	public:
		void read( utils::CStream &in )
		{
			uint32_t i,n;
			in >> n;
			std::map<K,V>::clear();
			for (i=0;i<n;i++)
			{
				K 	key_obj;
				in >> key_obj;
				// Create an pair (Key, empty), then read directly into the ".second":
				V	&val_obj = std::map<K,V>::operator[] (key_obj);
				in >> val_obj;
			}
		}
		void write( utils::CStream &out ) const
		{
			out << static_cast<uint32_t>(std::map<K,V>::size());
			for (typename std::map<K,V>::const_iterator i=std::map<K,V>::begin();i!=std::map<K,V>::end();i++)
				out << i->first << i->second;
		}
	};

	/** A std::multimap with basic read/write to ease implementing the mrpt::utils::CSerializable interface in classes having this type of data members.
	  */
	template <class K,class V>
	class multimap_serializable: public std::multimap<K,V>
	{
	public:
		void read( utils::CStream &in )
		{
			uint32_t i,n;
			in >> n;
			std::multimap<K,V>::clear();
			typename std::multimap<K,V>::iterator it;
			for (i=0;i<n;i++)
			{
				K 	key_obj;
				in >> key_obj;
				// Create an pair (Key, empty), then read directly into the ".second":
				it = std::multimap<K,V>::insert( std::pair<K,V>(key_obj,V()) );
				in >> it->second;
			}
		}
		void write( utils::CStream &out ) const
		{
			out << static_cast<uint32_t>(std::multimap<K,V>::size());
			for (typename std::multimap<K,V>::const_iterator i=std::multimap<K,V>::begin();i!=std::multimap<K,V>::end();i++)
				out << i->first << i->second;
		}
	};



	/** Returns the index of the value "T" in the container "vect", or string::npos if not found.
	  */
	template <class T>
	size_t find_in_vector(const T &value, const std::vector<T> &vect)
	{
		for (size_t i=0;i<vect.size();i++)
			if (vect[i]==value) return i;
		return std::string::npos;
	}


	} // End of namespace
} // End of namespace
#endif
