/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.
#include <mrpt/opengl/CGeneralizedCylinder.h>
#include <mrpt/math/CMatrixTemplate.h>
#include <mrpt/poses/CPose3D.h>

#include "opengl_internals.h"

using namespace mrpt;
using namespace mrpt::math;
using namespace mrpt::opengl;
using namespace mrpt::utils;
using namespace std;

IMPLEMENTS_SERIALIZABLE(CGeneralizedCylinder,CRenderizable,mrpt::opengl)

void CGeneralizedCylinder::TQuadrilateral::calculateNormal()	{
	float ax=points[1].x-points[0].x;
	float ay=points[1].y-points[0].y;
	float az=points[1].z-points[0].z;
	float bx=points[2].x-points[0].x;
	float by=points[2].y-points[0].y;
	float bz=points[2].z-points[0].z;
	normal[0]=ay*bz-az*by;
	normal[1]=az*bx-ax*bz;
	normal[2]=ax*by-ay*bx;
}

class FQuadrilateralRenderer	{
private:
	const mrpt::utils::TColorf &color;
public:
	void operator()(CGeneralizedCylinder::TQuadrilateral t) const	{
		glNormal3f(t.normal[0],t.normal[1],t.normal[2]);
		glColor4f(color.R,color.G,color.B,color.A);
		for (int i=0;i<4;i++) glVertex3f(t.points[i].x,t.points[i].y,t.points[i].z);
	}
	FQuadrilateralRenderer(const mrpt::utils::TColorf &c):color(c)	{}
	~FQuadrilateralRenderer()	{}
};

void CGeneralizedCylinder::render() const	{
	if (!meshUpToDate) updateMesh();
#if MRPT_HAS_OPENGL_GLUT
	checkOpenGLError();
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
	glShadeModel(GL_SMOOTH);
	glBegin(GL_QUADS);
	for_each(mesh.begin(),mesh.end(),FQuadrilateralRenderer(mrpt::utils::TColorf(m_color_R,m_color_G,m_color_B,m_color_A)));
	glEnd();
	glDisable(GL_BLEND);
#endif
}

bool CGeneralizedCylinder::traceRay(const CPose3D &o,float &dist) const	{
	//TODO
	return false;
}

void CGeneralizedCylinder::updateMesh() const	{
	size_t A=axis.size();
	vector_serializable<CPoint3D> genX=generatrix;
	if (closed&&genX.size()>2) genX.push_back(genX[0]);
	size_t G=genX.size();
	mesh.clear();
	if (A>1&&G>1)	{
		CMatrixTemplate<CPoint3D> pointsMesh(A,G);
		vector_float yaws;
		yaws.reserve(A);
		vector_serializable<CPoint3D>::const_iterator it1,it2;
		it1=axis.begin();
		for(;;)	{
			if ((it2=it1+1)==axis.end()) break;
			yaws.push_back(atan2(it2->y-it1->y,it2->x-it1->x));
			it1=it2;
		}
		yaws.push_back(*yaws.rbegin());
		for (size_t i=0;i<A;i++)	{
			CPose3D base=CPose3D(axis[i].x,axis[i].y,axis[i].z,yaws[i],0,0);
			for (size_t j=0;j<G;j++) pointsMesh(i,j)=base+genX[j];
		}
		mesh.reserve((A-1)*(G-1));
		for (size_t i=0;i<A-1;i++) for (size_t j=0;j<G-1;j++) mesh.push_back(CGeneralizedCylinder::TQuadrilateral(pointsMesh(i,j),pointsMesh(i,j+1),pointsMesh(i+1,j+1),pointsMesh(i+1,j)));
	}
	meshUpToDate=true;
}

void CGeneralizedCylinder::writeToStream(CStream &out,int *version) const	{
	if (version) *version=0;
	else	{
		writeToStreamRender(out);
		out<<axis<<generatrix;
	}
}

void CGeneralizedCylinder::readFromStream(CStream &in,int version)	{
	switch (version)	{
		case 0:
			readFromStreamRender(in);
			//version 0
			in>>axis>>generatrix;
			mesh.clear();
			meshUpToDate=false;
		default:
			MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)
	};
}

class FAddPose	{
public:
	CPose3D pose;
	CPoint3D operator()(const CPoint3D &p) const	{
		return pose+p;
	}
	FAddPose(const CPose3D &p):pose(p)	{}
	~FAddPose()	{}
};

void generatePolygon(CPolyhedronPtr &poly,const vector<CPoint3D> &profile,const CPoint3D &base,float yaw)	{
	FAddPose functor=FAddPose(CPose3D(base.x,base.y,base.z,yaw,0,0));
	vector<CPoint3D> p(profile.size());
	transform(profile.begin(),profile.end(),p.begin(),functor);
	vector<uint32_t> f(profile.size());
	for (uint32_t i=0;i<profile.size();i++) f[i]=i;
	vector<vector<uint32_t> > ff(1);
	ff[0]=f;
	poly=CPolyhedron::Create(p,ff);
}

void CGeneralizedCylinder::getOrigin(CPolyhedronPtr &poly) const	{
	if (axis.size()<2||generatrix.size()<3) throw std::logic_error("Not enough points.");
	CPoint3D p0=axis[0];
	CPoint3D p1=axis[1];
	generatePolygon(poly,generatrix,p0,atan2(p1.y-p0.y,p1.x-p0.x));
	poly->setPose(CPose3D(m_x,m_y,m_z,m_yaw,m_pitch,m_roll));
	poly->setColor(getColor());
}

void CGeneralizedCylinder::getEnd(CPolyhedronPtr &poly) const	{
	if (axis.size()<2||generatrix.size()<3) throw std::logic_error("Not enough points.");
	CPoint3D p0=axis[axis.size()-1];
	CPoint3D p1=axis[axis.size()-2];
	generatePolygon(poly,generatrix,p0,atan2(p0.y-p1.y,p0.x-p1.x));
	poly->setPose(CPose3D(m_x,m_y,m_z,m_yaw,m_pitch,m_roll));
	poly->setColor(getColor());
}
