/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/opengl/CSetOfLines.h>
#include "opengl_internals.h"


using namespace mrpt;
using namespace mrpt::opengl;
using namespace mrpt::utils;
using namespace mrpt::math;
using namespace std;

IMPLEMENTS_SERIALIZABLE( CSetOfLines, CRenderizable, mrpt::opengl )


/*---------------------------------------------------------------
							clear
  ---------------------------------------------------------------*/
void CSetOfLines::clear()
{
	m_x0.clear();	m_y0.clear();	m_z0.clear();
	m_x1.clear();	m_y1.clear();	m_z1.clear();
}

/*---------------------------------------------------------------
							appendLine
  ---------------------------------------------------------------*/
void CSetOfLines::appendLine(
	const float &x0,const float &y0, const float &z0,
	const float &x1,const float &y1, const float &z1 )
{
	m_x0.push_back(x0);	m_x1.push_back(x1);
	m_y0.push_back(y0);	m_y1.push_back(y1);
	m_z0.push_back(z0);	m_z1.push_back(z1);
}

/*---------------------------------------------------------------
							resize
  ---------------------------------------------------------------*/
void CSetOfLines::resize(const size_t &nLines)
{
	m_x0.resize(nLines);	m_y0.resize(nLines);	m_z0.resize(nLines);
	m_x1.resize(nLines);	m_y1.resize(nLines);	m_z1.resize(nLines);
}

/*---------------------------------------------------------------
							setLineByIndex
  ---------------------------------------------------------------*/
void CSetOfLines::setLineByIndex(
	const size_t &index,
	const float &x0,const float &y0, const float &z0,
	const float &x1,const float &y1, const float &z1 )
{
	MRPT_TRY_START
	if (index>=m_x0.size()) THROW_EXCEPTION("Index out of bounds");

	m_x0[index]=x0;	m_x1[index]=x1;
	m_y0[index]=y0;	m_y1[index]=y1;
	m_z0[index]=z0;	m_z1[index]=z1;

	MRPT_TRY_END
}

/*---------------------------------------------------------------
							render
  ---------------------------------------------------------------*/
void   CSetOfLines::render() const
{
#if MRPT_HAS_OPENGL_GLUT
	size_t	i,n = m_x0.size();

    glLineWidth(m_lineWidth);	checkOpenGLError();

    glBegin( GL_LINES );

	glColor4f( m_color_R,m_color_G,m_color_B,1.0f );

	for (i=0;i<n;i++)
	{
		glVertex3f( m_x0[i],m_y0[i],m_z0[i] );
		glVertex3f( m_x1[i],m_y1[i],m_z1[i] );
	}
	glEnd();
	checkOpenGLError();
#endif
}

/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  CSetOfLines::writeToStream(CStream &out,int *version) const
{

	if (version)
		*version = 1;
	else
	{
		writeToStreamRender(out);
		out << m_x0 << m_y0 << m_z0;
		out << m_x1 << m_y1 << m_z1;
		out << m_lineWidth;
	}
}

/*---------------------------------------------------------------
	Implements the reading from a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void  CSetOfLines::readFromStream(CStream &in,int version)
{
	switch(version)
	{
	case 0:
	case 1:
		{
			readFromStreamRender(in);
			in >> m_x0 >> m_y0 >> m_z0
			   >> m_x1 >> m_y1 >> m_z1;
			if (version>=1)
				in >> m_lineWidth;
			else m_lineWidth=1;
		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}

void CSetOfLines::reserve(size_t r)	{
	r+=m_x0.size();
	m_x0.reserve(r);
	m_y0.reserve(r);
	m_z0.reserve(r);
	m_x1.reserve(r);
	m_y1.reserve(r);
	m_z1.reserve(r);
}
