/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 


#include <mrpt/poses/CPose.h>
#include <mrpt/poses/CPose3D.h>
#include <mrpt/poses/CPoint3D.h>

using namespace mrpt;
using namespace mrpt::poses;
using namespace mrpt::math;
using namespace std;

IMPLEMENTS_VIRTUAL_SERIALIZABLE(CPose, CPoseOrPoint,mrpt::poses)

/*---------------------------------------------------------------
				pose3D = pose3D + pose3D
  ---------------------------------------------------------------*/
CPose3D  CPose::operator + (const CPose3D& b) const
{
	CPose3D		p;
	//p.HM = getHomogeneousMatrix() * b.HM;
	p.HM.multiply( getHomogeneousMatrix(), b.HM );

	// Update the x,y,z,yaw,pitch,roll fields:
	p.x = p.HM(0,3);
	p.y = p.HM(1,3);
	p.z = p.HM(2,3);
	p.getYawPitchRoll( p.yaw, p.pitch, p.roll );

	return p;
}

/*---------------------------------------------------------------
				point3D = pose3D + point3D
  ---------------------------------------------------------------*/
CPoint3D  CPose::operator + (const CPoint3D& b) const
{
/*	CMatrix			P(4,4);
	P.unit();
	P(0,3) = b.x;
	P(1,3) = b.y;
	P(2,3) = b.z;
	CMatrix		res = getHomogeneousMatrix() * P;
	return CPoint3D( res(0,3),res(1,3),res(2,3) );
*/
	CMatrixFloat HM( getHomogeneousMatrix() );
	return CPoint3D(
		HM.get_unsafe(0,0)*b.x + HM.get_unsafe(0,1)*b.y +HM.get_unsafe(0,2)*b.z + HM.get_unsafe(0,3),
		HM.get_unsafe(1,0)*b.x + HM.get_unsafe(1,1)*b.y +HM.get_unsafe(1,2)*b.z + HM.get_unsafe(1,3),
		HM.get_unsafe(2,0)*b.x + HM.get_unsafe(2,1)*b.y +HM.get_unsafe(2,2)*b.z + HM.get_unsafe(2,3) );
}

/*---------------------------------------------------------------
				point3D = pose3D + point3D
  ---------------------------------------------------------------*/
CPose3D  CPose::operator - (const CPose3D& b) const
{
	CPose3D		p;

	CMatrixDouble B_INV(4,4);
	b.getInverseHomogeneousMatrix( B_INV );
	p.HM = B_INV * getHomogeneousMatrix(); // (b.getInverseHomogeneousMatrix()) * getHomogeneousMatrix();

	// Update the x,y,z,yaw,pitch,roll fields:
	p.x = p.HM(0,3);
	p.y = p.HM(1,3);
	p.z = p.HM(2,3);
	p.getYawPitchRoll( p.yaw, p.pitch, p.roll );
	return p;

}

/*---------------------------------------------------------------
				point3D = pose3D + point3D
  ---------------------------------------------------------------*/
CPoint3D  CPose::operator - (const CPoint3D& b) const
{
/*	CMatrixDouble	P(4,4);
	P.unit();
	P(0,3) = x;
	P(1,3) = y;
	P(2,3) = z;
	CMatrixD		res =  (b.getInverseHomogeneousMatrix()) * P;
	return CPoint3D( res(0,3),res(1,3),res(2,3) );
	*/
	CMatrixDouble B_INV(4,4);
	b.getInverseHomogeneousMatrix( B_INV );
	return CPoint3D(
		B_INV.get_unsafe(0,0)*x + B_INV.get_unsafe(0,1)*y +B_INV.get_unsafe(0,2)*z + B_INV.get_unsafe(0,3),
		B_INV.get_unsafe(1,0)*x + B_INV.get_unsafe(1,1)*y +B_INV.get_unsafe(1,2)*z + B_INV.get_unsafe(1,3),
		B_INV.get_unsafe(2,0)*x + B_INV.get_unsafe(2,1)*y +B_INV.get_unsafe(2,2)*z + B_INV.get_unsafe(2,3)
		);
}

