/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.



#include <mrpt/random.h>
#include <mrpt/math/utils.h>

#include <mrpt/poses/CPose3DPDFGaussian.h>
#include <mrpt/poses/CPosePDF.h>
#include <mrpt/poses/CPosePDFGaussian.h>

using namespace mrpt::poses;
using namespace mrpt::utils;
using namespace std;

IMPLEMENTS_SERIALIZABLE( CPose3DPDFGaussian, CPose3DPDF, mrpt::poses )


/*---------------------------------------------------------------
	Constructor
  ---------------------------------------------------------------*/
CPose3DPDFGaussian::CPose3DPDFGaussian() : mean(0,0,0), cov(6,6)
{

}

/*---------------------------------------------------------------
	Constructor
  ---------------------------------------------------------------*/
CPose3DPDFGaussian::CPose3DPDFGaussian(
	const CPose3D	&init_Mean,
	const CMatrixD	&init_Cov ) : mean(init_Mean), cov(init_Cov)
{
	ASSERT_( init_Cov.getColCount()==6 && init_Cov.getRowCount()==6 );
}

/*---------------------------------------------------------------
	Copy Constructor from 2D PDF
  ---------------------------------------------------------------*/
CPose3DPDFGaussian::CPose3DPDFGaussian( const CPosePDFGaussian &o )
	 : mean( o.mean.x,o.mean.y,0,o.mean.phi,0,0 ),
	   cov(6,6)
{
	// Copy covariance:
	cov(0,0) = o.cov(0,0);
	cov(1,1) = o.cov(1,1);
	cov(3,3) = o.cov(2,2); // phi -> yaw

	cov(0,1) = cov(1,0) = o.cov(0,1);
	cov(0,3) = cov(3,0) = o.cov(0,2);
	cov(1,3) = cov(3,1) = o.cov(1,2);
}


/*---------------------------------------------------------------
	Constructor
  ---------------------------------------------------------------*/
CPose3DPDFGaussian::CPose3DPDFGaussian(
	const CPose3D	&init_Mean ) : mean(init_Mean), cov(6,6)
{
	cov.zeros();
}


/*---------------------------------------------------------------
						getEstimatedPose
  Returns an estimate of the pose, (the mean, or mathematical expectation of the PDF)
 ---------------------------------------------------------------*/
CPose3D  CPose3DPDFGaussian::getEstimatedPose() const
{
	return mean;
}

/*---------------------------------------------------------------
						getEstimatedCovariance
 ---------------------------------------------------------------*/
CMatrixD  CPose3DPDFGaussian::getEstimatedCovariance() const
{
	return cov;
}

/*---------------------------------------------------------------
						writeToStream
  ---------------------------------------------------------------*/
void  CPose3DPDFGaussian::writeToStream(CStream &out,int *version) const
{
	if (version)
		*version = 1;
	else
	{
		out << mean;
		out << cov(0,0) << cov(1,1) << cov(2,2) << cov(3,3) << cov(4,4) << cov(5,5);
		out << cov(0,1) << cov(0,2) << cov(0,3) << cov(0,4) << cov(0,5);
		out << cov(1,2) << cov(1,3) << cov(1,4) << cov(1,5);
		out << cov(2,3) << cov(2,4) << cov(2,5);
		out << cov(3,4) << cov(3,5);
		out << cov(4,5);
	}
}

/*---------------------------------------------------------------
						readFromStream
  ---------------------------------------------------------------*/
void  CPose3DPDFGaussian::readFromStream(CStream &in,int version)
{
	switch(version)
	{
	case 0:
		{
			float	x;
			in >> mean;

			cov.setSize(6,6);
			in >> x; cov(0,0) = x;
			in >> x; cov(1,1) = x;
			in >> x; cov(2,2) = x;
			in >> x; cov(3,3) = x;
			in >> x; cov(4,4) = x;
			in >> x; cov(5,5) = x;

			in >> x; cov(1,0) = x; cov(0,1) = x;
			in >> x; cov(2,0) = x; cov(0,2) = x;
			in >> x; cov(3,0) = x; cov(0,3) = x;
			in >> x; cov(4,0) = x; cov(0,4) = x;
			in >> x; cov(5,0) = x; cov(0,5) = x;

			in >> x; cov(2,1) = x; cov(1,2) = x;
			in >> x; cov(3,1) = x; cov(1,3) = x;
			in >> x; cov(4,1) = x; cov(1,4) = x;
			in >> x; cov(5,1) = x; cov(1,5) = x;

			in >> x; cov(3,2) = x; cov(2,3) = x;
			in >> x; cov(4,2) = x; cov(2,4) = x;
			in >> x; cov(5,2) = x; cov(2,5) = x;

			in >> x; cov(4,3) = x; cov(3,4) = x;
			in >> x; cov(5,3) = x; cov(3,5) = x;

			in >> x; cov(5,4) = x; cov(4,5) = x;

		} break;
	case 1:
		{
			double	x;
			in >> mean;

			cov.setSize(6,6);
			in >> x; cov(0,0) = x;
			in >> x; cov(1,1) = x;
			in >> x; cov(2,2) = x;
			in >> x; cov(3,3) = x;
			in >> x; cov(4,4) = x;
			in >> x; cov(5,5) = x;

			in >> x; cov(1,0) = x; cov(0,1) = x;
			in >> x; cov(2,0) = x; cov(0,2) = x;
			in >> x; cov(3,0) = x; cov(0,3) = x;
			in >> x; cov(4,0) = x; cov(0,4) = x;
			in >> x; cov(5,0) = x; cov(0,5) = x;

			in >> x; cov(2,1) = x; cov(1,2) = x;
			in >> x; cov(3,1) = x; cov(1,3) = x;
			in >> x; cov(4,1) = x; cov(1,4) = x;
			in >> x; cov(5,1) = x; cov(1,5) = x;

			in >> x; cov(3,2) = x; cov(2,3) = x;
			in >> x; cov(4,2) = x; cov(2,4) = x;
			in >> x; cov(5,2) = x; cov(2,5) = x;

			in >> x; cov(4,3) = x; cov(3,4) = x;
			in >> x; cov(5,3) = x; cov(3,5) = x;

			in >> x; cov(5,4) = x; cov(4,5) = x;

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}



/*---------------------------------------------------------------
						operator =
  ---------------------------------------------------------------*/
void  CPose3DPDFGaussian::copyFrom(const CPose3DPDF &o)
{
	if (this == &o) return;		// It may be used sometimes

	// Convert to gaussian pdf:
	mean = o.getEstimatedPose();
	cov  = o.getEstimatedCovariance();
}

/*---------------------------------------------------------------
						operator =
  ---------------------------------------------------------------*/
void  CPose3DPDFGaussian::copyFrom(const CPosePDF &o)
{
	// Convert to gaussian pdf:
	mean = CPose3D( o.getEstimatedPose() );

	CMatrixD C = o.getEstimatedCovariance();
	ASSERT_( C.IsSquare() && size(C,1)==3 );

	cov.zeros(6,6);
	cov.set_unsafe(0,0,   C.get_unsafe(0,0));
	cov.set_unsafe(1,1,   C.get_unsafe(1,1));
	cov.set_unsafe(3,3,   C.get_unsafe(2,2));

	cov.set_unsafe(0,1,   C.get_unsafe(0,1));
	cov.set_unsafe(1,0,   C.get_unsafe(0,1));

	cov.set_unsafe(0,3,   C.get_unsafe(0,2));
	cov.set_unsafe(3,0,   C.get_unsafe(0,2));

	cov.set_unsafe(1,3,   C.get_unsafe(1,2));
	cov.set_unsafe(3,1,   C.get_unsafe(1,2));
}


/*---------------------------------------------------------------

  ---------------------------------------------------------------*/
void  CPose3DPDFGaussian::saveToTextFile(const string &file) const
{
	FILE	*f=os::fopen(file.c_str(),"wt");
	if (!f) return;

	os::fprintf(f,"%e %e %e %e %e %e\n", mean.x, mean.y, mean.z, mean.yaw, mean.pitch, mean.roll );

	os::fprintf(f,"%e %e %e %e %e %e\n", cov(0,0),cov(0,1),cov(0,2),cov(0,3),cov(0,4),cov(0,5));
	os::fprintf(f,"%e %e %e %e %e %e\n", cov(1,0),cov(1,1),cov(1,2),cov(1,3),cov(1,4),cov(1,5));
	os::fprintf(f,"%e %e %e %e %e %e\n", cov(2,0),cov(2,1),cov(2,2),cov(2,3),cov(2,4),cov(2,5));
	os::fprintf(f,"%e %e %e %e %e %e\n", cov(3,0),cov(3,1),cov(3,2),cov(3,3),cov(3,4),cov(3,5));
	os::fprintf(f,"%e %e %e %e %e %e\n", cov(4,0),cov(4,1),cov(4,2),cov(4,3),cov(4,4),cov(4,5));
	os::fprintf(f,"%e %e %e %e %e %e\n", cov(5,0),cov(5,1),cov(5,2),cov(5,3),cov(5,4),cov(5,5));

	os::fclose(f);
}

/*---------------------------------------------------------------
						changeCoordinatesReference
 ---------------------------------------------------------------*/
void  CPose3DPDFGaussian::changeCoordinatesReference( const CPose3D &newReferenceBase )
{
	MRPT_TRY_START

	/** \todo Make an example and test this & 2DPDFGaussian! (19-jan-2008: equations are OK -> make examples!!)
	  */

	CMatrixDouble	M( newReferenceBase.getHomogeneousMatrix() );
	M.setSize(3,3);	// Clip the 4x4 matrix
	M.setSize(6,6);

	// The variance in yaw,pitch & roll is unmodified:
	M(3,3) = M(4,4) = M(5,5) = 1;

	// The mean:
	mean = newReferenceBase + mean;

	// The covariance:
	//cov = M * cov * (~M);
	CMatrixDouble OLD_COV(cov);
	M.multiplyByMatrixAndByTranspose( OLD_COV, cov );

	MRPT_TRY_END
}

/*---------------------------------------------------------------
						changeCoordinatesReference
 ---------------------------------------------------------------*/
void  CPose3DPDFGaussian::rotateCov(const double &ang)
{
	MRPT_UNUSED_PARAM(ang);
	THROW_EXCEPTION("TO DO!!!");

	/** To do!
	  */

/*	CMatrixD		rot(3,3);
	rot.zeros();
	rot(0,0)=rot(1,1)=cos(ang);
	rot(0,1)=-sin(ang);
	rot(1,0)=sin(ang);
	rot(2,2)=1;

	cov = rot * cov * (~rot);
*/
	assureSymmetry();
}

/*---------------------------------------------------------------
					drawSingleSample
 ---------------------------------------------------------------*/
void  CPose3DPDFGaussian::drawSingleSample( CPose3D &outPart ) const
{
	MRPT_UNUSED_PARAM(outPart);
	MRPT_TRY_START;

	vector_double	v;
	mrpt::random::randomNormalMultiDimensional( cov, v );

	outPart.setFromValues(
		mean.x + v[0],
		mean.y + v[1],
		mean.z + v[2],
		mean.yaw + v[3],
		mean.pitch + v[4],
		mean.roll + v[5] );

	MRPT_TRY_END_WITH_CLEAN_UP( \
        cov.saveToTextFile("__DEBUG_EXC_DUMP_drawSingleSample_COV.txt"); \
		);
}

/*---------------------------------------------------------------
					drawManySamples
 ---------------------------------------------------------------*/
void  CPose3DPDFGaussian::drawManySamples(
	size_t						N,
	vector<vector_double>	&outSamples ) const
{
	MRPT_TRY_START;

	mrpt::random::randomNormalMultiDimensionalMany(cov, N, outSamples );

	for (vector<vector_double>::iterator it=outSamples.begin();it!=outSamples.end();it++)
	{
		it->at(0) += mean.x;
		it->at(1) += mean.y;
		it->at(2) += mean.z;
		it->at(3) = math::wrapToPi( it->at(3) + mean.yaw );
		it->at(4) = math::wrapToPi( it->at(4) + mean.pitch );
		it->at(5) = math::wrapToPi( it->at(5) + mean.roll );
	}

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
					bayesianFusion
 ---------------------------------------------------------------*/
void  CPose3DPDFGaussian::bayesianFusion( CPose3DPDF &p1_, CPose3DPDF &p2_ )
{
	MRPT_UNUSED_PARAM(p1_); MRPT_UNUSED_PARAM(p2_);
	MRPT_TRY_START;

	THROW_EXCEPTION("TO DO!!!");

/*	ASSERT_( p1_.GetRuntimeClass() == CLASS_ID( CPose3DPDFGaussian )  );
	ASSERT_( p2_.GetRuntimeClass() == CLASS_ID( CPose3DPDFGaussian )  );

	CPose3DPDFGaussian	*p1 = (CPose3DPDFGaussian*) &p1_;
	CPose3DPDFGaussian	*p2 = (CPose3DPDFGaussian*) &p2_;


	CMatrixD	x1(3,1),x2(3,1),x(3,1);
	CMatrixD	C1( p1->cov );
	CMatrixD	C2( p2->cov );
	CMatrixD	C1_inv = C1.inv();
	CMatrixD	C2_inv = C2.inv();
	CMatrixD	C;

	x1(0,0) = p1->mean.x; x1(1,0) = p1->mean.y; x1(2,0) = p1->mean.phi;
	x2(0,0) = p2->mean.x; x2(1,0) = p2->mean.y; x2(2,0) = p2->mean.phi;

	C = !(C1_inv + C2_inv);

	this->cov = C;
	this->assureSymmetry();

	x = C * ( C1_inv*x1 + C2_inv*x2 );

	this->mean.x = x(0,0);
	this->mean.y = x(1,0);
	this->mean.phi = x(2,0);
	this->mean.normalizePhi();
*/
	MRPT_TRY_END;

}

/*---------------------------------------------------------------
					inverse
 ---------------------------------------------------------------*/
void	 CPose3DPDFGaussian::inverse(CPose3DPDF &o) const
{
	ASSERT_(o.GetRuntimeClass() == CLASS_ID(CPose3DPDFGaussian));
	CPose3DPDFGaussian	&out = static_cast<CPose3DPDFGaussian&>( o);

	// The mean:
	out.mean = CPose3D(0,0,0) - mean;

	// The covariance: Is the same:
	out.cov = cov;
}


/*---------------------------------------------------------------
							+=
 ---------------------------------------------------------------*/
void  CPose3DPDFGaussian::operator += ( const CPose3D &Ap)
{
	THROW_EXCEPTION("TO DO!!!");

	mean = mean + Ap;
//	rotateCov( Ap.phi );
}

/*---------------------------------------------------------------
					jacobiansPoseComposition
 ---------------------------------------------------------------*/
void CPose3DPDFGaussian::jacobiansPoseComposition(
	const CPose3DPDFGaussian &x,
	const CPose3DPDFGaussian &u,
	CMatrixDouble			 &df_dx,
	CMatrixDouble			 &df_du)
{
/*
	df_dx =
	[ 1, 0, 0, -sin(yaw)*cos(p)*xu+(-sin(yaw)*sin(p)*sin(r)-cos(yaw)*cos(r))*yu+(-sin(yaw)*sin(p)*cos(r)+cos(yaw)*sin(r))*zu, -cos(yaw)*sin(p)*xu+cos(yaw)*cos(p)*sin(r)*yu+cos(yaw)*cos(p)*cos(r)*zu, (cos(yaw)*sin(p)*cos(r)+sin(yaw)*sin(r))*yu+(-cos(yaw)*sin(p)*sin(r)+sin(yaw)*cos(r))*zu]
	[ 0, 1, 0,    cos(yaw)*cos(p)*xu+(cos(yaw)*sin(p)*sin(r)-sin(yaw)*cos(r))*yu+(cos(yaw)*sin(p)*cos(r)+sin(yaw)*sin(r))*zu, -sin(yaw)*sin(p)*xu+sin(yaw)*cos(p)*sin(r)*yu+sin(yaw)*cos(p)*cos(r)*zu, (sin(yaw)*sin(p)*cos(r)-cos(yaw)*sin(r))*yu+(-sin(yaw)*sin(p)*sin(r)-cos(yaw)*cos(r))*zu]
	[ 0, 0, 1,                                                                                                             0, -cos(p)*xu-sin(p)*sin(r)*yu-sin(p)*cos(r)*zu,                            cos(p)*cos(r)*yu-cos(p)*sin(r)*zu]
	[ 0, 0, 0, 1, 0, 0]
	[ 0, 0, 0, 0, 1, 0]
	[ 0, 0, 0, 0, 0, 1]
*/
	df_dx.unit(6);

	double   xu = u.mean.x;
	double   yu = u.mean.y;
	double   zu = u.mean.z;

	double   sy = sin(x.mean.yaw);	double   cy = cos(x.mean.yaw);
	double   sp = sin(x.mean.pitch);	double   cp = cos(x.mean.pitch);
	double   sr = sin(x.mean.roll);	double   cr = cos(x.mean.roll);

	df_dx(0,3) = -sy*cp*xu+(-sy*sp*sr-cy*cr)*yu+(-sy*sp*cr+cy*sr)*zu;
	df_dx(0,4) = -cy*sp*xu+cy*cp*sr*yu+cy*cp*cr*zu;
	df_dx(0,5) = (cy*sp*cr+sy*sr)*yu+(-cy*sp*sr+sy*cr)*zu;

	df_dx(1,3) = cy*cp*xu+(cy*sp*sr-sy*cr)*yu+(cy*sp*cr+sy*sr)*zu;
	df_dx(1,4) = -sy*sp*xu+sy*cp*sr*yu+sy*cp*cr*zu;
	df_dx(1,5) =  (sy*sp*cr-cy*sr)*yu+(-sy*sp*sr-cy*cr)*zu;

	df_dx(2,4) = -cp*xu-sp*sr*yu-sp*cr*zu;
	df_dx(2,5) = cp*cr*yu-cp*sr*zu;

/*
	df_du =
	[ cos(yaw)*cos(p), cos(yaw)*sin(p)*sin(r)-sin(yaw)*cos(r), cos(yaw)*sin(p)*cos(r)+sin(yaw)*sin(r), 0, 0, 0]
	[ sin(yaw)*cos(p), sin(yaw)*sin(p)*sin(r)+cos(yaw)*cos(r), sin(yaw)*sin(p)*cos(r)-cos(yaw)*sin(r), 0, 0, 0]
	[ -sin(p),         cos(p)*sin(r),                          cos(p)*cos(r),                          0, 0, 0]
	[ 0, 0, 0, 1, 0, 0]
	[ 0, 0, 0, 0, 1, 0]
	[ 0, 0, 0, 0, 0, 1]
*/
	// This is the homogeneous matrix of "x":
	df_du.unit(6);

	const CMatrixDouble *x_HM;
	x.mean.getHomogeneousMatrix(x_HM);

	df_du.insertMatrix(0,0, *x_HM );
	// But only the rows/cols [0-2]:
	df_du(0,3) =
	df_du(1,3) =
	df_du(2,3) = 0;
}


/*---------------------------------------------------------------
							+=
 ---------------------------------------------------------------*/
void  CPose3DPDFGaussian::operator += ( const CPose3DPDFGaussian &Ap)
{
	// MEAN:
	this->mean = this->mean + Ap.mean;

	// COV:
	CMatrixDouble  OLD_COV(this->cov);

	CMatrixDouble  df_dx, df_du;

	CPose3DPDFGaussian::jacobiansPoseComposition(
		*this,  // x
		Ap,     // u
		df_dx,
		df_du );

	// this->cov = H1*this->cov*~H1 + H2*Ap.cov*~H2;
	df_dx.multiplyByMatrixAndByTranspose( OLD_COV, cov );
	df_du.multiplyByMatrixAndByTranspose( Ap.cov,  cov, false, 0, true); // Accumulate result
}

/*---------------------------------------------------------------
						evaluatePDF
 ---------------------------------------------------------------*/
double  CPose3DPDFGaussian::evaluatePDF( const CPose3D &x ) const
{
	MRPT_UNUSED_PARAM(x);
	THROW_EXCEPTION("TO DO!!!");

/*	CMatrixD	X(6,1);
	X(0,0) = x.x;
	X(1,0) = x.y;
	X(2,0) = x.z;

	CMatrixD	MU(6,1);
	MU(0,0) = mean.x;
	MU(1,0) = mean.y;
	MU(2,0) = mean.z;

	return math::normalPDF( X, MU, this->cov );
*/
}

/*---------------------------------------------------------------
						evaluateNormalizedPDF
 ---------------------------------------------------------------*/
double  CPose3DPDFGaussian::evaluateNormalizedPDF( const CPose3D &x ) const
{
	MRPT_UNUSED_PARAM(x);
	THROW_EXCEPTION("TO DO!!!");
/*	CMatrixD	X(3,1);
	X(0,0) = x.x;
	X(1,0) = x.y;
	X(2,0) = x.phi;

	CMatrixD	MU(3,1);
	MU(0,0) = mean.x;
	MU(1,0) = mean.y;
	MU(2,0) = mean.phi;

	return math::normalPDF( X, MU, this->cov ) / math::normalPDF( MU, MU, this->cov );
*/
}

/*---------------------------------------------------------------
						assureSymmetry
 ---------------------------------------------------------------*/
void  CPose3DPDFGaussian::assureSymmetry()
{
	// Differences, when they exist, appear in the ~15'th significant
	//  digit, so... just take one of them arbitrarily!
	cov(0,1) = cov(1,0);
	cov(0,2) = cov(2,0);
	cov(0,3) = cov(3,0);
	cov(0,4) = cov(4,0);
	cov(0,5) = cov(5,0);

	cov(1,2) = cov(2,1);
	cov(1,3) = cov(3,1);
	cov(1,4) = cov(4,1);
	cov(1,5) = cov(5,1);

	cov(2,3) = cov(3,2);
	cov(2,4) = cov(4,2);
	cov(2,5) = cov(5,2);

	cov(3,4) = cov(4,3);
	cov(3,5) = cov(5,3);

	cov(4,5) = cov(5,4);
}

/*---------------------------------------------------------------
						mahalanobisDistanceTo
 ---------------------------------------------------------------*/
double  CPose3DPDFGaussian::mahalanobisDistanceTo( const CPose3DPDFGaussian& theOther )
{
	MRPT_TRY_START;

	CMatrixD		COV_( cov + theOther.cov );
	CMatrixD		M1( mean );
	CMatrixD		M2( theOther.mean );
	CMatrixD		MU(M2-M1);

	return sqrt( MU.multiplyByMatrixAndByTransposeScalar(!COV_) );

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						operator <<
 ---------------------------------------------------------------*/
ostream &   mrpt::poses::operator << (
	ostream		&out,
	const CPose3DPDFGaussian	&obj )
{
	out << "Mean: " << obj.mean << "\n";
	out << "Covariance:\n" << obj.cov << "\n";

	return out;
}

/*---------------------------------------------------------------
						operator +
 ---------------------------------------------------------------*/
CPose3DPDFGaussian mrpt::poses::operator +( const CPose3DPDFGaussian &x, const CPose3DPDFGaussian &u )
{
	CPose3DPDFGaussian 	res(x);
	res+=u;
	return res;
}

/*---------------------------------------------------------------
						getCovSubmatrix2D
 ---------------------------------------------------------------*/
void CPose3DPDFGaussian::getCovSubmatrix2D( CMatrixDouble &out_cov ) const
{
	out_cov.setSize(3,3);

	for (int i=0;i<3;i++)
	{
		int a = i==2 ? 3:i;
		for (int j=i;j<3;j++)
		{
			int b = j==2 ? 3:j;
			double f = cov.get_unsafe(a,b);
			out_cov.set_unsafe(i,j, f);
			out_cov.set_unsafe(j,i, f);
		}
	}
}
