/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.



#include <mrpt/config.h>

#include <mrpt/system/os.h>
#include <mrpt/system/os.h>
#include <mrpt/math/CMatrix.h>
#include <mrpt/math/CMatrixD.h>
#include <mrpt/random/RandomGenerators.h>

#ifdef MRPT_OS_WINDOWS
    #include <windows.h>
#else

#endif


using namespace mrpt;
using namespace mrpt::random;
using namespace mrpt::utils;
using namespace std;

#include <cmath>


// Data used internally by the MT19937 PRNG algorithm.
struct  TMT19937_data
{
	TMT19937_data() : index(0), seed_initialized(false)
	{}

	uint32_t	MT[624];
	uint32_t	index;
	bool		seed_initialized;
} MT19937_data;

void MT19937_generateNumbers();
void MT19937_initializeGenerator(const uint32_t &seed);


// MT19937 algorithm
// http://en.wikipedia.org/wiki/Mersenne_twister
// Initialize the generator from a seed
void MT19937_initializeGenerator(const uint32_t &seed)
{
	MT19937_data.seed_initialized = true;
	MT19937_data.MT[0] = seed;
	for (uint32_t i=1;i<624;i++)
		MT19937_data.MT[i] = static_cast<uint32_t>( 1812433253 * (MT19937_data.MT[i-1] ^ ( MT19937_data.MT[i-1] >> 30 )) + i); // 0x6c078965
}

// MT19937 algorithm
// http://en.wikipedia.org/wiki/Mersenne_twister
uint32_t mrpt::random::RandomUniInt()
{
	if (!MT19937_data.index)
		MT19937_generateNumbers();

	uint32_t	y = MT19937_data.MT[MT19937_data.index];
	y ^= y >> 11;
	y ^= (y << 7) & 2636928640U; // 0x9d2c5680
	y ^= (y << 15) & 4022730752U; // 0xefc60000
	y ^= (y >> 18);

	MT19937_data.index = (MT19937_data.index + 1) % 624;
	return y;
}

// Generate an array of 624 untempered numbers
void MT19937_generateNumbers()
{
	if (!MT19937_data.seed_initialized)	mrpt::random::Randomize();

	for (uint32_t i=0;i<624;i++)
	{
		uint32_t	y = (0x80000000 & MT19937_data.MT[i]) + (0x7FFFFFFF & MT19937_data.MT[(i+1) % 624]);
		MT19937_data.MT[i] = MT19937_data.MT[(i + 397) % 624] ^ (y >> 1);
		if (y % 2) // Odd
			MT19937_data.MT[i] ^= 2567483615U; // 0x9908b0df
	}
}

/*---------------------------------------------------------------
						Randomize
  ---------------------------------------------------------------*/
void	  mrpt::random::Randomize(const uint32_t &seed)
{
//	srand( (unsigned) seed );
	MT19937_initializeGenerator(seed);
	MT19937_data.index = 0;
}

/*---------------------------------------------------------------
						Randomize
  ---------------------------------------------------------------*/
void	 mrpt::random::Randomize()
{
	MT19937_initializeGenerator( static_cast<uint32_t>(system::getCurrentTime()) );
	MT19937_data.index = 0;
}

/*---------------------------------------------------------------
					normalizedGaussian
  This is a routine which has been extracted from page 217 in
      the Numerical Recipes in C, William H. Press
  ---------------------------------------------------------------*/
double  mrpt::random::normalizedGaussian( double *likelihood  )
{
   static int iset = 0;
   static double gset;
   double fac,r,v1,v2,x;

   if (iset == 0) {   /* We don't have an extra deviate handy so */
       do {
	   v1 = RandomUni(-1.0,1.0);   /* pick two uniform numbers in */
	   v2 = RandomUni(-1.0,1.0);   /* square extending from -1 to +1*/
	                                 /* in each direction, */

	   r = v1 * v1 + v2 * v2;        /* see if they are in the unitcircle*/
       } while (r >= 1.0 && r!=0);               /* and if they are not, try again. */
       fac = sqrt(-2.0*log(r)/r);
       /* Now make the Box-Muller transformation to get two normal deviates.
	  Return one and save the other for next time. */
       gset = v1 * fac;
       iset = 1;

	   if (likelihood)
	   {
		   x = v2*fac;
		   *likelihood = 0.39894228040143267794 * exp( -0.5*x*x );
		   return x;
	   }
	   else
	   {
	       return v2*fac;
	   }

   } else
   {      /* We have an extra deviate handy, */
       iset = 0; /* so unset the flag,              */

	   if (likelihood)
	   {
		   x = (float)gset;
		   *likelihood = 0.39894228040143267794 * exp( -0.5*x*x );
		   return x;
	   }
	   else
	   {
	       return (float)gset;
	   }
   }
}

/*---------------------------------------------------------------
						randomNormalMultiDimensionalMany
  ---------------------------------------------------------------*/
template <typename T>
void  mrpt::random::randomNormalMultiDimensionalMany(
	const CMatrixTemplateNumeric<T>	&cov,
	size_t							desiredSamples,
	std::vector< std::vector<T> >	&ret,
	std::vector<T>					*samplesLikelihoods)
{
	typename std::vector<std::vector<T> >::iterator		ret_it;
	typename std::vector<T>::iterator			liks_it;
	size_t					dim = cov.getColCount();
	std::vector<T>		res;
	CMatrixTemplateNumeric<T>	Z,D;

	MRPT_TRY_START;

	if (samplesLikelihoods)
	{
		samplesLikelihoods->clear();
		samplesLikelihoods->resize(desiredSamples, 0);
	}

	ret.resize(desiredSamples);
	ASSERT_(cov.getRowCount() == cov.getColCount() );

	// Set size of output vector:
	res.resize(dim,0);

	/** Computes the eigenvalues/eigenvector decomposition of this matrix,
	*    so that: M = Z · D · Z<sup>T</sup>, where columns in Z are the
	*	  eigenvectors and the diagonal matrix D contains the eigenvalues
	*    as diagonal elements, sorted in <i>ascending</i> order.
	*/
	cov.eigenVectors( Z, D );

	// Scale eigenvectors with eigenvalues:
	D.Sqrt();
	Z = Z * D;

	if (samplesLikelihoods)
		liks_it = samplesLikelihoods->begin();

	for (ret_it=ret.begin();ret_it!=ret.end();ret_it++)
	{
		size_t		i;
		double	likelihood_1, likelihood_all;

		// Reset vector to 0:
		ret_it->resize(dim,0);
		likelihood_all = 1.0;

		if (samplesLikelihoods)
		{
			// Save likelihoods also:
			// ------------------------------
			for (i=0;i<dim;i++)
			{
				T	rnd = normalizedGaussian(&likelihood_1);
				for (size_t d=0;d<dim;d++)	(*ret_it)[d]+=Z(d,i)*rnd;
				// Product of each independent gaussian sample:
				likelihood_all *= likelihood_1;
			}

			// Save the total likelihood of the generated vector:
			*liks_it = static_cast<T>(likelihood_all);
			liks_it++;
		}
		else
		{
			// DO NOT save likelihoods also, just the samples
			// ----------------------------------------------------
			for (i=0;i<dim;i++)
			{
				T rnd = normalizedGaussian();
				for (size_t d=0;d<dim;d++)	(*ret_it)[d]+=Z(d,i)*rnd;
			}
		}

	} // For each vector sample to generate...

	MRPT_TRY_END_WITH_CLEAN_UP( \
		printf("\nEXCEPTION: Dumping variables for debuging:\n"); \
		std::cout << "Z:\n" << Z << "D:\n" << D << "Cov:\n" << cov; \
		try \
		{
			cov.eigenVectors(Z,D); \
			std::cout << "Original Z:" << Z << "Original D:" << D;  \
		} \
		catch(...) {}; \
		);

}

// Instantiations:
template MRPTDLLIMPEXP void  mrpt::random::randomNormalMultiDimensionalMany<double>(
	const CMatrixTemplateNumeric<double> &,size_t,std::vector< std::vector<double> >	&,std::vector<double> *);
template MRPTDLLIMPEXP void  mrpt::random::randomNormalMultiDimensionalMany<float>(
	const CMatrixTemplateNumeric<float> &,size_t,std::vector< std::vector<float> >	&,std::vector<float> *);



/*---------------------------------------------------------------
						RandomNormal
  ---------------------------------------------------------------*/
double mrpt::random::RandomNormal( double mean, double std)
{
	return std * normalizedGaussian() + mean;
}

/*---------------------------------------------------------------
						randomNormalMultiDimensional
  ---------------------------------------------------------------*/
template <typename T>
void  mrpt::random::randomNormalMultiDimensional(
	const CMatrixTemplateNumeric<T>	&cov,
	std::vector<T>		&out_result)
{
	ASSERT_(cov.getRowCount() == cov.getColCount() );

	size_t						dim = cov.getColCount();
	CMatrixTemplateNumeric<T>	Z,D;

	MRPT_TRY_START;

	// Set size of output vector:
	out_result.clear();
	out_result.resize(dim,0);

	/** Computes the eigenvalues/eigenvector decomposition of this matrix,
	*    so that: M = Z · D · Z<sup>T</sup>, where columns in Z are the
	*	  eigenvectors and the diagonal matrix D contains the eigenvalues
	*    as diagonal elements, sorted in <i>ascending</i> order.
	*/
	cov.eigenVectors( Z, D );

	// Scale eigenvectors with eigenvalues:
	D.Sqrt();
	Z.multiply(Z,D);

	for (size_t i=0;i<dim;i++)
	{
		T rnd = normalizedGaussian();
		for (size_t d=0;d<dim;d++)
			out_result[d]+= ( Z(d,i)* rnd );
	}

	MRPT_TRY_END_WITH_CLEAN_UP( \
		printf("\nEXCEPTION: Dumping variables for debuging:\n"); \
		std::cout << "Z:\n" << Z << "D:\n" << D << "Cov:\n" << cov; \
		try \
		{ \
			cov.eigenVectors(Z,D); \
			std::cout << "Original Z:" << Z << "Original D:" << D; \
		} \
		catch(...)  {}; \
		);
}

// Instantiations:
template MRPTDLLIMPEXP void  mrpt::random::randomNormalMultiDimensional<double>( const CMatrixTemplateNumeric<double> &, std::vector<double> &);
template MRPTDLLIMPEXP void  mrpt::random::randomNormalMultiDimensional<float>( const CMatrixTemplateNumeric<float> &, std::vector<float> &);

