/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 

#include <mrpt/slam/CObservationStereoImages.h>
#include <mrpt/slam/CLandmarksMap.h>

using namespace mrpt::slam; using namespace mrpt::utils; using namespace mrpt::poses;

// This must be added to any CSerializable class implementation file.
IMPLEMENTS_SERIALIZABLE(CObservationStereoImages, CObservation,mrpt::slam)

/*---------------------------------------------------------------
					Constructor
 ---------------------------------------------------------------*/
CObservationStereoImages::CObservationStereoImages( void *iplImageLeft,void *iplImageRight ) :
	m_auxMap(),
	cameraPose(),
	intrinsicParams(3,3),
	imageLeft( iplImageLeft ),
	imageRight( iplImageRight ),
	rightCameraPose(),
	focalLength_meters(0.02)
{
}

/*---------------------------------------------------------------
					Default Constructor
 ---------------------------------------------------------------*/
CObservationStereoImages::CObservationStereoImages( ) :
	m_auxMap(),
	cameraPose(),
	intrinsicParams(3,3),
	imageLeft( NULL ),
	imageRight( NULL ),
	rightCameraPose(),
	focalLength_meters(0.02)
{
}

/*---------------------------------------------------------------
					Destructor
 ---------------------------------------------------------------*/
CObservationStereoImages::~CObservationStereoImages(  )
{
}

/*---------------------------------------------------------------
  Implements the writing to a CStream capability of CSerializable objects
 ---------------------------------------------------------------*/
void  CObservationStereoImages::writeToStream(CStream &out, int *version) const
{
	if (version)
		*version =4 ;
	else
	{
		// The data
		out << cameraPose << intrinsicParams << imageLeft << imageRight;

		out << timestamp;

		out << rightCameraPose;

		out << focalLength_meters;
		out << sensorLabel;
	}
}

/*---------------------------------------------------------------
  Implements the reading from a CStream capability of CSerializable objects
 ---------------------------------------------------------------*/
void  CObservationStereoImages::readFromStream(CStream &in, int version)
{
	switch(version)
	{
	case 0:
	case 1:
	case 2:
	case 3:
	case 4:
		{
			m_auxMap.clear();

			in >> cameraPose >> intrinsicParams >> imageLeft >> imageRight;

			if (version>=1)
					in >> timestamp;
			else 	timestamp = INVALID_TIMESTAMP;

			if (version>=2)
			{
				in >> rightCameraPose;
			}
			else
			{
				rightCameraPose = CPose3D(0.10f,0,0);
			}

			if (version>=3)
			{
				in >> focalLength_meters;
			}
			else
			{
				focalLength_meters = 0.002;
			}

			if (version>=4)
					in >> sensorLabel;
			else 	sensorLabel = "";


		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};

}

/*---------------------------------------------------------------
 Inserts a pure virtual method for finding the likelihood between this
   and another observation, probably only of the same derived class. The operator
   may be asymmetric.

 \param anotherObs The other observation to compute likelihood with.
 \param anotherObsPose If known, the belief about the robot pose when the other observation was taken can be supplied here, or NULL if it is unknown.

 \return Returns a likelihood measurement, in the range [0,1].
 \exception std::exception On any error, as another observation being of an invalid class.
  ---------------------------------------------------------------*/
float  CObservationStereoImages::likelihoodWith(
	const CObservation		*anotherObs,
	const CPosePDF			*anotherObsPose ) const
{
	MRPT_UNUSED_PARAM(anotherObs); MRPT_UNUSED_PARAM(anotherObsPose);
	return 0;
}


/*---------------------------------------------------------------
				buildAuxiliaryMap
---------------------------------------------------------------*/
const CLandmarksMap *CObservationStereoImages::buildAuxiliaryMap( mrpt::slam::CLandmark::TLandmarkID fID, const mrpt::slam::CLandmarksMap::TInsertionOptions *insOpts ) const
{
    if (!m_auxMap.get())
	{
		CLandmarksMap *auxMap = new CLandmarksMap();
		if( insOpts )
			auxMap->insertionOptions = *insOpts;

		auxMap->loadSiftFeaturesFromStereoImageObservation( *this, fID );
		m_auxMap.set(auxMap);
	} // end if

	return m_auxMap.get();
}

void CObservationStereoImages::CAuxMapWrapper::clear()
{
	if (auxMap) delete auxMap;
	auxMap=NULL;
}

void CObservationStereoImages::CAuxMapWrapper::set(CLandmarksMap	*m)
{
	if (auxMap) delete auxMap;
	auxMap = m;
}
