/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/slam/CRobotSimulator.h>
#include <mrpt/random.h>
#include <mrpt/math/utils.h>

using namespace mrpt::slam;
using namespace mrpt::utils;
using namespace mrpt::poses;


/*************************************************************************
        Constructor
*************************************************************************/
CRobotSimulator::CRobotSimulator(float TAU , float DELAY ) :
	X(0),Y(0),PHI(0),v(0),w(0),
	t(0),
	odo_x(0),odo_y(0),odo_phi(0),
	usar_error_odometrico(false),
	Command_Time(0),
	Command_v(0),Command_w(0),
	Command_v0(0),Command_w0(0),
	cTAU(TAU),
	cDELAY(DELAY)
{
	ResetStatus();
}

/*************************************************************************
        Destructor
*************************************************************************/
CRobotSimulator::~CRobotSimulator()
{


}

/*************************************************************************
                        Incremento de tiempo

 Simula el robot durante este tiempo: Velocidades, giros, movimientos,...
*************************************************************************/
void    CRobotSimulator::SimulateInterval( double At )
{
        double  dx,dy,dx_odo,dy_odo;          // Incremento de X e Y en este intervalo:
        double  curvatura;
        double  w_radianes;
        double  mod,ang;
        double  AAt, tt;

        tt = 0.0;
        AAt = 0.001;     // Paso minimo de tiempo a simular:

      while (tt<At)
      {
        t+= AAt;
        tt+=AAt;

        // Cambiar velocidades:
        // ----------------------------------------------------------------
        double t_transcurrido = t - Command_Time;
		t_transcurrido-=cDELAY; if (t_transcurrido<0) t_transcurrido=0;

		if (cTAU==0 && cDELAY==0)
		{
			v = Command_v;
			w = Command_w;
		}
		else
		{
			v = Command_v0 + (Command_v-Command_v0)*(1-exp(-t_transcurrido/cTAU));
			w = Command_w0 + (Command_w-Command_w0)*(1-exp(-t_transcurrido/cTAU));
		}


        // Simular movimiento en líneas o circulos durante el periodo At:
        // ----------------------------------------------------------------
        if ( v != 0.0 )
        {
                w_radianes = w;

                curvatura = w_radianes / v;

                if ( curvatura != 0.0 )
                {
                        // Curva circular:
                        double dx0 = dx = (sin(  w_radianes * AAt ) / curvatura);
                        double dy0 = dy = (1-cos( w_radianes * AAt ) ) / curvatura;

						mod = sqrt(dx*dx+dy*dy);
						ang = atan2(dy0, dx0);
						ang += PHI;
                        dx = mod * cos(ang);
                        dy = mod * sin(ang);

						ang = atan2(dy0, dx0);
                        ang+= odo_phi;
                        dx_odo = mod * cos(ang);
                        dy_odo = mod * sin(ang);
                }
                else
                {
                        // Linea recta:
                        dx =  cos( PHI ) * v * AAt;
                        dy =  sin( PHI ) * v * AAt;

                        dx_odo = cos( odo_phi ) * v * AAt;
                        dy_odo = sin( odo_phi ) * v * AAt;
                }

                // Las posiciones reales:
                X+= dx;
                Y+= dy;

                // y las odometricas (con ruido)
                odo_x+=dx_odo;
                odo_y+=dy_odo;
				if (usar_error_odometrico)
				{
					odo_x+= m_Ax_err_bias + m_Ax_err_std * mrpt::random::normalizedGaussian();
					odo_y+= m_Ay_err_bias + m_Ay_err_std * mrpt::random::normalizedGaussian();
				}
        }

        PHI+= w*AAt;
		PHI = math::wrapToPi(PHI);

        odo_phi+=w*AAt;
		if (usar_error_odometrico)
		{
			odo_phi+= m_Aphi_err_bias + m_Aphi_err_std * mrpt::random::normalizedGaussian();
		}
		odo_phi = math::wrapToPi(odo_phi);
      }

}

/*************************************************************************
        Dar un comando de movimiento al robot:

   Guarda los valores para ir ejecutando el movimiento poco a poco,
    teniendo en cuenta los tiempos de reaccion.
*************************************************************************/
void    CRobotSimulator::MovementCommand ( double lin_vel, double ang_vel )
{
        // Apuntar datos para ejecutarlos lentamente, segun
        //   la respuesta del robot:
        Command_Time = t;
        Command_v = lin_vel;
        Command_w = ang_vel;

        // Actuales:
        Command_v0 = v;
        Command_w0 = w;

}


void CRobotSimulator::ResetStatus()
{
	X=Y=PHI=t=0.0;
	odo_x=odo_y=odo_phi=0.0;
	v=w=0.0;
	Command_Time = 0.0;
	Command_v = Command_w = Command_v0 = Command_w0 = 0.0;
}

