/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/config.h>
#include <mrpt/utils/utils_defs.h>

#include <mrpt/utils/CTicTac.h>
#include <mrpt/utils/CFileOutputStream.h>
#include <mrpt/utils/CFileInputStream.h>
#include <mrpt/system/os.h>
#include <mrpt/synch.h>

#include <cstdlib>
#include <cstdarg>
#include <float.h>
#include <iostream>
#include <algorithm>
#include <cctype>
#include <ctime>
#include <cstdio>

// Visual Leak Detector
//  http://www.codeproject.com/tools/visualleakdetector.asp
#if MRPT_HAS_VLD
#include <vld.h>
#endif

#ifdef MRPT_OS_WINDOWS
    #include <conio.h>
	#include <windows.h>
	#include <process.h>
	#include <tlhelp32.h>
	#include <sys/utime.h>
	#include <io.h>
	#include <direct.h>
#else
    #include <pthread.h>
    #include <termios.h>
    #include <unistd.h>
    #include <sys/select.h>
    #include <sys/time.h>
	#include <sys/types.h>
    #include <time.h>
	#include <unistd.h>
    #include <sys/stat.h>
	#include <utime.h>
	#include <errno.h>
	#include <signal.h>

	#define _access access
	#define _rmdir rmdir
#endif


using namespace mrpt;
using namespace mrpt::utils;
using namespace mrpt::system;
using namespace std;

#ifdef MRPT_OS_LINUX
    /** Function published in http://www.linuxjournal.com/article/1138
      */
    int myKbhit(void)
    {
      struct timeval tv;
      fd_set read_fd;

      /* Do not wait at all, not even a microsecond */
      tv.tv_sec=0;
      tv.tv_usec=0;

      /* Must be done first to initialize read_fd */
      FD_ZERO(&read_fd);

      /* Makes select() ask if input is ready:
       * 0 is the file descriptor for stdin    */
      FD_SET(0,&read_fd);

      /* The first parameter is the number of the
       * largest file descriptor to check + 1. */
      if(select(1, &read_fd,
                NULL, /*No writes*/
                NULL, /*No exceptions*/
                &tv) == -1)
        return 0;	/* An error occured */

      /*	read_fd now holds a bit map of files that are
       * readable. We test the entry for the standard
       * input (file 0). */
      if(FD_ISSET(0,&read_fd))
        /* Character pending on stdin */
        return 1;

      /* no characters were pending */
      return 0;
    }

#endif


// --------------------------------------------------------------------------------------------------
// For UNIX only: If a fatal signal is caught, throw a MRPT exception to inform about the event:
//   Based on code from wxWidgets (utilsunx.cpp)
// --------------------------------------------------------------------------------------------------

#ifdef MRPT_OS_LINUX

// Use the wonderful wxWidgets stack walker!
#if MRPT_HAS_WXWIDGETS && wxUSE_STACKWALKER

#include <wx/string.h>
#include <wx/stackwalk.h>

/** A custom class that build a string representation of the stack frames
  */
class CMRPTStackWalker : public wxStackWalker
{
    public:
    CMRPTStackWalker() : m_stackDescription("Contents of the stack:\n")
    {
    }

    virtual ~CMRPTStackWalker()
    {
    }

    std::string  m_stackDescription;

    void OnStackFrame(const wxStackFrame& frame)
    {
        //cerr << format("%u\n",(unsigned int)frame.GetLevel());
        string filename(
            mrpt::system::extractFileName( string(frame.GetFileName().mb_str()) ) +
            string(".") +
            mrpt::system::extractFileExtension( string(frame.GetFileName().mb_str()) ) );

        m_stackDescription += format(
          "[%4u] 0x%p -> %s File: %s Function: %s Line: %u\n",
          (unsigned int)frame.GetLevel(),
          frame.GetAddress(),
          string(frame.GetModule().mb_str()).c_str(),
          filename.c_str(),
          string(frame.GetName().mb_str()).c_str(),
          (unsigned int)frame.GetLine()
          );
    }

};

#endif  // wxWidgets

extern "C" void MRPT_SIGNAL_HANDLER_SIG( int )
{
#if MRPT_HAS_WXWIDGETS && wxUSE_STACKWALKER
    CMRPTStackWalker    sw;
    sw.Walk();
    cerr << sw.m_stackDescription; cerr.flush();
    //THROW_EXCEPTION( "*FATAL*: Signal SIGSEGV caught!" );
    abort();
#else
	cerr << "*FATAL*: Signal SIGSEGV caught!" << endl;
    abort();
#endif
}

#endif // Linux


/*---------------------------------------------------------------
            registerFatalExceptionHandlers
  ---------------------------------------------------------------*/
void mrpt::system::registerFatalExceptionHandlers()
{
	static bool done = false;
	if (done) return;
	done = true;

#ifdef MRPT_OS_LINUX
    // install the signal handler
    struct sigaction act;
    // some systems extend it with non std fields, so zero everything
    memset(&act, 0, sizeof(act));

    sigemptyset(&act.sa_mask);
    act.sa_flags = 0;

    act.sa_handler = MRPT_SIGNAL_HANDLER_SIG; //MRPT_SIGNAL_HANDLER_SIGFPE;
    if (0!=sigaction(SIGFPE, &act, NULL) ) cerr << "[registerFatalExceptionHandlers] Cannot install signal handler!!" << endl;

    act.sa_handler = MRPT_SIGNAL_HANDLER_SIG; //MRPT_SIGNAL_HANDLER_SIGILL;
    if (0!=sigaction(SIGILL, &act, NULL) ) cerr << "[registerFatalExceptionHandlers] Cannot install signal handler!!" << endl;

    act.sa_handler = MRPT_SIGNAL_HANDLER_SIG; //MRPT_SIGNAL_HANDLER_SIGBUS;
    if (0!=sigaction(SIGBUS, &act, NULL) ) cerr << "[registerFatalExceptionHandlers] Cannot install signal handler!!" << endl;

    act.sa_handler = MRPT_SIGNAL_HANDLER_SIG; //MRPT_SIGNAL_HANDLER_SIGSEGV;
    if (0!=sigaction(SIGSEGV, &act, NULL) ) cerr << "[registerFatalExceptionHandlers] Cannot install signal handler!!" << endl;
#endif

}





/*---------------------------------------------------------------
					ExtractFileName
	Extracts just the name of a filename from a
      complete path plus name plus extension.
  ---------------------------------------------------------------*/
string  mrpt::system::extractFileName(const string& filePath)
{
	int	 i,dotPos = int(filePath.size());
	if (filePath.size()<2) return string("");

	for (i=(int)filePath.size()-1;i>=0 && !(filePath[i]=='\\' || filePath[i]=='/');i--)
		if (dotPos==int(filePath.size()) && filePath[i]=='.')
			dotPos = i;
	return filePath.substr(i+1,dotPos-i-1);
}

/*---------------------------------------------------------------
					ExtractFileDirectory
	Extracts just the directory of a filename from a
      complete path plus name plus extension.
  ---------------------------------------------------------------*/
string  mrpt::system::extractFileDirectory(const string& filePath)
{
	if (filePath.size()<2) return filePath;

	// Search the first "/" or "\" from the right:
	int i;
	for (i=(int)filePath.size()-1;i>0;i--)
		if (filePath[i]=='\\' || filePath[i]=='/')
			break;

	if (!i) return string("");
	else	return filePath.substr(0,i+1);
}


/*---------------------------------------------------------------
					ExtractFileName
	Extracts just the name of a filename from a
      complete path plus name plus extension.
  ---------------------------------------------------------------*/
string  mrpt::system::extractFileExtension(const string& filePath, bool ignore_gz)
{
	if (filePath.size()<2) return string("");

	size_t i_end = filePath.size()-1;

	int	i= (int)(i_end);
	while (i>0)
	{
		if (filePath[i]=='.')
		{
			string the_ext = filePath.substr(i+1,i_end-i);
			if (!ignore_gz || the_ext!="gz")
				return the_ext;
			else
			{
				i_end = --i;
			}
		}
		else	i--;
	}
	// No extension:
	return string("");
}

/*---------------------------------------------------------------
					FileExists
	Test if a given file exists
  ---------------------------------------------------------------*/
bool  mrpt::system::fileExists(const string& path)
{
	return 0 == _access(path.c_str(), 0x00 ); // 0x00 = Check for existence only!
}

/*---------------------------------------------------------------
					time_tToTimestamp
  ---------------------------------------------------------------*/
mrpt::system::TTimeStamp  mrpt::system::time_tToTimestamp(const time_t &t )
{
    return (((uint64_t)t) * (uint64_t)10000000) + ((uint64_t)116444736*1000000000);
}

/*---------------------------------------------------------------
					time_tToTimestamp
  ---------------------------------------------------------------*/
mrpt::system::TTimeStamp  mrpt::system::time_tToTimestamp(const double &t )
{
    return (uint64_t)(t*10000000.0)+((uint64_t)116444736*1000000000);
}

/*---------------------------------------------------------------
					timestampTotime_t
  ---------------------------------------------------------------*/
double mrpt::system::timestampTotime_t( const mrpt::system::TTimeStamp  &t )
{
    return double(t - ((uint64_t)116444736*1000000000)) / 10000000.0;
}

/*---------------------------------------------------------------
					Returns the current system time.
  ---------------------------------------------------------------*/
mrpt::system::TTimeStamp  mrpt::system::getCurrentTime( )
{
#ifdef MRPT_OS_WINDOWS
	FILETIME		t;
	GetSystemTimeAsFileTime(&t);
	return (((uint64_t)t.dwHighDateTime) << 32) | ((uint64_t)t.dwLowDateTime);
#else
    timespec  tim;
    clock_gettime((clockid_t)0, &tim);

	/** TODO: Do better! */
    return time_tToTimestamp( time(NULL) ) + tim.tv_nsec/100;
#endif
}

/*---------------------------------------------------------------
					timestampToParts
  ---------------------------------------------------------------*/
void mrpt::system::timestampToParts( TTimeStamp t, TTimeParts &p )
{
	double T = mrpt::system::timestampTotime_t(t);
	time_t tt = time_t(T);

	double sec_frac = T - tt;
	ASSERT_(sec_frac<1.0);

	struct tm * parts = gmtime(&tt);

	p.year		= parts->tm_year + 1900;
	p.month		= parts->tm_mon + 1;
	p.day		= parts->tm_mday;
	p.day_of_week = parts->tm_wday + 1;
	p.daylight_saving = parts->tm_isdst;
	p.hour		= parts->tm_hour;
	p.minute	= parts->tm_min;
	p.second	= parts->tm_sec + sec_frac;
}


#ifndef HAVE_TIMEGM
	time_t timegm(struct tm *tm)
	{
		time_t ret;
		char *tz;

		/* save current timezone and set UTC */
		tz = getenv("TZ");
		putenv("TZ=UTC");   /* use Coordinated Universal Time (i.e. zero offset) */
		tzset();

		ret = mktime(tm);
		if(tz)
		{
			char buf[256];
			mrpt::system::os::sprintf(buf, sizeof(buf), "TZ=%s", tz);
			putenv(buf);
		} else
			putenv("TZ=");
		tzset();

		return ret;
	}
#endif	// HAVE_TIMEGM


/*---------------------------------------------------------------
					buildTimestampFromParts
  ---------------------------------------------------------------*/
TTimeStamp mrpt::system::buildTimestampFromParts( const TTimeParts &p )
{
	struct tm parts;

	parts.tm_year  = p.year - 1900;
	parts.tm_mon   = p.month-1;
	parts.tm_mday  = p.day;
	parts.tm_wday  = p.day_of_week - 1;
	parts.tm_isdst = p.daylight_saving;
	parts.tm_hour  = p.hour;
	parts.tm_min   = p.minute;
	parts.tm_sec   = int(p.second);

	double sec_frac = p.second - parts.tm_sec;

	time_t  tt = timegm(&parts); // Local time: mktime

	return mrpt::system::time_tToTimestamp( double(tt) + sec_frac );
}

/*---------------------------------------------------------------
					buildTimestampFromPartsLocalTime
  ---------------------------------------------------------------*/
TTimeStamp mrpt::system::buildTimestampFromPartsLocalTime( const TTimeParts &p )
{
	struct tm parts;

	parts.tm_year  = p.year - 1900;
	parts.tm_mon   = p.month-1;
	parts.tm_mday  = p.day;
	parts.tm_wday  = p.day_of_week - 1;
	parts.tm_isdst = p.daylight_saving;
	parts.tm_hour  = p.hour;
	parts.tm_min   = p.minute;
	parts.tm_sec   = int(p.second);

	double sec_frac = p.second - parts.tm_sec;

	time_t  tt = mktime(&parts);

	return mrpt::system::time_tToTimestamp( double(tt) + sec_frac );
}

/*---------------------------------------------------------------
					Returns the current local time.
  ---------------------------------------------------------------*/
mrpt::system::TTimeStamp  mrpt::system::getCurrentLocalTime()
{
#ifdef MRPT_OS_WINDOWS
	FILETIME		tt,t;
	GetSystemTimeAsFileTime(&tt);
	FileTimeToLocalFileTime(&tt,&t);

	return (((uint64_t)t.dwHighDateTime) << 32) | ((uint64_t)t.dwLowDateTime);
#else
    timespec  tim;
    clock_gettime((clockid_t)0, &tim);

	time_t  tt;
	struct tm * timeinfo;
	time(&tt);
	timeinfo = localtime( &tt );

    return time_tToTimestamp( mktime(timeinfo) ) + tim.tv_nsec/100;
#endif
}


/*---------------------------------------------------------------
					timeDifference
  ---------------------------------------------------------------*/
double mrpt::system::timeDifference( const mrpt::system::TTimeStamp &t1, const mrpt::system::TTimeStamp &t2 )
{
	MRPT_TRY_START
	ASSERT_(t1!=INVALID_TIMESTAMP)
	ASSERT_(t2!=INVALID_TIMESTAMP)

	// The signed __int64 is for MSVC 6
	return ((double)((int64_t)(t2-t1)))/10000000.0;

	MRPT_TRY_END
}

/*---------------------------------------------------------------
					secondsToTimestamp
  ---------------------------------------------------------------*/
mrpt::system::TTimeStamp mrpt::system::secondsToTimestamp( const double &nSeconds )
{
	return (mrpt::system::TTimeStamp)(nSeconds*10000000.0);
}

/*---------------------------------------------------------------
					formatTimeInterval
  ---------------------------------------------------------------*/
string mrpt::system::formatTimeInterval( const double &t )
{
	double timeSeconds = (t<0) ? (-t) : t;

	unsigned int nHours =  (unsigned int)timeSeconds / 3600;
	unsigned int nMins  = ((unsigned int)timeSeconds % 3600) / 60 ;
	unsigned int nSecs  = (unsigned int)timeSeconds % 60;
	unsigned int milSecs= (unsigned int ) ( 1000*(timeSeconds - floor(timeSeconds))  / 1000.0 );

	return format(
		"%02u:%02u:%02u.%03u",
		nHours,
		nMins,
		nSecs,
		milSecs );
}


/*---------------------------------------------------------------
  Convert a timestamp into this textual form: YEAR/MONTH/DAY,HH:MM:SS.MMM
  ---------------------------------------------------------------*/
string  mrpt::system::dateTimeToString(const mrpt::system::TTimeStamp &t)
{
	if (t==INVALID_TIMESTAMP) return string("INVALID_TIMESTAMP");

	uint64_t        tmp = (t - ((uint64_t)116444736*1000000000));
    time_t          auxTime = tmp / (uint64_t)10000000;
    unsigned int	secFractions = (unsigned int)( 1000000 * (tmp % 10000000) / 10000000.0 );
    tm  *ptm = gmtime( &auxTime );

	return format(
		"%u/%02u/%02u,%02u:%02u:%02u.%06u",
		1900+ptm->tm_year,
		ptm->tm_mon+1,
		ptm->tm_mday,
		ptm->tm_hour,
		ptm->tm_min,
		(unsigned int)ptm->tm_sec,
		secFractions );
}

/*---------------------------------------------------------------
  Convert a timestamp into this textual form (in local time):
      YEAR/MONTH/DAY,HH:MM:SS.MMM
  ---------------------------------------------------------------*/
string  mrpt::system::dateTimeLocalToString(const mrpt::system::TTimeStamp &t)
{
	if (t==INVALID_TIMESTAMP) return string("INVALID_TIMESTAMP");

	uint64_t        tmp = (t - ((uint64_t)116444736*1000000000));
    time_t          auxTime = tmp / (uint64_t)10000000;
    unsigned int	secFractions = (unsigned int)( 1000000 * (tmp % 10000000) / 10000000.0 );
    tm  *ptm = localtime( &auxTime );

	return format(
		"%u/%02u/%02u,%02u:%02u:%02u.%06u",
		1900+ptm->tm_year,
		ptm->tm_mon+1,
		ptm->tm_mday,
		ptm->tm_hour,
		ptm->tm_min,
		(unsigned int)ptm->tm_sec,
		secFractions );
}

/*---------------------------------------------------------------
						extractDayTimeFromTimestamp
  ---------------------------------------------------------------*/
double  mrpt::system::extractDayTimeFromTimestamp(const mrpt::system::TTimeStamp &t)
{
	MRPT_TRY_START
	ASSERT_(t!=INVALID_TIMESTAMP)

#ifdef MRPT_OS_WINDOWS
	SYSTEMTIME		sysT;
	FileTimeToSystemTime( (FILETIME*)&t, &sysT );
	return sysT.wHour * 3600.0 + sysT.wMinute * 60.0 + sysT.wSecond + sysT.wMilliseconds * 0.001;
#else
    time_t      auxTime = (t - ((uint64_t)116444736*1000000000)) / (uint64_t)10000000;
    tm  *ptm = gmtime( &auxTime );
	return ptm->tm_hour * 3600.0 + ptm->tm_min * 60.0 + ptm->tm_sec;
#endif
	MRPT_TRY_END
}


/*---------------------------------------------------------------
  Convert a timestamp into this textual form: HH:MM:SS.MMM
  ---------------------------------------------------------------*/
string  mrpt::system::timeLocalToString(const mrpt::system::TTimeStamp &t)
{
	if (t==INVALID_TIMESTAMP) return string("INVALID_TIMESTAMP");

	uint64_t        tmp = (t - ((uint64_t)116444736*1000000000));
    time_t          auxTime = tmp / (uint64_t)10000000;
    unsigned int	secFractions = (unsigned int)( 1000000 * (tmp % 10000000) / 10000000.0 );
    tm  *ptm = localtime( &auxTime );

	return format(
		"%02u:%02u:%02u.%06u",
		ptm->tm_hour,
		ptm->tm_min,
		(unsigned int)ptm->tm_sec,
		secFractions );
}

/*---------------------------------------------------------------
  Convert a timestamp into this textual form: HH:MM:SS.MMM
  ---------------------------------------------------------------*/
string  mrpt::system::timeToString(const mrpt::system::TTimeStamp &t)
{
	if (t==INVALID_TIMESTAMP) return string("INVALID_TIMESTAMP");

	uint64_t        tmp = (t - ((uint64_t)116444736*1000000000));
    time_t          auxTime = tmp / (uint64_t)10000000;
    unsigned int	secFractions = (unsigned int)( 1000000 * (tmp % 10000000) / 10000000.0 );
    tm  *ptm = gmtime( &auxTime );

	return format(
		"%02u:%02u:%02u.%06u",
		ptm->tm_hour,
		ptm->tm_min,
		(unsigned int)ptm->tm_sec,
		secFractions );
}

/*---------------------------------------------------------------
  Convert a timestamp into this textual form: YEAR/MONTH/DAY
  ---------------------------------------------------------------*/
string  mrpt::system::dateToString(const mrpt::system::TTimeStamp &t)
{
	if (t==INVALID_TIMESTAMP) return string("INVALID_TIMESTAMP");

	uint64_t        tmp = (t - ((uint64_t)116444736*1000000000));
    time_t          auxTime = tmp / (uint64_t)10000000;
    tm  *ptm = gmtime( &auxTime );

	return format(
		"%u/%02u/%02u",
		1900+ptm->tm_year,
		ptm->tm_mon+1,
		ptm->tm_mday
		);
}


/*---------------------------------------------------------------
						vectorToTextFile
 ---------------------------------------------------------------*/
bool  mrpt::system::vectorToTextFile( const vector<float> &vec, const string &fileName, bool append, bool byRows )
{
	FILE	*f=os::fopen(fileName.c_str(), append ? "at" : "wt");
	if (!f) return false;

	for (vector<float>::const_iterator	it=vec.begin();it!=vec.end();it++)
		os::fprintf(f,byRows ? "%e ":"%e\n",*it);

	if (byRows) os::fprintf(f,"\n");

	os::fclose(f);
	return true;	// All ok.
}

/*---------------------------------------------------------------
						vectorToTextFile
 ---------------------------------------------------------------*/
bool  mrpt::system::vectorToTextFile( const vector<double> &vec, const string &fileName, bool append, bool byRows  )
{
	FILE	*f=os::fopen(fileName.c_str(),append ? "at" : "wt");
	if (!f) return false;

	for (vector<double>::const_iterator	it=vec.begin();it!=vec.end();it++)
		os::fprintf(f,byRows ? "%e ":"%e\n",*it);

	if (byRows) os::fprintf(f,"\n");

	os::fclose(f);
	return true;	// All ok.
}

/*---------------------------------------------------------------
						vectorToTextFile
 ---------------------------------------------------------------*/
bool  mrpt::system::vectorToTextFile( const vector<int> &vec, const string &fileName, bool append, bool byRows  )
{
	FILE	*f=os::fopen(fileName.c_str(),append ? "at" : "wt");
	if (!f) return false;

	for (vector<int>::const_iterator	it=vec.begin();it!=vec.end();it++)
		os::fprintf(f,byRows ? "%i ":"%i\n",*it);

	if (byRows) os::fprintf(f,"\n");

	os::fclose(f);
	return true;	// All ok.
}

/*---------------------------------------------------------------
						vectorToTextFile
 ---------------------------------------------------------------*/
bool  mrpt::system::vectorToTextFile( const vector<size_t> &vec, const string &fileName, bool append, bool byRows  )
{
	FILE	*f=os::fopen(fileName.c_str(),append ? "at" : "wt");
	if (!f) return false;

	for (vector<size_t>::const_iterator	it=vec.begin();it!=vec.end();it++)
		os::fprintf(f,byRows ? "%u ":"%u\n",static_cast<unsigned int>(*it));

	if (byRows) os::fprintf(f,"\n");

	os::fclose(f);
	return true;	// All ok.
}

#ifdef  MRPT_OS_WINDOWS
#include <windows.h>
	// Windows:
	typedef struct _PROCESS_MEMORY_COUNTERS {
	  DWORD cb;
	  DWORD PageFaultCount;
	  SIZE_T PeakWorkingSetSize;
	  SIZE_T WorkingSetSize;
	  SIZE_T QuotaPeakPagedPoolUsage;
	  SIZE_T QuotaPagedPoolUsage;
	  SIZE_T QuotaPeakNonPagedPoolUsage;
	  SIZE_T QuotaNonPagedPoolUsage;
	  SIZE_T PagefileUsage;
	  SIZE_T PeakPagefileUsage;
	} PROCESS_MEMORY_COUNTERS,
	*PPROCESS_MEMORY_COUNTERS;

	namespace mrpt
	{
		namespace system
		{
			/** This is an auxiliary class for mrpt::system::getMemoryUsage() under Windows.
			  *  It loads in runtime PSAPI.DLL. This is to avoid problems in some platforms, i.e Windows 2000,
			  *  where this DLL must not be present.
			  */
			class CAuxPSAPI_Loader
			{
			protected:
				typedef BOOL (WINAPI *TGetProcessMemoryInfo)(
				  HANDLE Process,
				  PPROCESS_MEMORY_COUNTERS ppsmemCounters,
				  DWORD cb );

				TGetProcessMemoryInfo		m_ptr;

			public:
				HMODULE m_dll;

				CAuxPSAPI_Loader()
				{
					m_ptr = NULL;

					m_dll = LoadLibraryA("PSAPI.DLL");
					if (m_dll)
					{
						m_ptr = (TGetProcessMemoryInfo) GetProcAddress(m_dll,"GetProcessMemoryInfo");
					}
				}
				~CAuxPSAPI_Loader()
				{
					if (m_dll)
					{
						FreeLibrary(m_dll);
						m_dll = NULL;
						m_ptr = NULL;
					}
				}

				BOOL WINAPI GetProcessMemoryInfo(
				  HANDLE Process,
				  PPROCESS_MEMORY_COUNTERS ppsmemCounters,
				  DWORD cb )
				{
					try
					{
						if (!m_ptr)
								return false;
						else	return (*m_ptr )(Process,ppsmemCounters,cb);
					}
					catch(...)
					{
						return false;
					}
				}
			};
		}
	}

#endif

/*---------------------------------------------------------------
						getMemoryUsage
 ---------------------------------------------------------------*/
unsigned long  mrpt::system::getMemoryUsage()
{
	MRPT_TRY_START
	unsigned long MEM = 0;

#ifdef  MRPT_OS_WINDOWS
	// Windows:
	static CAuxPSAPI_Loader		PSAPI_LOADER;

	PROCESS_MEMORY_COUNTERS		pmc;
	pmc.cb = sizeof(pmc);

	if ( PSAPI_LOADER.GetProcessMemoryInfo( GetCurrentProcess(),&pmc,sizeof(pmc)  ) )
	{
		MEM = (long)pmc.PagefileUsage;
	}
#else
	// Linux:
	//int page_size = getpagesize();

	FILE *f = ::fopen ("/proc/self/stat", "r");
	if (!f) return 0;

	// Note: some of these scanf specifiers would normally be 'long' versions if
	// not for the fact that we are using suppression (gcc warns).  see 'man
	// proc' for scanf specifiers and meanings.
	if (!::fscanf(f,
		"%*d %*s %*c %*d %*d %*d %*d %*d %*u %*u %*u %*u %*u %*u %*u %*d %*d %*d %*d %*d %*d %*u %lu", &MEM))
	{
		// Error parsing:
		MEM=0;
	}
	::fclose (f);
	//::system("cat /proc/self/statm");
#endif

	return MEM;
	MRPT_TRY_END
}

/*---------------------------------------------------------------
						createDirectory
 ---------------------------------------------------------------*/
bool  mrpt::system::createDirectory( const string &dirName )
{
#ifdef MRPT_OS_WINDOWS
	return 0!=CreateDirectoryA( dirName.c_str(), NULL );
#else
    int ret = mkdir( dirName.c_str(), S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH );
    if (ret && errno!=EEXIST) // We ignore this error...
    {
		string str = format("[createDirectory %s]",dirName.c_str());
		perror(str.c_str());
		return false;
    }
    else
		return true;	// OK
#endif
}

/*---------------------------------------------------------------
						deleteFile
 ---------------------------------------------------------------*/
bool  mrpt::system::deleteFile( const string &fileName )
{
	return 0==remove( fileName.c_str() );
}



/*---------------------------------------------------------------
					mrpt::system::deleteFiles
---------------------------------------------------------------*/
void mrpt::system::deleteFiles(const string &s)
{
	MRPT_TRY_START;
	size_t	len = s.size()+20;
	char	*aux=new char[len];
#ifdef MRPT_OS_WINDOWS
	os::sprintf(aux,len,"del %s",&s[0]);
	for (char *c=aux;*c;c++)
		if (*c=='/') *c = '\\';
	os::strcat(aux,len," /Q");
#else
	os::sprintf(aux,len,"rm %s",&s[0]);
#endif

	int res = ::system(aux);
	if (res)
	{
		cerr << "[mrpt::system::deleteFiles] Warning: error invoking: " << aux << endl;
	}
	delete[] aux;
	MRPT_TRY_END;
}

void mrpt::system::deleteFilesInDirectory(const string &s)
{
	MRPT_TRY_START;
	size_t	len = s.size()+100;
	char	*aux=new char[len];
#ifdef MRPT_OS_WINDOWS
	os::sprintf(aux,len,"del %s\\*.*",&s[0]);
	for (char *c=aux;*c;c++)
		if (*c=='/') *c = '\\';
	os::strcat(aux,len," /Q");
#else
	os::sprintf(aux,len,"find %s  | xargs rm -fr",&s[0]);
#endif

	int res = ::system(aux);
	if (res)
		cerr << "[mrpt::system::deleteFiles] Warning: error invoking: " << aux << endl;

	delete[] aux;
	MRPT_TRY_END;
}



/*---------------------------------------------------------------
						toLowerCase
  ---------------------------------------------------------------*/
string  mrpt::system::lowerCase(const string& str)
{ return toLowerCase(str); }

string  mrpt::system::toLowerCase(const string& str)
{
	string outStr( str );

	transform(
		outStr.begin(), outStr.end(),		// In
		outStr.begin(),			// Out
		(int(*)(int)) tolower );
	return outStr;
}

/*---------------------------------------------------------------
						toUpperCase
  ---------------------------------------------------------------*/
string  mrpt::system::upperCase(const string& str)
{ return toUpperCase(str); }

string  mrpt::system::toUpperCase(const string& str)
{
	string outStr( str );
	transform(
		outStr.begin(), outStr.end(),		// In
		outStr.begin(),			// Out
		(int(*)(int)) toupper );
	return outStr;
}


/*---------------------------------------------------------------
					mrpt::system::MRPT_getCompilationDate
---------------------------------------------------------------*/
string mrpt::system::MRPT_getCompilationDate()
{
	return string(__DATE__);
}

/*---------------------------------------------------------------
					mrpt::system::MRPT_getVersion
---------------------------------------------------------------*/
#include <mrpt/version.h>
string mrpt::system::MRPT_getVersion()
{
	return string( ::MRPT_version_str );
}


/*---------------------------------------------------------------
					encodeUTF8

    Author: Marius Bancila
    http://www.codeguru.com/cpp/misc/misc/multi-lingualsupport/article.php/c10451/
---------------------------------------------------------------*/
#define         MASKBITS                0x3F
#define         MASKBYTE                0x80
#define         MASK2BYTES              0xC0
#define         MASK3BYTES              0xE0
#define         MASK4BYTES              0xF0
#define         MASK5BYTES              0xF8
#define         MASK6BYTES              0xFC

void mrpt::system::encodeUTF8( const vector_word &input, std::string &output )
{
    output = ""; // output.clear();  VC6...
    output.reserve( input.size() );
    for(size_t i=0; i < input.size(); i++)
    {
        // 0xxxxxxx
        if(input[i] < 0x80)
        {
            output += (char)input[i];
        }
        // 110xxxxx 10xxxxxx
        else if(input[i] < 0x800)
        {
            output += (char)(MASK2BYTES | input[i] >> 6);
            output += (char)(MASKBYTE | (input[i] & MASKBITS) );
        }
        // 1110xxxx 10xxxxxx 10xxxxxx
        /*else if(input[i] < 0x10000)
        {
            output.push_back((char)(MASK3BYTES | input[i] >> 12));
            output.push_back((char)(MASKBYTE | input[i] >> 6 & MASKBITS));
            output.push_back((char)(MASKBYTE | input[i] & MASKBITS));
        }*/
    }
}

/*---------------------------------------------------------------
					decodeUTF8

    Author: Marius Bancila
    http://www.codeguru.com/cpp/misc/misc/multi-lingualsupport/article.php/c10451/
---------------------------------------------------------------*/
void mrpt::system::decodeUTF8( const std::string &input, vector_word &output )
{
    output.clear();
    output.reserve( input.size() );
    for(size_t i=0; i < input.size();)
    {
        uint16_t ch;

        // 1110xxxx 10xxxxxx 10xxxxxx
        if((input[i] & MASK3BYTES) == MASK3BYTES)
        {
            ch = ((input[i] & 0x0F) << 12) | (
            (input[i+1] & MASKBITS) << 6)
            | (input[i+2] & MASKBITS);
            i += 3;
        }
        // 110xxxxx 10xxxxxx
        else if((input[i] & MASK2BYTES) == MASK2BYTES)
        {
            ch = ((input[i] & 0x1F) << 6) | (input[i+1] & MASKBITS);
            i += 2;
        }
        // 0xxxxxxx
        else if( uint8_t(input[i]) < MASKBYTE)
        {
            ch = input[i];
            i += 1;
        }
        output.push_back(ch);
    }
}

/*---------------------------------------------------------------
			fileNameStripInvalidChars
---------------------------------------------------------------*/
std::string mrpt::system::fileNameStripInvalidChars( const std::string &filename)
{
	string	ret(filename);
	for (string::iterator c=ret.begin();c!=ret.end();c++)
	{
		if (!isalnum(*c) &&
			*c !='.' &&
			*c !='-' &&
			*c !='#' &&
			*c !='%' &&
			*c !='$' &&
			*c !='&' &&
			*c !='+' &&
			*c !='(' && *c !=')' &&
			*c !='[' && *c !=']' &&
			*c !='{' && *c !='{' )
				*c = '_';
	}
	return ret;
}

/*---------------------------------------------------------------
						sprintf
---------------------------------------------------------------*/
int os::sprintf(char *buf, size_t bufSize, const char *format, ...) MRPT_NO_THROWS
{
	MRPT_UNUSED_PARAM(bufSize);

	int			result;
	va_list		ap;
	va_start (ap, format);

#if defined(_MSC_VER) && (_MSC_VER>=1400)
	// Use a secure version in Visual Studio 2005:
	result = ::vsprintf_s (buf, bufSize, format, ap);
#else
	// Use standard version:
	result = ::vsprintf (buf, format, ap);
#endif

	va_end (ap);
	return result;
}

/*---------------------------------------------------------------
					vsprintf
---------------------------------------------------------------*/
int os::vsprintf(char *buf, size_t bufSize, const char *format, va_list args) MRPT_NO_THROWS
{
	MRPT_UNUSED_PARAM(bufSize);
#if defined(_MSC_VER) && (_MSC_VER>=1400)
	// Use a secure version in Visual Studio 2005:
	return ::vsprintf_s (buf, bufSize, format, args);
#else
	// Use standard version:
	return ::vsprintf (buf, format, args);
#endif
}

/*---------------------------------------------------------------
				vsnprintf
---------------------------------------------------------------*/
int os::vsnprintf(char *buf, size_t bufSize, const char *format, va_list args) MRPT_NO_THROWS
{
#if defined(_MSC_VER)
	#if (_MSC_VER>=1400)
		// Use a secure version in Visual Studio 2005:
		return ::vsnprintf_s (buf, bufSize, _TRUNCATE, format, args);
	#else
		return ::vsprintf(buf,format, args);
	#endif
#else
	// Use standard version:
	return ::vsnprintf(buf, bufSize,format, args);
#endif
}

/*---------------------------------------------------------------
					fopen
---------------------------------------------------------------*/
FILE * os::fopen(const std::string &fileName,const char *mode) MRPT_NO_THROWS
{
	return fopen(fileName.c_str(),mode);
}

/*---------------------------------------------------------------
					fopen
---------------------------------------------------------------*/
FILE * os::fopen(const char *fileName,const char *mode) MRPT_NO_THROWS
{
#if defined(_MSC_VER) && (_MSC_VER>=1400)
	// Use a secure version in Visual Studio 2005:
	FILE	*f;
	if ( 0 != ::fopen_s(&f,fileName,mode) )
			return NULL;
	else	return f;
#else
	// Use standard version:
	return ::fopen(fileName,mode);
#endif
}

/*---------------------------------------------------------------
					fclose
---------------------------------------------------------------*/
void os::fclose(FILE *f) MRPT_NO_THROWS
{
	::fclose(f);
}

/*---------------------------------------------------------------
						strcat
---------------------------------------------------------------*/
char * os::strcat(char *dest, size_t destSize, const char *source) MRPT_NO_THROWS
{
	MRPT_UNUSED_PARAM(destSize);

#if defined(_MSC_VER) && (_MSC_VER>=1400)
	::strcat_s(dest,destSize,source);
#else
	::strcat(dest,source);
#endif
	return dest;
}

/*---------------------------------------------------------------
						strcpy
---------------------------------------------------------------*/
char * os::strcpy(char *dest, size_t destSize, const char *source) MRPT_NO_THROWS
{
	MRPT_UNUSED_PARAM(destSize);

#if defined(_MSC_VER) && (_MSC_VER>=1400)
	::strcpy_s(dest,destSize,source);
#else
	::strcpy(dest,source);
#endif
	return dest;
}


/*---------------------------------------------------------------
						strcmp
---------------------------------------------------------------*/
int os::_strcmp(const char*str1,const char*str2) MRPT_NO_THROWS
{
	return ::strcmp( str1,str2 );
}

/*---------------------------------------------------------------
						strcmpi
---------------------------------------------------------------*/
int os::_strcmpi(const char*str1,const char*str2) MRPT_NO_THROWS
{
#ifdef MRPT_OS_WINDOWS
    #if defined(_MSC_VER) && (_MSC_VER>=1400)
        return ::_strcmpi( str1,str2 );
    #else
        return ::strcmpi( str1,str2 );
    #endif
#else
    return ::strcasecmp( str1,str2 );
#endif
}

/*---------------------------------------------------------------
						memcpy
---------------------------------------------------------------*/
void os::memcpy(
	void		*dest,
	size_t		destSize,
	const void	*src,
	size_t		copyCount ) MRPT_NO_THROWS
{
#if defined(_MSC_VER) && (_MSC_VER>=1400)
	::memcpy_s(dest,destSize,src,copyCount);
#else
	::memcpy( dest,src,copyCount );
#endif
}

/*---------------------------------------------------------------
						getch
---------------------------------------------------------------*/
int os::getch() MRPT_NO_THROWS
{
#ifdef MRPT_OS_WINDOWS
	return cin.get();
#else
	struct termios oldt,
	newt;
	int ch;
	tcgetattr( STDIN_FILENO, &oldt );
	newt = oldt;
	newt.c_lflag &= ~( ICANON | ECHO );
	tcsetattr( STDIN_FILENO, TCSANOW, &newt );
	ch = getchar();
	tcsetattr( STDIN_FILENO, TCSANOW, &oldt );
	return ch;
#endif
}

/*---------------------------------------------------------------
						kbhit
---------------------------------------------------------------*/
bool os::kbhit() MRPT_NO_THROWS
{
#ifdef MRPT_OS_WINDOWS
    #if defined(_MSC_VER) && (_MSC_VER>=1400)
        return ::_kbhit() != 0;
    #else
        return ::kbhit() != 0;
    #endif
#else
    return myKbhit();
#endif
}

/*---------------------------------------------------------------
						sleep
---------------------------------------------------------------*/
void mrpt::system::sleep( int time_ms ) MRPT_NO_THROWS
{
#ifdef MRPT_OS_WINDOWS
	Sleep( time_ms );
#else
	// We will wake up on signals: Assure the desired time has passed:
	CTicTac tictac;
	tictac.Tic();
	int timeLeft_ms = time_ms - (int)(tictac.Tac()*1000);
	while ( timeLeft_ms>0 )
	{
		usleep( timeLeft_ms * 1000 );
		timeLeft_ms = time_ms - (int)(tictac.Tac()*1000);
	}
#endif
}

/*---------------------------------------------------------------
						tokenize
---------------------------------------------------------------*/
char *mrpt::system::strtok( char *str, const char *strDelimit, char **context ) MRPT_NO_THROWS
{
#if defined(_MSC_VER) && (_MSC_VER>=1400)
	// Use a secure version in Visual Studio 2005:
	return ::strtok_s(str,strDelimit,context);
#else
	// Use standard version:
	return ::strtok(str,strDelimit);
#endif
}

/*---------------------------------------------------------------
						trim
---------------------------------------------------------------*/
std::string mrpt::system::trim(const std::string &str)
{
	if (str.empty())
	{
		return std::string();
	}
	else
	{
		size_t s = str.find_first_not_of(" ");
		size_t e = str.find_last_not_of(" ");
		if (s==std::string::npos || e==std::string::npos)
				return std::string();
		else	return str.substr( s, e-s+1);
	}
}

/*---------------------------------------------------------------
						tokenize
---------------------------------------------------------------*/
void  mrpt::system::tokenize(
	const std::string			&inString,
	const std::string			&inDelimiters,
	std::deque<std::string>	&outTokens ) MRPT_NO_THROWS
{
	char	*nextTok,*context;

	outTokens.clear();

	nextTok = strtok ((char*)inString.c_str(),inDelimiters.c_str(),&context);
	while (nextTok != NULL)
	{
		outTokens.push_back( std::string(nextTok) );
		nextTok = strtok (NULL,inDelimiters.c_str(),&context);
	};
}

/*---------------------------------------------------------------
						tokenize
---------------------------------------------------------------*/
void  mrpt::system::tokenize(
	const std::string			&inString,
	const std::string			&inDelimiters,
	std::vector<std::string>	&outTokens ) MRPT_NO_THROWS
{
	char	*nextTok,*context;

	outTokens.clear();

	nextTok = strtok ((char*)inString.c_str(),inDelimiters.c_str(),&context);
	while (nextTok != NULL)
	{
		outTokens.push_back( std::string(nextTok) );
		nextTok = strtok (NULL,inDelimiters.c_str(),&context);
	};
}

/*---------------------------------------------------------------
						isnan
---------------------------------------------------------------*/
bool  mrpt::system::isNaN(float  f) MRPT_NO_THROWS
{
#ifdef MRPT_OS_WINDOWS
    return 0!=_isnan(f);
#else
    return isnan(f);
#endif
}

/*---------------------------------------------------------------
						isnan
---------------------------------------------------------------*/
bool  mrpt::system::isNaN(double f) MRPT_NO_THROWS
{
#ifdef MRPT_OS_WINDOWS
    return 0!=_isnan(f);
#else
    return isnan(f);
#endif
}

/*---------------------------------------------------------------
						isFinite
---------------------------------------------------------------*/
bool  mrpt::system::isFinite(float f) MRPT_NO_THROWS
{
#ifdef MRPT_OS_WINDOWS
    return 0!=_finite(f);
#else
    return finite(f);
#endif
}


/*---------------------------------------------------------------
						isFinite
---------------------------------------------------------------*/
bool  mrpt::system::isFinite(double f) MRPT_NO_THROWS
{
#ifdef MRPT_OS_WINDOWS
    return 0!=_finite(f);
#else
    return finite(f);
#endif
}

#ifdef HAVE_LONG_DOUBLE
	/*---------------------------------------------------------------
							isnan
	---------------------------------------------------------------*/
	bool  mrpt::system::isNaN(long double f) MRPT_NO_THROWS
	{
	#ifdef MRPT_OS_WINDOWS
		return 0!=_isnan(f);
	#else
		return isnan(f);
	#endif
	}

	/*---------------------------------------------------------------
							isFinite
	---------------------------------------------------------------*/
	bool  mrpt::system::isFinite(long double f) MRPT_NO_THROWS
	{
	#ifdef MRPT_OS_WINDOWS
		return 0!=_finite(f);
	#else
		return finite(f);
	#endif
	}
#endif


/*---------------------------------------------------------------
						createThread
---------------------------------------------------------------*/
namespace mrpt
{
	namespace utils
	{
		struct TAuxThreadLaucher
		{
	#ifdef MRPT_OS_WINDOWS
			TAuxThreadLaucher() : win_sem(0,10)
			{
			}
	#else
			TAuxThreadLaucher() { };
	#endif
			void    (*ptrFunc) (void *);
			void    *param;

	#ifdef MRPT_OS_WINDOWS
			// These for windows only:
			unsigned long		myWindowsId;
			synch::CSemaphore	win_sem;
	#endif
		};

		void *auxiliary_thread_launcher_LIN( void *param )
		{
			try
			{
			TAuxThreadLaucher   *d = (TAuxThreadLaucher*) param;

			TAuxThreadLaucher	localCopy = *d;

	#ifdef MRPT_OS_WINDOWS
			// Signal that the thread has started:
			d->myWindowsId = (unsigned long)GetCurrentThreadId();
			d->win_sem.release();
			// Our parent thread will release the memory of "param".
	#else
			// LINUX: We have to free here this memory:
			delete d;
			d = NULL;
	#endif

			// Now start the user code:
			localCopy.ptrFunc( localCopy.param );
			}
			catch(std::exception &e)
			{
				std::cout << "Exception in [auxiliary_thread_launcher_LIN/WIN]!!!:\n" << e.what();
			}
			catch(...)
			{
				std::cout << "Untyped exception in [auxiliary_thread_launcher_LIN/WIN]!!!\n";
			}
			return NULL;
		}

		void auxiliary_thread_launcher_WIN( void *param )
		{
			auxiliary_thread_launcher_LIN(param);
		}
	} // end namespace
} // end namespace


// Frwd decl:
namespace mrpt {
	namespace system {
		TThreadHandle createThread_internal(void       ( *func )( void * ), void       *param );
	}
}


// This private version does not check: MRPT_HAS_NO_THREADS
mrpt::system::TThreadHandle mrpt::system::createThread_internal(
    void       ( *func )( void * ),
    void       *param
    )
{
	MRPT_TRY_START;

    TAuxThreadLaucher   *auxData=new TAuxThreadLaucher();
    auxData->ptrFunc = func;
    auxData->param = param;

#ifdef MRPT_OS_WINDOWS
	TThreadHandle		threadHandle;

	HANDLE h= (HANDLE)_beginthread( auxiliary_thread_launcher_WIN,0, auxData);
	if (h== ((HANDLE) -1))
	{
		delete auxData;
		THROW_EXCEPTION("Error creating new thread");
	}

	threadHandle.hThread = h;

	// Wait until the thread starts so we know its ID:
	auxData->win_sem.waitForSignal();
	threadHandle.idThread = auxData->myWindowsId;

	delete auxData; auxData = NULL;

	return threadHandle;

#else
	TThreadHandle		threadHandle;

    pthread_t   newThreadId;
    int iRet = pthread_create( &newThreadId,NULL,auxiliary_thread_launcher_LIN,auxData);
    ASSERT_(iRet==0);

	threadHandle.idThread = (unsigned long)newThreadId;
	return threadHandle;
#endif

	MRPT_TRY_END;
}



mrpt::system::TThreadHandle mrpt::system::createThread(
    void       ( *func )( void * ),
    void       *param
    )
{
	MRPT_TRY_START;

#if MRPT_HAS_NO_THREADS
	THROW_EXCEPTION("MRPT has been compiled with threads disabled!! If this is a BABEL application, please create threads using PERMANENT MONITORS.")
#endif

	return createThread_internal(func,param);

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						joinThread
---------------------------------------------------------------*/
void mrpt::system::joinThread( const TThreadHandle &threadHandle )
{
#ifdef MRPT_OS_WINDOWS
	DWORD ret = WaitForSingleObject( (HANDLE) threadHandle.hThread , INFINITE );
	if (ret!=WAIT_OBJECT_0)
		cerr << "[mrpt::system::joinThread] Error waiting for thread completion!" << endl;
#else
	pthread_join(threadHandle.idThread, NULL);
#endif
}

/*---------------------------------------------------------------
						getCurrentThreadId
---------------------------------------------------------------*/
unsigned long mrpt::system::getCurrentThreadId() MRPT_NO_THROWS
{
#ifdef MRPT_OS_WINDOWS
	return static_cast<unsigned long>(GetCurrentThreadId());
#else
	return static_cast<unsigned long>(pthread_self());
#endif
}

/*---------------------------------------------------------------
					changeThreadPriority
---------------------------------------------------------------*/
void MRPTDLLIMPEXP mrpt::system::changeThreadPriority(
	const TThreadHandle &threadHandle,
	TThreadPriority priority )
{
#ifdef MRPT_OS_WINDOWS
	SetThreadPriority( threadHandle.hThread, priority);
#else

#endif
}

/*---------------------------------------------------------------
					changeCurrentProcessPriority
---------------------------------------------------------------*/
void MRPTDLLIMPEXP mrpt::system::changeCurrentProcessPriority( TProcessPriority priority  )
{
#ifdef MRPT_OS_WINDOWS
	DWORD dwPri;
	switch (priority)
	{
	case ppIdle:	dwPri = IDLE_PRIORITY_CLASS; break;
	case ppNormal:	dwPri = NORMAL_PRIORITY_CLASS; break;
	case ppHigh:	dwPri = HIGH_PRIORITY_CLASS; break;
	case ppVeryHigh: dwPri= REALTIME_PRIORITY_CLASS; break;
	default:
		THROW_EXCEPTION("Invalid priority value");
	}
	SetPriorityClass( GetCurrentProcess(), dwPri );
#else

#endif
}

/*---------------------------------------------------------------
						os::fprintf
---------------------------------------------------------------*/
int os::fprintf(FILE *fil, const char *frm, ...) MRPT_NO_THROWS
{
	int			result;
	va_list		ap;
	va_start(ap, frm);

#if defined(_MSC_VER) && (_MSC_VER>=1400)
	// Use a secure version in Visual Studio 2005:
	result = ::vfprintf_s(fil, frm, ap);

#else
	// Use standard version:
	result = ::vfprintf(fil, frm, ap);
#endif

	va_end (ap);
	return result;
}

// Hack for Visual C++/Studio:
//  From http://www.gamedev.net/community/forums/topic.asp?topic_id=310888
#if defined(_MSC_VER)
#include <stdio.h>
#include <malloc.h>
	int vfscanf(FILE* file, const char *format, va_list argPtr) MRPT_NO_THROWS
	{
		// http://www.codeguru.com/Cpp/Cpp/string/comments.php/c5631/?thread=61724
		// Get an upper bound for the # of args
		size_t count = 0;
		const char* p = format;

		while(1)
		{
			char c = *(p++);
			if (c == 0)
				break;

			if (c == '%' && (p[0] != '*' && p[0] != '%'))
				++count;
		}

		if (count <= 0)
			return 0;

		int result;

		// copy stack pointer
		_asm
		{
			mov esi, esp;
		}

		// push variable parameters pointers on stack
		for (int i = ((int)count) - 1; i >= 0; --i)
		{
			_asm
			{
				mov eax, dword ptr[i];
				mov ecx, dword ptr [argPtr];
				mov edx, dword ptr [ecx+eax*4];
				push edx;
			}
		}

		int stackAdvance = (int)((2 + count) * 4);

		_asm
		{
			// now push on the fixed params
			mov eax, dword ptr [format];
			push eax;
			mov eax, dword ptr [file];
			push eax;

			// call fscanf, and more the result in to result
			call dword ptr [fscanf];
			mov result, eax;

			// restore stack pointer
			mov eax, dword ptr[stackAdvance];
			add esp, eax;
			//mov esp, esi;
		}

		return result;
	}
#endif

/*---------------------------------------------------------------
						mrpt::system::fscanf
---------------------------------------------------------------*/
int os::fscanf(FILE *fil, const char *frm, ...) MRPT_NO_THROWS
{
	int			result;
	va_list		ap;
	va_start(ap, frm);

#if defined(_MSC_VER)
	// Custom hacked implementation!! (See above)
	result = vfscanf(fil, frm, ap);
#else
	// Use standard version:
	result = ::vfscanf(fil, frm, ap);
#endif

	va_end (ap);
	return result;
}


/*---------------------------------------------------------------
					mrpt::system::getCurrentThreadTimes
---------------------------------------------------------------*/
void mrpt::system::getCurrentThreadTimes(
	time_t			&creationTime,
	time_t			&exitTime,
	double			&cpuTime )
{
	MRPT_TRY_START;

#ifdef MRPT_OS_WINDOWS
	FILETIME	timCreat,timExit, timKernel, timUser;
	uint64_t	t;

	HANDLE threadHandle;

#if !defined(HAVE_OPENTHREAD) // defined(_MSC_VER) && (_MSC_VER<1300)
	// In MSVC6/GCC the ID is just the HANDLE:
	threadHandle = (HANDLE) mrpt::system::getCurrentThreadId(); //threadId;
#else
	// Get the handle from the ID:
	threadHandle = OpenThread( READ_CONTROL | THREAD_QUERY_INFORMATION, FALSE,  mrpt::system::getCurrentThreadId() );  // threadId);
	if (!threadHandle)	 THROW_EXCEPTION("Cannot open the thread with the given 'threadId'");
#endif

	if (!GetThreadTimes( threadHandle , &timCreat, &timExit, &timKernel, &timUser ))
	{
		CloseHandle(threadHandle);
		THROW_EXCEPTION("Error accessing thread times!");
	}

#if !defined(HAVE_OPENTHREAD) // _MSC_VER) && (_MSC_VER>=1300)
	// From OpenThread...
	CloseHandle(threadHandle);
#endif

	// Formula is derived from:
	//  http://support.microsoft.com/kb/167296
	t = (((uint64_t)timCreat.dwHighDateTime) << 32) | timCreat.dwLowDateTime;
	creationTime = (t - 116444736000000000ULL)/10000000;

	t = (((uint64_t)timExit.dwHighDateTime) << 32) | timExit.dwLowDateTime;
	exitTime = (t - 116444736000000000ULL)/10000000;

	// CPU time is user+kernel:
	int64_t	t1 = (((uint64_t)timKernel.dwHighDateTime) << 32) | timKernel.dwLowDateTime;
	int64_t	t2 = (((uint64_t)timUser.dwHighDateTime) << 32) | timUser.dwLowDateTime;

	cpuTime = ((double)(t1+t2)) * 100e-9;	// FILETIME counts intervals of 100ns

#else
	// Unix:
#	ifdef HAVE_GETTID
		pid_t 	id = gettid();
#	else
		// gettid is:
		//  186 in 64bit
		//  224 in 32bit
		#if MRPT_WORD_SIZE==64
			pid_t 	id = (long int)syscall(186);
		#elif MRPT_WORD_SIZE==32
			pid_t 	id = (long int)syscall(224);
		#else
			#error MRPT_WORD_SIZE must be 32 or 64.
		#endif
#	endif

	// (JL) Refer to: /usr/src/linux/fs/proc/array.c
	long unsigned 	tms_utime=0, tms_stime=0;
	ifstream is(format("/proc/self/task/%i/stat", id).c_str() );

	if (is.is_open())
	{
		string s;
		getline(is,s);

		size_t idx = s.find(")");

		if (idx!=string::npos)
		{
			vector_string	tokens;
			mrpt::system::tokenize( string(s.c_str()+idx+1)," ",tokens);

			if (tokens.size()>=13)
			{
				sscanf(tokens[11].c_str(), "%lu" ,&tms_utime);
				sscanf(tokens[12].c_str(), "%lu", &tms_stime);
			}
		}
	}

	// Compute cpuTime:
	double clockTicksPerSecond = (double)sysconf(_SC_CLK_TCK);
	if (clockTicksPerSecond>0)
		cpuTime = (tms_utime + tms_stime) / clockTicksPerSecond;

#endif

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					mrpt::system::pause
---------------------------------------------------------------*/
void mrpt::system::pause() MRPT_NO_THROWS
{
	std::cout << "Press any key to continue..." << std::endl;
	os::getch();
}


std::string mrpt::system::getcwd()
{
	MRPT_TRY_START

#ifdef MRPT_OS_WINDOWS
	char	auxBuf[MAX_PATH] = "";
	if(!::GetCurrentDirectoryA(sizeof(auxBuf) - 1, auxBuf))
		THROW_EXCEPTION("Error getting current working directory!");
	return std::string(auxBuf);

#else
	size_t size = 100;
	for (;;)
	{
		char *buffer = (char *) malloc(size);
		if (::getcwd(buffer, size) == buffer)
		{
			std::string s(buffer);
			free(buffer);
			return s;
		}
		free (buffer);
		if (errno != ERANGE)
			THROW_EXCEPTION("Error getting current working directory!");
		size *= 2;
	}
#endif

	MRPT_TRY_END
}

/*---------------------------------------------------------------
					clearConsole
---------------------------------------------------------------*/
void mrpt::system::clearConsole()
{
#ifdef MRPT_OS_WINDOWS
	int ret=::system("cls");
#else
	int ret=::system("clear");
#endif
	if (ret)
		cerr << "[mrpt::system::clearConsole] Error invoking 'clear screen' " << endl;
}

/*---------------------------------------------------------------
					getTempFileName
  ---------------------------------------------------------------*/
std::string mrpt::system::getTempFileName()
{
#ifdef MRPT_OS_WINDOWS
	static int	uniq = 0;
	char	TMP_PATH[ MAX_PATH ];
	char	tmpPath[ MAX_PATH ];
	GetTempPathA(MAX_PATH,tmpPath);
	GetTempFileNameA(tmpPath,"mrpt", ++uniq, TMP_PATH);
	return std::string( TMP_PATH );
#else
	char tmp[] = "/tmp/mrpt_tempXXXXXX";
	int fd;
	fd = mkstemp(tmp);
	ASSERT_(fd>=0);
	close(fd);
	return std::string( tmp );
#endif
}

/*---------------------------------------------------------------
					loadBinaryFile
  ---------------------------------------------------------------*/
bool mrpt::system::loadBinaryFile( vector_byte &out_data, const std::string &fileName )
{
	try
	{
		CFileInputStream	fi(fileName);
		size_t  N = fi.getTotalBytesCount();

		out_data.resize(N);
		if (N)
		{
			size_t NN = fi.ReadBuffer( &out_data[0], N);
			return NN==N;
		}
		else return true;
	}
	catch(...) { return false; }
}

/*---------------------------------------------------------------
					vectorToBinaryFile
  ---------------------------------------------------------------*/
bool mrpt::system::vectorToBinaryFile( const vector_byte &vec, const std::string &fileName )
{
	try
	{
		mrpt::utils::CFileOutputStream	of(fileName);
		of.WriteBuffer( &vec[0], sizeof(vec[0])*vec.size() );
		return true;
	}
	catch(...) { return false; }
}

/*---------------------------------------------------------------
					_strtoll
  ---------------------------------------------------------------*/
int64_t mrpt::system::os::_strtoll(const char *nptr, char **endptr, int base)
{
#ifdef MRPT_OS_WINDOWS
	return (int64_t) ::strtol( nptr, endptr, base );
#else
	return (int64_t) ::strtoll( nptr, endptr, base );
#endif

}

/*---------------------------------------------------------------
					_strtoull
  ---------------------------------------------------------------*/
uint64_t mrpt::system::os::_strtoull(const char *nptr, char **endptr, int base)
{
#ifdef MRPT_OS_WINDOWS
	return (uint64_t) ::strtoul( nptr, endptr, base );
#else
	return (uint64_t) ::strtoull( nptr, endptr, base );
#endif

}


/** Renames a file - If the target path is different and the filesystem allows it, it will be moved to the new location.
  * \return false on any error. In that case, if a pointer to a receiver string is passed in error_msg, a description of the error is saved there.
  */
bool mrpt::system::renameFile( const string &oldFileName, const string &newFileName,  std::string *error_msg )
{
	bool ret_err = 0==rename( oldFileName.c_str(), newFileName.c_str() );

	if (error_msg)
	{
		if (ret_err)
			*error_msg= strerror(errno);
		else
			*error_msg="";
	}

	return ret_err;
}

