/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 


#include <mrpt/vision/CImageConvolution.h>
using namespace mrpt::vision;
using namespace std;

/*-------------------------------------------------------------
						convolutionHorizontal
-------------------------------------------------------------*/
void  CImageConvolution::convolutionHorizontal(
		CMRPTImageFloat				&inImg,
		CMRPTImageFloat				&outImg,
		CGaussianConvolutionKernel	&Kernel,
		TImageROI					ROI)
{

	// Version using ROI
	MRPT_TRY_START;

	vector_float					*kernel = &Kernel.m_kernel;
	float							*ptrRow;
	register float					*ptrOut;
	register float					sum;
	unsigned int					kernel_size = (unsigned int)kernel->size();
	register unsigned int			radius = (kernel_size-1) / 2;
	register unsigned int			ncols = inImg.getWidth();
	register unsigned int			nrows = inImg.getHeight();
	register unsigned int			i, j;
	std::vector<float>::iterator	it;

	ASSERT_( &inImg != &outImg );

	unsigned int idy_in;
	unsigned int input_xIni, input_xEnd, input_yIni, input_yEnd;
	unsigned int nCols_out, nRows_out;
	unsigned int nCols_out_radius;

	// Is there a ROI?
	if ( !(ROI.xMin == 0 && ROI.xMax == 0 && ROI.yMin == 0 && ROI.yMax == 0) )
	{
		// There is a ROI
		// Check ROI (z-dimension is ignored):
		ASSERT_( (ROI.xMin < ROI.xMax) && (ROI.yMin < ROI.yMax) );

		// Set indexes for the input image
		input_xIni	= (unsigned int)ROI.xMin;
		input_xEnd	= (unsigned int)ROI.xMax;
		input_yIni	= (unsigned int)ROI.yMin;
		input_yEnd	= (unsigned int)ROI.yMax;

		// Output image size
		nCols_out	= (unsigned int)(ROI.xMax - ROI.xMin);
		nRows_out	= (unsigned int)(ROI.yMax - ROI.yMin);

	} // end if
	else
	{
		// There is no ROI
		// Set indexes for the input image
		input_xIni	= 0;
		input_xEnd	= ncols;
		input_yIni	= 0;
		input_yEnd	= nrows;

		// Output image size
		nCols_out	= ncols;
		nRows_out	= nrows;

	} // end else
	nCols_out_radius = nCols_out - radius;

	// Prepare the output image:
	outImg.setSize( nCols_out, nRows_out );

	// If output image size is higher than the kernel
	if ( nCols_out > kernel_size )
	{
		// Two defined indexes: 'idy_in' for the input image and 'j' for the output one
		// For each row:
		for ( idy_in = input_yIni, j = 0 ; idy_in < input_yEnd ; idy_in++, j++ )
		{
			ptrOut = outImg(0,j);

			// Convolve firsts columns with kernel:
			for ( i = 0 ; i < radius ; i++ )
			{
				sum = 0.0;
				ptrRow = inImg( input_xIni, idy_in );
				for ( it = kernel->begin() + radius - i; it != kernel->end(); it++ )
				{
					sum += (*ptrRow++) * (*it);
				}
				*ptrOut++ = sum;
			}
			// Convolve middle columns with kernel:
			for ( i = radius ; i < nCols_out_radius ; i++ )
			{
				sum = 0.0;
				ptrRow = inImg( i-radius + input_xIni, idy_in );
				for (it=kernel->begin(); it!=kernel->end(); it++)
				{
					sum += (*ptrRow++) * (*it);
				}
				*ptrOut++ = sum;
			}
			// Convolve lasts columns with kernel:
			for (i = nCols_out_radius ; i < nCols_out ; i++)
			{
				sum = 0.0;
				ptrRow = inImg( i-radius + input_xIni, idy_in );
				for (it=kernel->begin(); it!=kernel->end()-radius+(nCols_out-i-1); it++)
				{
					sum += (*ptrRow++) * (*it);
				}
				*ptrOut++ = sum;
			}
		} // end for each row
	}
	else	//This case is used when the inImg is quite short (width < kernel_size)
	{
		for (idy_in = input_yIni, j = 0 ; j < input_yEnd ; idy_in++, j++)
		{
			ptrOut = outImg(0,j);

			// Convolve firsts columns with kernel:
			for (i = 0; i < nCols_out_radius ; i++)
			{
				sum = 0.0;
				ptrRow = inImg( input_xIni, idy_in );
				for (it=kernel->begin()+radius-i; it!=kernel->end(); it++)
				{
					sum += (*ptrRow++) * (*it);
				}
				*ptrOut++ = sum;
			}
			// Convolve middle columns with kernel:
			for (i = nCols_out_radius; i < radius ; i++)
			{
				sum = 0.0;
				ptrRow = inImg( i-nCols_out_radius+input_xIni, idy_in );
				for (it=kernel->begin()+radius-i; it!=kernel->end()-radius+(nCols_out-i-1); it++)
				{
					sum += (*ptrRow++) * (*it);
				}
				*ptrOut++ = sum;
			}
			// Convolve lasts columns with kernel:
			for (i=radius ; i < nCols_out ; i++)
			{
				sum = 0.0;
				ptrRow = inImg( i-radius+input_xIni, idy_in );
				for (it=kernel->begin(); it!=kernel->end()-radius+(nCols_out-i-1); it++)
				{
					sum += (*ptrRow++) * (*it);
				}
				*ptrOut++ = sum;
			}
		} // end for each row
	}
	MRPT_TRY_END;

	//MRPT_UNUSED_PARAM(ROI);
	//try
	//{
	//	vector_float					*kernel = &Kernel.m_kernel;
	//	float							*ptrRow;
	//	register float					*ptrOut;
	//	register float					sum;
	//	unsigned int					kernel_size = (unsigned int)kernel->size();
	//	register unsigned int			radius = (kernel_size-1)  / 2;
	//	register unsigned int			ncols = inImg.getWidth();
	//	register unsigned int			nrows = inImg.getHeight();
	//	register unsigned int			i, j;
	//	unsigned int					nCols_radius = ncols - radius;
	//	std::vector<float>::iterator	it; //, itIni,itEnd;

	//	ASSERT_( &inImg != &outImg );

	//	// Prepare the output image:
	//	outImg.setSize( inImg.getWidth(), inImg.getHeight() );

	//	// For each row:
	//	if (inImg.getWidth()>kernel_size)
	//	{
	//		for (j = 0 ; j < nrows ; j++)
	//		{
	//			ptrOut = outImg(0,j);

	//			// Convolve firsts columns with kernel:
	//			for (i=0 ; i < radius ; i++)
	//			{
	//				sum = 0.0;
	//				ptrRow = inImg(0,j);
	//				for (it=kernel->begin()+radius-i; it!=kernel->end(); it++)
	//				{
	//					sum += (*ptrRow++) * (*it);
	//				}
	//				*ptrOut++ = sum;
	//			}
	//			// Convolve middle columns with kernel:
	//			for (i = radius ; i < nCols_radius ; i++)
	//			{
	//				sum = 0.0;
	//				ptrRow = inImg(i-radius,j);
	//				for (it=kernel->begin(); it!=kernel->end(); it++)
	//				{
	//					sum += (*ptrRow++) * (*it);
	//				}
	//				*ptrOut++ = sum;
	//			}
	//			// Convolve lasts columns with kernel:
	//			for (i=nCols_radius ; i < ncols ; i++)
	//			{
	//				sum = 0.0;
	//				ptrRow = inImg(i-radius,j);
	//				for (it=kernel->begin(); it!=kernel->end()-radius+(ncols-i-1); it++)
	//				{
	//					sum += (*ptrRow++) * (*it);
	//				}
	//				*ptrOut++ = sum;
	//			}
	//		} // end for each row
	//	}
	//	else	//This case is used when the inImg is quite short (width < kernel_size)
	//	{
	//		for (j = 0 ; j < nrows ; j++)
	//		{
	//			ptrOut = outImg(0,j);

	//			// Convolve firsts columns with kernel:
	//			for (i=0 ; i < nCols_radius ; i++)
	//			{
	//				sum = 0.0;
	//				ptrRow = inImg(0,j);
	//				for (it=kernel->begin()+radius-i; it!=kernel->end(); it++)
	//				{
	//					sum += (*ptrRow++) * (*it);
	//				}
	//				*ptrOut++ = sum;
	//			}
	//			// Convolve middle columns with kernel:
	//			for (i=nCols_radius ; i < radius ; i++)
	//			{
	//				sum = 0.0;
	//				ptrRow = inImg(i-nCols_radius,j);
	//				for (it=kernel->begin()+radius-i; it!=kernel->end()-radius+(ncols-i-1); it++)
	//				{
	//					sum += (*ptrRow++) * (*it);
	//				}
	//				*ptrOut++ = sum;
	//			}
	//			// Convolve lasts columns with kernel:
	//			for (i=radius ; i < ncols ; i++)
	//			{
	//				sum = 0.0;
	//				ptrRow = inImg(i-radius,j);
	//				for (it=kernel->begin(); it!=kernel->end()-radius+(ncols-i-1); it++)
	//				{
	//					sum += (*ptrRow++) * (*it);
	//				}
	//				*ptrOut++ = sum;
	//			}
	//		} // end for each row
	//	}
	//}
	//catch(std::exception &e)
	//{
	//	THROW_STACKED_EXCEPTION(e);
	//}
	//catch(...)
	//{
	//	THROW_EXCEPTION("Unexpected runtime error!");
	//}

}

/*-------------------------------------------------------------
					convolutionVertical
-------------------------------------------------------------*/
void  CImageConvolution::convolutionVertical(
		CMRPTImageFloat				&inImg,
		CMRPTImageFloat				&outImg,
		CGaussianConvolutionKernel	&Kernel,
		TImageROI					ROI)
{
	MRPT_TRY_START;

	vector_float					*kernel = &Kernel.m_kernel;
	float							*ptrCol;
	register float					*ptrOut;
	register float					sum;
	unsigned int					kernel_size = (unsigned int)kernel->size();
	register unsigned int			radius = (kernel_size-1)  / 2;
	register unsigned int			ncols = inImg.getWidth();
	register unsigned int			nrows = inImg.getHeight();
	register unsigned int			i, j;
	std::vector<float>::iterator	it; //, itIni,itEnd;

	ASSERT_( &inImg != &outImg );

	unsigned int idx_in;
	unsigned int input_xIni, input_xEnd, input_yIni, input_yEnd;
	unsigned int nCols_out, nRows_out;
	unsigned int nRows_out_radius;

	// Is there a ROI?
	if ( !(ROI.xMin == 0 && ROI.xMax == 0 && ROI.yMin == 0 && ROI.yMax == 0) )
	{
		// There is a ROI
		// Check ROI (z-dimension is ignored):
		ASSERT_( (ROI.xMin < ROI.xMax) && (ROI.yMin < ROI.yMax) );

		// Set indexes for the input image
		input_xIni	= (unsigned int)ROI.xMin;
		input_xEnd	= (unsigned int)ROI.xMax;
		input_yIni	= (unsigned int)ROI.yMin;
		input_yEnd	= (unsigned int)ROI.yMax;

		// Output image size
		nCols_out	= (unsigned int)(ROI.xMax - ROI.xMin);
		nRows_out	= (unsigned int)(ROI.yMax - ROI.yMin);

	} // end if
	else
	{
		// There is no ROI
		// Set indexes for the input image
		input_xIni	= 0;
		input_xEnd	= ncols;
		input_yIni	= 0;
		input_yEnd	= nrows;

		// Output image size
		nCols_out	= ncols;
		nRows_out	= nrows;

	} // end else
	nRows_out_radius = nRows_out - radius;

	// Prepare the output image:
	outImg.setSize( nCols_out, nRows_out );

	if ( nRows_out > kernel_size )
	{
		// For each col:
		// Two indexes, 'idx_in' for the input image and 'i' for the output one
		for (idx_in = input_xIni, i = 0 ; idx_in < input_xEnd ; idx_in++, i++)
		{
			ptrOut = outImg( i, 0 );

			// Convolve firsts rows with kernel:
			for (j=0 ; j < radius ; j++)
			{
				sum = 0.0;
				ptrCol = inImg( idx_in, input_yIni );
				for (it=kernel->begin()+radius-j; it!=kernel->end(); it++)
				{
					sum += (*ptrCol) * (*it);
					ptrCol += ncols;
				}
				*ptrOut = sum;
				ptrOut += nCols_out;
			}
			// Convolve middle rows with kernel:
			for (j = radius ; j < nRows_out_radius ; j++)
			{
				sum = 0.0;
				ptrCol = inImg( idx_in, j - radius + input_yIni );
				for (it=kernel->begin(); it!=kernel->end(); it++)
				{
					sum += *ptrCol * (*it);
					ptrCol += ncols;
				}
				*ptrOut = sum;
				ptrOut+= nCols_out;
			}
			// Convolve lasts rows with kernel:
			for (j = nRows_out_radius ; j < nRows_out ; j++)
			{
				sum = 0.0;
				ptrCol = inImg( idx_in, j - radius + input_yIni);
				for (it=kernel->begin(); it!=kernel->end()-radius+(nRows_out-j-1); it++)
				{
					sum += (*ptrCol) * (*it);
					ptrCol+= ncols;
				}
				*ptrOut = sum;
				ptrOut+= nCols_out;
			}
		} // end for each col
	}
	else		//This case is used when the inImg is quite short (height < kernel_size)
	{
		// For each col:
		for (idx_in = input_xIni, i = 0 ; idx_in < input_xEnd; idx_in++, i++)
		{
			ptrOut = outImg(i,0);

			// Convolve firsts rows with kernel:
			for (j=0 ; j < nRows_out_radius ; j++)
			{
				sum = 0.0;
				ptrCol = inImg( idx_in, input_yIni );
				for (it=kernel->begin()+radius-j; it!=kernel->end(); it++)
				{
					sum += (*ptrCol) * (*it);
					ptrCol += ncols;
				}
				*ptrOut = sum;
				ptrOut += nCols_out;
			}
			// Convolve middle rows with kernel:
			for (j=nRows_out_radius ; j < radius ; j++)
			{
				sum = 0.0;
				ptrCol = inImg( idx_in, j-nRows_out_radius+input_yIni);
				for (it=kernel->begin()+radius-j; it!=kernel->end()-radius+(nRows_out-j-1); it++)
				{
					sum += *ptrCol * (*it);
					ptrCol += ncols;
				}
				*ptrOut = sum;
				ptrOut+= nCols_out;
			}
			// Convolve lasts rows with kernel:
			for (j=radius ; j < nrows ; j++)
			{
				sum = 0.0;
				ptrCol = inImg(idx_in,j-radius+input_yIni);
				for (it=kernel->begin(); it!=kernel->end()-radius+(nRows_out-j-1); it++)
				{
					sum += (*ptrCol) * (*it);
					ptrCol+= ncols;
				}
				*ptrOut = sum;
				ptrOut+= nCols_out;
			}
		} // end for each col
	}

	MRPT_TRY_END;

	//MRPT_UNUSED_PARAM(ROI);

	//try
	//{
	//	vector_float					*kernel = &Kernel.m_kernel;
	//	float							*ptrCol;
	//	register float					*ptrOut;
	//	register float					sum;
	//	unsigned int					kernel_size = (unsigned int)kernel->size();
	//	register unsigned int			radius = (kernel_size-1)  / 2;
	//	register unsigned int			ncols = inImg.getWidth();
	//	register unsigned int			nrows = inImg.getHeight();
	//	register unsigned int			i, j;
	//	unsigned int					nRows_radius = nrows - radius;
	//	std::vector<float>::iterator	it; //, itIni,itEnd;

	//	ASSERT_( &inImg != &outImg );

	//	// Prepare the output image:
	//	outImg.setSize( inImg.getWidth(), inImg.getHeight() );

	//	if (inImg.getHeight()>kernel_size)
	//	{
	//		// For each col:
	//		for (i = 0 ; i < ncols ; i++)
	//		{
	//			ptrOut = outImg(i,0);

	//			// Convolve firsts rows with kernel:
	//			for (j=0 ; j < radius ; j++)
	//			{
	//				sum = 0.0;
	//				ptrCol = inImg(i,0);
	//				for (it=kernel->begin()+radius-j; it!=kernel->end(); it++)
	//				{
	//					sum += (*ptrCol) * (*it);
	//					ptrCol += ncols;
	//				}
	//				*ptrOut = sum;
	//				ptrOut += ncols;
	//			}
	//			// Convolve middle rows with kernel:
	//			for (j=radius ; j < nRows_radius ; j++)
	//			{
	//				sum = 0.0;
	//				ptrCol = inImg(i,j-radius);
	//				for (it=kernel->begin(); it!=kernel->end(); it++)
	//				{
	//					sum += *ptrCol * (*it);
	//					ptrCol += ncols;
	//				}
	//				*ptrOut = sum;
	//				ptrOut+= ncols;
	//			}
	//			// Convolve lasts rows with kernel:
	//			for (j=nRows_radius ; j < nrows ; j++)
	//			{
	//				sum = 0.0;
	//				ptrCol = inImg(i,j-radius);
	//				for (it=kernel->begin(); it!=kernel->end()-radius+(nrows-j-1); it++)
	//				{
	//					sum += (*ptrCol) * (*it);
	//					ptrCol+= ncols;
	//				}
	//				*ptrOut = sum;
	//				ptrOut+= ncols;
	//			}
	//		} // end for each col
	//	}
	//	else		//This case is used when the inImg is quite short (height < kernel_size)
	//	{
	//		// For each col:
	//		for (i = 0 ; i < ncols ; i++)
	//		{
	//			ptrOut = outImg(i,0);

	//			// Convolve firsts rows with kernel:
	//			for (j=0 ; j < nRows_radius ; j++)
	//			{
	//				sum = 0.0;
	//				ptrCol = inImg(i,0);
	//				for (it=kernel->begin()+radius-j; it!=kernel->end(); it++)
	//				{
	//					sum += (*ptrCol) * (*it);
	//					ptrCol += ncols;
	//				}
	//				*ptrOut = sum;
	//				ptrOut += ncols;
	//			}
	//			// Convolve middle rows with kernel:
	//			for (j=nRows_radius ; j < radius ; j++)
	//			{
	//				sum = 0.0;
	//				ptrCol = inImg(i,j-nRows_radius);
	//				for (it=kernel->begin()+radius-j; it!=kernel->end()-radius+(nrows-j-1); it++)
	//				{
	//					sum += *ptrCol * (*it);
	//					ptrCol += ncols;
	//				}
	//				*ptrOut = sum;
	//				ptrOut+= ncols;
	//			}
	//			// Convolve lasts rows with kernel:
	//			for (j=radius ; j < nrows ; j++)
	//			{
	//				sum = 0.0;
	//				ptrCol = inImg(i,j-radius);
	//				for (it=kernel->begin(); it!=kernel->end()-radius+(nrows-j-1); it++)
	//				{
	//					sum += (*ptrCol) * (*it);
	//					ptrCol+= ncols;
	//				}
	//				*ptrOut = sum;
	//				ptrOut+= ncols;
	//			}
	//		} // end for each col
	//	}
	//}
	//catch(std::exception &e)
	//{
	//	THROW_STACKED_EXCEPTION(e);
	//}
	//catch(...)
	//{
	//	THROW_EXCEPTION("Unexpected runtime error!");
	//}

}

/*-------------------------------------------------------------
					convolutionHorzVert
-------------------------------------------------------------*/
void  CImageConvolution::convolutionHorzVert(
		CMRPTImageFloat				&inImg,
		CMRPTImageFloat				&outImg,
		CGaussianConvolutionKernel	&Kernel)
{
	CMRPTImageFloat		temp(inImg.getWidth(), inImg.getHeight() );

	convolutionVertical(inImg, temp, Kernel );
	convolutionHorizontal(temp, outImg, Kernel );
}

/*-------------------------------------------------------------
					convolutionSeparate
-------------------------------------------------------------*/
/**** FAMD ******/
// ROI added
void  CImageConvolution::convolutionSeparate(
				CMRPTImageFloat				&inImg,
				CMRPTImageFloat				&outImg,
				CGaussianConvolutionKernel	&kernelHorz,
				CGaussianConvolutionKernel	&kernelVert,
				TImageROI					ROI)
{
	CMRPTImageFloat		temp(inImg.getWidth(), inImg.getHeight() );

	convolutionHorizontal(inImg,temp, kernelHorz, ROI );
	convolutionVertical(temp,outImg, kernelVert, ROI );
}
/***** END FAMD *****/
/******** FAMD *******/
/*-------------------------------------------------------------
					convolutionGeneric
-------------------------------------------------------------*/
void  CImageConvolution::convolutionGeneric(
				const CMRPTImageFloat	&inImg,
				CMRPTImageFloat			&outImg,
				CMatrix					&kernel,
				TImageROI				myROI)
{
	MRPT_TRY_START;

	int kernelLen = (int)kernel.getColCount();
	int	radius = (int)(kernel.getColCount()-1)/2;

	ASSERT_( kernel.getColCount() == kernel.getRowCount() );

	ASSERT_( &inImg != &outImg );

	unsigned int nRows = inImg.getHeight();
	unsigned int nCols = inImg.getWidth();
	unsigned int x_iniR1, x_endR1, y_iniR1, y_endR1;	// ROI limits
	unsigned int x_iniR2, x_endR2, y_iniR2, y_endR2;	// ROI limits
	unsigned int x_iniR3, x_endR3, y_iniR3, y_endR3;	// ROI limits
	unsigned int x_iniR4, x_endR4, y_iniR4, y_endR4;	// ROI limits
	unsigned int x_iniR5, x_endR5, y_iniR5, y_endR5;	// ROI limits
	unsigned int outImg_cols, outImg_rows;
	unsigned int i, j, y;							// Counters
	bool		 ROI = false;

	// Prepare the output image
	if ((myROI.xMin == 0) && (myROI.xMax == 0) && (myROI.yMin == 0) && (myROI.yMax == 0))
	{	// There is no ROI. We divide the image in five sections for speeding
		// Output image has the same sections
		x_iniR1 = radius;
		x_endR1 = nCols-radius;
		y_iniR1 = radius;
		y_endR1 = nRows-radius;

		x_iniR2 = 0;
		x_endR2 = nCols;
		y_iniR2 = 0;
		y_endR2 = radius;

		x_iniR3 = 0;
		x_endR3 = nCols;
		y_iniR3 = nRows-radius;
		y_endR3 = nRows;

		x_iniR4 = 0;
		x_endR4 = radius;
		y_iniR4 = radius;
		y_endR4 = nRows-radius;

		x_iniR5 = nCols-radius;
		x_endR5 = nCols;
		y_iniR5 = radius;
		y_endR5 = nRows-radius;

		outImg_cols = nCols;
		outImg_rows = nRows;
	}
	else
	{	// There is a ROI (TODO: Check parameters)
		// Define sections
		// Input image
		ROI = true;
		x_iniR1 = max((unsigned int)radius, (unsigned int)myROI.xMin);
		x_endR1 = min(nCols-(unsigned int)radius, (unsigned int)myROI.xMax);
		y_iniR1 = max((unsigned int)radius, (unsigned int)myROI.yMin);
		y_endR1 = min(nRows-(unsigned int)radius, (unsigned int)myROI.yMax);

		x_iniR2 = x_endR2 = y_iniR2 = y_endR2 = 0;
		x_iniR3 = x_endR3 = y_iniR3 = y_endR3 = 0;
		x_iniR4 = x_endR4 = y_iniR4 = y_endR4 = 0;
		x_iniR5 = x_endR5 = y_iniR5 = y_endR5 = 0;

		// Define necessary sections
		// R2: top
		if ( myROI.yMin < (unsigned int)radius )
		{
			x_iniR2 = (unsigned int)myROI.xMin;
			x_endR2 = (unsigned int)myROI.xMax;
			y_iniR2 = (unsigned int)myROI.yMin;
			y_endR2 = radius;
		}
		// R3: bottom
		if ( myROI.yMax > nRows - (unsigned int)radius )
		{
			x_iniR3 = (unsigned int)myROI.xMin;
			x_endR3 = (unsigned int)myROI.xMax;
			y_iniR3 = radius;
			y_endR3 = (unsigned int)myROI.yMax;
		}
		// R4: left
		if ( myROI.xMin < (unsigned int)radius )
		{
			x_iniR4 = (unsigned int)myROI.xMin;
			x_endR4 = radius;
			y_iniR4 = max( (unsigned int)myROI.yMin, (unsigned int)radius );
			y_endR4 = min( (unsigned int)myROI.yMax, nRows - (unsigned int)radius );
		}
		// R5: right
		if ( (unsigned int)myROI.xMax > nCols - (unsigned int)radius )
		{
			x_iniR5 = nCols - (unsigned int)radius;
			x_endR5 = (unsigned int)myROI.xMax;
			y_iniR5 = max( (unsigned int)myROI.yMin, (unsigned int)radius );
			y_endR5 = min( (unsigned int)myROI.yMax, nRows - (unsigned int)radius );
		}

		outImg_cols = (unsigned int)myROI.xMax - (unsigned int)myROI.xMin;
		outImg_rows = (unsigned int)myROI.yMax - (unsigned int)myROI.yMin;
	}

	outImg.setSize( outImg_cols, outImg_rows );

	int c,r;
	// Convolve the image with the kernel
	// R1: Main section
	for (j = y_iniR1, y = 0; j < y_endR1; j++, y++)
	{
		float *ptr = outImg( x_iniR1 - (unsigned int)myROI.xMin, j - (unsigned int)myROI.yMin);
		for (i = x_iniR1; i < x_endR1; i++)
		{
			for (r = 0; r < kernelLen; r++) {
				float *ptImg = inImg(i-radius, j+r-radius);
				for (c = kernelLen-1; c >= 0; c--)
					(*ptr) += kernel( r, c )*(*ptImg++);
			}
			ptr++;
		}
	} // end for

	// R2: Top section
	int aux_idy = 0;
	int aux_idx = 0;
	for (j = y_iniR2, y = 0; j < y_endR2; j++, y++)
	{
		float *ptr = outImg( x_iniR2 - (unsigned int)myROI.xMin, j - (unsigned int)myROI.yMin); // Pointer to output image
		for (i = x_iniR2; i < x_endR2; i++)
		{
			for (r = 0; r < kernelLen; r++)
			{
				for (c = kernelLen-1; c >= 0; c--) {
					aux_idx = i-c+kernelLen-1-radius;
					aux_idy = j+r-radius;
					if ( (aux_idx >= (int)x_iniR2) && (aux_idx < (int)x_endR2) && (aux_idy >= (int)y_iniR2) )
					{
						// If it is into the image limits
						float *ptImg = inImg( aux_idx, aux_idy );
						(*ptr) += kernel( r, c )*(*ptImg);
					}	// end if

				} // end for c
			} // end for
			ptr++;
		} // end for 'i'
	} // end for

	// R3: Bottom section
	for (j = y_iniR3, y = 0; j < y_endR3; j++, y++)
	{
		float *ptr = outImg( x_iniR3 - (unsigned int)myROI.xMin, j - (unsigned int)myROI.yMin);
		for (i = x_iniR3; i < x_endR3; i++)
		{
			for (r = 0; r < kernelLen; r++)
			{
				for (c = kernelLen-1; c >= 0; c--) {
					aux_idx = i-c+kernelLen-1-radius;
					aux_idy = j+r-radius;
					if ( (aux_idx >= (int)x_iniR3) && (aux_idx < (int)x_endR3) && (aux_idy < (int)y_endR3) ) {
						float *ptImg = inImg( aux_idx, aux_idy );
						(*ptr) += kernel( r, c )*(*ptImg);
					} // end if
				} // end for
			}
			ptr++;
		}
	}

	// R4: Left section
	for (j = y_iniR4, y = 0; j < y_endR4; j++, y++)
	{
		float *ptr = outImg( x_iniR4 - (unsigned int)myROI.xMin, j - (unsigned int)myROI.yMin);
		for (i = x_iniR4; i < x_endR4; i++)
		{
			for (r = 0; r < kernelLen; r++)
			{
					for (c = kernelLen-1; c >= 0; c--)
					{
						aux_idy = j+r-radius;
						aux_idx = i-c+kernelLen-1-radius;
						if (aux_idx >= (int)x_iniR4)
						{
							float *ptImg = inImg( aux_idx, aux_idy );
							(*ptr) += kernel( r, c )*(*ptImg);
						} // end if
					} // end for
			}
			ptr++;
		}
	}

	// R5: Right section
	for (j = y_iniR5, y = 0; j < y_endR5; j++, y++)
	{
		float *ptr = outImg( x_iniR5 - (unsigned int)myROI.xMin, j - (unsigned int)myROI.yMin);
		for (i = x_iniR5; i < x_endR5; i++)
		{
			for (r = 0; r < kernelLen; r++)
			{
				for (c = kernelLen-1; c >= 0; c--)
				{
					aux_idy = j+r-radius;
					aux_idx = i-c+kernelLen-1-radius;
					if (aux_idx < (int)x_endR5)
					{
						float *ptImg = inImg( aux_idx, aux_idy );
						(*ptr) += kernel( r, c )*(*ptImg);
					} // end if
				} // end for
			}
			ptr++;
		}
	}

	MRPT_TRY_END;
}
/***** END FAMD *****/


void  CImageConvolution::convolutionGeneric(
				const CMRPTImageFloat	&inImg,
				CMRPTImageFloat			&outImg,
				CMatrix					&kernel,
				unsigned int			x_min,
				unsigned int			x_length,
				unsigned int			y_min,
				unsigned int			y_length)
{
	MRPT_TRY_START;

	unsigned int x_max = x_min + x_length;
	unsigned int y_max = y_min + y_length;
	int kernelLen = (int)kernel.getColCount();
	int	radius = (int)(kernel.getColCount()-1)/2;

	ASSERT_(kernel.getColCount() == kernel.getRowCount());

	ASSERT_( &inImg != &outImg );

	unsigned int nRows = inImg.getHeight();
	unsigned int nCols = inImg.getWidth();
	unsigned int x_iniR1, x_endR1, y_iniR1, y_endR1;	// ROI limits
	unsigned int x_iniR2, x_endR2, y_iniR2, y_endR2;	// ROI limits
	unsigned int x_iniR3, x_endR3, y_iniR3, y_endR3;	// ROI limits
	unsigned int x_iniR4, x_endR4, y_iniR4, y_endR4;	// ROI limits
	unsigned int x_iniR5, x_endR5, y_iniR5, y_endR5;	// ROI limits
	unsigned int outImg_cols, outImg_rows;
	unsigned int i, j, y;							// Counters
	bool		 ROI = false;

	// Prepare the output image
	if ((x_min == 0) && (x_max == 0) && (y_min == 0) && (y_max == 0))
	{	// There is no ROI. We divide the image in five sections for speeding
		// Output image has the same sections
		x_iniR1 = radius;
		x_endR1 = nCols-radius;
		y_iniR1 = radius;
		y_endR1 = nRows-radius;

		x_iniR2 = 0;
		x_endR2 = nCols;
		y_iniR2 = 0;
		y_endR2 = radius;

		x_iniR3 = 0;
		x_endR3 = nCols;
		y_iniR3 = nRows-radius;
		y_endR3 = nRows;

		x_iniR4 = 0;
		x_endR4 = radius;
		y_iniR4 = radius;
		y_endR4 = nRows-radius;

		x_iniR5 = nCols-radius;
		x_endR5 = nCols;
		y_iniR5 = radius;
		y_endR5 = nRows-radius;

		outImg_cols = nCols;
		outImg_rows = nRows;
	}
	else
	{	// There is a ROI (TODO: Check parameters)
		// Define sections
		// Input image
		ROI = true;
		x_iniR1 = max((unsigned int)radius, x_min);
		x_endR1 = min(nCols-(unsigned int)radius, x_max);
		y_iniR1 = max((unsigned int)radius, y_min);
		y_endR1 = min(nRows-(unsigned int)radius, y_max);

		x_iniR2 = x_endR2 = y_iniR2 = y_endR2 = 0;
		x_iniR3 = x_endR3 = y_iniR3 = y_endR3 = 0;
		x_iniR4 = x_endR4 = y_iniR4 = y_endR4 = 0;
		x_iniR5 = x_endR5 = y_iniR5 = y_endR5 = 0;

		// Define necessary sections
		// R2: top
		if ( y_min < (unsigned int)radius )
		{
			x_iniR2 = x_min;
			x_endR2 = x_max;
			y_iniR2 = y_min;
			y_endR2 = radius;
		}
		// R3: bottom
		if ( y_max > nRows - (unsigned int)radius )
		{
			x_iniR3 = x_min;
			x_endR3 = x_max;
			y_iniR3 = radius;
			y_endR3 = y_max;
		}
		// R4: left
		if ( x_min < (unsigned int)radius )
		{
			x_iniR4 = x_min;
			x_endR4 = radius;
			y_iniR4 = max( y_min, (unsigned int)radius );
			y_endR4 = min( y_max, nRows - (unsigned int)radius );
		}
		// R5: right
		if ( x_max > nCols - (unsigned int)radius )
		{
			x_iniR5 = nCols - (unsigned int)radius;
			x_endR5 = x_max;
			y_iniR5 = max( y_min, (unsigned int)radius );
			y_endR5 = min( y_max, nRows - (unsigned int)radius );
		}

		outImg_cols = x_max - x_min;
		outImg_rows = y_max - y_min;
	}

	outImg.setSize( outImg_cols, outImg_rows );

	int c,r;
	// Convolve the image with the kernel
	// R1: Main section
	for (j = y_iniR1, y = 0; j < y_endR1; j++, y++)
	{
		float *ptr = outImg( x_iniR1 - x_min, j - y_min);
		for (i = x_iniR1; i < x_endR1; i++)
		{
			for (r = 0; r < kernelLen; r++) {
				float *ptImg = inImg(i-radius, j+r-radius);
				for (c = kernelLen-1; c >= 0; c--)
					(*ptr) += kernel( r, c )*(*ptImg++);
			}
			ptr++;
		}
	} // end for

	// R2: Top section
	int aux_idy = 0;
	int aux_idx = 0;
	for (j = y_iniR2, y = 0; j < y_endR2; j++, y++)
	{
		float *ptr = outImg( x_iniR2 - x_min, j - y_min); // Pointer to output image
		for (i = x_iniR2; i < x_endR2; i++)
		{
			for (r = 0; r < kernelLen; r++)
			{
				for (c = kernelLen-1; c >= 0; c--) {
					aux_idx = i-c+kernelLen-1-radius;
					aux_idy = j+r-radius;
					if ( (aux_idx >= (int)x_iniR2) && (aux_idx < (int)x_endR2) && (aux_idy >= (int)y_iniR2) )
					{
						// If it is into the image limits
						float *ptImg = inImg( aux_idx, aux_idy );
						(*ptr) += kernel( r, c )*(*ptImg);
					}	// end if

				} // end for c
			} // end for
			ptr++;
		} // end for 'i'
	} // end for

	// R3: Bottom section
	for (j = y_iniR3, y = 0; j < y_endR3; j++, y++)
	{
		float *ptr = outImg( x_iniR3 - x_min, j - y_min);
		for (i = x_iniR3; i < x_endR3; i++)
		{
			for (r = 0; r < kernelLen; r++)
			{
				for (c = kernelLen-1; c >= 0; c--) {
					aux_idx = i-c+kernelLen-1-radius;
					aux_idy = j+r-radius;
					if ( (aux_idx >= (int)x_iniR3) && (aux_idx < (int)x_endR3) && (aux_idy < (int)y_endR3) ) {
						float *ptImg = inImg( aux_idx, aux_idy );
						(*ptr) += kernel( r, c )*(*ptImg);
					} // end if
				} // end for
			}
			ptr++;
		}
	}

	// R4: Left section
	for (j = y_iniR4, y = 0; j < y_endR4; j++, y++)
	{
		float *ptr = outImg( x_iniR4 - x_min, j - y_min);
		for (i = x_iniR4; i < x_endR4; i++)
		{
			for (r = 0; r < kernelLen; r++)
			{
					for (c = kernelLen-1; c >= 0; c--)
					{
						aux_idy = j+r-radius;
						aux_idx = i-c+kernelLen-1-radius;
						if (aux_idx >= (int)x_iniR4)
						{
							float *ptImg = inImg( aux_idx, aux_idy );
							(*ptr) += kernel( r, c )*(*ptImg);
						} // end if
					} // end for
			}
			ptr++;
		}
	}

	// R5: Right section
	for (j = y_iniR5, y = 0; j < y_endR5; j++, y++)
	{
		float *ptr = outImg( x_iniR5 - x_min, j - y_min);
		for (i = x_iniR5; i < x_endR5; i++)
		{
			for (r = 0; r < kernelLen; r++)
			{
				for (c = kernelLen-1; c >= 0; c--)
				{
					aux_idy = j+r-radius;
					aux_idx = i-c+kernelLen-1-radius;
					if (aux_idx < (int)x_endR5)
					{
						float *ptImg = inImg( aux_idx, aux_idy );
						(*ptr) += kernel( r, c )*(*ptImg);
					} // end if
				} // end for
			}
			ptr++;
		}
	}

	MRPT_TRY_END;
}

/*-------------------------------------------------------------
					computeGradients
-------------------------------------------------------------*/
void  CImageConvolution::computeGradients(
				CMRPTImageFloat				&inImg,
				float						sigma,
				CMRPTImageFloat				&outGradientX,
				CMRPTImageFloat				&outGradientY)
{
	static CGaussianConvolutionKernel	kernelGauss(sigma,false);
	static CGaussianConvolutionKernel	kernelDOG(sigma,true);

	// Just in case...
	kernelGauss.changeSigma(sigma);
	kernelDOG.changeSigma(sigma);

	// Filter:
	convolutionSeparate(inImg, outGradientX, kernelDOG, kernelGauss );
	convolutionSeparate(inImg, outGradientY, kernelGauss, kernelDOG );
}

/*-------------------------------------------------------------
					computeSmoothedImage
-------------------------------------------------------------*/
/**** FAMD ******/
// ROI added
void  CImageConvolution::computeSmoothedImage(
				CMRPTImageFloat				&inImg,
				float						sigma,
				CMRPTImageFloat				&outImg,
				TImageROI					ROI)
{
	static CGaussianConvolutionKernel	kernelGauss(sigma,false);

	// Just in case...
	kernelGauss.changeSigma(sigma);

	// Filter:
	convolutionSeparate(inImg, outImg, kernelGauss, kernelGauss, ROI  );
}
/****END FAMD***/
