/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>						// Only for precomp. headers, include all libmrpt-core headers.
#include <mrpt/vision/CVisualOdometryStereo.h>
#include <algorithm>

using namespace mrpt;
using namespace mrpt::vision;
using namespace mrpt::gui;

// *********************************************
// Default options initializer
// *********************************************
TOdometryOptions::TOdometryOptions() :
	covMethod( TCov_Linear ),
	minNumFeatures( 25 ),
	minDispX( 0 ),
	minDispY( 0 ),
	minSetDist( 0 ),
	VERBOSE( false ),
	PLOT_IMAGES( false ),
	SAVE_FILES( false )
{
} // end of 'default options initializer'

// *********************************************
// Constructor
// *********************************************
CVisualOdometryStereo::CVisualOdometryStereo( )
{
#if VISUALODOMETRY_USES_GUI
	wind1 = NULL;
	wind2 = NULL;
#endif
	firstObs = true;
	nIter = 0;
} // end of 'constructor'

// *********************************************
// Destructor
// *********************************************
CVisualOdometryStereo::~CVisualOdometryStereo( )
{
#if VISUALODOMETRY_USES_GUI
	if( wind1 != NULL )
		delete wind1;
	if( wind2 != NULL )
		delete wind2;
#endif
} // end of 'destructor'

// *********************************************
// Reset
// *********************************************
void  CVisualOdometryStereo::reset( )
{
	firstObs = false;
	nIter = 0;
} // end of 'reset' method

/*-------------------------------------------------------------
			TOdometryOptions: loadFromConfigFile
-------------------------------------------------------------*/
void TOdometryOptions::loadFromConfigFile(
			const utils::CConfigFileBase	&iniFile,
			const std::string		&section)
{
	int cov = iniFile.read_int(section.c_str(),"covMethod",covMethod);
	switch (cov)
	{
		case 0:
			covMethod = TCov_Linear;
			break;
		case 1:
			covMethod = TCov_UT;
			break;
		case 2:
			covMethod = TCov_Particles;
			break;
	} // end switch

	MRPT_LOAD_CONFIG_VAR( VERBOSE, bool,  iniFile, section );
	MRPT_LOAD_CONFIG_VAR( PLOT_IMAGES, bool,  iniFile, section );
	MRPT_LOAD_CONFIG_VAR( SAVE_FILES, bool,  iniFile, section );
	MRPT_LOAD_CONFIG_VAR( minNumFeatures, int,  iniFile, section );

} // end of loadFromConfigFile

/*---------------------------------------------------------------
					TStereoSystemParams: dumpToTextStream
  ---------------------------------------------------------------*/
void  TOdometryOptions::dumpToTextStream(CStream	&out)
{
	out.printf("\n----------- [vision::TOdometryOptions] ------------ \n\n");

	out.printf("Covariance computation	= ");
	switch (covMethod)
	{
	case TCov_Linear:
		out.printf("Linear\n");
		break;
	case TCov_UT:
		out.printf("UT\n");
		break;
	case TCov_Particles:
		out.printf("Particles\n");
		break;
	} // end switch

	LOADABLEOPTS_DUMP_VAR(minNumFeatures, int )
	LOADABLEOPTS_DUMP_VAR(VERBOSE, bool )
	LOADABLEOPTS_DUMP_VAR(PLOT_IMAGES, bool )
	LOADABLEOPTS_DUMP_VAR(SAVE_FILES, bool )

}

// *********************************************
// storeMatchedFeatureList
// *********************************************
void  CVisualOdometryStereo::storeMatchedFeatureList( const CMatchedFeatureList &mfList )
{
	CMatchedFeatureList::const_iterator	itMatch;

	// Clear and resize the feature lists
	leftFeats.clear();
	rightFeats.clear();

	// Fill the feature lists
	for( itMatch = mfList.begin(); itMatch != mfList.end(); itMatch++)
	{
		leftFeats.insert( itMatch->first );
		rightFeats.insert( itMatch->second );
	}
} // end 'storeMathedFeatureList'

// *********************************************
// addPointsToLists
// *********************************************
void  CVisualOdometryStereo::addMatches( const CMatchedFeatureList &mfList )
{
	//// Add the new points to the current KLT lists
	//TFeatureID idx1 = leftFeats.getMaxID();
	//TFeatureID idx2 = rightFeats.getMaxID();

	//ASSERT_( idx1 == idx2 );
	CMatchedFeatureList::const_iterator itList;
	for( itList = mfList.begin(); itList != mfList.end(); itList++)
	{
		//(*itList->first).ID = ++idx1;
		//(*itList->second).ID = ++idx2;

		leftFeats.insert( itList->first );
		rightFeats.insert( itList->second );
	} // end for

} // end of 'addPointsToLists' method

// *********************************************
// computeRTCovariance
// *********************************************
void  CVisualOdometryStereo::computeRTCovariance(
	const poses::CPose3D &meanPose,
	const slam::CMetricMap::TMatchingPairList &list,
	const slam::CObservationVisualLandmarks &Pre_Cloud,
	const slam::CObservationVisualLandmarks &Cur_Cloud,
	math::CMatrixD &RTCov )
{
	switch( odometryOptions.covMethod )
	{

	// *********************************************
	// Linear propagation of the uncertainty
	// *********************************************
	case vision::TOdometryOptions::TCov_Linear:
	{
		// ITERATORS
		slam::CLandmarksMap::TCustomSequenceLandmarks::const_iterator	itLand1, itLand2;

		CMatrixD C(6,6);
		for(itLand1 = Pre_Cloud.landmarks.landmarks.begin(); itLand1 != Pre_Cloud.landmarks.landmarks.end(); itLand1++)
			 for( itLand2 = Cur_Cloud.landmarks.landmarks.begin(); itLand2 != Cur_Cloud.landmarks.landmarks.end(); itLand2++ )
			 {
				 if( itLand1->ID != itLand2->ID )
					continue;

				 // COVARIANCE MATRIX (Si = S1 + S2)
				 CMatrixD Si(3,3);
				 Si(0,0) = itLand1->pose_cov_11 + itLand2->pose_cov_11;
				 Si(0,1) = Si(1,0) = itLand1->pose_cov_12 + itLand2->pose_cov_12;
				 Si(0,2) = Si(2,0) = itLand1->pose_cov_13 + itLand2->pose_cov_13;

				 Si(1,1) = itLand1->pose_cov_22 + itLand2->pose_cov_22;
				 Si(1,2) = Si(2,1) = itLand1->pose_cov_23 + itLand2->pose_cov_23;

				 Si(2,2) = itLand1->pose_cov_33 + itLand2->pose_cov_33;

				 // DEBUG
				 /** /
				 FILE *mf = mrpt::system::os::fopen("si.txt", "wt");
				 mrpt::system::os::fprintf( mf, "%e\t%e\t%e\n%e\t%e\t%e\n%e\t%e\t%e\n",
					 Si(0,0), Si(0,1), Si(0,2),
					 Si(1,0), Si(1,1), Si(1,2),
					 Si(2,0), Si(2,1), Si(2,2));
				 mrpt::system::os::fclose( mf );
				/ **/

				 // H MATRIX
				 CMatrixD Hi(3,6);
				 Hi(0,0) = 1;	Hi(0,1) = 0;	Hi(0,2) = 0;
				 Hi(1,0) = 0;	Hi(1,1) = 1;	Hi(1,2) = 0;
				 Hi(2,0) = 0;	Hi(2,1) = 0;	Hi(2,2) = 1;

				 Hi(0,3) =	-sin(meanPose.yaw)*cos(meanPose.pitch)*itLand1->pose_mean_x +
						   (-sin(meanPose.yaw)*sin(meanPose.pitch)*sin(meanPose.roll) - cos(meanPose.yaw)*cos(meanPose.roll))*itLand1->pose_mean_y +
						   (-sin(meanPose.yaw)*sin(meanPose.pitch)*cos(meanPose.roll) + cos(meanPose.yaw)*sin(meanPose.roll))*itLand1->pose_mean_z;

				 Hi(0,4) =  -cos(meanPose.yaw)*sin(meanPose.pitch)*itLand1->pose_mean_x +
							 cos(meanPose.yaw)*cos(meanPose.pitch)*sin(meanPose.roll)*itLand1->pose_mean_y +
							 cos(meanPose.yaw)*cos(meanPose.pitch)*cos(meanPose.roll)*itLand1->pose_mean_z;

				 Hi(0,5) =	(cos(meanPose.yaw)*sin(meanPose.pitch)*cos(meanPose.roll) + sin(meanPose.yaw)*sin(meanPose.roll))*itLand1->pose_mean_y +
						   (-cos(meanPose.yaw)*sin(meanPose.pitch)*sin(meanPose.roll) + sin(meanPose.yaw)*cos(meanPose.roll))*itLand1->pose_mean_z;

				 Hi(1,3) =	 cos(meanPose.yaw)*cos(meanPose.pitch)*itLand1->pose_mean_x +
							(cos(meanPose.yaw)*sin(meanPose.pitch)*sin(meanPose.roll) - sin(meanPose.yaw)*cos(meanPose.roll))*itLand1->pose_mean_y +
							(cos(meanPose.yaw)*sin(meanPose.pitch)*cos(meanPose.roll) + sin(meanPose.yaw)*sin(meanPose.roll))*itLand1->pose_mean_z;

				 Hi(1,4) =	-sin(meanPose.yaw)*sin(meanPose.pitch)*itLand1->pose_mean_x +
							 sin(meanPose.yaw)*cos(meanPose.pitch)*sin(meanPose.roll)*itLand1->pose_mean_y +
							 sin(meanPose.yaw)*cos(meanPose.pitch)*cos(meanPose.roll)*itLand1->pose_mean_z;

				 Hi(1,5) =  (sin(meanPose.yaw)*sin(meanPose.pitch)*cos(meanPose.roll) - cos(meanPose.yaw)*sin(meanPose.roll))*itLand1->pose_mean_y +
						   (-sin(meanPose.yaw)*sin(meanPose.pitch)*sin(meanPose.roll) - cos(meanPose.yaw)*cos(meanPose.roll))*itLand1->pose_mean_z;

				 Hi(2,3) =	0;

				 Hi(2,4) =  -cos(meanPose.pitch)*itLand1->pose_mean_x -
							 sin(meanPose.pitch)*sin(meanPose.roll)*itLand1->pose_mean_y -
							 sin(meanPose.pitch)*cos(meanPose.roll)*itLand1->pose_mean_z;

				 Hi(2,5) =   cos(meanPose.pitch)*cos(meanPose.roll)*itLand1->pose_mean_y -
							 cos(meanPose.pitch)*sin(meanPose.roll)*itLand1->pose_mean_z;

				 //Si.inv().
				 C += (~Hi)*Si.inv()*Hi;

				 // DEBUG
				/** /
				 FILE *mf2 = mrpt::system::os::fopen("hi.txt", "wt");
				 mrpt::system::os::fprintf( mf2, "%e\t%e\t%e\t%e\t%e\t%e\n%e\t%e\t%e\t%e\t%e\t%e\n%e\t%e\t%e\t%e\t%e\t%e\n",
					 Hi(0,0), Hi(0,1), Hi(0,2), Hi(0,3), Hi(0,4), Hi(0,5),
					 Hi(1,0), Hi(1,1), Hi(1,2), Hi(1,3), Hi(1,4), Hi(1,5),
					 Hi(2,0), Hi(2,1), Hi(2,2), Hi(2,3), Hi(2,4), Hi(2,5));
				 mrpt::system::os::fclose( mf2 );

				 FILE *mf3 = mrpt::system::os::fopen("c.txt", "wt");
				 mrpt::system::os::fprintf( mf3, "%e\t%e\t%e\t%e\t%e\t%e\n%e\t%e\t%e\t%e\t%e\t%e\n%e\t%e\t%e\t%e\t%e\t%e\n%e\t%e\t%e\t%e\t%e\t%e\n%e\t%e\t%e\t%e\t%e\t%e\n%e\t%e\t%e\t%e\t%e\t%e\n",
					 C(0,0), C(0,1), C(0,2), C(0,3), C(0,4), C(0,5),
					 C(1,0), C(1,1), C(1,2), C(1,3), C(1,4), C(1,5),
					 C(2,0), C(2,1), C(2,2), C(2,3), C(2,4), C(2,5),
					 C(3,0), C(3,1), C(3,2), C(3,3), C(3,4), C(3,5),
					 C(4,0), C(4,1), C(4,2), C(4,3), C(4,4), C(4,5),
					 C(5,0), C(5,1), C(5,2), C(5,3), C(5,4), C(5,5));
				 mrpt::system::os::fclose( mf3 );

				 break;
				 / **/
			 } // end for

		RTCov = C.inv();	// Output covariance matrix
		// TEST:
		//RTCov(0,0) += 0.0004;	RTCov(1,1) += 0.0004;	RTCov(2,2) += 0.0004;
		//RTCov(3,3) += 0.000064;	RTCov(4,4) += 0.000064;	RTCov(5,5) += 0.000064;
		//std::cout << "Final: " << std::endl << RTCov << std::endl;
	}
		break;

	// *********************************************
	// Using the Unscented Transformation (UT)
	// *********************************************
	case vision::TOdometryOptions::TCov_UT:
	{
		// ***********************************************************************************
		// ***********************************************************************************
		// USING UNSCENDED TRANSFORM (UT)
		// Pa = computeCovariance( list, cloud1, cloud2 )
		// C = propagateMeanAndCovarianceTroughUT( list, Pa )

		// a = { x11,y11,z11,x21,y21,z21, x12,y12,z12,x22,y22,z22, ..., x1N,y1N,z1N,x2N,y2N,z2N }
		// NA = 6*nCorrs

		// g(·) = ICP6D(·)

		// b = { x,y,z,yaw,pitch,roll }
		// NB = 6

		size_t					NA = 6*list.size();
		double					k = (double)NA/2;

		double					w0 = k/(NA + k);
		double					w1 = 1/(2*(NA + k));

		// ***********************************************************************************
		// "Pa" and "L" Matrixes
		// ***********************************************************************************
		//			[ sxx1 sxy1 sxz1	0     0     0	... ]
		//			[ syx1 syy1 syz1    0     0     0   ... ]
		//			[ szx1 szy1 szz1    0     0     0   ... ]
		//	Pa =	[    0    0    0 sxx1' sxy1' sxz1'  ... ]	for each pair
		//			[    0    0    0 syx1' syy1' syz1'  ... ]
		//			[    0    0    0 szx1' szy1' szz1'  ... ]
		//			[  ...  ...  ...  ...   ...   ...   ... ]
		// ***********************************************************************************

		// Since Pa is block diagonal... the cholesky decomposition come to:
		//		 (	[ A1	0		0  ] )		[ chol(A1)	0			0		 ]
		//	chol (	[ 0		A2		0  ] )  =	[ 0			chol(A2)	0		 ]
		//		 (	[ 0		0		A3 ] )		[ 0			0			chol(A3) ]
		// ***********************************************************************************

		CMatrixD					L( NA, NA );

		CMatrixD					pPa(3,3);
		CMatrixD					pL(3,3);

		L.zeros();
		unsigned int			i;
		slam::CMetricMap::TMatchingPairList::const_iterator itList;

		// For each pair in the list... insert covariances in 'Pa' matrix
		for(i = 0, itList = list.begin(); itList != list.end(); itList++, i++)
		{
			const slam::CLandmark *lm1 = Pre_Cloud.landmarks.landmarks.getByID( itList->this_idx );
			const slam::CLandmark *lm2 = Cur_Cloud.landmarks.landmarks.getByID( itList->other_idx );
			if( lm1 && lm2 )
			{
				// LM1 ---------------------------------------------------------------
				pPa(0,0) = ( NA + k ) * lm1->pose_cov_11;
				pPa(0,1) = pPa(1,0) = ( NA + k ) * lm1->pose_cov_12;
				pPa(0,2) = pPa(2,0) = ( NA + k ) * lm1->pose_cov_13;

				pPa(1,1) = ( NA + k ) * lm1->pose_cov_22;
				pPa(1,2) = pPa(1,2) = ( NA + k ) * lm1->pose_cov_23;

				pPa(2,2) = ( NA + k ) * lm1->pose_cov_33;

				// Cholesky of pPa and inster into pL
				math::chol( pPa, pL );
				L.insertMatrix( i*6, i*6, pL );

				// LM2 ---------------------------------------------------------------
				pPa(0,0) = ( NA + k ) * lm2->pose_cov_11;
				pPa(0,1) = pPa(1,0) = ( NA + k ) * lm2->pose_cov_12;
				pPa(0,2) = pPa(2,0) = ( NA + k ) * lm2->pose_cov_13;

				pPa(1,1) = ( NA + k ) * lm2->pose_cov_22;
				pPa(1,2) = pPa(1,2) = ( NA + k ) * lm2->pose_cov_23;

				pPa(2,2) = ( NA + k ) * lm2->pose_cov_33;

				// Cholesky of pPa and instert into pL
				math::chol( pPa, pL );
				L.insertMatrix( i*6+3, i*6+3, pL );

			}
			else
				THROW_EXCEPTION("There are no landmarks in 'clouds'");

		}

		// ***********************************************************************************
		// "B" parameters
		// ***********************************************************************************
		poses::CPose3DPDFParticles				B( 2*NA + 1 );
		double	log_w1 = log(w1);

		*(B.m_particles[0].d) = meanPose;						// Mean List through the function
		B.m_particles[0].log_w = log(w0);						// Mean List weight

		slam::CMetricMap::TMatchingPairList				auxList1, auxList2;
		slam::CMetricMap::TMatchingPairList::iterator	itAuxList1, itAuxList2;
		auxList1.resize( list.size() );
		auxList2.resize( list.size() );

		for( i = 1; i <= NA; i++ )
		{
			vector_float						vAux;

			vAux.resize( L.getColCount() );
			L.extractRow( i-1, vAux );										// Extract the proper row

			unsigned int m = 0;
			for( itList = list.begin(), itAuxList1 = auxList1.begin(), itAuxList2 = auxList2.begin();
				 itList != list.end(), itAuxList1 != auxList1.end(), itAuxList2 != auxList2.end();
				 m = m + 6, itList++, itAuxList1++, itAuxList2++ )
			{
				itAuxList1->this_idx = itList->this_idx;					// Modify List1
				itAuxList1->this_x = itList->this_x + vAux[ m ];
				itAuxList1->this_y = itList->this_y + vAux[ m+1 ];
				itAuxList1->this_z = itList->this_z + vAux[ m+2 ];

				itAuxList1->other_idx = itList->other_idx;
				itAuxList1->other_x = itList->other_x + vAux[ m+3 ];
				itAuxList1->other_y = itList->other_y + vAux[ m+4 ];
				itAuxList1->other_z = itList->other_z + vAux[ m+5 ];

				itAuxList2->this_idx = itList->this_idx;					// Modify List2
				itAuxList2->this_x = itList->this_x - vAux[ m ];
				itAuxList2->this_y = itList->this_y - vAux[ m+1 ];
				itAuxList2->this_z = itList->this_z - vAux[ m+2 ];

				itAuxList2->other_idx = itList->other_idx;
				itAuxList2->other_x = itList->other_x - vAux[ m+3 ];
				itAuxList2->other_y = itList->other_y - vAux[ m+4 ];
				itAuxList2->other_z = itList->other_z - vAux[ m+5 ];
			} // end for

			double scale;
			scan_matching::leastSquareErrorRigidTransformation6D( auxList1, *B.m_particles[i].d, scale);		// Compute ICP6D for each
			B.m_particles[i].log_w = log_w1;

			scan_matching::leastSquareErrorRigidTransformation6D( auxList2, *B.m_particles[i+NA].d, scale );	// Compute ICP6D for each
			B.m_particles[i+NA].log_w = log_w1;

		} // end for

		// Convert meanB into a vector
		poses::CPose3D	meanB = B.getEstimatedPose();

		CMatrixD	Pb;

		Pb.zeros(6,6);

		for( i = 0; i <= 2*NA; i++ )
		{

			CMatrixD	v(6,1);											// v = Bi - meanB;
			double	Wi = exp( B.m_particles[i].log_w );

			// Spatial
			v(0,0) = B.m_particles[i].d->x - meanB.x;
			v(1,0) = B.m_particles[i].d->y - meanB.y;
			v(2,0) = B.m_particles[i].d->z - meanB.z;

			// Angles
			v(3,0) = B.m_particles[i].d->yaw - meanB.yaw;
			v(4,0) = B.m_particles[i].d->pitch - meanB.pitch;
			v(5,0) = B.m_particles[i].d->roll - meanB.roll;

			// Covariance
			Pb = Pb + Wi*(v*~v);
		} // end for

		RTCov = Pb;

	}
		break;

	case vision::TOdometryOptions::TCov_Particles:
	{

	}
		break;


	} // end swith 'method'

} // end of 'computeRTCovariance' method

// *********************************************
//				getMatchedList
// *********************************************
void CVisualOdometryStereo::getMatchedList( CMatchedFeatureList &mfList )
{
	//CMatchedFeatureList::iterator	itList;
	CFeatureList::iterator			itLList, itRList;

	mfList.clear();		// Clear the output list
	// mfList.resize( leftFeats.size() );

	for( itLList = leftFeats.begin(), itRList = rightFeats.begin();
		 itLList != leftFeats.end(), itRList != rightFeats.end();
		 itLList++, itRList++)
	{
		std::pair<CFeaturePtr,CFeaturePtr> m_pair( *itLList, *itRList );
		mfList.insert( m_pair );
	} // end for

/** /
	for( itList = mfList.begin(), itLList = leftFeats.begin(), itRList = rightFeats.begin();
		 itList != mfList.end(), itLList != leftFeats.end(), itRList != rightFeats.end();
		 itList++, itLList++, itRList++)
	{
		// List of features from the left image
		itList->first.ID = itLList->ID;
		itList->first.x = itLList->x;
		itList->first.y = itLList->y;
		itList->first.type = itLList->type;

		// List of features from the right image
		itList->second.ID = itRList->ID;
		itList->second.x = itRList->x;
		itList->second.y = itRList->y;
		itList->second.type = itRList->type;
	}
/ **/
} // end of 'getMatchedList'


// *********************************************
// Process
// *********************************************
bool  CVisualOdometryStereo::process( const slam::CObservationStereoImages &inObs, poses::CPose3DPDFGaussian &outEst )
{
	// Point Lists
	CFeatureList			lFeats, rFeats;
	CMatchedFeatureList		mfList;
	unsigned int			nMatches = 0;
	FILE					*fposes, *fcov;

	system::createDirectory("LOG_VODOMETRY");										// Create LOG directory
	system::deleteFiles("./LOG_VODOMETRY/*.txt");									// Clear LOG directory

	fposes	= mrpt::system::os::fopen( "./LOG_VODOMETRY/meanPose.txt", "at" );
	fcov	= mrpt::system::os::fopen( "./LOG_VODOMETRY/RTcov.txt", "at" );

	nIter++;	// Increment the iteration number

	if( odometryOptions.VERBOSE )
		std::cout << "------ Iteration " << nIter << " ------" << std::endl << std::endl;

	if( firstObs )																// If it is the first observation...
	{
		firstObs = false;														// The next one won't be the first observation (unless a reset is done)

		// Stereo system parameters initialization
		stereoParams.baseline = (float)inObs.rightCameraPose.x;
		stereoParams.K = inObs.intrinsicParams;

		// Odometry options related to the image size:
		odometryOptions.minDispX = inObs.imageLeft.getWidth()/4;
		odometryOptions.minDispY = inObs.imageLeft.getHeight()/4;
		odometryOptions.minSetDist = inObs.imageLeft.getHeight()/4;

		// Extract KLT (default option) features from the images
		fExtract.options.method = CFeatureExtraction::extMethodKLT;
		//fExtract.options.SIFTOptions.implementation = CFeatureExtraction::CSBinary;
		fExtract.detectFeatures( inObs.imageLeft, lFeats );
		//fExtract.computeSiftDescriptors( inObs.imageLeft, lFeats );
		fExtract.detectFeatures( inObs.imageRight, rFeats );
		//fExtract.computeSiftDescriptors( inObs.imageRight, rFeats );

		// Match KLT features
		nMatches = vision::matchFeatures(lFeats, rFeats, mfList, matchingOptions);		// TO DO: Default options in call to function

		// SHOW INFORMATION
		if( odometryOptions.SAVE_FILES )
		{

			inObs.imageLeft.saveToFile("./LOG_VODOMETRY/leftImg.jpg");
			inObs.imageRight.saveToFile("./LOG_VODOMETRY/rightImg.jpg");

			lFeats.saveToTextFile( format("./LOG_VODOMETRY/lFeats_%04d.txt", nIter) );
			rFeats.saveToTextFile( format("./LOG_VODOMETRY/rFeats_%04d.txt", nIter) );
			mfList.saveToTextFile( format("./LOG_VODOMETRY/matches_%04d.txt" ,nIter) );
		}

		if( odometryOptions.VERBOSE )
		{
			std::cout << ".: Extract features :." << std::endl;
			std::cout << "\tLeft features: " << lFeats.size() << std::endl;
			std::cout << "\tRight features: " << rFeats.size() << std::endl << std::endl;
			std::cout << ".: Match features :." << std::endl;
			std::cout << "\t# of matches: " << nMatches << "\n\n";
		}

		if( odometryOptions.PLOT_IMAGES )
		{
		#if VISUALODOMETRY_USES_GUI
			if(wind1 == NULL)
				wind1 = new CDisplayWindow();
			if(wind2 == NULL)
				wind2 = new CDisplayWindow();

			CMatchedFeatureList::iterator	it;
			vector_float					xL, yL, xR, yR;
			unsigned int					k;
			size_t							tam = mfList.size();

			xL.resize( tam );
			yL.resize( tam );
			xR.resize( tam );
			yR.resize( tam );

			for(k = 0, it = mfList.begin(); it != mfList.end(); it++, k++)
			{
				xL[k] = (it->first)->x;
				yL[k] = (it->first)->y;

				xR[k] = (it->second)->x;
				yR[k] = (it->second)->y;

			}
			wind1->setPos( 680, 600 );
			wind1->setWindowTitle("Left Image");
			wind1->showImageAndPoints( inObs.imageLeft, xL, yL );
			wind2->setPos( 1100, 600 );
			wind2->setWindowTitle("Right Image");
			wind2->showImageAndPoints( inObs.imageRight, xR, yR );
		#endif
		}

		// Project the matches
		vision::projectMatchedFeatures( mfList, stereoParams, Prev_cloud.landmarks );

		if( odometryOptions.SAVE_FILES )
			Prev_cloud.landmarks.saveToMATLABScript3D( format("./LOG_VODOMETRY/cloud_%04d.m",nIter) );

		if( odometryOptions.VERBOSE )
		{
			std::cout << ".: Project features :." << std::endl;
			std::cout << "\t# of projected features: " << Prev_cloud.landmarks.size() << "\n\n";
		}

		// Store the matched features
		storeMatchedFeatureList( mfList );

		// Insert additional data into the CLandmarksMap
		Prev_cloud.timestamp = inObs.timestamp;
		Prev_cloud.landmarks.changeCoordinatesReference( inObs.cameraPose );

		imgL = inObs.imageLeft;
		imgR = inObs.imageRight;

		// AT THIS POINT, WE HAVE STORED A CLOUD OF 3D POINTS, TWO SETS OF KLT POINTS AND TWO IMAGES.
		mrpt::system::os::fclose( fposes );
		mrpt::system::os::fclose( fcov );
		return true;
	} // end if 'First observation'
	else
	{
		// Track features
		vision::trackFeatures( imgL, inObs.imageLeft, leftFeats );			// LEFT IMAGE
		vision::trackFeatures( imgR, inObs.imageRight, rightFeats );		// RIGHT IMAGE

		if( odometryOptions.VERBOSE )
		{
			std::cout << ".: Track features :." << std::endl;
			std::cout << "\tLeft features: " << leftFeats.size() << std::endl;
			std::cout << "\tRight features: " << rightFeats.size() << std::endl;
		}

		// Delete bad tracked features
		vision::checkTrackedFeatures( leftFeats, rightFeats, matchingOptions );
		if( odometryOptions.VERBOSE )
		{
			std::cout << ".: Track checking performed :." << std::endl;
			std::cout << "\t# good matches: " << rightFeats.size() << std::endl;
		}

		// Save Files
		if( odometryOptions.SAVE_FILES )
		{
			leftFeats.saveToTextFile( format("./LOG_VODOMETRY/ltFeats_%04d.txt",nIter) );
			rightFeats.saveToTextFile( format("./LOG_VODOMETRY/rtFeats_%04d.txt",nIter) );
		}

		vector_float std, mean;
		vision::getDispersion( leftFeats, std, mean );

		double currDist = sqrt( square( mean[0] - inObs.imageLeft.getWidth()/2 ) + square( mean[1] - inObs.imageLeft.getHeight()/2 ) );

		// Are there enough features?
		unsigned int nCurrentFeatures = (unsigned int)leftFeats.size();

		if( nCurrentFeatures < odometryOptions.minNumFeatures ||
			currDist > odometryOptions.minSetDist ||
			std[0] < odometryOptions.minDispX ||
			std[1] < odometryOptions.minDispY ) // If there are not enough features or they are not scattered enough, we search for more
		{
			CMatchedFeatureList auxMFList;
			CFeatureList		auxLList, auxRList;

			// Find more features in the images
			fExtract.findMoreFeatures( inObs.imageLeft, leftFeats, auxLList );
			fExtract.findMoreFeatures( inObs.imageRight, rightFeats, auxRList );

			// Match them
			vision::matchFeatures( auxLList, auxRList, auxMFList, matchingOptions );

			// Add matches
			addMatches( auxMFList );

			if( odometryOptions.VERBOSE )
			{
				std::cout << ".: Find more features :." << std::endl;
				std::cout << "\t# new matches: " << auxMFList.size() << std::endl;
			}
		} // end if 'are there enough features?'

		CMatchedFeatureList matchedList;
		getMatchedList( matchedList );

		// PLOT IMAGES
		if( odometryOptions.PLOT_IMAGES )
		{
		#if VISUALODOMETRY_USES_GUI
			if(wind1 == NULL)
				wind1 = new CDisplayWindow();
			if(wind2 == NULL)
				wind2 = new CDisplayWindow();

			CMatchedFeatureList::iterator	it;
			vector_float					xL, yL, xR, yR;
			unsigned int					k;
			size_t							tam = matchedList.size();

			xL.resize( tam );
			yL.resize( tam );
			xR.resize( tam );
			yR.resize( tam );

			for(k = 0, it = matchedList.begin(); it != matchedList.end(); it++, k++)
			{
				xL[k] = (it->first)->x;
				yL[k] = (it->first)->y;

				xR[k] = (it->second)->x;
				yR[k] = (it->second)->y;

			}

			wind1->setPos( 680, 600 );
			wind1->setWindowTitle("Left Image");
			wind1->showImageAndPoints( inObs.imageLeft, xL, yL );
			wind2->setPos( 1100, 600 );
			wind2->setWindowTitle("Right Image");
			wind2->showImageAndPoints( inObs.imageRight, xR, yR );
		#endif
		}


		// PROBABILISTIC ESTIMATION OF THE CHANGE IN POSE
		// Computation of the current cloud of points
		slam::CObservationVisualLandmarks	Cur_cloud;

		// Project matched features
		vision::projectMatchedFeatures( matchedList, stereoParams, Cur_cloud.landmarks );

		if( odometryOptions.SAVE_FILES )
			Cur_cloud.landmarks.saveToMATLABScript3D( format("./LOG_VODOMETRY/cloud_%04d.m",nIter) );

		Cur_cloud.timestamp = inObs.timestamp;
		Cur_cloud.landmarks.changeCoordinatesReference( inObs.cameraPose );

		// Get the list of matched 3D points which act as inputs of the algorithm
		slam::CMetricMap::TMatchingPairList		list;
		list.clear();

		// *********************************************
		// DEBUG ***************************************
		// *********************************************
		/** /
		CMatrixD	p1(15,3), p2(15,3);
		for( unsigned int i = 0; i < 15; i++)
		{
			p1(i,0) = mrpt::math::random::RandomUni( 0, 6 );
			p1(i,1) = mrpt::math::random::RandomUni( 0, 6 );
			p1(i,2) = mrpt::math::random::RandomUni( 0, 6 );

			p2(i,0) = p1(i,0)+6;
			p2(i,1) = p1(i,1);
			p2(i,2) = p1(i,2);
		} // end for

		Prev_cloud.landmarks.clear();
		Cur_cloud.landmarks.clear();

		for (unsigned int i = 0; i < 15; i++)
		{
			slam::CLandmark lm1, lm2;
			lm1.ID = lm2.ID = i;

			lm1.pose_mean_x = p1(i,0);		lm2.pose_mean_x = p2(i,0);
			lm1.pose_mean_y = p1(i,1);		lm2.pose_mean_y = p2(i,1);
			lm1.pose_mean_z = 0;			lm2.pose_mean_z = 0;

			lm1.pose_cov_11 = 1;			lm2.pose_cov_11 = 1;
			lm1.pose_cov_12 = 0;			lm2.pose_cov_12 = 0;
			lm1.pose_cov_13 = 0;			lm2.pose_cov_13 = 0;
			lm1.pose_cov_22 = 1;			lm2.pose_cov_22 = 1;
			lm1.pose_cov_23 = 0;			lm2.pose_cov_23 = 0;
			lm1.pose_cov_33 = 1;			lm2.pose_cov_33 = 1;

			Prev_cloud.landmarks.landmarks.push_back( &lm1 );
			Cur_cloud.landmarks.landmarks.push_back( &lm2 );
		}

		/ **/

		vision::cloudsToMatchedList( Prev_cloud, Cur_cloud, list );

		// Compute the mean of the change in pose
		poses::CPose3D	meanPose;
		//vision::filterBadCorrsByDistance( list, 3 );

		if( odometryOptions.SAVE_FILES )
			list.dumpToFile( format("./LOG_VODOMETRY/matchedList_%04d.txt",nIter).c_str() );

		double		scale;
		vector_int	inliers;

		scan_matching::leastSquareErrorRigidTransformation6DRANSAC( list, meanPose, scale, inliers );
		//scan_matching::leastSquareErrorRigidTransformation6D( list, meanPose, scale ); // WITHOUT RANSAC

		// ***************************************************************
		// Delete outliers! NOT EFFICIENT!!
		// ***************************************************************
		slam::CMetricMap::TMatchingPairList::iterator	it;
		vector_int::iterator							vIt;
		slam::CMetricMap::TMatchingPairList				auxMatchedList;
		for( vIt = inliers.begin(); vIt != inliers.end(); vIt++ )
			auxMatchedList.push_back( list[ *vIt ] );

		list.clear();
		list = auxMatchedList;

		// ***************************************************************
		// ***************************************************************

		// Compute the covariance of the change in pose
		CMatrixD RTCov;
		RTCov.zeros( 6, 6 );
		computeRTCovariance( meanPose, list, Prev_cloud, Cur_cloud, RTCov );

		if( odometryOptions.SAVE_FILES )
		{
			list.dumpToFile( format("./LOG_VODOMETRY/RANSACmatchedList_%04d.txt",nIter).c_str() );

			mrpt::system::os::fprintf( fposes, "%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t%.5f\n",
				meanPose.x, meanPose.y, meanPose.z,
				meanPose.yaw, meanPose.pitch, meanPose.roll);

			mrpt::system::os::fprintf( fcov, "%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t", RTCov(0,0), RTCov(0,1), RTCov(0,2), RTCov(0,3), RTCov(0,4), RTCov(0,5) );
			mrpt::system::os::fprintf( fcov, "%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t", RTCov(1,0), RTCov(1,1), RTCov(1,2), RTCov(1,3), RTCov(1,4), RTCov(1,5) );
			mrpt::system::os::fprintf( fcov, "%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t", RTCov(2,0), RTCov(2,1), RTCov(2,2), RTCov(2,3), RTCov(2,4), RTCov(2,5) );
			mrpt::system::os::fprintf( fcov, "%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t", RTCov(3,0), RTCov(3,1), RTCov(3,2), RTCov(3,3), RTCov(3,4), RTCov(3,5) );
			mrpt::system::os::fprintf( fcov, "%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t", RTCov(4,0), RTCov(4,1), RTCov(4,2), RTCov(4,3), RTCov(4,4), RTCov(4,5) );
			mrpt::system::os::fprintf( fcov, "%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t%.5f\n", RTCov(5,0), RTCov(5,1), RTCov(5,2), RTCov(5,3), RTCov(5,4), RTCov(5,5) );

		}

		// Fill the output 'outEst'
		outEst.mean = meanPose;
		outEst.cov = RTCov;

		std::cout << "COV: " << std::endl << RTCov << std::endl;

		imgL = inObs.imageLeft;
		imgR = inObs.imageRight;

		// Storage of the current cloud of points
		Prev_cloud = Cur_cloud;
		std::cout << meanPose << std::endl;

		mrpt::system::os::fclose( fposes );
		mrpt::system::os::fclose( fcov );

		return true;
	} // end else
} // end of 'process' method
