/* +---------------------------------------------------------------------------+
   |          The Mobile m_robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/config.h>

#include <mrpt/hwdrivers/CActivMediaRobotBase.h>
#include <mrpt/utils/CTicTac.h>
#include <mrpt/slam/CObservationRange.h>

#if MRPT_HAS_ARIA
	// Specific definitions for ARIA, before including Aria.h:
	// -------------------------------------------------------------
//	#if defined(_MSC_VER) && !defined(MRPT_BUILT_AS_DLL)
//		#define ARIA_STATIC
//	#endif

	#include <mrpt/otherlibs/aria/Aria.h>
#endif


IMPLEMENTS_GENERIC_SENSOR(CActivMediaRobotBase,mrpt::hwdrivers)

using namespace mrpt;
using namespace mrpt::utils;
using namespace mrpt::system;
using namespace mrpt::slam;
using namespace mrpt::hwdrivers;
using namespace std;

#define THE_ROBOT	static_cast<ArRobot*>(m_robot)

/*-------------------------------------------------------------
						Constructor
-------------------------------------------------------------*/
CActivMediaRobotBase::CActivMediaRobotBase() :
	m_com_port			(),
	m_robotBaud			( 115200 ),
	m_firstIncreOdometry(true),
	m_enableSonars		(false),
	m_robot				(NULL),
	m_sonarDev			(NULL),
	m_simpleConnector	(NULL)
{
#if MRPT_HAS_ARIA
	Aria::init();

	m_robot		= (void*)new ArRobot();
	m_sonarDev	= (void*)new ArSonarDevice();

	// add the sonar to the m_robot
    THE_ROBOT->addRangeDevice((ArSonarDevice*)m_sonarDev);

#ifdef MRPT_OS_WINDOWS
	m_com_port = "COM1";
#else
	m_com_port = "/dev/ttyS1";
#endif

#else
	THROW_EXCEPTION("MRPT has been compiled with 'MRPT_BUILD_ARIA'=OFF, so this class cannot be used.");
#endif
}

/*-------------------------------------------------------------
						Destructor
-------------------------------------------------------------*/
CActivMediaRobotBase::~CActivMediaRobotBase()
{
#if MRPT_HAS_ARIA
	// Disconnect comms:
	disconnectAndDisableMotors();

	// Destroy object:
//	cout << "[~CActivMediaRobotBase] Destroying connector object" << endl;
//	delete (ArSimpleConnector*)m_simpleConnector;
	m_simpleConnector = NULL;

//	cout << "[~CActivMediaRobotBase] Destroying robot object" << endl;
//	delete THE_ROBOT;
	m_robot = NULL;

//	cout << "[~CActivMediaRobotBase] Destroying sonar object" << endl;
//	delete (ArSonarDevice*)m_sonarDev;
	m_sonarDev = NULL;

	// Shutdown ARIA threads:
//	cout << "[~CActivMediaRobotBase] ARIA shutdown" << endl;
	Aria::shutdown();
	cout << "[~CActivMediaRobotBase] ARIA shutdown done" << endl;
#else
	THROW_EXCEPTION("MRPT has been compiled with 'MRPT_BUILD_ARIA'=OFF, so this class cannot be used.");
#endif
}

/*-------------------------------------------------------------
						loadConfig
-------------------------------------------------------------*/
void  CActivMediaRobotBase::loadConfig(
	const mrpt::utils::CConfigFileBase &configSource,
	const std::string	  &iniSection )
{
#ifdef MRPT_OS_WINDOWS
	m_com_port = configSource.read_string(iniSection,"robotPort_WIN",m_com_port,true);
#else
	m_com_port = configSource.read_string(iniSection,"robotPort_LIN",m_com_port,true);
#endif

	m_robotBaud = configSource.read_int(iniSection, "robotBaud", m_robotBaud, true );

}

/*-------------------------------------------------------------
						setSerialPortConfig
-------------------------------------------------------------*/
void  CActivMediaRobotBase::setSerialPortConfig(
	const std::string  &portName,
	int		portBaudRate )
{
	m_com_port = portName;
	m_robotBaud = portBaudRate;
}

/*-------------------------------------------------------------
						initialize
-------------------------------------------------------------*/
void CActivMediaRobotBase::initialize()
{
#if MRPT_HAS_ARIA
	char		*args[10];
	int			nArgs;

	char		strBaud[100];
	os::sprintf(strBaud,sizeof(strBaud),"%i", m_robotBaud);

	char		strCOM[100];
	os::sprintf(strCOM,sizeof(strCOM),"%s", m_com_port.c_str());

	args[0] = (char*)"mrpt";

	args[1] = (char*)"-robotPort";	args[2] = strCOM;
	args[3] = (char*)"-robotBaud"; args[4] = strBaud;
	args[5] = NULL;
	nArgs = 5;

	if (m_simpleConnector) delete (ArSimpleConnector*)m_simpleConnector;

	m_simpleConnector = new ArSimpleConnector( &nArgs, args );
	static_cast<ArSimpleConnector*>(m_simpleConnector)->parseArgs();

	connectAndEnableMotors();
	m_state = ssWorking;
#else
	THROW_EXCEPTION("MRPT has been compiled with 'MRPT_BUILD_ARIA'=OFF, so this class cannot be used.");
#endif
}


/*-------------------------------------------------------------
						connectAndEnableMotors
-------------------------------------------------------------*/
void CActivMediaRobotBase::connectAndEnableMotors()
{
#if MRPT_HAS_ARIA
	// Establecimiento de la conexión con el pioneer
	if (!static_cast<ArSimpleConnector*>(m_simpleConnector)->connectRobot( THE_ROBOT ))
	{
		THROW_EXCEPTION_CUSTOM_MSG1("[CActivMediaRobotBase] Coldn't connect to robot thru %s", m_com_port.c_str() )
	}
	else
	{
		// Enable processing thread:
		THE_ROBOT->lock();
		THE_ROBOT->runAsync(true);
		THE_ROBOT->unlock();

		system::sleep(500);

		CTicTac	tictac;
		tictac.Tic();

		if (!THE_ROBOT->areMotorsEnabled())  // Are the motors already enabled?
		{
			THE_ROBOT->enableMotors();
			system::sleep(500);

			bool	success = false;

			while (tictac.Tac()<4000)
			{
			    THE_ROBOT->lock();
			    if (!THE_ROBOT->isRunning())
				{
					THROW_EXCEPTION("ARIA robot is not running");
				}
				if (THE_ROBOT->areMotorsEnabled())
				{
					THE_ROBOT->unlock();
					success = true;
					break;
				}
				THE_ROBOT->unlock();
				system::sleep(100);
			}

			if (!success)
			{
				disconnectAndDisableMotors();
				THROW_EXCEPTION("Couldn't enable robot motors");
			}

			// Start continuous stream of packets:
			THE_ROBOT->requestEncoderPackets();

			if (m_enableSonars)
					THE_ROBOT->enableSonar();
			else	THE_ROBOT->disableSonar();

			m_firstIncreOdometry = true;
		}
	}
#else
	THROW_EXCEPTION("MRPT has been compiled with 'MRPT_BUILD_ARIA'=OFF, so this class cannot be used.");
#endif
}

/*-------------------------------------------------------------
				disconnectAndDisableMotors
-------------------------------------------------------------*/
void CActivMediaRobotBase::disconnectAndDisableMotors()
{
#if MRPT_HAS_ARIA
	THE_ROBOT->stopEncoderPackets();

	if (THE_ROBOT->areMotorsEnabled())	//Si están activos los motores
	{
		//Uso exclusivo del m_robot (bloqueo)
		THE_ROBOT->lock();
			THE_ROBOT->stopRunning();		// Detiene movimiento
			cout << "[CActivMediaRobotBase] Disabling motors...";
			THE_ROBOT->disableMotors();		// Desactiva los motores
			cout << "Ok" << endl;
		THE_ROBOT->unlock();
	}

	THE_ROBOT->lock();
		cout << "[CActivMediaRobotBase] Disconnecting...";
		THE_ROBOT->disconnect();
		cout << "Ok" << endl;
	THE_ROBOT->unlock();
#else
	THROW_EXCEPTION("MRPT has been compiled with 'MRPT_BUILD_ARIA'=OFF, so this class cannot be used.");
#endif
}

/*-------------------------------------------------------------
					doProcess
-------------------------------------------------------------*/
void CActivMediaRobotBase::doProcess()
{
#if MRPT_HAS_ARIA
	THROW_EXCEPTION("to do");
#else
	THROW_EXCEPTION("MRPT has been compiled with 'MRPT_BUILD_ARIA'=OFF, so this class cannot be used.");
#endif
}

/*-------------------------------------------------------------
					changeOdometry
-------------------------------------------------------------*/
void CActivMediaRobotBase::changeOdometry(const mrpt::poses::CPose2D &newOdometry)
{
#if MRPT_HAS_ARIA
	ArPose	pos_actual;

	THE_ROBOT->lock();
		pos_actual.setPose( newOdometry.x*1000, newOdometry.y*1000, RAD2DEG( newOdometry.phi ) );
		THE_ROBOT->setDeadReconPose( pos_actual );
	THE_ROBOT->unlock();
#else
	THROW_EXCEPTION("MRPT has been compiled with 'MRPT_BUILD_ARIA'=OFF, so this class cannot be used.");
#endif
}


/*-------------------------------------------------------------
					getOdometry
-------------------------------------------------------------*/
void CActivMediaRobotBase::getOdometry(poses::CPose2D &out_odom)
{
#if MRPT_HAS_ARIA
	THE_ROBOT->lock();

	ArPose	pose = THE_ROBOT->getEncoderPose();
	out_odom.x = pose.getX() * 0.001;
	out_odom.y = pose.getY() * 0.001;
	out_odom.phi = DEG2RAD( pose.getTh() );

	THE_ROBOT->unlock();
#else
	THROW_EXCEPTION("MRPT has been compiled with 'MRPT_BUILD_ARIA'=OFF, so this class cannot be used.");
#endif
}

/*-------------------------------------------------------------
					getOdometryFull
-------------------------------------------------------------*/
void CActivMediaRobotBase::getOdometryFull(
	poses::CPose2D	&out_odom,
	double			&out_lin_vel,
	double			&out_ang_vel,
	int64_t			&out_left_encoder_ticks,
	int64_t			&out_right_encoder_ticks
	)
{
#if MRPT_HAS_ARIA
	THE_ROBOT->lock();

	// Odometry:
	ArPose	pose = THE_ROBOT->getEncoderPose();
	out_odom.x = pose.getX() * 0.001;
	out_odom.y = pose.getY() * 0.001;
	out_odom.phi = DEG2RAD( pose.getTh() );

	// Velocities:
	out_lin_vel = THE_ROBOT->getVel() * 0.001;
	out_ang_vel = DEG2RAD( THE_ROBOT->getRotVel() );

	// Encoders:
	out_left_encoder_ticks	= THE_ROBOT->getLeftEncoder();
	out_right_encoder_ticks	= THE_ROBOT->getRightEncoder();

	THE_ROBOT->unlock();
#else
	THROW_EXCEPTION("MRPT has been compiled with 'MRPT_BUILD_ARIA'=OFF, so this class cannot be used.");
#endif
}

/*-------------------------------------------------------------
					getOdometryIncrement
-------------------------------------------------------------*/
void CActivMediaRobotBase::getOdometryIncrement(
	poses::CPose2D	&out_incr_odom,
	double			&out_lin_vel,
	double			&out_ang_vel,
	int64_t			&out_incr_left_encoder_ticks,
	int64_t			&out_incr_right_encoder_ticks
	)
{
#if MRPT_HAS_ARIA
	THE_ROBOT->lock();

	static CPose2D	last_pose;
	static int64_t  last_left_ticks=0, last_right_ticks=0;

	CPose2D		cur_pose;
	int64_t		cur_left_ticks, cur_right_ticks;

	// Velocities:
	out_lin_vel = THE_ROBOT->getVel() * 0.001;
	out_ang_vel = DEG2RAD( THE_ROBOT->getRotVel() );

	// Current odometry:
	ArPose	pose = THE_ROBOT->getEncoderPose();
	cur_pose.x = pose.getX() * 0.001;
	cur_pose.y = pose.getY() * 0.001;
	cur_pose.phi = DEG2RAD( pose.getTh() );

	// Current encoders:
	cur_left_ticks	= THE_ROBOT->getLeftEncoder();
	cur_right_ticks	= THE_ROBOT->getRightEncoder();

	// Compute increment:
	if (m_firstIncreOdometry)
	{
		// First time:
		m_firstIncreOdometry = false;
		out_incr_odom = CPose2D(0,0,0);
		out_incr_left_encoder_ticks  = 0;
		out_incr_right_encoder_ticks = 0;
	}
	else
	{
		// Normal case:
		out_incr_odom = cur_pose - last_pose;
		out_incr_left_encoder_ticks = cur_left_ticks - last_left_ticks;
		out_incr_right_encoder_ticks = cur_right_ticks - last_right_ticks;
	}

	// save for the next time:
	last_pose		 = cur_pose;
	last_left_ticks  = cur_left_ticks;
	last_right_ticks = cur_right_ticks;

	THE_ROBOT->unlock();
#else
	THROW_EXCEPTION("MRPT has been compiled with 'MRPT_BUILD_ARIA'=OFF, so this class cannot be used.");
#endif
}

/*-------------------------------------------------------------
					getSonarsReadings
-------------------------------------------------------------*/
void CActivMediaRobotBase::getSonarsReadings( bool &thereIsObservation, CObservationRange	&obs )
{
#if MRPT_HAS_ARIA
	THE_ROBOT->lock();

	obs.minSensorDistance = 0;
	obs.maxSensorDistance = 30;

	int		i,N =THE_ROBOT->getNumSonar();

	obs.sensorLabel = "BASE_SONARS";
	obs.sensorConeApperture = DEG2RAD( 30 );
	obs.timestamp = system::now();

	// Resize buffers:
	obs.sensedData.resize(N);

	thereIsObservation = true;

	unsigned int time_cnt = THE_ROBOT->getCounter();

	for (i=0;i<N;i++)
	{
		ArSensorReading		*sr = THE_ROBOT->getSonarReading(i);

		if (!sr->isNew(time_cnt))
		{
			thereIsObservation = false;
			break;
		}

		obs.sensedData[i].sensorID = i;
		obs.sensedData[i].sensorPose.setFromValues(
			0.001*sr->getSensorX(),
			0.001*sr->getSensorY(),
			0.001*sr->getSensorX(),
			DEG2RAD( sr->getSensorTh() ) );

		obs.sensedData[i].sensedDistance = 0.001*THE_ROBOT->getSonarRange(i);
	}

	THE_ROBOT->unlock();
#else
	THROW_EXCEPTION("MRPT has been compiled with 'MRPT_BUILD_ARIA'=OFF, so this class cannot be used.");
#endif
}

/*-------------------------------------------------------------
					getBatteryCharge
-------------------------------------------------------------*/
void CActivMediaRobotBase::getBatteryCharge( double &out_batery_volts )
{
#if MRPT_HAS_ARIA
	THE_ROBOT->lock();
		out_batery_volts = THE_ROBOT->getBatteryVoltageNow();
	THE_ROBOT->unlock();
#else
	THROW_EXCEPTION("MRPT has been compiled with 'MRPT_BUILD_ARIA'=OFF, so this class cannot be used.");
#endif
}

/*-------------------------------------------------------------
					setVelocities
-------------------------------------------------------------*/
void CActivMediaRobotBase::setVelocities( const double &lin_vel, const double &ang_vel)
{
#if MRPT_HAS_ARIA
	THE_ROBOT->lock();
		THE_ROBOT->setVel( lin_vel*1000 );
		THE_ROBOT->setRotVel( RAD2DEG( ang_vel ) );
	THE_ROBOT->unlock();
#else
	THROW_EXCEPTION("MRPT has been compiled with 'MRPT_BUILD_ARIA'=OFF, so this class cannot be used.");
#endif
}
