/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef  conversions_H
#define  conversions_H

#include <mrpt/utils/utils_defs.h>
#include <mrpt/math/lightweight_geom_data.h>

namespace mrpt
{
	/** This namespace provides topography helper functions, coordinate transformations.
	 */
	namespace topography
	{

		/** Coordinates transformation from longitude/latitude/height to X/Y/Z coordinates
		  *  The WGS84 ellipsoid is used for the transformation. The coordinates are in 3D
		  *   relative to some user-provided point, with local X axis being east-ward, Y north-ward, Z up-ward.
		  *  For an explanation, refer to http://en.wikipedia.org/wiki/Reference_ellipsoid
		  * \sa coordinatesTransformation_WGS84_geocentric, ENU_axes_from_WGS84
		  */
		void  MRPTDLLIMPEXP coordinatesTransformation_WGS84(
			double		in_longitude_degrees,
			double		in_latitude_degrees,
			double		in_height_meters,
			double		&out_x_meters,
			double		&out_y_meters,
			double		&out_z_meters,
			double		in_longitude_reference_degrees,
			double		in_latitude_reference_degrees,
			double		in_height_reference_meters
			);

		/** Returns the East-North-Up (ENU) coordinate system associated to the given point.
		  * This is the reference employed in coordinatesTransformation_WGS84
		  * \param only_angles If set to true, the (x,y,z) fields will be left zeroed.
		  * \sa coordinatesTransformation_WGS84
		  */
		void MRPTDLLIMPEXP ENU_axes_from_WGS84(
			double		in_longitude_reference_degrees,
			double		in_latitude_reference_degrees,
			double		in_height_reference_meters,
			mrpt::math::TPose3D &out_ENU,
			bool		only_angles = false
			);

		/** Coordinates transformation from longitude/latitude/height to geocentric X/Y/Z coordinates.
		  *  The WGS84 ellipsoid is used for the transformation. The coordinates are in 3D
		  *   where the reference is the center of the Earth.
		  *  For an explanation, refer to http://en.wikipedia.org/wiki/Reference_ellipsoid
		  * \sa coordinatesTransformation_WGS84
		  */
		void  MRPTDLLIMPEXP coordinatesTransformation_WGS84_geocentric(
			double		in_longitude_degrees,
			double		in_latitude_degrees,
			double		in_height_meters,
			double		&out_x_meters,
			double		&out_y_meters,
			double		&out_z_meters
			);

		/** Convert latitude and longitude coordinates into UTM coordinates, computing the corresponding UTM zone and latitude band.
		  *   This method is based on public code by Gabriel Ruiz Martinez and Rafael Palacios.
		  *   Example:
\code
Input:
 Lat=40.3154333    Lon=-3.4857166
Output:
 x = 458731
 y = 4462881
 utm_zone = 30
 utm_band = T
\endcode
		   \sa http://www.mathworks.com/matlabcentral/fileexchange/10915
		  */
		void  MRPTDLLIMPEXP LatLonToUTM(
			double		in_latitude_degrees,
			double		in_longitude_degrees,
			double    	&out_UTM_x,
			double    	&out_UTM_y,
			int    		&out_UTM_zone,
			char    	&out_UTM_latitude_band );

	} // End of namespace

} // End of namespace

#endif
