/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/core.h>

using namespace mrpt::utils;
using namespace mrpt::gui;
using namespace mrpt::vision;
using namespace std;

#include <mrpt/examples_config.h>
string   myDataDir( MRPT_EXAMPLES_BASE_DIRECTORY + string("imageConvolutionFFT/") );

void TestTrackFeatures()
{

	CImage im1, im2;
	im1.loadFromFile("J:/Trabajo/Experimentos/[2009] vOdometry Characterization/right1.jpg");
	im2.loadFromFile("J:/Trabajo/Experimentos/[2009] vOdometry Characterization/right2.jpg");

	CFeatureExtraction	fExt;
	CFeatureList		feats;

	fExt.options.featsType = featKLT;
	fExt.detectFeatures( im1, feats );
	feats.saveToTextFile("J:/Trabajo/Experimentos/[2009] vOdometry Characterization/before.txt");

	vision::trackFeatures( im1, im2, feats );

	feats.saveToTextFile("J:/Trabajo/Experimentos/[2009] vOdometry Characterization/after.txt");

}

void TestRectifyImages()
{
    CImage im;
    CMatrix cam_matrix(3,3), dist_coeff(1,4);

    im.loadFromFile("/home/paco/Documents/Images/calib/L01.bmp");
    cam_matrix(0,0) = 938.8868; cam_matrix(0,1) = 0;            cam_matrix(0,2) = 367.8682;
    cam_matrix(1,0) = 0;        cam_matrix(1,1) = 938.8868;     cam_matrix(1,2) = 303.2578;
    cam_matrix(2,0) = 0;        cam_matrix(2,1) = 0;            cam_matrix(2,2) = 1;

    dist_coeff(0,0) = -0.3202480;
    dist_coeff(0,1) = -0.3470451;
    dist_coeff(0,2) = 0;
    dist_coeff(0,3) = 0;

    im.rectifyImageInPlace( cam_matrix, dist_coeff );

    im.saveToFile("/home/paco/Documents/Images/calib/L01_REC.bmp");
} // end TestRectifyImages

// ------------------------------------------------------
//				TestCapture
// ------------------------------------------------------
void TestExtractMatchProjectAndPaint()
{
	CDisplayWindow3D	wind;
	CFeatureExtraction	fExt;
	CFeatureList		featsHarris_L, featsHarris_R;
	CMatchedFeatureList	mHarris, mSIFT, mSURF;
	CImage				imL, imR;

	string imgL = MRPT_EXAMPLES_BASE_DIRECTORY + string("feature_extraction/") + string("imgs/imL_p01.jpg");		// Left image
	string imgR = MRPT_EXAMPLES_BASE_DIRECTORY + string("feature_extraction/") + string("imgs/imR_p01.jpg");		// Right image

	// Load and check images
	if (!imL.loadFromFile( imgL ))
	{
		cerr << "Cannot load " << imgL  << endl;
		return;
	}
	cout << "Loaded test image: " << imgL << endl;

	if (!imR.loadFromFile( imgR ))
	{
		cerr << "Cannot load " << imgR  << endl;
		return;
	}
	cout << "Loaded test image: " << imgR << endl;

	cout << "***************************************************" << endl;
	cout << "***************************************************" << endl;

	// Extract features:
	// HARRIS
	cout << "Detecting HARRIS features in LEFT image" << endl;
	fExt.options.featsType = featKLT;
	fExt.detectFeatures( imL, featsHarris_L );
	cout << "Detected " << featsHarris_L.size() << endl;

	cout << "Detecting HARRIS features in RIGHT image" << endl;
	fExt.detectFeatures( imR, featsHarris_R );
	cout << "Detected " << featsHarris_R.size() << endl;

	cout << "***************************************************" << endl;
	cout << "***************************************************" << endl;

	// Match features:
	size_t nMatches;
	TMatchingOptions opt;

	// HARRIS
	cout << "Matching HARRIS features by CORRELATION" << endl;
	nMatches = matchFeatures2( featsHarris_L, featsHarris_R, mHarris );
	cout << "Matches found: " << mHarris.size() << endl;
	
	cout << "***************************************************" << endl;

} // end TestExtractMatchProjectAndPaint

// ------------------------------------------------------
//				TestCapture
// ------------------------------------------------------
void TestMatchFeatures()
{
	CDisplayWindow		wind;
	CFeatureExtraction	fExt;
	CFeatureList		featsHarris_L, featsHarris_R, featsSIFT_L, featsSIFT_R, featsSURF_L, featsSURF_R;
	CMatchedFeatureList	mHarris, mSIFT, mSURF;
	CImage				imL, imR;

	string imgL = MRPT_EXAMPLES_BASE_DIRECTORY + string("feature_extraction/") + string("imgs/imL_p01.jpg");		// Left image
	string imgR = MRPT_EXAMPLES_BASE_DIRECTORY + string("feature_extraction/") + string("imgs/imR_p01.jpg");		// Right image

	// Load and check images
	if (!imL.loadFromFile( imgL ))
	{
		cerr << "Cannot load " << imgL  << endl;
		return;
	}
	cout << "Loaded test image: " << imgL << endl;

	if (!imR.loadFromFile( imgR ))
	{
		cerr << "Cannot load " << imgR  << endl;
		return;
	}
	cout << "Loaded test image: " << imgR << endl;

	cout << "***************************************************" << endl;
	cout << "***************************************************" << endl;

	// Extract features:
	// HARRIS
	cout << "Detecting HARRIS features in LEFT image" << endl;
	fExt.options.featsType = featHarris;
	fExt.detectFeatures( imL, featsHarris_L );
	cout << "Detected " << featsHarris_L.size() << endl;

	cout << "Detecting HARRIS features in RIGHT image" << endl;
	fExt.detectFeatures( imR, featsHarris_R );
	cout << "Detected " << featsHarris_R.size() << endl;

	cout << "***************************************************" << endl;

	// SIFT
	cout << "Detecting SIFT features in LEFT image" << endl;
	fExt.options.featsType = featSIFT;
	fExt.options.SIFTOptions.implementation = CFeatureExtraction::Hess;
	fExt.detectFeatures( imL, featsSIFT_L );
	cout << "Detected " << featsSIFT_L.size() << endl;
	
	cout << "Detecting SIFT features in RIGHT image" << endl;
	fExt.options.featsType = featSIFT;
	fExt.options.SIFTOptions.implementation = CFeatureExtraction::Hess;
	fExt.detectFeatures( imR, featsSIFT_R );
	cout << "Detected " << featsSIFT_R.size() << endl;

	cout << "***************************************************" << endl;

	// SURF
	cout << "Detecting SURF features in LEFT image" << endl;
	fExt.options.featsType = featSURF;
	fExt.detectFeatures( imL, featsSURF_L );
	cout << "Detected " << featsSURF_L.size() << endl;
	
	cout << "Detecting SURF features in RIGHT image" << endl;
	fExt.detectFeatures( imR, featsSURF_R );
	cout << "Detected " << featsSURF_R.size() << endl;

	cout << "***************************************************" << endl;
	cout << "***************************************************" << endl;

	// Match features:
	size_t nMatches;
	TMatchingOptions opt;

	// HARRIS
	cout << "Matching HARRIS features by CORRELATION" << endl;
	nMatches = matchFeatures2( featsHarris_L, featsHarris_R, mHarris );
	cout << "Matches found: " << mHarris.size() << endl;
	
	cout << "***************************************************" << endl;

	// SIFT
	cout << "Matching SIFT features by DESCRIPTOR" << endl;
	opt.matching_method = TMatchingOptions::mmDescriptorSIFT;
	nMatches = matchFeatures2( featsSIFT_L, featsSIFT_R, mSIFT, opt );
	cout << "Matches found: " << mSIFT.size() << endl;

	cout << "***************************************************" << endl;
	
	// SURF
	cout << "Matching SURF features by DESCRIPTOR" << endl;
	opt.matching_method = TMatchingOptions::mmDescriptorSURF;
	nMatches = matchFeatures2( featsSURF_L, featsSURF_R, mSURF, opt );
	cout << "Matches found: " << mSURF.size() << endl;

	wind.showImagesAndMatchedPoints( imL, imR, mSURF, TColor(0,0,255) );

} // end TestMatchFeatures

// ------------------------------------------------------
//				TestCapture
// ------------------------------------------------------
void TestExtractFeatures()
{
	CDisplayWindow		wind1,wind2,wind3,wind4;
	CFeatureExtraction	fExt;
	CFeatureList		featsHarris, featsKLT, featsSIFT_Hess, featsSIFT_Lowe, featsSIFT_Vedaldi, featsSURF;
	CImage				img;

	string the_img = myDataDir+string("test_image.jpg");

	if (!img.loadFromFile(the_img ))
	{
		cerr << "Cannot load " << the_img  << endl;
		return;
	}
	cout << "Loaded test image: " << the_img << endl;

	CTicTac	tictac;

	cout << "Extracting Harris features... [f_harris.txt]";
	tictac.Tic();
	fExt.options.featsType = featHarris;
	fExt.detectFeatures( img, featsHarris );
	cout << "Detected " << featsHarris.size() << " features in " << endl;
	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
	featsHarris.saveToTextFile("f_harris.txt");
	wind1.setWindowTitle("Harris detected features");
	wind1.showImageAndPoints( img, featsHarris );

	cout << "Computing SIFT descriptors only ... [f_harris+sift.txt]";
	tictac.Tic();
	fExt.options.SIFTOptions.implementation = CFeatureExtraction::Hess;
	fExt.computeDescriptors( img, featsHarris, descSIFT );
	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
	featsHarris.saveToTextFile("f_harris+sift.txt");

	cout << "Extracting KLT features... [f_klt.txt]";
	tictac.Tic();
	fExt.options.featsType = featKLT;
	fExt.detectFeatures( img, featsKLT );
	cout << "Detected " << featsKLT.size() << " features in " << endl;
	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
	featsKLT.saveToTextFile("f_klt.txt");
	wind2.setWindowTitle("KLT detected features");
	wind2.showImageAndPoints( img, featsKLT );

	cout << "Extracting SIFT features... [f_sift_hess.txt]";
	tictac.Tic();
	fExt.options.featsType = featSIFT;
	fExt.options.SIFTOptions.implementation = CFeatureExtraction::Hess;
	fExt.detectFeatures( img, featsSIFT_Hess );
	cout << "Detected " << featsSIFT_Hess.size() << " features in " << endl;
	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
	featsSIFT_Hess.saveToTextFile("f_sift_hess.txt");
	wind3.setWindowTitle("SIFT Hess detected features");
	wind3.showImageAndPoints( img, featsSIFT_Hess );

	cout << "Extracting SURF features... [f_surf.txt]";
	tictac.Tic();
	fExt.options.featsType = featSURF;
	fExt.detectFeatures( img, featsSURF );
	cout << "Detected " << featsSURF.size() << " features in " << endl;
	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
	featsSURF.saveToTextFile("f_surf.txt");
	wind4.setWindowTitle("SURF detected features");
	wind4.showImageAndPoints( img, featsSURF );

	cout << "Computing spin images descriptors only ... [f_harris+spinimgs.txt]";
	tictac.Tic();
	fExt.options.SpinImagesOptions.radius = 13;
	fExt.options.SpinImagesOptions.hist_size_distance  = 10;
	fExt.options.SpinImagesOptions.hist_size_intensity = 10;

	fExt.computeDescriptors( img, featsHarris, descSpinImages );

	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
	featsHarris.saveToTextFile("f_harris+spinimgs.txt");

	mrpt::system::pause();

	return; 
}

// ------------------------------------------------------
//				TestCapture
// ------------------------------------------------------
//void TestExtractFeatures()
//{
//	CDisplayWindow		wind1;
//	CFeatureExtraction	fExt;
//	CFeatureList		fHarris1, fHarris2;
//	CMatchedFeatureList	fMatched;
//	CImage				lImg, rImg;
//
//	string left_img		= myDataDir+string("left.jpg");
//	string right_img	= myDataDir+string("right.jpg");
//
//	if( !lImg.loadFromFile( left_img ) || !rImg.loadFromFile( right_img ) )
//	{
//		cerr << "Cannot load " << left_img << " or " << right_img << endl;
//		return;
//	}
//	cout << "Loaded test images" << endl;
//
//	CTicTac	tictac;
//
//	cout << "Extracting Harris features in left image ...";
//	fExt.options.featsType = featHarris;
//	fExt.detectFeatures( lImg, fHarris1 );
//	cout << "Detected " << featsHarris.size() << " features in " << endl;
//
//	cout << "Extracting Harris features in right image ...";
//	fExt.detectFeatures( rImg, fHarris2 );
//	cout << "Detected " << featsHarris.size() << " features in " << endl;
//
//	cout << "Matching features ..." << endl;
//	cout << "Method #1" << endl;
//	
//	tictac.Tic();
//	unsigned int nMatches = mrpt::vision::matchFeatures2( fHarris1, fHarris2, fMatched );
//	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
//	cout << "Matched " << featsHarris.size() << " features in " << endl;
//
//
//	featsHarris.saveToTextFile("f_harris.txt");
//	wind1.setWindowTitle("Harris detected features");
//	wind1.showImageAndPoints( img, featsHarris );
//
//	cout << "Computing SIFT descriptors only ... [f_harris+sift.txt]";
//	tictac.Tic();
//	fExt.options.SIFTOptions.implementation = CFeatureExtraction::Hess;
//	fExt.computeDescriptors( img, featsHarris, descSIFT );
//	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
//	featsHarris.saveToTextFile("f_harris+sift.txt");
//
//	cout << "Extracting KLT features... [f_klt.txt]";
//	tictac.Tic();
//	fExt.options.featsType = featKLT;
//	fExt.detectFeatures( img, featsKLT );
//	cout << "Detected " << featsKLT.size() << " features in " << endl;
//	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
//	featsKLT.saveToTextFile("f_klt.txt");
//	wind2.setWindowTitle("KLT detected features");
//	wind2.showImageAndPoints( img, featsKLT );
//
//	cout << "Extracting SIFT features... [f_sift_hess.txt]";
//	tictac.Tic();
//	fExt.options.featsType = featSIFT;
//	fExt.options.SIFTOptions.implementation = CFeatureExtraction::Hess;
//	fExt.detectFeatures( img, featsSIFT_Hess );
//	cout << "Detected " << featsSIFT_Hess.size() << " features in " << endl;
//	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
//	featsSIFT_Hess.saveToTextFile("f_sift_hess.txt");
//	wind3.setWindowTitle("SIFT Hess detected features");
//	wind3.showImageAndPoints( img, featsSIFT_Hess );
//
//	cout << "Extracting SURF features... [f_surf.txt]";
//	tictac.Tic();
//	fExt.options.featsType = featSURF;
//	fExt.detectFeatures( img, featsSURF );
//	cout << "Detected " << featsSURF.size() << " features in " << endl;
//	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
//	featsSURF.saveToTextFile("f_surf.txt");
//	wind4.setWindowTitle("SURF detected features");
//	wind4.showImageAndPoints( img, featsSURF );
//
//	cout << "Computing spin images descriptors only ... [f_harris+spinimgs.txt]";
//	tictac.Tic();
//	fExt.options.SpinImagesOptions.radius = 13;
//	fExt.options.SpinImagesOptions.hist_size_distance  = 10;
//	fExt.options.SpinImagesOptions.hist_size_intensity = 10;
//
//	fExt.computeDescriptors( img, featsHarris, descSpinImages );
//
//	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
//	featsHarris.saveToTextFile("f_harris+spinimgs.txt");
//
//	mrpt::system::pause();
//
//	return; 
//}

// ------------------------------------------------------
//				TestCapture
// ------------------------------------------------------
void TestExtractFeaturesTile()
{
	CDisplayWindow		wind1,wind2;
	CFeatureExtraction	fExt;
	CFeatureList		featsHarris;
	CImage				img;

	string the_img = myDataDir+string("test_image.jpg");

	if (!img.loadFromFile(the_img ))
	{
		cerr << "Cannot load " << the_img  << endl;
		return;
	}
	cout << "Loaded test image: " << the_img << endl;

	CTicTac	tictac;

	cout << "Extracting Harris features (tiled)... [f_harris_tiled.txt]";

	fExt.options.featsType = featHarris;
	fExt.options.harrisOptions.tile_image = true;

	tictac.Tic();
	fExt.detectFeatures( img, featsHarris );
	cout << format("  %.03fms",tictac.Tac()*1000) << endl;

	cout << "Detected " << featsHarris.size() << " features in " << endl;
	featsHarris.saveToTextFile("f_harris_tiled.txt");
	wind1.setWindowTitle("Harris detected features (Tiled image)");
	wind1.showTiledImageAndPoints( img, featsHarris );

	cout << "Extracting Harris features... [f_harris.txt]";

	fExt.options.harrisOptions.tile_image = false;
	
	tictac.Tic();
	fExt.detectFeatures( img, featsHarris );
	cout << format("  %.03fms",tictac.Tac()*1000) << endl;

	featsHarris.saveToTextFile("f_harris.txt");
	wind2.setWindowTitle("Harris detected features");
	wind2.showTiledImageAndPoints( img, featsHarris );

	mrpt::system::pause();

	return; 
}

int main(int argc, char **argv)
{
	try
	{
		//TestMatchFeatures();
		//TestExtractFeatures();
		//TestExtractFeaturesTile();
		//TestRectifyImages();
		TestTrackFeatures();

		return 0;
	} catch (std::exception &e)
	{
		std::cout << "MRPT exception caught: " << e.what() << std::endl;
		return -1;
	}
	catch (...)
	{
		printf("Another exception!!");
		return -1;
	}

}
