/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/slam/CRangeBearingKFSLAM.h>
#include <mrpt/poses/CPosePDF.h>
#include <mrpt/poses/CPosePDFGaussian.h>
#include <mrpt/slam/CActionRobotMovement3D.h>

#include <mrpt/math/utils.h>
#include <mrpt/math/CMatrix.h>
#include <mrpt/math/CMatrixD.h>
#include <mrpt/utils/CTicTac.h>

using namespace mrpt::slam;
using namespace mrpt::poses;
using namespace mrpt::utils;
using namespace std;


/*---------------------------------------------------------------
							Constructor
  ---------------------------------------------------------------*/
CRangeBearingKFSLAM::CRangeBearingKFSLAM( ) :
	options(),
	m_action(),m_SF(),
	m_IDs(),
	mapPartitioner(),
	m_SFs(),
	m_lastPartitionSet()
{
	reset();
}

/*---------------------------------------------------------------
							reset
  ---------------------------------------------------------------*/
void CRangeBearingKFSLAM::reset( )
{
	m_action.clear_unique();
	m_SF.clear_unique();
	m_IDs.clear();
	m_SFs.clear();
	mapPartitioner.clear();
	m_lastPartitionSet.clear();

	// -----------------------
	// INIT KF STATE
	m_xkk.assign(6,0);	// State: 6D pose (x,y,z,yaw,pitch,roll)

	// Initial cov:  Very small diagonal
	m_pkk.setSize(6,6);
	m_pkk.zeros();
	// -----------------------

	// Use SF-based matching (faster & easier for bearing-range observations with ID).
	mapPartitioner.options.useMapMatching  = false;
}

/*---------------------------------------------------------------
							Destructor
  ---------------------------------------------------------------*/
CRangeBearingKFSLAM::~CRangeBearingKFSLAM()
{

}

/*---------------------------------------------------------------
							getCurrentRobotPose
  ---------------------------------------------------------------*/
void CRangeBearingKFSLAM::getCurrentRobotPose(
	CPose3DPDFGaussian      &out_robotPose ) const
{
	MRPT_START

	ASSERT_(m_xkk.size()>=6);

	// Set 6D pose mean:
	out_robotPose.mean.setFromValues(
		m_xkk[0],
		m_xkk[1],
		m_xkk[2],
		m_xkk[3],
		m_xkk[4],
		m_xkk[5] );
	// and cov:
	CMatrixTemplateNumeric<kftype>  COV(6,6);
	m_pkk.extractMatrix(0,0,COV);
	out_robotPose.cov = COV;

	MRPT_END
}


/*---------------------------------------------------------------
							getCurrentState
  ---------------------------------------------------------------*/
void CRangeBearingKFSLAM::getCurrentState(
	CPose3DPDFGaussian      &out_robotPose,
	std::vector<CPoint3D>  &out_landmarksPositions,
	std::map<unsigned int,CLandmark::TLandmarkID> &out_landmarkIDs,
	CVectorDouble      &out_fullState,
	CMatrixDouble      &out_fullCovariance ) const
{
	MRPT_START

	ASSERT_(m_xkk.size()>=6);

	// Set 6D pose mean:
	out_robotPose.mean.setFromValues(
		m_xkk[0],
		m_xkk[1],
		m_xkk[2],
		m_xkk[3],
		m_xkk[4],
		m_xkk[5] );
	// and cov:
	CMatrixTemplateNumeric<kftype>  COV(6,6);
	m_pkk.extractMatrix(0,0,COV);
	out_robotPose.cov = COV;

	// Landmarks:
	ASSERT_( ((m_xkk.size() - 6) % 3)==0 );
	size_t i,nLMs = (m_xkk.size() - 6) / 3;
	out_landmarksPositions.resize(nLMs);
	for (i=0;i<nLMs;i++)
	{
		out_landmarksPositions[i].x( m_xkk[6+i*3+0] );
		out_landmarksPositions[i].y( m_xkk[6+i*3+1] );
		out_landmarksPositions[i].z( m_xkk[6+i*3+2] );
	} // end for i

	// IDs:
	out_landmarkIDs = m_IDs.getInverseMap(); //m_IDs_inverse;

    // Full state:
    out_fullState.resize( m_xkk.size() );
    for (i=0;i<m_xkk.size();i++)
		out_fullState[i] = m_xkk[i];

	// Full cov:
	out_fullCovariance = m_pkk;

	MRPT_END
}


/*---------------------------------------------------------------
						processActionObservation
  ---------------------------------------------------------------*/
void  CRangeBearingKFSLAM::processActionObservation(
    CActionCollectionPtr	&action,
    CSensoryFramePtr		&SF )
{
	MRPT_START

	m_action = action;
	m_SF = SF;

	// Sanity check:
	ASSERT_( m_IDs.size() == (m_pkk.getColCount() - get_vehicle_size() )/get_feature_size() );

	// ===================================================================================================================
	// Here's the meat!: Call the main method for the KF algorithm, which will call all the callback methods as required:
	// ===================================================================================================================
	runOneKalmanIteration();


	// =============================================================
	//  ADD TO SFs SEQUENCE
	// =============================================================
	CPose3DPDFGaussianPtr	auxPosePDF = CPose3DPDFGaussian::Create();
	getCurrentRobotPose( *auxPosePDF );

	if (options.create_simplemap)
	{
		m_SFs.insert( CPose3DPDFPtr(auxPosePDF) , SF );
	}

	// =============================================================
	//  UPDATE THE PARTITION GRAPH EXPERIMENT
	// =============================================================
	if (options.doPartitioningExperiment)
	{
	    if (options.partitioningMethod==0)
	    {
	        // Use spectral-graph technique:
            mapPartitioner.addMapFrame( SF, auxPosePDF );

            vector<vector_uint>   partitions;
            mapPartitioner.updatePartitions( partitions );
            m_lastPartitionSet = partitions;
	    }
	    else
	    {
	        // Fixed partitions every K observations:
	        vector<vector_uint>   partitions;
	        vector_uint           tmpCluster;

	        ASSERT_(options.partitioningMethod>1);
	        size_t    K = options.partitioningMethod;

	        for (size_t i=0;i<m_SFs.size();i++)
	        {
	            tmpCluster.push_back(i);
	            if ( (i % K) == 0 )
	            {
	                partitions.push_back(tmpCluster);
	                tmpCluster.clear();
	                tmpCluster.push_back(i);  // This observation "i" is shared between both clusters
	            }
	        }
            m_lastPartitionSet = partitions;
	    }

		printf("Partitions: %u\n", static_cast<unsigned>(m_lastPartitionSet.size() ));
	}

    MRPT_END
}



/** Must return the action vector u.
  * \param out_u The action vector which will be passed to OnTransitionModel
  */
void CRangeBearingKFSLAM::OnGetAction( KFArray_ACT &u) const
{
	// Get odometry estimation:
	CActionRobotMovement2DPtr actMov2D = m_action->getBestMovementEstimation();
	CActionRobotMovement3DPtr actMov3D = m_action->getActionByClass<CActionRobotMovement3D>();
	if (actMov3D)
	{
		u[0]=actMov3D->poseChange.mean.x();
		u[1]=actMov3D->poseChange.mean.y();
		u[2]=actMov3D->poseChange.mean.z();
		u[3]=actMov3D->poseChange.mean.yaw();
		u[4]=actMov3D->poseChange.mean.pitch();
		u[5]=actMov3D->poseChange.mean.roll();
	}
	else
	if (actMov2D)
	{
		CPose2D estMovMean;
		actMov2D->poseChange->getMean(estMovMean);
		u[0]=estMovMean.x();
		u[1]=estMovMean.y();
		u[2]=0;
		u[3]=estMovMean.phi();
		u[4]=0;
		u[5]=0;
	}
	else
	{
		// Left u as zeros
		for (size_t i=0;i<6;i++) u[i]=0;
	}
}


/** This virtual function musts implement the prediction model of the Kalman filter.
 */
void  CRangeBearingKFSLAM::OnTransitionModel(
	const KFArray_ACT &u,
	KFArray_VEH       &xv,
	bool &out_skipPrediction ) const
{
	MRPT_START

	CPose3D  robotPose(xv[0],xv[1],xv[2],xv[3],xv[4],xv[5]);
	CPose3D  odoIncrement(u[0],u[1],u[2],u[3],u[4],u[5]);

	// Pose composition:
	robotPose = robotPose + odoIncrement;

	xv[0]=robotPose.x();
	xv[1]=robotPose.y();
	xv[2]=robotPose.z();
	xv[3]=robotPose.yaw();
	xv[4]=robotPose.pitch();
	xv[5]=robotPose.roll();

	// Do not update the vehicle pose & its covariance until we have some landmakrs in the map,
	// otherwise, we are imposing a lower bound to the best uncertainty from now on:
	if (m_xkk.size() == get_vehicle_size() )
	{
		out_skipPrediction = true;
		return;
	}

	MRPT_END
}

/** This virtual function musts calculate the Jacobian F of the prediction model.
 */
void  CRangeBearingKFSLAM::OnTransitionJacobian( KFMatrix_VxV  &F ) const
{
	MRPT_START

	// The jacobian of the transition function:  dfv_dxv
	CActionRobotMovement2DPtr act2D = m_action->getBestMovementEstimation();
	CActionRobotMovement3DPtr act3D = m_action->getActionByClass<CActionRobotMovement3D>();

	if (act3D && act2D)  THROW_EXCEPTION("Both 2D & 3D odometry are present!?!?")

	TPoint3D  Ap;

	if (act3D)
	{
		Ap = TPoint3D(CPoint3D(act3D->poseChange.mean));
	}
	else if (act2D)
	{
		Ap = TPoint3D(CPoint3D(act2D->poseChange->getMeanVal()));
	}
	else
	{
		// No odometry at all:
		F.unit(); // Unit diagonal
		return;
	}

	kftype cy = cos(m_xkk[3]);	kftype sy = sin(m_xkk[3]);
	kftype cp = cos(m_xkk[4]);	kftype sp = sin(m_xkk[4]);
	kftype cr = cos(m_xkk[5]);	kftype sr = sin(m_xkk[5]);

	kftype xu = Ap.x;
	kftype yu = Ap.y;
	kftype zu = Ap.z;

	F.unit(); // Unit diagonal

	F (0,3) = -sy*cp*xu+(-sy*sp*sr-cy*cr)*yu+(-sy*sp*cr+cy*sr)*zu;
	F (0,4) = -cy*sp*xu + cy*cp*sr*yu+cy*cp*cr*zu;
	F (0,5) = (cy*sp*cr+sy*sr)*yu+(-cy*sp*sr+sy*cr)*zu;
	F (1,3) = cy*cp*xu+(cy*sp*sr-sy*cr)*yu+(cy*sp*cr+sy*sr)*zu;
	F (1,4) = -sy*sp*xu+sy*cp*sr*yu+sy*cp*cr*zu;
	F (1,5) = (sy*sp*cr-cy*sr)*yu+(-sy*sp*sr-cy*cr)*zu;
	F (2,4) = -cp*xu-sp*sr*yu-sp*cr*zu;
	F (2,5) = cp*cr*yu-cp*sr*zu;

	MRPT_END
}

/** This virtual function musts calculate de noise matrix of the prediction model.
 */
void  CRangeBearingKFSLAM::OnTransitionNoise( KFMatrix_VxV &Q ) const
{
	MRPT_START

	// The uncertainty of the 2D odometry, projected from the current position:
	CActionRobotMovement2DPtr act2D = m_action->getBestMovementEstimation();
	CActionRobotMovement3DPtr act3D = m_action->getActionByClass<CActionRobotMovement3D>();

	if (act3D && act2D)  THROW_EXCEPTION("Both 2D & 3D odometry are present!?!?")

	CPose3DPDFGaussian odoIncr;

	if (!act3D && !act2D)
	{
		// Use constant Q:
		Q.zeros();
		ASSERT_(options.stds_Q_no_odo.size()==Q.getColCount())

		for (size_t i=0;i<6;i++)
			Q(i,i) = square( options.stds_Q_no_odo[i]);
		return;
	}
	else
	{
		if (act2D)
		{
			// 2D odometry:
			CPosePDFGaussian odoIncr2D;
			odoIncr2D.copyFrom( *act2D->poseChange );
			odoIncr = odoIncr2D;

			// Avoid null values in the diagonal!
			odoIncr.cov(2,2)=        // z
			odoIncr.cov(4,4)=        // pitch
			odoIncr.cov(5,5)=1e-20f; // roll
		}
		else
		{
			// 3D odometry:
			odoIncr = act3D->poseChange;
		}
	}

	odoIncr.cov(2,2) +=  square(options.std_odo_z_additional);

	odoIncr.changeCoordinatesReference( CPose3D( m_xkk[0],m_xkk[1],m_xkk[2],m_xkk[3],m_xkk[4],m_xkk[5] ) );

	Q = odoIncr.cov;

	MRPT_END
}


void CRangeBearingKFSLAM::OnObservationModel(
	const vector_size_t       &idx_landmarks_to_predict,
	std::vector<KFArray_OBS>  &out_predictions ) const
{
	MRPT_START

	// Get the sensor pose relative to the robot:
	CObservationBearingRangePtr obs = m_SF->getObservationByClass<CObservationBearingRange>();
	ASSERT_(obs);
	const TPose3D  sensorPoseOnRobot = TPose3D( obs->sensorLocationOnRobot );

	/* -------------------------------------------
       Equations, obtained using matlab, of the relative 3D position of a landmark (xi,yi,zi), relative
          to a robot 6D pose (x0,y0,z0,y,p,r)
        Refer to technical report "6D EKF derivation...", 2008

        x0 y0 z0 y p r         % Robot's 6D pose
        x0s y0s z0s ys ps rs   % Sensor's 6D pose relative to robot
        xi yi zi % Absolute 3D landmark coordinates:

        Hx : dh_dxv   -> Jacobian of the observation model wrt the robot pose
        Hy : dh_dyi   -> Jacobian of the observation model wrt each landmark mean position

        Sizes:
	     h:  Lx3
         Hx: 3Lx6
         Hy: 3Lx3

          L=# of landmarks in the map (ALL OF THEM)
	  ------------------------------------------- */

	// Mean of the prior of the robot pose:
	CPose3D  robotPose( m_xkk[0],m_xkk[1],m_xkk[2],m_xkk[3],m_xkk[4],m_xkk[5] );

    const size_t  vehicle_size = get_vehicle_size();
    //const size_t  obs_size  = get_observation_size();
    const size_t  feature_size = get_feature_size();

    // Robot 6D pose:
//    kftype x0 = m_xkk[0];
//    kftype y0 = m_xkk[1];
//    kftype z0 = m_xkk[2];
//    kftype y  = m_xkk[3];
//    kftype p  = m_xkk[4];
//    kftype r  = m_xkk[5];

//    kftype cy = cos(y); kftype sy = sin(y);
//    kftype cp = cos(p); kftype sp = sin(p);
//    kftype cr = cos(r); kftype sr = sin(r);

    // Sensor 6D pose on robot:
//    kftype x0s = sensorPoseOnRobot.x;
//    kftype y0s = sensorPoseOnRobot.y;
//    kftype z0s = sensorPoseOnRobot.z;
//    kftype ys = sensorPoseOnRobot.yaw;
//    kftype ps = sensorPoseOnRobot.pitch;
//    kftype rs = sensorPoseOnRobot.roll;
//
//    kftype cys = cos(ys); kftype sys = sin(ys);
//    kftype cps = cos(ps); kftype sps = sin(ps);
//    kftype crs = cos(rs); kftype srs = sin(rs);

	CPose3D  sensorPoseAbs( robotPose + sensorPoseOnRobot );

	const size_t N = idx_landmarks_to_predict.size();
	out_predictions.resize(N);
	for (size_t i=0;i<N;i++)
	{
		const size_t row_in = feature_size*idx_landmarks_to_predict[i];

		// Landmark absolute 3D position in the map:
		kftype xi = m_xkk[ vehicle_size + row_in + 0 ];
		kftype yi = m_xkk[ vehicle_size + row_in + 1 ];
		kftype zi = m_xkk[ vehicle_size + row_in + 2 ];

		// Generate Range, yaw, pitch
		// ---------------------------------------------------
		CPoint3D	mapEst(xi,yi,zi);

		sensorPoseAbs.sphericalCoordinates(
			mapEst,
			out_predictions[i][0], // range
			out_predictions[i][1], // yaw
			out_predictions[i][2]  // pitch
			);
	}

	MRPT_END
}

void CRangeBearingKFSLAM::OnObservationJacobians(
	const size_t &idx_landmark_to_predict,
	KFMatrix_OxV &Hx,
	KFMatrix_OxF &Hy ) const
{
	MRPT_START

	// Get the sensor pose relative to the robot:
	CObservationBearingRangePtr obs = m_SF->getObservationByClass<CObservationBearingRange>();
	ASSERT_(obs);
	const TPose3D  sensorPoseOnRobot = TPose3D( obs->sensorLocationOnRobot );

	/* -------------------------------------------
       Equations, obtained using matlab, of the relative 3D position of a landmark (xi,yi,zi), relative
          to a robot 6D pose (x0,y0,z0,y,p,r)
        Refer to technical report "6D EKF derivation...", 2008

        x0 y0 z0 y p r         % Robot's 6D pose
        x0s y0s z0s ys ps rs   % Sensor's 6D pose relative to robot
        xi yi zi % Absolute 3D landmark coordinates:

        Hx : dh_dxv   -> Jacobian of the observation model wrt the robot pose
        Hy : dh_dyi   -> Jacobian of the observation model wrt each landmark mean position

        Sizes:
	     h:  Lx3
         Hx: 3Lx6
         Hy: 3Lx3

          L=# of landmarks in the map (ALL OF THEM)
	  ------------------------------------------- */

	// Mean of the prior of the robot pose:
	CPose3D  robotPose( m_xkk[0],m_xkk[1],m_xkk[2],m_xkk[3],m_xkk[4],m_xkk[5] );

    const size_t  vehicle_size = get_vehicle_size();
    //const size_t  obs_size  = get_observation_size();
    const size_t  feature_size = get_feature_size();

    // Robot 6D pose:
    kftype x0 = m_xkk[0];
    kftype y0 = m_xkk[1];
    kftype z0 = m_xkk[2];
    kftype y  = m_xkk[3];
    kftype p  = m_xkk[4];
    kftype r  = m_xkk[5];

    kftype cy = cos(y); kftype sy = sin(y);
    kftype cp = cos(p); kftype sp = sin(p);
    kftype cr = cos(r); kftype sr = sin(r);

    // Sensor 6D pose on robot:
    kftype x0s = sensorPoseOnRobot.x;
    kftype y0s = sensorPoseOnRobot.y;
    kftype z0s = sensorPoseOnRobot.z;
    kftype ys = sensorPoseOnRobot.yaw;
    kftype ps = sensorPoseOnRobot.pitch;
    kftype rs = sensorPoseOnRobot.roll;

    kftype cys = cos(ys); kftype sys = sin(ys);
    kftype cps = cos(ps); kftype sps = sin(ps);
    kftype crs = cos(rs); kftype srs = sin(rs);

	CPose3D  sensorPoseAbs( robotPose + sensorPoseOnRobot );

	const size_t row_in = feature_size*idx_landmark_to_predict;

	// Landmark absolute 3D position in the map:
	kftype xi = m_xkk[ vehicle_size + row_in + 0 ];
	kftype yi = m_xkk[ vehicle_size + row_in + 1 ];
	kftype zi = m_xkk[ vehicle_size + row_in + 2 ];

	// ---------------------------------------------------
	// Generate dhi_dxv: A 6x3 block
	// ---------------------------------------------------

	kftype EXPRE14 = (sy*cr*cys*crs-cy*sp*sr*cys*crs+cy*cp*sys*crs+sy*cr*sys*sps*srs-cps*srs*sy*sr-cy*cp*cys*sps*srs-cps*srs*cy*sp*cr-cy*sp*sr*sys*sps*srs);

	kftype EXPRE17 = EXPRE14*x0+sy*y0*cp*sys*crs+sys*crs*x0s+sps*sp*cys*srs*z0+(sy*cp*cys*sps*srs-sy*cp*sys*crs+sy*sp*sr*sys*sps*srs+sy*sp*sr*cys*crs+cy*cr*sys*sps*srs+cy*cr*cys*crs+cps*srs*sy*sp*cr-cps*srs*cy*sr)*yi+(-sp*cys*sps*srs+sp*sys*crs+cp*sr*sys*sps*srs+cp*sr*cys*crs+cp*cr*cps*srs)*zi-sp*sys*crs*z0-sps*cy*cr*y0*sys*srs-cps*z0s*srs+(cy*cp*cys*sps*srs-cy*cp*sys*crs+cy*sp*sr*sys*sps*srs+cy*sp*sr*cys*crs-sy*cr*sys*sps*srs-sy*cr*cys*crs+cps*srs*cy*sp*cr+cps*srs*sy*sr)*xi+cps*cy*sr*y0*srs-sps*y0s*sys*srs-sps*cys*srs*x0s-sps*sy*y0*cp*cys*srs-sy*sr*y0*sp*cys*crs-sps*sr*cp*sys*srs*z0-sps*sy*sr*y0*sp*sys*srs-cps*cr*cp*srs*z0-cps*sy*cr*y0*sp*srs-sr*cp*cys*crs*z0-y0s*cys*crs-cy*cr*y0*cys*crs;

	kftype EXPRE0 = square(EXPRE17);

	kftype EXPRE18 = (-sy*cr*cys*srs+cy*sp*sr*cys*srs-cy*sp*sr*sys*sps*crs+sy*cr*sys*sps*crs-cps*crs*sy*sr-cps*crs*cy*sp*cr-cy*cp*sys*srs-cy*cp*cys*sps*crs);

	kftype EXPRE35 = (cy*cp*cys*sps*crs+cy*cp*sys*srs+cy*sp*sr*sys*sps*crs-cy*sp*sr*cys*srs-sy*cr*sys*sps*crs+sy*cr*cys*srs+cps*crs*cy*sp*cr+cps*crs*sy*sr);


	kftype EXPRE11 = EXPRE18*x0-cps*z0s*crs-sys*srs*x0s+y0s*cys*srs+sr*cp*cys*srs*z0+cy*cr*y0*cys*srs+cps*cy*sr*y0*crs+sps*sp*cys*crs*z0+EXPRE35*xi+(-sp*cys*sps*crs-sp*sys*srs+cp*sr*sys*sps*crs-cp*sr*cys*srs+cp*cr*cps*crs)*zi+(sy*cp*cys*sps*crs+sy*cp*sys*srs+sy*sp*sr*sys*sps*crs-sy*sp*sr*cys*srs+cy*cr*sys*sps*crs-cy*cr*cys*srs+cps*crs*sy*sp*cr-cps*crs*cy*sr)*yi-sps*sr*cp*sys*crs*z0-sps*sy*sr*y0*sp*sys*crs-sps*cy*cr*y0*sys*crs+sp*sys*srs*z0-sps*sy*y0*cp*cys*crs-cps*cr*cp*crs*z0-sps*y0s*sys*crs-sps*cys*crs*x0s-sy*y0*cp*sys*srs-cps*sy*cr*y0*sp*crs+sy*sr*y0*sp*cys*srs;

	kftype EXPRE20 = (sy*sys*cr*cps-cy*cp*cys*cps+cy*cr*sp*sps-sp*sys*cps*cy*sr+sy*sr*sps);

	kftype EXPRE25 = (-sp*sy*cr*sps+cy*sys*cr*cps+sp*sys*cps*sy*sr+cy*sr*sps+sy*cp*cys*cps);

	kftype EXPRE15 = EXPRE20*x0+(-cy*cr*sp*sps+cy*cp*cys*cps-sy*sr*sps+sp*sys*cps*cy*sr-sy*sys*cr*cps)*xi+EXPRE25*yi+(-sp*cys*cps+cp*sr*sys*cps-cp*cr*sps)*zi-sp*sys*cps*sy*sr*y0+cp*cr*sps*z0-cp*sys*cps*sr*z0+sp*cys*cps*z0+sp*sy*cr*sps*y0-cys*cps*x0s-sys*cps*y0s-sys*cps*cy*cr*y0-cy*sr*sps*y0+z0s*sps-cp*cys*cps*sy*y0;

	kftype EXPRE1 = (0.5)/sqrt(square(EXPRE15)+EXPRE0+square(EXPRE11));
	kftype EXPRE2 = 2*(( sy*sys*cr*cps-cy*cp*cys*cps+cy*cr*sp*sps-sp*sys*cps*cy*sr+sy*sr*sps)*x0+(-cy*cr*sp*sps+cy*cp*cys*cps-sy*sr*sps+sp*sys*cps*cy*sr-sy*sys*cr*cps)*xi+EXPRE25*yi+(-sp*cys*cps+cp*sr*sys*cps-cp*cr*sps)*zi-sp*sys*cps*sy*sr*y0+cp*cr*sps*z0-cp*sys*cps*sr*z0+sp*cys*cps*z0+sp*sy*cr*sps*y0-cys*cps*x0s-sys*cps*y0s-sys*cps*cy*cr*y0-cy*sr*sps*y0+z0s*sps-cp*cys*cps*sy*y0);
	kftype K_05 = 0.5;
	kftype K_15 = 1.5;

	kftype EXPRE12 = (sy*cp*sys*crs-cy*cr*sys*sps*srs+cps*srs*cy*sr-sy*cp*cys*sps*srs-sy*sp*sr*cys*crs-sy*sp*sr*sys*sps*srs-cps*srs*sy*sp*cr-cy*cr*cys*crs);

	kftype EXPRE13 = ((-sy*sr*cys*crs-cy*sp*cr*cys*crs-sy*sr*sys*sps*srs-cps*srs*sy*cr+cps*srs*cy*sp*sr-cy*sp*cr*sys*sps*srs)*x0+(sy*sp*cr*sys*sps*srs+sy*sp*cr*cys*crs-cy*sr*sys*sps*srs-cy*sr*cys*crs-cps*srs*sy*sp*sr-cps*srs*cy*cr)*yi+(cp*cr*sys*sps*srs+cp*cr*cys*crs-cp*sr*cps*srs)*zi+sps*cy*sr*y0*sys*srs+(cy*sp*cr*sys*sps*srs+cy*sp*cr*cys*crs+sy*sr*sys*sps*srs+sy*sr*cys*crs-cps*srs*cy*sp*sr+cps*srs*sy*cr)*xi+cps*cy*cr*y0*srs-sy*cr*y0*sp*cys*crs-sps*cr*cp*sys*srs*z0-sps*sy*cr*y0*sp*sys*srs+cps*sr*cp*srs*z0+cps*sy*sr*y0*sp*srs-cr*cp*cys*crs*z0+cy*sr*y0*cys*crs);

	kftype EXPRE16 = ((sy*cp*cys*sps*crs+sy*cp*sys*srs+sy*sp*sr*sys*sps*crs-sy*sp*sr*cys*srs+cy*cr*sys*sps*crs-cy*cr*cys*srs+cps*crs*sy*sp*cr-cps*crs*cy*sr)*x0-sy*cr*cys*srs*y0-cps*crs*sy*sr*y0+(cy*cr*cys*srs+cps*crs*cy*sr-sy*sp*sr*sys*sps*crs-cy*cr*sys*sps*crs-sy*cp*cys*sps*crs-sy*cp*sys*srs-cps*crs*sy*sp*cr+sy*sp*sr*cys*srs)*xi+EXPRE35*yi-cy*sp*sr*sys*sps*crs*y0+sy*cr*sys*sps*crs*y0-cy*cp*cys*sps*crs*y0-cy*cp*sys*srs*y0-cps*crs*cy*sp*cr*y0+cy*sp*sr*cys*srs*y0);

	kftype EXPRE21 = (-sp*sys*cps*sy*sr+sp*sy*cr*sps-cy*sys*cr*cps-cy*sr*sps-sy*cp*cys*cps);

	kftype EXPRE22 = (EXPRE25*x0+EXPRE21*xi+(-cy*cr*sp*sps+cy*cp*cys*cps-sy*sr*sps+sp*sys*cps*cy*sr-sy*sys*cr*cps)*yi-sp*sys*cps*cy*sr*y0+sp*cy*cr*sps*y0+sys*cps*sy*cr*y0+sy*sr*sps*y0-cp*cys*cps*cy*y0);

	kftype EXPRE30 = ((cy*sp*cys*cps+cy*cr*cp*sps-cp*sys*cps*cy*sr)*x0+(-cy*cr*cp*sps-cy*sp*cys*cps+cp*sys*cps*cy*sr)*xi+(-cp*sy*cr*sps+cp*sys*cps*sy*sr-sy*sp*cys*cps)*yi+(-cp*cys*cps-sp*sr*sys*cps+sp*cr*sps)*zi-cp*sys*cps*sy*sr*y0-sp*cr*sps*z0+sp*sys*cps*sr*z0+cp*cys*cps*z0+cp*sy*cr*sps*y0+sp*cys*cps*sy*y0);


	Hx.get_unsafe(0,0)= EXPRE1*(EXPRE2*EXPRE20+2*(EXPRE17)*EXPRE14+2*(EXPRE11)*EXPRE18);
	Hx.get_unsafe(0,1)= EXPRE1*(EXPRE2*EXPRE21+2*(EXPRE17)*EXPRE12+2*(EXPRE11)*(cy*cr*cys*srs+cps*crs*cy*sr-sy*sp*sr*sys*sps*crs-cy*cr*sys*sps*crs-sy*cp*cys*sps*crs-sy*cp*sys*srs-cps*crs*sy*sp*cr+sy*sp*sr*cys*srs));
	Hx.get_unsafe(0,2)= EXPRE1*(EXPRE2*(cp*cr*sps-cp*sr*sys*cps+sp*cys*cps)+2*(EXPRE17)*(sp*cys*sps*srs-sp*sys*crs-cp*sr*sys*sps*srs-cp*cr*cps*srs-cp*sr*cys*crs)+2*(EXPRE11)*(cp*sr*cys*srs+sp*cys*sps*crs-cp*sr*sys*sps*crs+sp*sys*srs-cp*cr*cps*crs));
	Hx.get_unsafe(0,3)= EXPRE1*(EXPRE2*EXPRE22+2*(EXPRE17)*((sy*cp*cys*sps*srs-sy*cp*sys*crs+sy*sp*sr*sys*sps*srs+sy*sp*sr*cys*crs+cy*cr*sys*sps*srs+cy*cr*cys*crs+cps*srs*sy*sp*cr-cps*srs*cy*sr)*x0+cy*cp*sys*crs*y0+(cy*cp*cys*sps*srs-cy*cp*sys*crs+cy*sp*sr*sys*sps*srs+cy*sp*sr*cys*crs-sy*cr*sys*sps*srs-sy*cr*cys*crs+cps*srs*cy*sp*cr+cps*srs*sy*sr)*yi+sy*cr*sys*sps*srs*y0+EXPRE12*xi-cps*srs*sy*sr*y0-cy*cp*cys*sps*srs*y0-sp*cy*sr*cys*crs*y0-sp*cy*sr*sys*sps*srs*y0-cps*srs*cy*cr*sp*y0+sy*cr*cys*crs*y0)+2*(EXPRE11)*EXPRE16);
	Hx.get_unsafe(0,4)= EXPRE1*(EXPRE2*EXPRE30+2*(EXPRE17)*((-cy*cp*sr*cys*crs-cy*sp*sys*crs+cy*sp*cys*sps*srs-cps*srs*cy*cp*cr-cy*cp*sr*sys*sps*srs)*x0-sy*y0*sp*sys*crs+sps*cp*cys*srs*z0+(-sy*sp*cys*sps*srs+sy*sp*sys*crs+sy*cp*sr*sys*sps*srs+sy*cp*sr*cys*crs+cps*srs*sy*cp*cr)*yi+(-cp*cys*sps*srs+cp*sys*crs-sp*sr*sys*sps*srs-sp*sr*cys*crs-sp*cr*cps*srs)*zi-cp*sys*crs*z0+(-cy*sp*cys*sps*srs+cy*sp*sys*crs+cy*cp*sr*sys*sps*srs+cy*cp*sr*cys*crs+cps*srs*cy*cp*cr)*xi+sps*sy*y0*sp*cys*srs-sy*sr*y0*cp*cys*crs+sps*sr*sp*sys*srs*z0-sps*sy*sr*y0*cp*sys*srs+cps*cr*sp*srs*z0-cps*sy*cr*y0*cp*srs+sr*sp*cys*crs*z0)+2*(EXPRE11)*((cy*cp*sr*cys*srs-cy*cp*sr*sys*sps*crs-cps*crs*cy*cp*cr+cy*sp*sys*srs+cy*sp*cys*sps*crs)*x0-sr*sp*cys*srs*z0+sps*cp*cys*crs*z0+(-cy*sp*cys*sps*crs-cy*sp*sys*srs+cy*cp*sr*sys*sps*crs-cy*cp*sr*cys*srs+cps*crs*cy*cp*cr)*xi+(-cp*cys*sps*crs-cp*sys*srs-sp*sr*sys*sps*crs+sp*sr*cys*srs-sp*cr*cps*crs)*zi+(-sy*sp*cys*sps*crs-sy*sp*sys*srs+sy*cp*sr*sys*sps*crs-sy*cp*sr*cys*srs+cps*crs*sy*cp*cr)*yi+sps*sr*sp*sys*crs*z0-sps*sy*sr*y0*cp*sys*crs+cp*sys*srs*z0+sps*sy*y0*sp*cys*crs+cps*cr*sp*crs*z0+sy*y0*sp*sys*srs-cps*sy*cr*y0*cp*crs+sy*sr*y0*cp*cys*srs));
	Hx.get_unsafe(0,5)= EXPRE1*(EXPRE2*((-sy*sys*sr*cps-cy*sr*sp*sps-sp*sys*cps*cy*cr+sy*cr*sps)*x0+(cy*sr*sp*sps-sy*cr*sps+sp*sys*cps*cy*cr+sy*sys*sr*cps)*xi+(sp*sy*sr*sps-cy*sys*sr*cps+sp*sys*cps*sy*cr+cy*cr*sps)*yi+(cp*cr*sys*cps+cp*sr*sps)*zi-sp*sys*cps*sy*cr*y0-cp*sr*sps*z0-cp*sys*cps*cr*z0-sp*sy*sr*sps*y0+sys*cps*cy*sr*y0-cy*cr*sps*y0)+2*(EXPRE17)*EXPRE13+2*(EXPRE11)*((sy*sr*cys*srs+cy*sp*cr*cys*srs-cy*sp*cr*sys*sps*crs-sy*sr*sys*sps*crs-cps*crs*sy*cr+cps*crs*cy*sp*sr)*x0+cr*cp*cys*srs*z0-cy*sr*y0*cys*srs+cps*cy*cr*y0*crs+(cy*sp*cr*sys*sps*crs-cy*sp*cr*cys*srs+sy*sr*sys*sps*crs-sy*sr*cys*srs-cps*crs*cy*sp*sr+cps*crs*sy*cr)*xi+(cp*cr*sys*sps*crs-cp*cr*cys*srs-cp*sr*cps*crs)*zi+(sy*sp*cr*sys*sps*crs-sy*sp*cr*cys*srs-cy*sr*sys*sps*crs+cy*sr*cys*srs-cps*crs*sy*sp*sr-cps*crs*cy*cr)*yi-sps*cr*cp*sys*crs*z0-sps*sy*cr*y0*sp*sys*crs+sps*cy*sr*y0*sys*crs+cps*sr*cp*crs*z0+cps*sy*sr*y0*sp*crs+sy*cr*y0*sp*cys*srs));

	Hx.get_unsafe(1,0)= (EXPRE14/(EXPRE15)-(EXPRE17)/square(EXPRE15)*EXPRE20)/(1+EXPRE0/square(EXPRE15));
	Hx.get_unsafe(1,1)= (EXPRE12/(EXPRE15)-(EXPRE17)/square(EXPRE15)*EXPRE21)/(1+EXPRE0/square(EXPRE15));
	Hx.get_unsafe(1,2)= ((sp*cys*sps*srs-sp*sys*crs-cp*sr*sys*sps*srs-cp*cr*cps*srs-cp*sr*cys*crs)/(EXPRE15)-(EXPRE17)/square(EXPRE15)*(cp*cr*sps-cp*sr*sys*cps+sp*cys*cps))/(1+EXPRE0/square(EXPRE15));

	Hx.get_unsafe(1,3)= (((sy*cp*cys*sps*srs-sy*cp*sys*crs+sy*sp*sr*sys*sps*srs+sy*sp*sr*cys*crs+cy*cr*sys*sps*srs+cy*cr*cys*crs+cps*srs*sy*sp*cr-cps*srs*cy*sr)*x0+cy*cp*sys*crs*y0+(cy*cp*cys*sps*srs-cy*cp*sys*crs+cy*sp*sr*sys*sps*srs+cy*sp*sr*cys*crs-sy*cr*sys*sps*srs-sy*cr*cys*crs+cps*srs*cy*sp*cr+cps*srs*sy*sr)*yi+sy*cr*sys*sps*srs*y0+EXPRE12*xi-cps*srs*sy*sr*y0-cy*cp*cys*sps*srs*y0-sp*cy*sr*cys*crs*y0-sp*cy*sr*sys*sps*srs*y0-cps*srs*cy*cr*sp*y0+sy*cr*cys*crs*y0)/(EXPRE15)-(EXPRE17)/square(EXPRE15)*EXPRE22)/(1+EXPRE0/square(EXPRE15));
	Hx.get_unsafe(1,4)= (((-cy*cp*sr*cys*crs-cy*sp*sys*crs+cy*sp*cys*sps*srs-cps*srs*cy*cp*cr-cy*cp*sr*sys*sps*srs)*x0-sy*y0*sp*sys*crs+sps*cp*cys*srs*z0+(-sy*sp*cys*sps*srs+sy*sp*sys*crs+sy*cp*sr*sys*sps*srs+sy*cp*sr*cys*crs+cps*srs*sy*cp*cr)*yi+(-cp*cys*sps*srs+cp*sys*crs-sp*sr*sys*sps*srs-sp*sr*cys*crs-sp*cr*cps*srs)*zi-cp*sys*crs*z0+(-cy*sp*cys*sps*srs+cy*sp*sys*crs+cy*cp*sr*sys*sps*srs+cy*cp*sr*cys*crs+cps*srs*cy*cp*cr)*xi+sps*sy*y0*sp*cys*srs-sy*sr*y0*cp*cys*crs+sps*sr*sp*sys*srs*z0-sps*sy*sr*y0*cp*sys*srs+cps*cr*sp*srs*z0-cps*sy*cr*y0*cp*srs+sr*sp*cys*crs*z0)/(EXPRE15)-(EXPRE17)/square(EXPRE15)*EXPRE30)/(1+EXPRE0/square(EXPRE15));
	Hx.get_unsafe(1,5)= (EXPRE13/(EXPRE15)-(EXPRE17)/square(EXPRE15)*((-sy*sys*sr*cps-cy*sr*sp*sps-sp*sys*cps*cy*cr+sy*cr*sps)*x0+(cy*sr*sp*sps-sy*cr*sps+sp*sys*cps*cy*cr+sy*sys*sr*cps)*xi+(sp*sy*sr*sps-cy*sys*sr*cps+sp*sys*cps*sy*cr+cy*cr*sps)*yi+(cp*cr*sys*cps+cp*sr*sps)*zi-sp*sys*cps*sy*cr*y0-cp*sr*sps*z0-cp*sys*cps*cr*z0-sp*sy*sr*sps*y0+sys*cps*cy*sr*y0-cy*cr*sps*y0))/(1+EXPRE0/square(EXPRE15));

	Hx.get_unsafe(2,0)= -(EXPRE18/sqrt(square(EXPRE15)+EXPRE0)-K_05*(EXPRE11)/pow(square(EXPRE15)+EXPRE0,K_15)*(EXPRE2*EXPRE20+2*(EXPRE17)*EXPRE14))/(1+square(EXPRE11)/(square(EXPRE15)+EXPRE0));
	Hx.get_unsafe(2,1)= -((cy*cr*cys*srs+cps*crs*cy*sr-sy*sp*sr*sys*sps*crs-cy*cr*sys*sps*crs-sy*cp*cys*sps*crs-sy*cp*sys*srs-cps*crs*sy*sp*cr+sy*sp*sr*cys*srs)/sqrt(square(EXPRE15)+EXPRE0)-K_05*(EXPRE11)/pow(square(EXPRE15)+EXPRE0,K_15)*(EXPRE2*EXPRE21+2*(EXPRE17)*EXPRE12))/(1+square(EXPRE11)/(square(EXPRE15)+(EXPRE17)));
	Hx.get_unsafe(2,2)= -((cp*sr*cys*srs+sp*cys*sps*crs-cp*sr*sys*sps*crs+sp*sys*srs-cp*cr*cps*crs)/sqrt(square(EXPRE15)+EXPRE0)-K_05*(EXPRE11)/pow(square(EXPRE15)+EXPRE0,K_15)*(EXPRE2*(cp*cr*sps-cp*sr*sys*cps+sp*cys*cps)+2*(EXPRE17)*(sp*cys*sps*srs-sp*sys*crs-cp*sr*sys*sps*srs-cp*cr*cps*srs-cp*sr*cys*crs)))/(1+square(EXPRE11)/(square(EXPRE15)+EXPRE0));
	Hx.get_unsafe(2,3)= -(EXPRE16/sqrt(square(EXPRE15)+EXPRE0)-K_05*(EXPRE11)/pow(square(EXPRE15)+EXPRE0,K_15)*(EXPRE2*EXPRE22+2*(EXPRE17)*((sy*cp*cys*sps*srs-sy*cp*sys*crs+sy*sp*sr*sys*sps*srs+sy*sp*sr*cys*crs+cy*cr*sys*sps*srs+cy*cr*cys*crs+cps*srs*sy*sp*cr-cps*srs*cy*sr)*x0+cy*cp*sys*crs*y0+(cy*cp*cys*sps*srs-cy*cp*sys*crs+cy*sp*sr*sys*sps*srs+cy*sp*sr*cys*crs-sy*cr*sys*sps*srs-sy*cr*cys*crs+cps*srs*cy*sp*cr+cps*srs*sy*sr)*yi+sy*cr*sys*sps*srs*y0+EXPRE12*xi-cps*srs*sy*sr*y0-cy*cp*cys*sps*srs*y0-sp*cy*sr*cys*crs*y0-sp*cy*sr*sys*sps*srs*y0-cps*srs*cy*cr*sp*y0+sy*cr*cys*crs*y0)))/(1+square(EXPRE11)/(square(EXPRE15)+EXPRE0));
	Hx.get_unsafe(2,4)= -(((cy*cp*sr*cys*srs-cy*cp*sr*sys*sps*crs-cps*crs*cy*cp*cr+cy*sp*sys*srs+cy*sp*cys*sps*crs)*x0-sr*sp*cys*srs*z0+sps*cp*cys*crs*z0+(-cy*sp*cys*sps*crs-cy*sp*sys*srs+cy*cp*sr*sys*sps*crs-cy*cp*sr*cys*srs+cps*crs*cy*cp*cr)*xi+(-cp*cys*sps*crs-cp*sys*srs-sp*sr*sys*sps*crs+sp*sr*cys*srs-sp*cr*cps*crs)*zi+(-sy*sp*cys*sps*crs-sy*sp*sys*srs+sy*cp*sr*sys*sps*crs-sy*cp*sr*cys*srs+cps*crs*sy*cp*cr)*yi+sps*sr*sp*sys*crs*z0-sps*sy*sr*y0*cp*sys*crs+cp*sys*srs*z0+sps*sy*y0*sp*cys*crs+cps*cr*sp*crs*z0+sy*y0*sp*sys*srs-cps*sy*cr*y0*cp*crs+sy*sr*y0*cp*cys*srs)/sqrt(square(EXPRE15)+EXPRE0)-K_05*(EXPRE11)/pow(square(EXPRE15)+EXPRE0,K_15)*(EXPRE2*EXPRE30+2*(EXPRE17)*((-cy*cp*sr*cys*crs-cy*sp*sys*crs+cy*sp*cys*sps*srs-cps*srs*cy*cp*cr-cy*cp*sr*sys*sps*srs)*x0-sy*y0*sp*sys*crs+sps*cp*cys*srs*z0+(-sy*sp*cys*sps*srs+sy*sp*sys*crs+sy*cp*sr*sys*sps*srs+sy*cp*sr*cys*crs+cps*srs*sy*cp*cr)*yi+(-cp*cys*sps*srs+cp*sys*crs-sp*sr*sys*sps*srs-sp*sr*cys*crs-sp*cr*cps*srs)*zi-cp*sys*crs*z0+(-cy*sp*cys*sps*srs+cy*sp*sys*crs+cy*cp*sr*sys*sps*srs+cy*cp*sr*cys*crs+cps*srs*cy*cp*cr)*xi+sps*sy*y0*sp*cys*srs-sy*sr*y0*cp*cys*crs+sps*sr*sp*sys*srs*z0-sps*sy*sr*y0*cp*sys*srs+cps*cr*sp*srs*z0-cps*sy*cr*y0*cp*srs+sr*sp*cys*crs*z0)))/(1+square(EXPRE11)/(square(EXPRE15)+EXPRE0));
	Hx.get_unsafe(2,5)= -(((sy*sr*cys*srs+cy*sp*cr*cys*srs-cy*sp*cr*sys*sps*crs-sy*sr*sys*sps*crs-cps*crs*sy*cr+cps*crs*cy*sp*sr)*x0+cr*cp*cys*srs*z0-cy*sr*y0*cys*srs+cps*cy*cr*y0*crs+(cy*sp*cr*sys*sps*crs-cy*sp*cr*cys*srs+sy*sr*sys*sps*crs-sy*sr*cys*srs-cps*crs*cy*sp*sr+cps*crs*sy*cr)*xi+(cp*cr*sys*sps*crs-cp*cr*cys*srs-cp*sr*cps*crs)*zi+(sy*sp*cr*sys*sps*crs-sy*sp*cr*cys*srs-cy*sr*sys*sps*crs+cy*sr*cys*srs-cps*crs*sy*sp*sr-cps*crs*cy*cr)*yi-sps*cr*cp*sys*crs*z0-sps*sy*cr*y0*sp*sys*crs+sps*cy*sr*y0*sys*crs+cps*sr*cp*crs*z0+cps*sy*sr*y0*sp*crs+sy*cr*y0*sp*cys*srs)/sqrt(square(EXPRE15)+EXPRE0)-K_05*(EXPRE11)/pow(square(EXPRE15)+EXPRE0,K_15)*(EXPRE2*((-sy*sys*sr*cps-cy*sr*sp*sps-sp*sys*cps*cy*cr+sy*cr*sps)*x0+(cy*sr*sp*sps-sy*cr*sps+sp*sys*cps*cy*cr+sy*sys*sr*cps)*xi+(sp*sy*sr*sps-cy*sys*sr*cps+sp*sys*cps*sy*cr+cy*cr*sps)*yi+(cp*cr*sys*cps+cp*sr*sps)*zi-sp*sys*cps*sy*cr*y0-cp*sr*sps*z0-cp*sys*cps*cr*z0-sp*sy*sr*sps*y0+sys*cps*cy*sr*y0-cy*cr*sps*y0)+2*(EXPRE17)*EXPRE13))/(1+square(EXPRE11)/(square(EXPRE15)+EXPRE0));

	// ---------------------------------------------------
	// Generate dhi_dyi: A 3x3 block
	// ---------------------------------------------------
	// Generate dhi_dyi: A 3x3 block
	Hy.get_unsafe(0,0)=EXPRE1*(EXPRE2*(-cy*cr*sp*sps+cy*cp*cys*cps-sy*sr*sps+sp*sys*cps*cy*sr-sy*sys*cr*cps)+2*(EXPRE17)*(cy*cp*cys*sps*srs-cy*cp*sys*crs+cy*sp*sr*sys*sps*srs+cy*sp*sr*cys*crs-sy*cr*sys*sps*srs-sy*cr*cys*crs+cps*srs*cy*sp*cr+cps*srs*sy*sr)+2*(EXPRE11)*EXPRE35);
	Hy.get_unsafe(0,1)=EXPRE1*(EXPRE2*EXPRE25+2*(EXPRE17)*(sy*cp*cys*sps*srs-sy*cp*sys*crs+sy*sp*sr*sys*sps*srs+sy*sp*sr*cys*crs+cy*cr*sys*sps*srs+cy*cr*cys*crs+cps*srs*sy*sp*cr-cps*srs*cy*sr)+2*(EXPRE11)*(sy*cp*cys*sps*crs+sy*cp*sys*srs+sy*sp*sr*sys*sps*crs-sy*sp*sr*cys*srs+cy*cr*sys*sps*crs-cy*cr*cys*srs+cps*crs*sy*sp*cr-cps*crs*cy*sr));
	Hy.get_unsafe(0,2)=EXPRE1*(EXPRE2*(-sp*cys*cps+cp*sr*sys*cps-cp*cr*sps)+2*(EXPRE17)*(-sp*cys*sps*srs+sp*sys*crs+cp*sr*sys*sps*srs+cp*sr*cys*crs+cp*cr*cps*srs)+2*(EXPRE11)*(-sp*cys*sps*crs-sp*sys*srs+cp*sr*sys*sps*crs-cp*sr*cys*srs+cp*cr*cps*crs));

	Hy.get_unsafe(1,0)=((cy*cp*cys*sps*srs-cy*cp*sys*crs+cy*sp*sr*sys*sps*srs+cy*sp*sr*cys*crs-sy*cr*sys*sps*srs-sy*cr*cys*crs+cps*srs*cy*sp*cr+cps*srs*sy*sr)/(EXPRE15)-(EXPRE17)/square(EXPRE15)*(-cy*cr*sp*sps+cy*cp*cys*cps-sy*sr*sps+sp*sys*cps*cy*sr-sy*sys*cr*cps))/(1+EXPRE0/square(EXPRE15));
	Hy.get_unsafe(1,1)=((sy*cp*cys*sps*srs-sy*cp*sys*crs+sy*sp*sr*sys*sps*srs+sy*sp*sr*cys*crs+cy*cr*sys*sps*srs+cy*cr*cys*crs+cps*srs*sy*sp*cr-cps*srs*cy*sr)/(EXPRE15)-(EXPRE17)/square(EXPRE15)*EXPRE25)/(1+EXPRE0/square(EXPRE15));
	Hy.get_unsafe(1,2)=((-sp*cys*sps*srs+sp*sys*crs+cp*sr*sys*sps*srs+cp*sr*cys*crs+cp*cr*cps*srs)/(EXPRE15)-(EXPRE17)/square(EXPRE15)*(-sp*cys*cps+cp*sr*sys*cps-cp*cr*sps))/(1+EXPRE0/square(EXPRE15));

	Hy.get_unsafe(2,0)=-(EXPRE35/sqrt(square(EXPRE15)+EXPRE0)-K_05*(EXPRE11)/pow(square(EXPRE15)+EXPRE0,K_15)*(EXPRE2*(-cy*cr*sp*sps+cy*cp*cys*cps-sy*sr*sps+sp*sys*cps*cy*sr-sy*sys*cr*cps)+2*(EXPRE17)*(cy*cp*cys*sps*srs-cy*cp*sys*crs+cy*sp*sr*sys*sps*srs+cy*sp*sr*cys*crs-sy*cr*sys*sps*srs-sy*cr*cys*crs+cps*srs*cy*sp*cr+cps*srs*sy*sr)))/(1+square(EXPRE11)/(square(EXPRE15)+EXPRE0));
	Hy.get_unsafe(2,1)=-((sy*cp*cys*sps*crs+sy*cp*sys*srs+sy*sp*sr*sys*sps*crs-sy*sp*sr*cys*srs+cy*cr*sys*sps*crs-cy*cr*cys*srs+cps*crs*sy*sp*cr-cps*crs*cy*sr)/sqrt(square(EXPRE15)+EXPRE0)-K_05*(EXPRE11)/pow(square(EXPRE15)+EXPRE0,K_15)*(EXPRE2*EXPRE25+2*(EXPRE17)*(sy*cp*cys*sps*srs-sy*cp*sys*crs+sy*sp*sr*sys*sps*srs+sy*sp*sr*cys*crs+cy*cr*sys*sps*srs+cy*cr*cys*crs+cps*srs*sy*sp*cr-cps*srs*cy*sr)))/(1+square(EXPRE11)/(square(EXPRE15)+EXPRE0));
	Hy.get_unsafe(2,2)=-((-sp*cys*sps*crs-sp*sys*srs+cp*sr*sys*sps*crs-cp*sr*cys*srs+cp*cr*cps*crs)/sqrt(square(EXPRE15)+EXPRE0)-K_05*(EXPRE11)/pow(square(EXPRE15)+EXPRE0,K_15)*(EXPRE2*(-sp*cys*cps+cp*sr*sys*cps-cp*cr*sps)+2*(EXPRE17)*(-sp*cys*sps*srs+sp*sys*crs+cp*sr*sys*sps*srs+cp*sr*cys*crs+cp*cr*cps*srs)))/(1+square(EXPRE11)/(square(EXPRE15)+EXPRE0));

	MRPT_END
}


/** This is called between the KF prediction step and the update step, and the application must return the observations and, when applicable, the data association between these observations and the current map.
  *
  * \param out_z N vectors, each for one "observation" of length OBS_SIZE, N being the number of "observations": how many observed landmarks for a map, or just one if not applicable.
  * \param out_data_association An empty vector or, where applicable, a vector where the i'th element corresponds to the position of the observation in the i'th row of out_z within the system state vector (in the range [0,getNumberOfLandmarksInTheMap()-1]), or -1 if it is a new map element and we want to insert it at the end of this KF iteration.
  * \param in_S The full covariance matrix of the observation predictions (i.e. the "innovation covariance matrix"). This is a M·O x M·O matrix with M=length of "in_lm_indices_in_S".
  * \param in_lm_indices_in_S The indices of the map landmarks (range [0,getNumberOfLandmarksInTheMap()-1]) that can be found in the matrix in_S.
  *
  *  This method will be called just once for each complete KF iteration.
  * \note It is assumed that the observations are independent, i.e. there are NO cross-covariances between them.
  */
void CRangeBearingKFSLAM::OnGetObservationsAndDataAssociation(
	std::vector<KFArray_OBS>    &Z,
	vector_int                  &data_association,
	const vector<KFArray_OBS>   &all_predictions,
	const KFMatrix              &S,
	const vector_size_t         &lm_indices_in_S,
	const KFMatrix_OxO          &R
	)
{
	MRPT_START

    //static const size_t vehicle_size = get_vehicle_size();
    static const size_t obs_size  = get_observation_size();

	// Z: Observations
	CObservationBearingRange::TMeasurementList::const_iterator itObs;

	CObservationBearingRangePtr obs = m_SF->getObservationByClass<CObservationBearingRange>();
	ASSERT_(obs);

	const size_t N = obs->sensedData.size();
	Z.resize(N);

	size_t row;
	for (row=0,itObs = obs->sensedData.begin();itObs!=obs->sensedData.end();itObs++,row++)
	{
		// Fill one row in Z:
		Z[row][0] =itObs->range;
		Z[row][1] =itObs->yaw;
		Z[row][2] =itObs->pitch;
	}

	// Data association:
	// ---------------------
	data_association.assign(N,-1);  // Initially, all new landmarks

	// For each observed LM:
	vector_size_t   obs_idxs_needing_data_assoc;
	obs_idxs_needing_data_assoc.reserve(N);

	{
		vector_int::iterator itDA;
		CObservationBearingRange::TMeasurementList::const_iterator itObs;
		size_t row;
		for (row=0,itObs = obs->sensedData.begin(),itDA=data_association.begin();itObs!=obs->sensedData.end();itObs++,itDA++,row++)
		{
    		// Fill data asociation: Using IDs!
			if (itObs->landmarkID<0)
				obs_idxs_needing_data_assoc.push_back(row);
			else
			{
    			mrpt::utils::bimap<CLandmark::TLandmarkID,unsigned int>::iterator itID;
    			if ( (itID = m_IDs.find_key( itObs->landmarkID )) != m_IDs.end()  )
    				*itDA = itID->second; // This row in Z corresponds to the i'th map element in the state vector:
			}
		}
	}

	// ---- Perform data association ----
	//  Only for observation indices in "obs_idxs_needing_data_assoc"
	if (obs_idxs_needing_data_assoc.empty())
	{
		// We don't need to do DA:
		m_last_data_association = TDataAssocInfo();
	}
	else
	{
		// Build a Z matrix with the observations that need dat.assoc:
		const size_t nObsDA = obs_idxs_needing_data_assoc.size();

		CMatrixTemplateNumeric<kftype>  Z_obs_means(nObsDA,obs_size);
		for (size_t i=0;i<nObsDA;i++)
		{
			const size_t idx = obs_idxs_needing_data_assoc[i];
			for (unsigned k=0;k<obs_size;k++)
				Z_obs_means.get_unsafe(i,k) = Z[idx][k];
		}

		// Vehicle uncertainty
		KFMatrix_VxV  Pxx(UNINITIALIZED_MATRIX  );
		m_pkk.extractMatrix(0,0, Pxx);

		// Build predictions:
		// ---------------------------

		const size_t nPredictions = lm_indices_in_S.size();
		m_last_data_association.clear();

		// S is the covariance of the predictions:
		m_last_data_association.Y_pred_covs = S;

		// The means:
		m_last_data_association.Y_pred_means.setSize(nPredictions, obs_size);
		for (size_t q=0;q<nPredictions;q++)
		{
			const size_t i = lm_indices_in_S[q];
			for (size_t w=0;w<obs_size;w++)
				m_last_data_association.Y_pred_means.get_unsafe(q,w) = all_predictions[i][w];
			m_last_data_association.predictions_IDs.push_back( i ); // for the conversion of indices...
		}

		// Do Dat. Assoc :
		// ---------------------------
		if (nPredictions)
		{
			CMatrixDouble  Z_obs_cov = CMatrixDouble(R);

			mrpt::slam::data_association_full_covariance(
				Z_obs_means, //Z_obs_cov,
				m_last_data_association.Y_pred_means,m_last_data_association.Y_pred_covs,
				m_last_data_association.results,
				options.data_assoc_method,
				options.data_assoc_metric,
				options.data_assoc_IC_chi2_thres,
				true,   // Use KD-tree
				m_last_data_association.predictions_IDs,
				options.data_assoc_IC_metric,
				options.data_assoc_IC_ml_threshold
				);

			// Return pairings to the main KF algorithm:
			for (map<size_t,size_t>::const_iterator it= m_last_data_association.results.associations.begin();it!=m_last_data_association.results.associations.end();it++)
				data_association[ it->first ] = it->second;
		}
	}
	// ---- End of data association ----

	MRPT_END
}


/** This virtual function musts normalize the state vector and covariance matrix (only if its necessary).
 */
void  CRangeBearingKFSLAM::OnNormalizeStateVector()
{
	MRPT_START

	// Check angles:
    m_xkk[3] = math::wrapToPi( m_xkk[3] );
    m_xkk[4] = math::wrapToPi( m_xkk[4] );
    m_xkk[5] = math::wrapToPi( m_xkk[5] );

	MRPT_END
}

/*---------------------------------------------------------------
						loadOptions
  ---------------------------------------------------------------*/
void CRangeBearingKFSLAM::loadOptions( const mrpt::utils::CConfigFileBase &ini )
{
	// Main
	options.loadFromConfigFile( ini, "RangeBearingKFSLAM" );
	KF_options.loadFromConfigFile( ini, "RangeBearingKFSLAM_KalmanFilter" );
	// partition algorithm:
	mapPartitioner.options.loadFromConfigFile(ini,"RangeBearingKFSLAM");
}

/*---------------------------------------------------------------
						loadOptions
  ---------------------------------------------------------------*/
void CRangeBearingKFSLAM::TOptions::loadFromConfigFile(
	const mrpt::utils::CConfigFileBase	&source,
	const std::string		&section)
{
	stds_Q_no_odo[3] = RAD2DEG(stds_Q_no_odo[3]);
	stds_Q_no_odo[4] = RAD2DEG(stds_Q_no_odo[4]);
	stds_Q_no_odo[5] = RAD2DEG(stds_Q_no_odo[5]);

	source.read_vector(section,"stds_Q_no_odo", stds_Q_no_odo, stds_Q_no_odo );
	ASSERT_(stds_Q_no_odo.size()==6)

	stds_Q_no_odo[3] = DEG2RAD(stds_Q_no_odo[3]);
	stds_Q_no_odo[4] = DEG2RAD(stds_Q_no_odo[4]);
	stds_Q_no_odo[5] = DEG2RAD(stds_Q_no_odo[5]);

	std_sensor_range	= source.read_float(section,"std_sensor_range", std_sensor_range);
	std_sensor_yaw		= DEG2RAD( source.read_float(section,"std_sensor_yaw_deg", RAD2DEG(std_sensor_yaw)));
	std_sensor_pitch	= DEG2RAD( source.read_float(section,"std_sensor_pitch_deg", RAD2DEG(std_sensor_pitch) ));

	std_odo_z_additional = source.read_float(section,"std_odo_z_additional", std_odo_z_additional);

	MRPT_LOAD_CONFIG_VAR(doPartitioningExperiment,bool, source,section);
	MRPT_LOAD_CONFIG_VAR(partitioningMethod,int,  source,section);

	MRPT_LOAD_CONFIG_VAR(create_simplemap,bool, source,section);

	MRPT_LOAD_CONFIG_VAR_CAST(data_assoc_method,int, TDataAssociationMethod,   source,section);
	MRPT_LOAD_CONFIG_VAR_CAST(data_assoc_metric,int, TDataAssociationMetric,   source,section);
	MRPT_LOAD_CONFIG_VAR(data_assoc_IC_chi2_thres,double,  source, section);
	MRPT_LOAD_CONFIG_VAR_CAST(data_assoc_IC_metric,int, TDataAssociationMetric,   source,section);
	MRPT_LOAD_CONFIG_VAR(data_assoc_IC_ml_threshold,double,  source, section);

	MRPT_LOAD_CONFIG_VAR(quantiles_3D_representation, float, source,section);

}

/*---------------------------------------------------------------
						Constructor
  ---------------------------------------------------------------*/
CRangeBearingKFSLAM::TOptions::TOptions() :
	stds_Q_no_odo(6,0),
	std_sensor_range ( 0.01f),
	std_sensor_yaw   ( DEG2RAD( 0.2f )),
	std_sensor_pitch ( DEG2RAD( 0.2f )),
	std_odo_z_additional ( 0),
	doPartitioningExperiment(false),
	quantiles_3D_representation ( 3),
	partitioningMethod(0),
	data_assoc_method			(assocNN),
	data_assoc_metric			(metricMaha),
	data_assoc_IC_chi2_thres	(0.99),
	data_assoc_IC_metric		(metricMaha),
	data_assoc_IC_ml_threshold	(0.0),
	create_simplemap			(false)
{
	stds_Q_no_odo[0]=
	stds_Q_no_odo[1]=
	stds_Q_no_odo[2]=0.10f;
	stds_Q_no_odo[3]=
	stds_Q_no_odo[4]=
	stds_Q_no_odo[5]=DEG2RAD(4.0f);
}

/*---------------------------------------------------------------
						dumpToTextStream
  ---------------------------------------------------------------*/
void CRangeBearingKFSLAM::TOptions::dumpToTextStream( CStream &out)const
{
	out.printf("\n----------- [CRangeBearingKFSLAM::TOptions] ------------ \n\n");

	//out.printf("stds_Q_no_odo                           = "); stds_Q_no_odo

	out.printf("doPartitioningExperiment                = %c\n",	doPartitioningExperiment ? 'Y':'N' );
	out.printf("partitioningMethod                      = %i\n",	partitioningMethod );

}

void CRangeBearingKFSLAM::OnInverseObservationModel(
	const KFArray_OBS & in_z,
	KFArray_FEAT  & yn,
	KFMatrix_FxV  & dyn_dxv,
	KFMatrix_FxO  & dyn_dhn ) const
{
	MRPT_START

	/* -------------------------------------------
       Equations, obtained using matlab
        Refer to technical report "6D EKF derivation...", 2008

        x0 y0 z0 y p r         % Robot's 6D pose
        x0s y0s z0s ys ps rs   % Sensor's 6D pose relative to robot
        hn_r hn_y hn_p         % Observation hn: range, yaw, pitch

        xn yn zn               % Absolute 3D landmark coordinates:

        dyn_dxv   -> Jacobian of the inv. observation model wrt the robot pose
        dyn_dhn   -> Jacobian of the inv. observation model wrt each landmark observation

        Sizes:
	     hn:      1x3  <--
         yn:      1x3  -->
         dyn_dxv: 3x6  -->
         dyn_dhn: 3x3  -->
	  ------------------------------------------- */

	CObservationBearingRangePtr obs = m_SF->getObservationByClass<CObservationBearingRange>();
	ASSERT_(obs);
	const TPose3D   sensorPoseOnRobot = TPose3D( obs->sensorLocationOnRobot );

	// Mean of the prior of the robot pose:
	const TPose3D  robotPose( m_xkk[0],m_xkk[1],m_xkk[2],m_xkk[3],m_xkk[4],m_xkk[5] );

    // Robot 6D pose:
    kftype x0 = m_xkk[0];
    kftype y0 = m_xkk[1];
    kftype z0 = m_xkk[2];
    kftype y  = m_xkk[3];
    kftype p  = m_xkk[4];
    kftype r  = m_xkk[5];

    kftype cy = cos(y); kftype sy = sin(y);
    kftype cp = cos(p); kftype sp = sin(p);
    kftype cr = cos(r); kftype sr = sin(r);

    // Sensor 6D pose on robot:
    kftype x0s = sensorPoseOnRobot.x;
    kftype y0s = sensorPoseOnRobot.y;
    kftype z0s = sensorPoseOnRobot.z;
    kftype ys = sensorPoseOnRobot.yaw;
    kftype ps = sensorPoseOnRobot.pitch;
    kftype rs = sensorPoseOnRobot.roll;

    kftype cys = cos(ys); kftype sys = sin(ys);
    kftype cps = cos(ps); kftype sps = sin(ps);
    kftype crs = cos(rs); kftype srs = sin(rs);

    kftype hn_r = in_z[0];
    kftype hn_y = in_z[1];
    kftype hn_p = in_z[2];

    kftype chn_y = cos(hn_y);  kftype shn_y = sin(hn_y);
    kftype chn_p = cos(hn_p);  kftype shn_p = sin(hn_p);

    // Compute the mean 3D absolute coordinates:
    yn[0] = (cy*cp*cys*cps+(cy*sp*sr-sy*cr)*sys*cps-(cy*sp*cr+sy*sr)*sps)*hn_r*chn_y*chn_p+(cy*cp*(cys*sps*srs-sys*crs)+(cy*sp*sr-sy*cr)*(sys*sps*srs+cys*crs)+(cy*sp*cr+sy*sr)*cps*srs)*hn_r*shn_y*chn_p-(cy*cp*(cys*sps*crs+sys*srs)+(cy*sp*sr-sy*cr)*(sys*sps*crs-cys*srs)+(cy*sp*cr+sy*sr)*cps*crs)*hn_r*shn_p+cy*cp*x0s+(cy*sp*sr-sy*cr)*y0s+(cy*sp*cr+sy*sr)*z0s+x0;
    yn[1] = (sy*cp*cys*cps+(sy*sp*sr+cy*cr)*sys*cps-(sy*sp*cr-cy*sr)*sps)*hn_r*chn_y*chn_p+(sy*cp*(cys*sps*srs-sys*crs)+(sy*sp*sr+cy*cr)*(sys*sps*srs+cys*crs)+(sy*sp*cr-cy*sr)*cps*srs)*hn_r*shn_y*chn_p-(sy*cp*(cys*sps*crs+sys*srs)+(sy*sp*sr+cy*cr)*(sys*sps*crs-cys*srs)+(sy*sp*cr-cy*sr)*cps*crs)*hn_r*shn_p+sy*cp*x0s+(sy*sp*sr+cy*cr)*y0s+(sy*sp*cr-cy*sr)*z0s+y0;
    yn[2] = (-sp*cys*cps+cp*sr*sys*cps-cp*cr*sps)*hn_r*chn_y*chn_p+(-sp*(cys*sps*srs-sys*crs)+cp*sr*(sys*sps*srs+cys*crs)+cp*cr*cps*srs)*hn_r*shn_y*chn_p-(-sp*(cys*sps*crs+sys*srs)+cp*sr*(sys*sps*crs-cys*srs)+cp*cr*cps*crs)*hn_r*shn_p-sp*x0s+cp*sr*y0s+cp*cr*z0s+z0;

    // Jacobian wrt xv:
    dyn_dxv(0,0) = 1;
    dyn_dxv(0,1) = 0;
    dyn_dxv(0,2) = 0;
    dyn_dxv(0,3) = (-sy*cp*cys*cps+(-sy*sp*sr-cy*cr)*sys*cps-(-sy*sp*cr+cy*sr)*sps)*hn_r*chn_y*chn_p+(-sy*cp*(cys*sps*srs-sys*crs)+(-sy*sp*sr-cy*cr)*(sys*sps*srs+cys*crs)+(-sy*sp*cr+cy*sr)*cps*srs)*hn_r*shn_y*chn_p-(-sy*cp*(cys*sps*crs+sys*srs)+(-sy*sp*sr-cy*cr)*(sys*sps*crs-cys*srs)+(-sy*sp*cr+cy*sr)*cps*crs)*hn_r*shn_p-sy*cp*x0s+(-sy*sp*sr-cy*cr)*y0s+(-sy*sp*cr+cy*sr)*z0s;
    dyn_dxv(0,4) = (-cy*sp*cys*cps+cy*cp*sr*sys*cps-cy*cp*cr*sps)*hn_r*chn_y*chn_p+(-cy*sp*(cys*sps*srs-sys*crs)+cy*cp*sr*(sys*sps*srs+cys*crs)+cy*cp*cr*cps*srs)*hn_r*shn_y*chn_p-(-cy*sp*(cys*sps*crs+sys*srs)+cy*cp*sr*(sys*sps*crs-cys*srs)+cy*cp*cr*cps*crs)*hn_r*shn_p-cy*sp*x0s+cy*cp*sr*y0s+cy*cp*cr*z0s;
    dyn_dxv(0,5) = ((cy*sp*cr+sy*sr)*sys*cps-(-cy*sp*sr+sy*cr)*sps)*hn_r*chn_y*chn_p+((cy*sp*cr+sy*sr)*(sys*sps*srs+cys*crs)+(-cy*sp*sr+sy*cr)*cps*srs)*hn_r*shn_y*chn_p-((cy*sp*cr+sy*sr)*(sys*sps*crs-cys*srs)+(-cy*sp*sr+sy*cr)*cps*crs)*hn_r*shn_p+(cy*sp*cr+sy*sr)*y0s+(-cy*sp*sr+sy*cr)*z0s;

    dyn_dxv(1,0) = 0;
    dyn_dxv(1,1) = 1;
    dyn_dxv(1,2) = 0;
    dyn_dxv(1,3) = (cy*cp*cys*cps+(cy*sp*sr-sy*cr)*sys*cps-(cy*sp*cr+sy*sr)*sps)*hn_r*chn_y*chn_p+(cy*cp*(cys*sps*srs-sys*crs)+(cy*sp*sr-sy*cr)*(sys*sps*srs+cys*crs)+(cy*sp*cr+sy*sr)*cps*srs)*hn_r*shn_y*chn_p-(cy*cp*(cys*sps*crs+sys*srs)+(cy*sp*sr-sy*cr)*(sys*sps*crs-cys*srs)+(cy*sp*cr+sy*sr)*cps*crs)*hn_r*shn_p+cy*cp*x0s+(cy*sp*sr-sy*cr)*y0s+(cy*sp*cr+sy*sr)*z0s;
    dyn_dxv(1,4) = (-sy*sp*cys*cps+sy*cp*sr*sys*cps-sy*cp*cr*sps)*hn_r*chn_y*chn_p+(-sy*sp*(cys*sps*srs-sys*crs)+sy*cp*sr*(sys*sps*srs+cys*crs)+sy*cp*cr*cps*srs)*hn_r*shn_y*chn_p-(-sy*sp*(cys*sps*crs+sys*srs)+sy*cp*sr*(sys*sps*crs-cys*srs)+sy*cp*cr*cps*crs)*hn_r*shn_p-sy*sp*x0s+sy*cp*sr*y0s+sy*cp*cr*z0s;
    dyn_dxv(1,5) = ((sy*sp*cr-cy*sr)*sys*cps-(-sy*sp*sr-cy*cr)*sps)*hn_r*chn_y*chn_p+((sy*sp*cr-cy*sr)*(sys*sps*srs+cys*crs)+(-sy*sp*sr-cy*cr)*cps*srs)*hn_r*shn_y*chn_p-((sy*sp*cr-cy*sr)*(sys*sps*crs-cys*srs)+(-sy*sp*sr-cy*cr)*cps*crs)*hn_r*shn_p+(sy*sp*cr-cy*sr)*y0s+(-sy*sp*sr-cy*cr)*z0s;

    dyn_dxv(2,0) = 0;
    dyn_dxv(2,1) = 0;
    dyn_dxv(2,2) = 1;
    dyn_dxv(2,3) = 0;
    dyn_dxv(2,4) = (-cp*cys*cps-sp*sr*sys*cps+sp*cr*sps)*hn_r*chn_y*chn_p+(-cp*(cys*sps*srs-sys*crs)-sp*sr*(sys*sps*srs+cys*crs)-sp*cr*cps*srs)*hn_r*shn_y*chn_p-(-cp*(cys*sps*crs+sys*srs)-sp*sr*(sys*sps*crs-cys*srs)-sp*cr*cps*crs)*hn_r*shn_p-cp*x0s-sp*sr*y0s-sp*cr*z0s;
    dyn_dxv(2,5) = (cp*cr*sys*cps+cp*sr*sps)*hn_r*chn_y*chn_p+(cp*cr*(sys*sps*srs+cys*crs)-cp*sr*cps*srs)*hn_r*shn_y*chn_p-(cp*cr*(sys*sps*crs-cys*srs)-cp*sr*cps*crs)*hn_r*shn_p+cp*cr*y0s-cp*sr*z0s;

    // Jacobian wrt hn:
    dyn_dhn(0,0) = (cy*cp*cys*cps+(cy*sp*sr-sy*cr)*sys*cps-(cy*sp*cr+sy*sr)*sps)*chn_y*chn_p+(cy*cp*(cys*sps*srs-sys*crs)+(cy*sp*sr-sy*cr)*(sys*sps*srs+cys*crs)+(cy*sp*cr+sy*sr)*cps*srs)*shn_y*chn_p-(cy*cp*(cys*sps*crs+sys*srs)+(cy*sp*sr-sy*cr)*(sys*sps*crs-cys*srs)+(cy*sp*cr+sy*sr)*cps*crs)*shn_p;
    dyn_dhn(0,1) = -(cy*cp*cys*cps+(cy*sp*sr-sy*cr)*sys*cps-(cy*sp*cr+sy*sr)*sps)*hn_r*shn_y*chn_p+(cy*cp*(cys*sps*srs-sys*crs)+(cy*sp*sr-sy*cr)*(sys*sps*srs+cys*crs)+(cy*sp*cr+sy*sr)*cps*srs)*hn_r*chn_y*chn_p;
    dyn_dhn(0,2) = -(cy*cp*cys*cps+(cy*sp*sr-sy*cr)*sys*cps-(cy*sp*cr+sy*sr)*sps)*hn_r*chn_y*shn_p-(cy*cp*(cys*sps*srs-sys*crs)+(cy*sp*sr-sy*cr)*(sys*sps*srs+cys*crs)+(cy*sp*cr+sy*sr)*cps*srs)*hn_r*shn_y*shn_p-(cy*cp*(cys*sps*crs+sys*srs)+(cy*sp*sr-sy*cr)*(sys*sps*crs-cys*srs)+(cy*sp*cr+sy*sr)*cps*crs)*hn_r*chn_p;

    dyn_dhn(1,0) = (sy*cp*cys*cps+(sy*sp*sr+cy*cr)*sys*cps-(sy*sp*cr-cy*sr)*sps)*chn_y*chn_p+(sy*cp*(cys*sps*srs-sys*crs)+(sy*sp*sr+cy*cr)*(sys*sps*srs+cys*crs)+(sy*sp*cr-cy*sr)*cps*srs)*shn_y*chn_p-(sy*cp*(cys*sps*crs+sys*srs)+(sy*sp*sr+cy*cr)*(sys*sps*crs-cys*srs)+(sy*sp*cr-cy*sr)*cps*crs)*shn_p;
    dyn_dhn(1,1) =-(sy*cp*cys*cps+(sy*sp*sr+cy*cr)*sys*cps-(sy*sp*cr-cy*sr)*sps)*hn_r*shn_y*chn_p+(sy*cp*(cys*sps*srs-sys*crs)+(sy*sp*sr+cy*cr)*(sys*sps*srs+cys*crs)+(sy*sp*cr-cy*sr)*cps*srs)*hn_r*chn_y*chn_p;
    dyn_dhn(1,2) = -(sy*cp*cys*cps+(sy*sp*sr+cy*cr)*sys*cps-(sy*sp*cr-cy*sr)*sps)*hn_r*chn_y*shn_p-(sy*cp*(cys*sps*srs-sys*crs)+(sy*sp*sr+cy*cr)*(sys*sps*srs+cys*crs)+(sy*sp*cr-cy*sr)*cps*srs)*hn_r*shn_y*shn_p-(sy*cp*(cys*sps*crs+sys*srs)+(sy*sp*sr+cy*cr)*(sys*sps*crs-cys*srs)+(sy*sp*cr-cy*sr)*cps*crs)*hn_r*chn_p;

    dyn_dhn(2,0) =(-sp*cys*cps+cp*sr*sys*cps-cp*cr*sps)*chn_y*chn_p+(-sp*(cys*sps*srs-sys*crs)+cp*sr*(sys*sps*srs+cys*crs)+cp*cr*cps*srs)*shn_y*chn_p-(-sp*(cys*sps*crs+sys*srs)+cp*sr*(sys*sps*crs-cys*srs)+cp*cr*cps*crs)*shn_p;
    dyn_dhn(2,1) =-(-sp*cys*cps+cp*sr*sys*cps-cp*cr*sps)*hn_r*shn_y*chn_p+(-sp*(cys*sps*srs-sys*crs)+cp*sr*(sys*sps*srs+cys*crs)+cp*cr*cps*srs)*hn_r*chn_y*chn_p;
    dyn_dhn(2,2) =-(-sp*cys*cps+cp*sr*sys*cps-cp*cr*sps)*hn_r*chn_y*shn_p-(-sp*(cys*sps*srs-sys*crs)+cp*sr*(sys*sps*srs+cys*crs)+cp*cr*cps*srs)*hn_r*shn_y*shn_p-(-sp*(cys*sps*crs+sys*srs)+cp*sr*(sys*sps*crs-cys*srs)+cp*cr*cps*crs)*hn_r*chn_p;

    MRPT_END
}


/** If applicable to the given problem, do here any special handling of adding a new landmark to the map.
  * \param in_obsIndex The index of the observation whose inverse sensor is to be computed. It corresponds to the row in in_z where the observation can be found.
  * \param in_idxNewFeat The index that this new feature will have in the state vector (0:just after the vehicle state, 1: after that,...). Save this number so data association can be done according to these indices.
  * \sa OnInverseObservationModel
  */
void CRangeBearingKFSLAM::OnNewLandmarkAddedToMap(
	const size_t	in_obsIdx,
	const size_t	in_idxNewFeat )
{
	MRPT_START

	CObservationBearingRangePtr obs = m_SF->getObservationByClass<CObservationBearingRange>();
	ASSERT_(obs);

	// ----------------------------------------------
	// introduce in the lists of ID<->index in map:
	// ----------------------------------------------
	ASSERT_( in_obsIdx < obs->sensedData.size() )

	if (obs->sensedData[in_obsIdx].landmarkID>=0)
	{
		// The sensor provides us a LM ID... use it:
		m_IDs.insert(
			obs->sensedData[in_obsIdx].landmarkID,
			in_idxNewFeat );
	}
	else
	{
		// Features do not have IDs... use indices:
		m_IDs.insert( in_idxNewFeat, in_idxNewFeat);
	}

    MRPT_END
}


/*---------------------------------------------------------------
						getAs3DObject
  ---------------------------------------------------------------*/
void  CRangeBearingKFSLAM::getAs3DObject( mrpt::opengl::CSetOfObjectsPtr	&outObj ) const
{
    outObj->clear();

	// ------------------------------------------------
	//  Add the XYZ corner for the current area:
	// ------------------------------------------------
	outObj->insert( opengl::stock_objects::CornerXYZ() );


	// 3D ellipsoid for robot pose:
	CPointPDFGaussian	pointGauss;
    pointGauss.mean.x( m_xkk[0] );
    pointGauss.mean.y( m_xkk[1] );
    pointGauss.mean.z( m_xkk[2] );
    CMatrixTemplateNumeric<kftype>  COV;
    m_pkk.extractMatrix(0,0,3,3, COV);
    pointGauss.cov = COV;

    {
		opengl::CEllipsoidPtr ellip = opengl::CEllipsoid::Create();

		ellip->setPose(pointGauss.mean);
		ellip->setCovMatrix(pointGauss.cov);
		ellip->enableDrawSolid3D(false);
		ellip->setQuantiles( options.quantiles_3D_representation );
		ellip->set3DsegmentsCount(10);
		ellip->setColor(1,0,0);

		outObj->insert( ellip );
    }


	// 3D ellipsoids for landmarks:
	size_t i,nLMs = (m_xkk.size()-6)/3;
	for (i=0;i<nLMs;i++)
	{
        pointGauss.mean.x( m_xkk[6+3*i+0] );
        pointGauss.mean.y( m_xkk[6+3*i+1] );
        pointGauss.mean.z( m_xkk[6+3*i+2] );
        m_pkk.extractMatrix(6+3*i,6+3*i,3,3, COV);
        pointGauss.cov = COV;

		opengl::CEllipsoidPtr ellip = opengl::CEllipsoid::Create();

		ellip->setName( format( "%"PRIuPTR,i ) );
		ellip->enableShowName(true);
		ellip->setPose( pointGauss.mean );
		ellip->setCovMatrix( pointGauss.cov );
		ellip->enableDrawSolid3D(false);
		ellip->setQuantiles( options.quantiles_3D_representation );
		ellip->set3DsegmentsCount(10);

		ellip->setColor(0,0,1);

		// Set color depending on partitions?
		if (options.doPartitioningExperiment)
		{
		    // This is done for each landmark:
			map<int,bool>    belongToPartition;
			const CSensFrameProbSequence *SFs = &m_SFs; //mapPartitioner.getSequenceOfFrames();

			for (size_t p=0;p<m_lastPartitionSet.size();p++)
			{
				for (size_t w=0;w<m_lastPartitionSet[p].size();w++)
				{
					// Check if landmark #i is in the SF of m_lastPartitionSet[p][w]:
					CLandmark::TLandmarkID  i_th_ID = m_IDs.inverse(i);

					// Look for the lm_ID in the SF:
					CPose3DPDFPtr    pdf;
					CSensoryFramePtr SF_i;
					SFs->get( m_lastPartitionSet[p][w], pdf, SF_i);

					CObservationBearingRangePtr obs = SF_i->getObservationByClass<CObservationBearingRange>();

					for (size_t o=0;o<obs->sensedData.size();o++)
					{
						if ( obs->sensedData[o].landmarkID == i_th_ID )
						{
							belongToPartition[p]=true;
							break;
						}
					} // end for o
				} // end for w
			} // end for p

			// Build label:
			string strParts("[");

			for (map<int,bool>::iterator it=belongToPartition.begin();it!=belongToPartition.end();++it)
			{
				if (it!=belongToPartition.begin()) strParts += string(",");
				strParts += format("%i",it->first);
			}

			ellip->setName( ellip->getName() + strParts + string("]") );

		}

		outObj->insert( ellip );
	}
}


/*---------------------------------------------------------------
						getLastPartitionLandmarks
  ---------------------------------------------------------------*/
void CRangeBearingKFSLAM::getLastPartitionLandmarksAsIfFixedSubmaps( size_t K, std::vector<vector_uint>	&landmarksMembership )
{
    // temporary copy:
    std::vector<vector_uint> tmpParts = m_lastPartitionSet;

    // Fake "m_lastPartitionSet":

    // Fixed partitions every K observations:
    vector<vector_uint>   partitions;
    vector_uint           tmpCluster;

    for (size_t i=0;i<m_SFs.size();i++)
    {
        tmpCluster.push_back(i);
        if ( (i % K) == 0 )
        {
            partitions.push_back(tmpCluster);
            tmpCluster.clear();
            tmpCluster.push_back(i);  // This observation "i" is shared between both clusters
        }
    }
    m_lastPartitionSet = partitions;

    // Call the actual method:
    getLastPartitionLandmarks(landmarksMembership);

    // Replace copy:
    m_lastPartitionSet = tmpParts;
}

/*---------------------------------------------------------------
						getLastPartitionLandmarks
  ---------------------------------------------------------------*/
void CRangeBearingKFSLAM::getLastPartitionLandmarks(
    std::vector<vector_uint>	&landmarksMembership ) const
{
    landmarksMembership.clear();

    // All the computation is made based on "m_lastPartitionSet"

    if (!options.doPartitioningExperiment) return;

	size_t i,nLMs = (m_xkk.size()-6)/3;
	for (i=0;i<nLMs;i++)
	{
        map<int,bool>    belongToPartition;
        const CSensFrameProbSequence *SFs = &m_SFs; //mapPartitioner.getSequenceOfFrames();

        for (size_t p=0;p<m_lastPartitionSet.size();p++)
        {
            for (size_t w=0;w<m_lastPartitionSet[p].size();w++)
            {
                // Check if landmark #i is in the SF of m_lastPartitionSet[p][w]:
                CLandmark::TLandmarkID  i_th_ID = m_IDs.inverse(i);

                // Look for the lm_ID in the SF:
                CPose3DPDFPtr pdf;
                CSensoryFramePtr SF_i;
                SFs->get( m_lastPartitionSet[p][w], pdf, SF_i);

                CObservationBearingRangePtr obs = SF_i->getObservationByClass<CObservationBearingRange>();

                for (size_t o=0;o<obs->sensedData.size();o++)
                {
                    if ( obs->sensedData[o].landmarkID == i_th_ID )
                    {
                        belongToPartition[p]=true;
                        break;
                    }
                } // end for o
            } // end for w
        } // end for p

        // Build membership list:
        vector_uint membershipOfThisLM;

        for (map<int,bool>::iterator it=belongToPartition.begin();it!=belongToPartition.end();++it)
            membershipOfThisLM.push_back(it->first);

        landmarksMembership.push_back( membershipOfThisLM );
	} // end for i
}


/*---------------------------------------------------------------
            computeOffDiagonalBlocksApproximationError
  ---------------------------------------------------------------*/
double CRangeBearingKFSLAM::computeOffDiagonalBlocksApproximationError(
    const std::vector<vector_uint>	&landmarksMembership ) const
{
    MRPT_START

    // Compute the information matrix:
    CMatrixTemplateNumeric<kftype> fullCov( m_pkk );
	size_t i;
    for (i=0;i<6;i++)
        fullCov(i,i) = max(fullCov(i,i), 1e-6);

    CMatrixTemplateNumeric<kftype>		H( fullCov.inv() );
    H.Abs();        // Replace by absolute values:

    double sumOffBlocks = 0;
    unsigned int nLMs = landmarksMembership.size();

    ASSERT_((6+nLMs*3)==fullCov.getColCount());

    for (i=0;i<nLMs;i++)
    {
        for (size_t j=i+1;j<nLMs;j++)
        {
            // Sum the cross cov. between LM(i) and LM(j)??
            // --> Only if there is no common cluster:
            if ( 0==math::countCommonElements( landmarksMembership[i],landmarksMembership[j] ) )
            {
                size_t col = 6+i*3;
                size_t row = 6+j*3;
                sumOffBlocks += 2 * H.sum( row,col,row+2,col+2 );
            }
        }
    }

    return sumOffBlocks / H.sum(6,6);  // Starting (6,6)-end
    MRPT_END
}

/*---------------------------------------------------------------
              reconsiderPartitionsNow
  ---------------------------------------------------------------*/
void CRangeBearingKFSLAM::reconsiderPartitionsNow()
{
	mapPartitioner.markAllNodesForReconsideration();

    vector<vector_uint>   partitions; // A different buffer for making this thread-safe some day...

    mapPartitioner.updatePartitions( partitions );

    m_lastPartitionSet = partitions;
}

/*---------------------------------------------------------------
              saveMapAndPathRepresentationAsMATLABFile
  ---------------------------------------------------------------*/
void CRangeBearingKFSLAM::saveMapAndPath2DRepresentationAsMATLABFile(
	const string &fil,
	float stdCount,
	const string &styleLandmarks,
	const string &stylePath,
	const string &styleRobot ) const
{
	FILE	*f= os::fopen(fil.c_str(),"wt");
	if (!f) return;

	CMatrixDouble cov(2,2);
	CVectorDouble mean(2);

	// Header:
	os::fprintf(f,"%%--------------------------------------------------------------------\n");
	os::fprintf(f,"%% File automatically generated using the MRPT method:\n");
	os::fprintf(f,"%% 'CRangeBearingKFSLAM::saveMapAndPath2DRepresentationAsMATLABFile'\n");
	os::fprintf(f,"%%\n");
	os::fprintf(f,"%%                        ~ MRPT ~\n");
	os::fprintf(f,"%%  Jose Luis Blanco Claraco, University of Malaga @ 2008\n");
	os::fprintf(f,"%%  http://mrpt.sourceforge.net/ \n");
	os::fprintf(f,"%%--------------------------------------------------------------------\n");

	// Main code:
	os::fprintf(f,"hold on;\n\n");

	size_t i, nLMs = (m_xkk.size()-get_vehicle_size())/get_feature_size();

	for (i=0;i<nLMs;i++)
	{
		size_t idx = 6+i*3;

		cov(0,0) = m_pkk(idx+0,idx+0);
		cov(1,1) = m_pkk(idx+1,idx+1);
		cov(0,1) = cov(1,0) = m_pkk(idx+0,idx+1);

		mean[0] = m_xkk[idx+0];
		mean[1] = m_xkk[idx+1];

		// Command to draw the 2D ellipse:
		os::fprintf(f, "%s", math::MATLAB_plotCovariance2D( cov, mean, stdCount,styleLandmarks ).c_str() );
	}


	// Now: the robot path:
	// ------------------------------
	if (m_SFs.size())
	{
		os::fprintf(f,"\nROB_PATH=[");
		for (i=0;i<m_SFs.size();i++)
		{
			CSensoryFramePtr dummySF;
			CPose3DPDFPtr pdf3D;
			m_SFs.get(i,pdf3D,dummySF);

			CPose3D p;
			pdf3D->getMean(p);
			CPoint3D pnt3D(p); // 6D -> 3D only

			os::fprintf(f,"%.04f %.04f", pnt3D.x(), pnt3D.y() );
			if (i<(m_SFs.size()-1))
				os::fprintf(f,";");
		}
		os::fprintf(f,"];\n");

		os::fprintf(f,"plot(ROB_PATH(:,1),ROB_PATH(:,2),'%s');\n",stylePath.c_str());
	}

	// The robot pose:
	cov(0,0) = m_pkk(0,0);
	cov(1,1) = m_pkk(1,1);
	cov(0,1) = cov(1,0) = m_pkk(0,1);

	mean[0] = m_xkk[0];
	mean[1] = m_xkk[1];

	os::fprintf(f, "%s", math::MATLAB_plotCovariance2D( cov, mean, stdCount, styleRobot ).c_str() );

	os::fprintf(f,"\naxis equal;\n");
	os::fclose(f);
}



/** Computes A=A-B, which may need to be re-implemented depending on the topology of the individual scalar components (eg, angles).
  */
void CRangeBearingKFSLAM::OnSubstractObservationVectors(KFArray_OBS &A, const KFArray_OBS &B) const
{
	A-=B;
	mrpt::math::wrapToPiInPlace(A[1]);
	mrpt::math::wrapToPiInPlace(A[2]);
}


/** Return the observation NOISE covariance matrix, that is, the model of the Gaussian additive noise of the sensor.
  * \param out_R The noise covariance matrix. It might be non diagonal, but it'll usually be.
  */
void CRangeBearingKFSLAM::OnGetObservationNoise(KFMatrix_OxO &out_R) const
{
	out_R(0,0)= square( options.std_sensor_range );
	out_R(1,1)= square( options.std_sensor_yaw );
	out_R(2,2)= square( options.std_sensor_pitch );
}

/** This will be called before OnGetObservationsAndDataAssociation to allow the application to reduce the number of covariance landmark predictions to be made.
  *  For example, features which are known to be "out of sight" shouldn't be added to the output list to speed up the calculations.
  * \param in_all_prediction_means The mean of each landmark predictions; the computation or not of the corresponding covariances is what we're trying to determined with this method.
  * \param out_LM_indices_to_predict The list of landmark indices in the map [0,getNumberOfLandmarksInTheMap()-1] that should be predicted.
  * \note This is not a pure virtual method, so it should be implemented only if desired. The default implementation returns a vector with all the landmarks in the map.
  * \sa OnGetObservations, OnDataAssociation
  */
void CRangeBearingKFSLAM::OnPreComputingPredictions(
	const vector<KFArray_OBS>	&prediction_means,
	vector_size_t				&out_LM_indices_to_predict ) const
{
	CObservationBearingRangePtr obs = m_SF->getObservationByClass<CObservationBearingRange>();
	ASSERT_(obs);

	const double sensor_max_range = obs->maxSensorDistance;
	const double fov_yaw   = obs->fieldOfView_yaw;
	const double fov_pitch = obs->fieldOfView_pitch;

	const double max_vehicle_loc_uncertainty = 4 * std::sqrt( m_pkk.get_unsafe(0,0) + m_pkk.get_unsafe(1,1)+m_pkk.get_unsafe(2,2) );
	const double max_vehicle_ang_uncertainty = 4 * std::sqrt( m_pkk.get_unsafe(3,3) + m_pkk.get_unsafe(4,4)+m_pkk.get_unsafe(5,5) );

	out_LM_indices_to_predict.clear();
	for (size_t i=0;i<prediction_means.size();i++)
		if (      prediction_means[i][0] < (       1.5  + sensor_max_range + max_vehicle_loc_uncertainty + 4*options.std_sensor_range) &&
		    fabs(prediction_means[i][1]) < (DEG2RAD(20) + 0.5*fov_yaw      + max_vehicle_ang_uncertainty + 4*options.std_sensor_yaw)   &&
			fabs(prediction_means[i][2]) < (DEG2RAD(20) + 0.5*fov_pitch    + max_vehicle_ang_uncertainty + 4*options.std_sensor_pitch)
			)
		{
			out_LM_indices_to_predict.push_back(i);
		}
}
