/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/core.h>
#include <mrpt/hwdrivers.h>
#include <mrpt/hwdrivers/CRovio.h>
#include <mrpt/utils/net_utils.h>
#include <mrpt/slam/CObservationImage.h>

using namespace mrpt::utils;
using namespace mrpt::slam;
using namespace mrpt::gui;
using namespace mrpt::hwdrivers;
using namespace mrpt::utils::net;


/*-------------------------------------------------
                     INITIALIZE
  -----------------------------------------------*/
void CRovio::Initialize(string &errormsg_out, string url_out, string user_out, string password_out)
{
	//string response;
	TOptions.IP=url_out;
	TOptions.user=user_out;
	TOptions.password=password_out;
	http_get (format("http://%s/rev.cgi?Cmd=nav&action=1",TOptions.IP.c_str()), response, errormsg_out, 80, TOptions.user, TOptions.password);

	if (!response.empty())
		cout<<"Response:\n"<<response<<endl;
	if (!errormsg_out.empty())
		cout<<"Error msg:\n"<<errormsg_out<<endl;
	else
	{
//		camerathread=true;
//		cam_hd=createThread(StartCamera2,this);
		loadCameraParam();
	}
}


/*-------------------------------------------------
                     MOVING
  -----------------------------------------------*/
void CRovio::Moving(int action, int direction, int speed, string &response, string &errormsg)
{
	string MF;
	if(action==18){ //3 argument command
		MF=format("http://%s/rev.cgi?Cmd=nav&action=%i&drive=%i&speed=%i",TOptions.IP.c_str(),action,direction,speed); //direction: movement direction or auxiliar task. (e.g. turn on light)
		http_get (MF, response, errormsg, 80, TOptions.user, TOptions.password);
	}else{	//1 argument command
		MF=format("http://%s/rev.cgi?Cmd=nav&action=%i",TOptions.IP.c_str(),action);
		http_get (MF, response, errormsg, 80, TOptions.user, TOptions.password);
	}
}

/*-------------------------------------------------
                     IP CAMERA
  -----------------------------------------------*/
void CRovio::StartCamera(void* ref)	//This function takes a frame and waits until getLastImage ask for it, and so on.
{
	try
	{
		CRovio *obj=(CRovio*) ref;

		CFFMPEG_InputStream		in_video;
		string video_url=format("rtsp://%s/webcam",obj->TOptions.IP.c_str());

		if (!in_video.openURL(video_url,false /*grayscale*/, false /* verbose */ ))
			return;

		while(obj->camerathread)
		{
			CObservationImagePtr  obs = CObservationImage::Create();

			if (in_video.retrieveFrame(obs->image) && !(obj->newFrame))
			{

				//Critical section
				mrpt::synch::CCriticalSectionLocker cs( &obj->buffer_img_cs );
	//			obj->buffer_img_cs.enter();
	//			obj->buffer_img.clear();
				obj->buffer_img = obs;
				obj->newFrame.set(true);
	//			obj->buffer_img_cs.leave();
	//			obs.clear_unique();	//Clearing the obs pointer.
				cout<<"Image grabbed\n";
			}
			else
			{
				obs.clear();	//If no image was copied, destroy the object.
				cout<<"The program doesn't receive any image\n";
			}
			mrpt::system::sleep(30);
		}//end while

		in_video.close();
	}
	catch(std::exception &e)
	{
		cout<<"Error in StartCamera thread";
		cerr << e.what() << endl;
	}
	catch(...)
	{
		cout<<"Error in StartCamera thread";
	}
}

void CRovio::StartCamera2(void* ref)		//This function is always receiving the video stream
{
	try
	{
		CRovio *obj=(CRovio*) ref;

		CFFMPEG_InputStream		in_video;
		string video_url=format("rtsp://%s/webcam",obj->TOptions.IP.c_str());

		if (!in_video.openURL(video_url,false /*grayscale*/, false /* verbose */ ))
			return;

		while(obj->camerathread)
		{
			CObservationImagePtr  obs = CObservationImage::Create();

			if (in_video.retrieveFrame(obs->image))
			{
				//Critical section
				{
					mrpt::synch::CCriticalSectionLocker cs( &obj->buffer_img_cs );
					obj->buffer_img = obs;
				}

//				static int i=0;
//				cout<<"Image grabbed " << ++i << endl;
			}
			else
				cout << "Error grabbing image" << endl;
			mrpt::system::sleep(100);
		}//end while

		in_video.close();
	}
	catch(std::exception &e)
	{
		cout<<"Error in StartCamera thread";
		cerr << e.what() << endl;
	}
	catch(...)
	{
		cout<<"Error in StartCamera thread";
	}

}
/*-------------------------------------------------
                     GET LAST IMAGE
  -----------------------------------------------*/
void CRovio::getLastImage(CObservationImagePtr& lastImage )	//This function grabbes the images taken by StartCamera
{
	mrpt::synch::CCriticalSectionLocker cs( &buffer_img_cs );
	if(newFrame)
	{
		//Critical Section
//		buffer_img_cs.enter();
		lastImage=buffer_img;
		newFrame.set(false);
//		buffer_img_cs.leave();
	}
}

void CRovio::getLastImage2(CObservationImagePtr& lastImage )		//This function grabbes the images taken by StartCamera2
{
	mrpt::synch::CCriticalSectionLocker cs( &buffer_img_cs );
		lastImage=buffer_img;
}

/*-------------------------------------------------
                   CAPTURE PICTURE
  -----------------------------------------------*/
void CRovio::captureImage(CImage & picture)
{
	try
	{
		vector_byte resp;
		string MF=format("http://%s/Jpeg/CamImg[0000].jpg",TOptions.IP.c_str());
		http_get (MF, resp, errormsg, 80, TOptions.user, TOptions.password);

		CMemoryStream stream( &resp[0],  resp.size()  );
		picture.loadFromStreamAsJPEG(stream);
		picture.saveToFile("0000.jpg");
		//cout<<"Response:\n"<<response<<endl;
	}
	catch(std::exception &e)
	{
		cerr << e.what() << endl;
		//return -1;
	}
}

//Load the camera parameters.
void CRovio::loadCameraParam()
{
	CMatrix distortion_matrix(1,4);
	intrinsic_matrix.loadFromTextFile("intrinsic_matrix.txt");
	distortion_matrix.loadFromTextFile("distortion_matrix.txt");
	distortion.resize(4);
	distortion[0] = distortion_matrix.get_unsafe(0,0);
	distortion[1] = distortion_matrix.get_unsafe(0,1);
	distortion[2] = distortion_matrix.get_unsafe(0,2);
	distortion[3] = distortion_matrix.get_unsafe(0,3);
}

void CRovio::captureImageRect(CImage & picture)
{
	try
	{
		vector_byte resp;
		string MF=format("http://%s/Jpeg/CamImg[0000].jpg",TOptions.IP.c_str());
		http_get (MF, resp, errormsg, 80, TOptions.user, TOptions.password);

		CMemoryStream stream( &resp[0],  resp.size()  );
		picture.loadFromStreamAsJPEG(stream);
		picture.rectifyImageInPlace(intrinsic_matrix, distortion);
		picture.saveToFile("0000.jpg");
		//cout<<"Response:\n"<<response<<endl;
	}
	catch(std::exception &e)
	{
		cerr << e.what() << endl;
		//return -1;
	}
}
/*-------------------------------------------------
                     GET ENCODERS
  -----------------------------------------------*/
long convertToLong(char *sLong)
{
	char * result=strpbrk(sLong,"-0123456789ABCDEF");
	char * stop;
	return strtol(result,&stop,16);
}

int* CRovio::getEncoders()
{
	//VAR
	string resp, error, field;
	//string field_name[12]={"Packet length","Not Used","Left Wheel:Dir Rotation","Left Wheel:Ticks","Right Wheel:Dir Rotation","Right Wheel:Ticks","Rear Wheel:Dir Rotation","Rear Wheel:Ticks","Not used","Head Position","Batery","Config Status"};
	size_t pos, length;
	int *a_enc = new int[11];	//12 encoder's fields
	long l_value;
	//Code
	Moving(20,0,0, resp, error);	//get Encoders string
	if(error.empty())
	{
		pos=(resp.find("responses =")+12);
		for(int i=0;i<=11;i++)
		{
			if ( (i==3)||(i==5)||(i==7) )
				length=4;
			else
				length=2;

			field = resp.substr(pos,length);
			pos+=length;

			/*---------- String to binary conv------------------*/
			char* cstr = new char [field.size()+1];
			strcpy (cstr, field.c_str());
			l_value = convertToLong(cstr);

			if ( (i==2)||(i==4)||(i==6) )	//just interested in bit(2)-> "0000X0"
				l_value=( (l_value>>1) & 0x1);
			a_enc[i]=l_value;
		}

	}else{
		cout <<"\n---------------------------------------------------" << endl;
		cout << "ERROR->" <<error <<endl;
	}
	return a_enc;
}

/*-------------------------------------------------
        GET POSITION WITH NORTHSTAR SYSTEM
  -----------------------------------------------*/
mrpt::math::TPose2D CRovio::getPosition()
{
	size_t x_pos, y_pos, theta_pos, lenght;
	string x_value, y_value, theta_value, MF, response, errormsg;
	mrpt::math::TPose2D pose;
	MF=format("http://%s/rev.cgi?Cmd=nav&action=1",TOptions.IP.c_str());
	http_get (MF, response, errormsg, 80, TOptions.user, TOptions.password); //Generating response

	//Getting x value
	x_pos=response.find("x=");
	x_value=response.substr((x_pos+2),8);
	lenght=x_value.find('|');
	x_value=x_value.substr(0,lenght);
	char* x_char=new char [lenght];
	strcpy(x_char, x_value.c_str());
	pose.x=atof(x_char);

	//Getting y value
	y_pos=response.find("y=");
	y_value=response.substr((y_pos+2),8);
	lenght=y_value.find('|');
	y_value=y_value.substr(0,lenght);
	char* y_char=new char [lenght];
	strcpy(y_char, y_value.c_str());
	pose.y=atof(y_char);

	//Getting theta value
	theta_pos=response.find("theta=");
	theta_value=response.substr((theta_pos+6),8);
	lenght=theta_value.find('|');
	theta_value=theta_value.substr(0,lenght);
	char* theta_char=new char [lenght];
	strcpy(theta_char, theta_value.c_str());
	pose.phi=atof(theta_char);

	return pose;
}

CRovio::CRovio() : newFrame(0)
{

}

CRovio::~CRovio()
{
	camerathread=false;	//kill camera Thread
	joinThread(cam_hd); //Wait till thread finish
}

/*-------------------------------------------------------------
Movement Control Commands
/rev.cgi?Cmd=nav&action=
-------------------------------------------------------------*/

