/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/stereoslam/CStereoSLAMEKF.h>

using namespace mrpt;
using namespace mrpt::vision;
using namespace mrpt::stereoslam;
using namespace mrpt::gui;
using namespace mrpt::slam;

// *********************************************
// TStereoSLAMEKFOptions: Default options initializer
// *********************************************
TStereoSLAMEKFOptions::TStereoSLAMEKFOptions() 
{
	sigmas.resize(3,0);
}

// *********************************************
// TStereoSLAMEKFOptions: loadFromConfigFile
// *********************************************
void TStereoSLAMEKFOptions::loadFromConfigFile(
	const utils::CConfigFileBase	&iniFile,
	const std::string				&section )
{
	iniFile.read_vector( section.c_str(), "sigmas", vector_double(3), sigmas ); 
} // end of loadFromConfigFile

// *********************************************
// TStereoSLAMEKFOptions: dumpToTextStream
// *********************************************
void TStereoSLAMEKFOptions::dumpToTextStream( 
	CStream &out ) const
{
	out.printf( "\n----------- [vision::TStereoSLAMEKFOptions] ------------ \n\n" );
	out.printf( "Considered sigmas [c,r,d] = %.1f %.1f %.1f", sigmas[0], sigmas[1], sigmas[2] );
}

// *********************************************
// Constructor
// *********************************************
CStereoSLAMEKF::CStereoSLAMEKF() :
	m_RBEKFSLAM ( ),
	m_options ( ) {}
// *********************************************
// Destructor
// *********************************************
CStereoSLAMEKF::~CStereoSLAMEKF() {}

// *********************************************
// processActionObservation
// *********************************************
void CStereoSLAMEKF::processActionObservation(
	CActionCollectionPtr &action, 
	CSensoryFramePtr &SF )
{
	MRPT_START

	CSensoryFramePtr newSF = CSensoryFrame::Create();
	ASSERT_( !SF->getObservationByClass<CObservationBearingRange>() );							// Ensure that there is no more Range & Bearing
	CObservationStereoImagesPtr obs = SF->getObservationByClass<CObservationStereoImages>();	// Get the stereo images observation of the SF
	if( !obs )
		return;

	// Compute the change in pose
	CActionRobotMovement3D odVisualAction;
	m_vOdometer.process( obs, odVisualAction.poseChange );										// Get the change in pose of the robot
	const TOdometryInfo odInfo = m_vOdometer.getInfo();											// Get the info of the odometry

	odVisualAction.estimationMethod = CActionRobotMovement3D::emVisualOdometry;											
	
	// Insert action into the action collection
	action->insert( odVisualAction );

	// Convert the stereo observation onto a bearing & range observation
	CObservationBearingRangePtr outObs;
	vision::StereoObs2BRObs( 
		odInfo.m_mfList,				// The list of matched features (with IDs), extracted from the odometer
		obs->intrinsicParams,			// The intrinsic parameters of the stereo system
		obs->rightCameraPose.x(),		// The baseline of the stereo camera
		m_options.sigmas,				// The assumed sigmas of the column, row and disparity
		*outObs );						// The output bearing & range observation

	newSF = SF;
	newSF->insert( outObs );

	m_RBEKFSLAM.processActionObservation( action, newSF );

    MRPT_END
}
