/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/stereoslam/CVisualOdometryStereo.h>
#include <mrpt/system/filesystem.h>
#include <algorithm>

using namespace mrpt;
using namespace mrpt::vision;
using namespace mrpt::stereoslam;
using namespace mrpt::gui;

// *********************************************
// Default options initializer
// *********************************************
TOdometryOptions::TOdometryOptions() :
	covMethod( TCov_Linear ),
	minNumFeatures( 25 ),
	minDispX( 0 ),
	minDispY( 0 ),
	minSetDist( 0 ),
	debug_nsamples( 1e4 ),
	VERBOSE( false ),
	PLOT_IMAGES( false ),
	SAVE_FILES( false ),
	USE_DISPERSION( false ),
	DEBUG_ESTIMATE_COVARIANCE( false ),
	USE_RANSAC( false ),
	TIME_INFO( false ),
	OUT_DIR( "LOG_VODOMETRY" )
{
} // end of 'default options initializer'

// *********************************************
// Default odometry info initializer
// *********************************************
TOdometryInfo::TOdometryInfo() :
	m_obs( ),
	m_mfList( ),
	m_leftFeats( ),
	m_rightFeats( ),
	m_Prev_cloud( )
{

}

// *********************************************
// Constructor
// *********************************************
CVisualOdometryStereo::CVisualOdometryStereo() :
	stereoParams( ),
	matchingOptions( ),
	odometryOptions( ),
	m_nIter( 0 ),
	m_firstObs( true ),
	m_fExtract( ),
	m_dbg_outEstimations( ),
	m_dbg_sampleCount( 0 )
{

#if VISUALODOMETRY_USES_GUI
	m_wind1 = NULL;
	m_wind2 = NULL;
#endif
} // end of 'constructor'

// *********************************************
// Destructor
// *********************************************
CVisualOdometryStereo::~CVisualOdometryStereo( )
{
#if VISUALODOMETRY_USES_GUI
	if( m_wind1 != NULL )
		delete m_wind1;
	if( m_wind2 != NULL )
		delete m_wind2;
#endif
} // end of 'destructor'

// *********************************************
// Reset
// *********************************************
void CVisualOdometryStereo::reset( )
{
	m_odInfo.m_leftFeats.clear();
	m_odInfo.m_rightFeats.clear();

	m_firstObs			= false;
	m_nIter				= 0;
	m_dbg_sampleCount	= 0;

} // end of 'reset' method

// *********************************************
// loadOptions
// *********************************************
void CVisualOdometryStereo::loadOptions( const std::string &configFile )
{
	ASSERT_( system::fileExists(configFile) );
	CConfigFile		cfg(configFile);

	odometryOptions.loadFromConfigFile( cfg, "OdometryOptions" );
	stereoParams.loadFromConfigFile( cfg, "StereoParameters" );
	matchingOptions.loadFromConfigFile( cfg, "MatchingOptions" );
}

/*-------------------------------------------------------------
			TOdometryOptions: loadFromConfigFile
-------------------------------------------------------------*/
void TOdometryOptions::loadFromConfigFile(
			const utils::CConfigFileBase	&iniFile,
			const std::string		&section)
{
	int cov = iniFile.read_int(section.c_str(),"covMethod",covMethod);
	switch (cov)
	{
		case 0:
			covMethod = TCov_Linear;
			break;
		case 1:
			covMethod = TCov_UT;
			break;
		case 2:
			covMethod = TCov_Particles;
			break;
	} // end switch

	MRPT_LOAD_CONFIG_VAR( minNumFeatures, int,  iniFile, section );
	MRPT_LOAD_CONFIG_VAR( minDispX, double,  iniFile, section );
	MRPT_LOAD_CONFIG_VAR( minDispY, double,  iniFile, section );
	MRPT_LOAD_CONFIG_VAR( minSetDist, double,  iniFile, section );
	MRPT_LOAD_CONFIG_VAR( debug_nsamples, int,  iniFile, section );
	MRPT_LOAD_CONFIG_VAR( VERBOSE, bool,  iniFile, section );
	MRPT_LOAD_CONFIG_VAR( PLOT_IMAGES, bool,  iniFile, section );
	MRPT_LOAD_CONFIG_VAR( SAVE_FILES, bool,  iniFile, section );
	MRPT_LOAD_CONFIG_VAR( USE_DISPERSION, bool, iniFile, section );
	MRPT_LOAD_CONFIG_VAR( DEBUG_ESTIMATE_COVARIANCE, bool, iniFile, section );
	MRPT_LOAD_CONFIG_VAR( USE_RANSAC, bool, iniFile, section );
	MRPT_LOAD_CONFIG_VAR( TIME_INFO, bool, iniFile, section );
	MRPT_LOAD_CONFIG_VAR( OUT_DIR, string, iniFile, section );

} // end of loadFromConfigFile

/*---------------------------------------------------------------
					TStereoSystemParams: dumpToTextStream
  ---------------------------------------------------------------*/
void  TOdometryOptions::dumpToTextStream(CStream	&out) const
{
	out.printf("\n----------- [vision::TOdometryOptions] ------------ \n\n");

	out.printf("Covariance computation	= ");
	switch (covMethod)
	{
	case TCov_Linear:
		out.printf("Linear\n");
		break;
	case TCov_UT:
		out.printf("UT\n");
		break;
	case TCov_Particles:
		out.printf("Particles\n");
		break;
	} // end switch

	LOADABLEOPTS_DUMP_VAR(minNumFeatures, int )
	LOADABLEOPTS_DUMP_VAR(minDispX, double )
	LOADABLEOPTS_DUMP_VAR(minDispY, double )
	LOADABLEOPTS_DUMP_VAR(minSetDist, double )
	LOADABLEOPTS_DUMP_VAR(VERBOSE, bool )
	LOADABLEOPTS_DUMP_VAR(PLOT_IMAGES, bool )
	LOADABLEOPTS_DUMP_VAR(SAVE_FILES, bool )
	LOADABLEOPTS_DUMP_VAR(USE_DISPERSION, bool )
	LOADABLEOPTS_DUMP_VAR(DEBUG_ESTIMATE_COVARIANCE, bool )
	LOADABLEOPTS_DUMP_VAR(debug_nsamples, int )
	LOADABLEOPTS_DUMP_VAR(USE_RANSAC, bool )
	LOADABLEOPTS_DUMP_VAR(TIME_INFO, bool )
	LOADABLEOPTS_DUMP_VAR(OUT_DIR, string )

}

// *********************************************
// storeMatchedFeatureList
// *********************************************
void  CVisualOdometryStereo::storeMatchedFeatureList( const CMatchedFeatureList &mfList )
{
	CMatchedFeatureList::const_iterator	itMatch;

	// Clear and resize the feature lists
	m_odInfo.m_leftFeats.clear();
	m_odInfo.m_rightFeats.clear();

	// Fill the feature lists
	for( itMatch = mfList.begin(); itMatch != mfList.end(); itMatch++)
	{
		m_odInfo.m_leftFeats.push_back( itMatch->first );
		m_odInfo.m_rightFeats.push_back( itMatch->second );
	}
} // end 'storeMathedFeatureList'

// *********************************************
// addPointsToLists
// *********************************************
void  CVisualOdometryStereo::addMatches( const CMatchedFeatureList &mfList )
{
	// Add the new points to the current KLT lists
	//TFeatureID idx1 = leftFeats.getMaxID();
	//TFeatureID idx2 = rightFeats.getMaxID();

	//ASSERT_( idx1 == idx2 );
	CMatchedFeatureList::const_iterator itList;
	for( itList = mfList.begin(); itList != mfList.end(); itList++)
	{
		//(*itList->first).ID = ++idx1;
		//(*itList->second).ID = ++idx2;

		m_odInfo.m_leftFeats.push_back( itList->first );
		m_odInfo.m_rightFeats.push_back( itList->second );
	} // end for

} // end of 'addPointsToLists' method

// *********************************************
// computeRTCovariance
// *********************************************
void  CVisualOdometryStereo::computeRTCovariance(
	const poses::CPose3D &meanPose_,
	const slam::CMetricMap::TMatchingPairList &list,
	const slam::CObservationVisualLandmarks &Pre_Cloud,
	const slam::CObservationVisualLandmarks &Cur_Cloud,
	math::CMatrixD &RTCov )
{
	const TPose3D meanPose = TPose3D(meanPose_);

	switch( odometryOptions.covMethod )
	{

	// *********************************************
	// Linear propagation of the uncertainty
	// *********************************************
	case TOdometryOptions::TCov_Linear:
	{
		// ITERATORS
		slam::CLandmarksMap::TCustomSequenceLandmarks::const_iterator	itLand1, itLand2;

		CMatrixDouble C(6,6);
		for(itLand1 = Pre_Cloud.landmarks.landmarks.begin(); itLand1 != Pre_Cloud.landmarks.landmarks.end(); itLand1++)
			 for( itLand2 = Cur_Cloud.landmarks.landmarks.begin(); itLand2 != Cur_Cloud.landmarks.landmarks.end(); itLand2++ )
			 {
				 if( itLand1->ID != itLand2->ID )
					continue;

				 // COVARIANCE MATRIX (Si = S1 + S2)
				 CMatrixDouble Si(3,3);
				 Si(0,0) = itLand1->pose_cov_11 + itLand2->pose_cov_11;
				 Si(0,1) = Si(1,0) = itLand1->pose_cov_12 + itLand2->pose_cov_12;
				 Si(0,2) = Si(2,0) = itLand1->pose_cov_13 + itLand2->pose_cov_13;

				 Si(1,1) = itLand1->pose_cov_22 + itLand2->pose_cov_22;
				 Si(1,2) = Si(2,1) = itLand1->pose_cov_23 + itLand2->pose_cov_23;

				 Si(2,2) = itLand1->pose_cov_33 + itLand2->pose_cov_33;

				 // DEBUG
				 /** /
				 FILE *mf = mrpt::system::os::fopen("si.txt", "wt");
				 mrpt::system::os::fprintf( mf, "%e\t%e\t%e\n%e\t%e\t%e\n%e\t%e\t%e\n",
					 Si(0,0), Si(0,1), Si(0,2),
					 Si(1,0), Si(1,1), Si(1,2),
					 Si(2,0), Si(2,1), Si(2,2));
				 mrpt::system::os::fclose( mf );
				/ **/

				 // H MATRIX
				 CMatrixDouble Hi(3,6);
				 Hi(0,0) = 1;	Hi(0,1) = 0;	Hi(0,2) = 0;
				 Hi(1,0) = 0;	Hi(1,1) = 1;	Hi(1,2) = 0;
				 Hi(2,0) = 0;	Hi(2,1) = 0;	Hi(2,2) = 1;

				 Hi(0,3) =	-sin(meanPose.yaw)*cos(meanPose.pitch)*itLand1->pose_mean.x +
						   (-sin(meanPose.yaw)*sin(meanPose.pitch)*sin(meanPose.roll) - cos(meanPose.yaw)*cos(meanPose.roll))*itLand1->pose_mean.y +
						   (-sin(meanPose.yaw)*sin(meanPose.pitch)*cos(meanPose.roll) + cos(meanPose.yaw)*sin(meanPose.roll))*itLand1->pose_mean.z;

				 Hi(0,4) =  -cos(meanPose.yaw)*sin(meanPose.pitch)*itLand1->pose_mean.x +
							 cos(meanPose.yaw)*cos(meanPose.pitch)*sin(meanPose.roll)*itLand1->pose_mean.y +
							 cos(meanPose.yaw)*cos(meanPose.pitch)*cos(meanPose.roll)*itLand1->pose_mean.z;

				 Hi(0,5) =	(cos(meanPose.yaw)*sin(meanPose.pitch)*cos(meanPose.roll) + sin(meanPose.yaw)*sin(meanPose.roll))*itLand1->pose_mean.y +
						   (-cos(meanPose.yaw)*sin(meanPose.pitch)*sin(meanPose.roll) + sin(meanPose.yaw)*cos(meanPose.roll))*itLand1->pose_mean.z;

				 Hi(1,3) =	 cos(meanPose.yaw)*cos(meanPose.pitch)*itLand1->pose_mean.x +
							(cos(meanPose.yaw)*sin(meanPose.pitch)*sin(meanPose.roll) - sin(meanPose.yaw)*cos(meanPose.roll))*itLand1->pose_mean.y +
							(cos(meanPose.yaw)*sin(meanPose.pitch)*cos(meanPose.roll) + sin(meanPose.yaw)*sin(meanPose.roll))*itLand1->pose_mean.z;

				 Hi(1,4) =	-sin(meanPose.yaw)*sin(meanPose.pitch)*itLand1->pose_mean.x +
							 sin(meanPose.yaw)*cos(meanPose.pitch)*sin(meanPose.roll)*itLand1->pose_mean.y +
							 sin(meanPose.yaw)*cos(meanPose.pitch)*cos(meanPose.roll)*itLand1->pose_mean.z;

				 Hi(1,5) =  (sin(meanPose.yaw)*sin(meanPose.pitch)*cos(meanPose.roll) - cos(meanPose.yaw)*sin(meanPose.roll))*itLand1->pose_mean.y +
						   (-sin(meanPose.yaw)*sin(meanPose.pitch)*sin(meanPose.roll) - cos(meanPose.yaw)*cos(meanPose.roll))*itLand1->pose_mean.z;

				 Hi(2,3) =	0;

				 Hi(2,4) =  -cos(meanPose.pitch)*itLand1->pose_mean.x -
							 sin(meanPose.pitch)*sin(meanPose.roll)*itLand1->pose_mean.y -
							 sin(meanPose.pitch)*cos(meanPose.roll)*itLand1->pose_mean.z;

				 Hi(2,5) =   cos(meanPose.pitch)*cos(meanPose.roll)*itLand1->pose_mean.y -
							 cos(meanPose.pitch)*sin(meanPose.roll)*itLand1->pose_mean.z;

				 //Si.inv().
				 C += (~Hi)*Si.inv()*Hi;

				 // DEBUG
				/** /
				 FILE *mf2 = mrpt::system::os::fopen("hi.txt", "wt");
				 mrpt::system::os::fprintf( mf2, "%e\t%e\t%e\t%e\t%e\t%e\n%e\t%e\t%e\t%e\t%e\t%e\n%e\t%e\t%e\t%e\t%e\t%e\n",
					 Hi(0,0), Hi(0,1), Hi(0,2), Hi(0,3), Hi(0,4), Hi(0,5),
					 Hi(1,0), Hi(1,1), Hi(1,2), Hi(1,3), Hi(1,4), Hi(1,5),
					 Hi(2,0), Hi(2,1), Hi(2,2), Hi(2,3), Hi(2,4), Hi(2,5));
				 mrpt::system::os::fclose( mf2 );

				 FILE *mf3 = mrpt::system::os::fopen("c.txt", "wt");
				 mrpt::system::os::fprintf( mf3, "%e\t%e\t%e\t%e\t%e\t%e\n%e\t%e\t%e\t%e\t%e\t%e\n%e\t%e\t%e\t%e\t%e\t%e\n%e\t%e\t%e\t%e\t%e\t%e\n%e\t%e\t%e\t%e\t%e\t%e\n%e\t%e\t%e\t%e\t%e\t%e\n",
					 C(0,0), C(0,1), C(0,2), C(0,3), C(0,4), C(0,5),
					 C(1,0), C(1,1), C(1,2), C(1,3), C(1,4), C(1,5),
					 C(2,0), C(2,1), C(2,2), C(2,3), C(2,4), C(2,5),
					 C(3,0), C(3,1), C(3,2), C(3,3), C(3,4), C(3,5),
					 C(4,0), C(4,1), C(4,2), C(4,3), C(4,4), C(4,5),
					 C(5,0), C(5,1), C(5,2), C(5,3), C(5,4), C(5,5));
				 mrpt::system::os::fclose( mf3 );

				 break;
				 / **/
			 } // end for

		RTCov = C.inv();	// Output covariance matrix
		// TEST:
		//RTCov(0,0) += 0.0004;	RTCov(1,1) += 0.0004;	RTCov(2,2) += 0.0004;
		//RTCov(3,3) += 0.000064;	RTCov(4,4) += 0.000064;	RTCov(5,5) += 0.000064;
		//std::cout << "Final: " << std::endl << RTCov << std::endl;
	}
		break;

	// *********************************************
	// Using the Unscented Transformation (UT)
	// *********************************************
	case TOdometryOptions::TCov_UT:
	{
		// ***********************************************************************************
		// ***********************************************************************************
		// USING UNSCENTED TRANSFORM (UT)
		// Pa = computeCovariance( list, cloud1, cloud2 )
		// C = propagateMeanAndCovarianceTroughUT( list, Pa )

		// a = { x11,y11,z11,x21,y21,z21, x12,y12,z12,x22,y22,z22, ..., x1N,y1N,z1N,x2N,y2N,z2N }
		// NA = 6*nCorrs

		// g(·) = ICP6D(·)

		// b = { x,y,z,yaw,pitch,roll }
		// NB = 6

		size_t					NA = 6*list.size();
		double					k = (double)NA/2;

		double					w0 = k/(NA + k);
		double					w1 = 1/(2*(NA + k));

		// ***********************************************************************************
		// "Pa" and "L" Matrixes
		// ***********************************************************************************
		//			[ sxx1 sxy1 sxz1	0     0     0	... ]
		//			[ syx1 syy1 syz1    0     0     0   ... ]
		//			[ szx1 szy1 szz1    0     0     0   ... ]
		//	Pa =	[    0    0    0 sxx1' sxy1' sxz1'  ... ]	for each pair
		//			[    0    0    0 syx1' syy1' syz1'  ... ]
		//			[    0    0    0 szx1' szy1' szz1'  ... ]
		//			[  ...  ...  ...  ...   ...   ...   ... ]
		// ***********************************************************************************

		// Since Pa is block diagonal... the cholesky decomposition comes to:
		//		 (	[ A1	0		0  ] )		[ chol(A1)	0			0		 ]
		//	chol (	[ 0		A2		0  ] )  =	[ 0			chol(A2)	0		 ]
		//		 (	[ 0		0		A3 ] )		[ 0			0			chol(A3) ]
		// ***********************************************************************************

		CMatrixD					L( NA, NA );

		CMatrixD					pPa(3,3);
		CMatrixD					pL(3,3);

		L.zeros();
		unsigned int			i;
		slam::CMetricMap::TMatchingPairList::const_iterator itList;

		// For each pair in the list... insert covariances in 'Pa' matrix
		for(i = 0, itList = list.begin(); itList != list.end(); itList++, i++)
		{
			const slam::CLandmark *lm1 = Pre_Cloud.landmarks.landmarks.getByID( itList->this_idx );
			const slam::CLandmark *lm2 = Cur_Cloud.landmarks.landmarks.getByID( itList->other_idx );
			if( lm1 && lm2 )
			{
				// LM1 ---------------------------------------------------------------
				pPa(0,0) = ( NA + k ) * lm1->pose_cov_11;
				pPa(0,1) = pPa(1,0) = ( NA + k ) * lm1->pose_cov_12;
				pPa(0,2) = pPa(2,0) = ( NA + k ) * lm1->pose_cov_13;

				pPa(1,1) = ( NA + k ) * lm1->pose_cov_22;
				pPa(1,2) = pPa(1,2) = ( NA + k ) * lm1->pose_cov_23;

				pPa(2,2) = ( NA + k ) * lm1->pose_cov_33;

				// Cholesky of pPa and insert into pL
				pPa.chol(pL);
				L.insertMatrix( i*6, i*6, pL );

				// LM2 ---------------------------------------------------------------
				pPa(0,0) = ( NA + k ) * lm2->pose_cov_11;
				pPa(0,1) = pPa(1,0) = ( NA + k ) * lm2->pose_cov_12;
				pPa(0,2) = pPa(2,0) = ( NA + k ) * lm2->pose_cov_13;

				pPa(1,1) = ( NA + k ) * lm2->pose_cov_22;
				pPa(1,2) = pPa(1,2) = ( NA + k ) * lm2->pose_cov_23;

				pPa(2,2) = ( NA + k ) * lm2->pose_cov_33;

				// Cholesky of pPa and insert into pL
				pPa.chol(pL);
				L.insertMatrix( i*6+3, i*6+3, pL );

			}
			else
				THROW_EXCEPTION("There are no landmarks in 'clouds'");

		}

		// ***********************************************************************************
		// "B" parameters
		// ***********************************************************************************
		poses::CPose3DPDFParticles				B( 2*NA + 1 );
		double	log_w1 = log(w1);

		*(B.m_particles[0].d) = meanPose;						// Mean List through the function
		B.m_particles[0].log_w = log(w0);						// Mean List weight

		slam::CMetricMap::TMatchingPairList				auxList1, auxList2;
		slam::CMetricMap::TMatchingPairList::iterator	itAuxList1, itAuxList2;
		auxList1.resize( list.size() );
		auxList2.resize( list.size() );

		for( i = 1; i <= NA; i++ )
		{
			vector_float						vAux;

			vAux.resize( L.getColCount() );
			L.extractRow( i-1, vAux );										// Extract the proper row

			unsigned int m = 0;
			for( itList = list.begin(), itAuxList1 = auxList1.begin(), itAuxList2 = auxList2.begin();
				 itList != list.end(), itAuxList1 != auxList1.end(), itAuxList2 != auxList2.end();
				 m = m + 6, itList++, itAuxList1++, itAuxList2++ )
			{
				itAuxList1->this_idx = itList->this_idx;					// Modify List1
				itAuxList1->this_x = itList->this_x + vAux[ m ];
				itAuxList1->this_y = itList->this_y + vAux[ m+1 ];
				itAuxList1->this_z = itList->this_z + vAux[ m+2 ];

				itAuxList1->other_idx = itList->other_idx;
				itAuxList1->other_x = itList->other_x + vAux[ m+3 ];
				itAuxList1->other_y = itList->other_y + vAux[ m+4 ];
				itAuxList1->other_z = itList->other_z + vAux[ m+5 ];

				itAuxList2->this_idx = itList->this_idx;					// Modify List2
				itAuxList2->this_x = itList->this_x - vAux[ m ];
				itAuxList2->this_y = itList->this_y - vAux[ m+1 ];
				itAuxList2->this_z = itList->this_z - vAux[ m+2 ];

				itAuxList2->other_idx = itList->other_idx;
				itAuxList2->other_x = itList->other_x - vAux[ m+3 ];
				itAuxList2->other_y = itList->other_y - vAux[ m+4 ];
				itAuxList2->other_z = itList->other_z - vAux[ m+5 ];
			} // end for

			double scale;
			scan_matching::leastSquareErrorRigidTransformation6D( auxList1, *B.m_particles[i].d, scale);		// Compute ICP6D for each
			B.m_particles[i].log_w = log_w1;

			scan_matching::leastSquareErrorRigidTransformation6D( auxList2, *B.m_particles[i+NA].d, scale );	// Compute ICP6D for each
			B.m_particles[i+NA].log_w = log_w1;

		} // end for

		// Convert meanB into a vector
		poses::CPose3D	meanB;
		B.getMean(meanB);

		CMatrixD	Pb;

		Pb.zeros(6,6);

		for( i = 0; i <= 2*NA; i++ )
		{

			CMatrixD	v(6,1);											// v = Bi - meanB;
			double	Wi = exp( B.m_particles[i].log_w );

			// Spatial
			v(0,0) = B.m_particles[i].d->x() - meanB.x();
			v(1,0) = B.m_particles[i].d->y() - meanB.y();
			v(2,0) = B.m_particles[i].d->z() - meanB.z();

			// Angles
			v(3,0) = B.m_particles[i].d->yaw() - meanB.yaw();
			v(4,0) = B.m_particles[i].d->pitch() - meanB.pitch();
			v(5,0) = B.m_particles[i].d->roll() - meanB.roll();

			// Covariance
			Pb = Pb + Wi*(v*~v);
		} // end for

		RTCov = Pb;

	}
		break;

	case TOdometryOptions::TCov_Particles:
	{

	}
		break;


	} // end swith 'method'

} // end of 'computeRTCovariance' method

// *********************************************
//				getMatchedList
// *********************************************
void CVisualOdometryStereo::getMatchedList( CMatchedFeatureList &mfList )
{
	CFeatureList::iterator			itLList, itRList;

	mfList.clear();					// Clear the output list

	for( itLList = m_odInfo.m_leftFeats.begin(), itRList = m_odInfo.m_rightFeats.begin();
		 itLList != m_odInfo.m_leftFeats.end(), itRList != m_odInfo.m_rightFeats.end();
		 itLList++, itRList++)
	{
		std::pair<CFeaturePtr,CFeaturePtr> pair( *itLList, *itRList );
		mfList.push_back( pair );
	} // end for

} // end of 'getMatchedList'

// *********************************************
// dbgGetSampledList
// *********************************************
void CVisualOdometryStereo::dbgGetSampledList( const CFeatureList &inList, CFeatureList &outList, const double sigma )
{
	outList.clear();
	CFeatureList::const_iterator itList;
	for( itList = inList.begin(); itList != inList.end(); itList++ )
	{
		CPoint2DPDFGaussian aux;
		CPoint2D			auxOut;

		aux.mean.x( (*itList)->x );					// X Mean value
		aux.mean.y( (*itList)->y );					// Y Mean value
		aux.cov(0,0) = aux.cov(1,1) = square(sigma);// Sigma

		aux.drawSingleSample( auxOut );				// Draw a sample

		CFeaturePtr ft = CFeature::Create();
		ft->type		= (*itList)->type;			// The type of the feature
		ft->x			= auxOut.x();				// X position (drawn from the distribution with mean the initial value and std = 0.5 px)
		ft->y			= auxOut.y();				// Y position (drawn from the distribution with mean the initial value and std = 0.5 px)
		ft->KLT_status	= (*itList)->KLT_status;	// Feature Status
		ft->KLT_val		= (*itList)->KLT_val;		// A value proportional to the quality of the feature (unused yet)
		ft->ID			= (*itList)->ID;			// Feature ID into extraction
		ft->patchSize	= (*itList)->patchSize;		// The size of the feature patch
		ft->patch		= (*itList)->patch;			// The patch

		outList.push_back( ft );
	} // end for itList
} // end dbgGetSampledList

// *********************************************
// checkTrackedFeatures
// *********************************************
void CVisualOdometryStereo::checkTrackedFeatures( CFeatureList &leftList, CFeatureList &rightList, CMatchedFeatureList &mfList )
{
	ASSERT_( leftList.size() == rightList.size() );

	// Clear output structure
	mfList.clear();

	CFeatureList::iterator	itLeft, itRight;
	size_t					u,v;
	double					res;
	for( itLeft = leftList.begin(), itRight = rightList.begin(); itLeft != leftList.end(); )
	{
		bool del = false;
		// Feature out-of-image or epipolar constraint not fullfiled
		if( (*itLeft)->x < 0 || (*itLeft)->y < 0 ||	(*itRight)->x < 0 || (*itRight)->y < 0 || fabs( (*itLeft)->y - (*itRight)->y ) > matchingOptions.rowCheck_TH )
		{
			del = true;
			res = 0;
		}
		else
		{
			// Compute cross correlation:
			openCV_cross_correlation( (*itLeft)->patch, (*itRight)->patch, u, v, res );
			if( res < matchingOptions.ccCheck_TH ) { del = true; }
		}

		if( del )
		{
			// Show reason
			//std::cout << "Bad tracked match:";

			//if( res < matchingOptions.ccCheck_TH )
			//	std::cout << " CC Value: " << res << std::endl;
			//if( (*itLeft)->x < 0 || (*itLeft)->y < 0 || (*itRight)->x < 0 || (*itRight)->y < 0 )
			//	std::cout << " Out of bounds: (" << (*itLeft)->x << "," << (*itLeft)->y << " & (" << (*itRight)->x << "," << (*itRight)->y << ")";
			//if( fabs( (*itLeft)->y - (*itRight)->y ) > matchingOptions.rowCheck_TH )
			//	std::cout << " Bad row checking: " << fabs( (*itLeft)->y - (*itRight)->y );

			//std::cout << std::endl;

			// Erase the pair of features
			itLeft = leftList.erase( itLeft );
			itRight = rightList.erase( itRight );

		} // end if
		else
		{
			std::pair<CFeaturePtr,CFeaturePtr> mPair( *itLeft, *itRight );
			mfList.push_back( mPair );

			itLeft++;
			itRight++;
		} // end else
	} // end for
} // end checkTrackedFeatures

// *********************************************
// process_dbg
// *********************************************
void CVisualOdometryStereo::process_dbg( const slam::CObservationStereoImages &inObs1, const slam::CObservationStereoImages &inObs2, std::vector<CPose3D> &outEst )
{
	CFeatureList				fLeft, fRight;
	CFeatureList				auxfLeft1, auxfRight1, auxfLeft2, auxfRight2;
	CMatchedFeatureList			mfList;
	CPose3D						sampleRT;
	unsigned int				cont = 0;
	CObservationVisualLandmarks	cloud1, cloud2;
	CDisplayWindow				win1, win2, win3, win4;

	std::vector<double>			outScale;

	// Prepare output
	outEst.clear();
	outEst.resize( odometryOptions.debug_nsamples );
	outScale.resize( odometryOptions.debug_nsamples );

	win1.setWindowTitle("Left Image");
	win2.setWindowTitle("Right Image");

	stereoParams.K = inObs1.intrinsicParams;
	stereoParams.baseline = inObs1.rightCameraPose.x();
	stereoParams.stdPixel = 0.2;
	stereoParams.stdDisp = 0.4;

	m_fExtract.detectFeatures( inObs1.imageLeft, fLeft );		// Original keypoints
	m_fExtract.detectFeatures( inObs1.imageRight, fRight );

	while( cont < odometryOptions.debug_nsamples )
	{
		std::cout << "Sample: " << cont << std::endl << std::endl;

		auxfLeft1.clear();
		auxfRight1.clear();

		dbgGetSampledList( fLeft, auxfLeft1, 0.5 );													// Sample the list (now using auxfLeft1)
		dbgGetSampledList( fRight, auxfRight1, 0.5 );												// Sample the list (now using auxfRight1)
		size_t nMatches = vision::matchFeatures( auxfLeft1, auxfRight1, mfList, matchingOptions );	// Match the samples
		vision::projectMatchedFeatures( mfList, stereoParams, cloud1.landmarks );					// Project the matches

		//mfList.saveToTextFile("matches.txt");
		//inObs1.imageLeft.saveToFile("left.jpg");
		//inObs1.imageRight.saveToFile("right.jpg");

		//win1.showImageAndPoints( inObs1.imageLeft, fLeft );
		//win2.showImageAndPoints( inObs1.imageRight, fRight );

		// Delete non-matched features
		// ***********************************************************
		CMatchedFeatureList::iterator	itMatch;

		auxfLeft1.clear();																			// Clear the feature lists
		auxfRight1.clear();

		for( itMatch = mfList.begin(); itMatch != mfList.end(); itMatch++)							// Fill the feature lists
		{
			auxfLeft1.push_back( itMatch->first );
			auxfRight1.push_back( itMatch->second );
		}
		// ***********************************************************

		// Display
		win1.showImageAndPoints( inObs1.imageLeft, auxfLeft1 );
		win2.showImageAndPoints( inObs1.imageRight, auxfRight1 );

		vision::trackFeatures( inObs1.imageLeft, inObs2.imageLeft, auxfLeft1 );						// Track the features (only the matched ones)
		vision::trackFeatures( inObs1.imageRight, inObs2.imageRight, auxfRight1 );					// Track the features (only the matched ones)

		dbgGetSampledList( auxfLeft1, auxfLeft2, 0.5 );												// Sample the tracked list (now using auxfLeft2)
		dbgGetSampledList( auxfRight1, auxfRight2, 0.5 );											// Sample the tracked list (now using auxfRight2)

		// Display
		win3.showImageAndPoints( inObs2.imageLeft, auxfLeft2 );
		win4.showImageAndPoints( inObs2.imageRight, auxfRight2 );

		checkTrackedFeatures( auxfLeft2, auxfRight2, mfList );										// Check tracked features and convert to a matched list

		//mfList.saveToTextFile("tracked_matches.txt");
		//inObs1.imageLeft.saveToFile("new_left.jpg");
		//inObs1.imageRight.saveToFile("new_right.jpg");

		vision::projectMatchedFeatures( mfList, stereoParams, cloud2.landmarks );					// Project the tracked matches

		slam::CMetricMap::TMatchingPairList		list;												// Get the list of matched 3D points which act as inputs of the algorithm
		list.clear();

		vision::cloudsToMatchedList( cloud1, cloud2, list );										// Build the matched list of features
		list.saveAsMATLABScript("3dmatches.m");

		// Compute the least square error rigid transformation
		double		scale;
		vector_int	inliers;
		scan_matching::leastSquareErrorRigidTransformation6DRANSAC( list, sampleRT, scale, inliers, NULL, NULL, true );
		//scan_matching::leastSquareErrorRigidTransformation6D( list, sampleRT, scale );
		outEst[cont] = sampleRT;			// Store the estimated movement
		outScale[cont] = scale;
		//mrpt::system::pause();

		cont++;

	} // end while

	// Save results
	FILE *fresults = mrpt::system::os::fopen( "RTSamples.txt", "wt" );
	FILE *fscale = mrpt::system::os::fopen( "ScaleSamples.txt", "wt" );
	std::vector<CPose3D>::iterator it;
	std::vector<double>::iterator itScale;
	for( it = outEst.begin(), itScale = outScale.begin(); it != outEst.end(); it++, itScale++ )
	{
		mrpt::system::os::fprintf( fresults, "%.4f %.4f %.4f %.4f %.4f %.4f\n", it->x(), it->y(), it->z(), it->yaw(), it->pitch(), it->roll() );
		mrpt::system::os::fprintf( fscale, "%.4f\n", *itScale );
	}
	mrpt::system::os::fclose( fresults );
	mrpt::system::os::fclose( fscale );

}

// *********************************************
// process
// *********************************************
bool  CVisualOdometryStereo::process( const slam::CObservationStereoImagesPtr &inObs, poses::CPose3DPDFGaussian &outEst )
{
	// Point Lists
	CMatchedFeatureList		mfList;
	unsigned int			nMatches = 0;
	FILE					*fposes = NULL, *fcov = NULL, *ftime1 = NULL, *ftime2 = NULL;
	CTicTac					tictac;
	double					t1=0,t2=0,t3=0,t4=0,t5=0,t6=0,t7=0,t8=0,t9=0,t10=0,t11=0,t12=0;

	if( odometryOptions.SAVE_FILES || odometryOptions.TIME_INFO )
	{
		if( !system::fileExists( odometryOptions.OUT_DIR ))
			system::createDirectory( odometryOptions.OUT_DIR );		// Create LOG directory

		fposes	= mrpt::system::os::fopen( format( "./%s/meanPose.txt", odometryOptions.OUT_DIR.c_str() ), "at" );
		fcov	= mrpt::system::os::fopen( format( "./%s/RTcov.txt", odometryOptions.OUT_DIR.c_str() ), "at" );
		ftime1	= mrpt::system::os::fopen( format( "./%s/times1.txt", odometryOptions.OUT_DIR.c_str() ), "at" );
	}

	m_nIter++;	// Increment the iteration number

	if( odometryOptions.VERBOSE )
		std::cout << "------ Iteration " << m_nIter << " ------" << std::endl << std::endl;

	// ------------------------------------
	// FIRST ITERATION
	// ------------------------------------
	if( m_firstObs )
	{
		m_firstObs = false;														// The next one won't be the first observation (unless a reset is done)

		// Get stereo system parameters initialization
		stereoParams.baseline = (float)inObs->rightCameraPose.x();
		stereoParams.K = inObs->intrinsicParams;

		// If we are using Dispersion to detect a non-enough number of features
		if( odometryOptions.USE_DISPERSION )
		{
			// Odometry options related to the image size:
			odometryOptions.minDispX	= inObs->imageLeft.getWidth()/4;
			odometryOptions.minDispY	= inObs->imageLeft.getHeight()/4;
			odometryOptions.minSetDist	= inObs->imageLeft.getHeight()/4;
		}

		// Extract KLT (default option) features from the images
		m_fExtract.options.featsType = featKLT;
		//if( odometryOptions.DEBUG_ESTIMATE_COVARIANCE )
		//{
		//	m_fExtract.detectFeatures( inObs->imageLeft, m_dbg_orLF );		// These lists will act as the origin of the drawn samples
		//	m_fExtract.detectFeatures( inObs->imageRight, m_dbg_orRF );
		//}
		//else
		//{
		if( odometryOptions.TIME_INFO ) { tictac.Tic(); }
		m_fExtract.detectFeatures( inObs->imageLeft, m_odInfo.m_leftFeats );		// These lists will be modified through time
		m_fExtract.detectFeatures( inObs->imageRight, m_odInfo.m_rightFeats );
		if( odometryOptions.TIME_INFO ) { t1 = 1000.0f*tictac.Tac(); }

		//m_leftFeats.saveToTextFile( format( "./%s/TEST_LEFT.txt", odometryOptions.OUT_DIR.c_str() ) );
		//m_rightFeats.saveToTextFile( format( "./%s/TEST_RIGHT.txt", odometryOptions.OUT_DIR.c_str() ) );

		//}

		//if( m_dbg_sampleCount < odometryOptions.debug_nsamples )	// We extract a new sample list from the origin

		//CFeatureList auxLList, auxRList;		// Auxiliary lists for storing drawn features
		//if( odometryOptions.DEBUG_ESTIMATE_COVARIANCE )
		//{
		//	m_dbg_outEstimations.resize( odometryOptions.debug_nsamples );							// Resize the output vector if necessary

		//	dbgGetSampledList( m_dbg_orLF, auxLList );
		//	dbgGetSampledList( m_dbg_orRF, auxRList );
		//	nMatches = vision::matchFeatures(auxLList, auxRList, mfList, matchingOptions);			// TO DO: Default options in call to function
		//}
		//else
		if( odometryOptions.TIME_INFO ) { tictac.Tic();	}
		nMatches = vision::matchFeatures( m_odInfo.m_leftFeats, m_odInfo.m_rightFeats, mfList, matchingOptions);	// TO DO: Default options in call to function
		if( odometryOptions.TIME_INFO ) { t2 = 1000.0f*tictac.Tac(); }
		//mfList.saveToTextFile( format( "./%s/TEST_MATCH.txt", odometryOptions.OUT_DIR.c_str() ) );

		// Project the matches
		if( odometryOptions.TIME_INFO ) { tictac.Tic(); }
		vision::projectMatchedFeatures( mfList, stereoParams, m_odInfo.m_Prev_cloud.landmarks );
		if( odometryOptions.TIME_INFO ) { t3 = 1000.0f*tictac.Tac(); }

		// -------------------------------------
		// SHOW INFORMATION
		// -------------------------------------
		if( odometryOptions.PLOT_IMAGES )
		{
		#if VISUALODOMETRY_USES_GUI
			if(m_wind1 == NULL)
				m_wind1 = new CDisplayWindow();
			if(m_wind2 == NULL)
				m_wind2 = new CDisplayWindow();

			CMatchedFeatureList::iterator	it;
			vector_float					xL, yL, xR, yR;
			unsigned int					k;
			size_t							tam = mfList.size();

			xL.resize( tam );
			yL.resize( tam );
			xR.resize( tam );
			yR.resize( tam );

			for(k = 0, it = mfList.begin(); it != mfList.end(); it++, k++)
			{
				xL[k] = (it->first)->x;
				yL[k] = (it->first)->y;

				xR[k] = (it->second)->x;
				yR[k] = (it->second)->y;

			}
			m_wind1->setPos( 680, 600 );
			m_wind1->setWindowTitle("Left Image");
			m_wind1->showImageAndPoints( inObs->imageLeft, xL, yL );
			m_wind2->setPos( 1100, 600 );
			m_wind2->setWindowTitle("Right Image");
			m_wind2->showImageAndPoints( inObs->imageRight, xR, yR );
		#endif
		}

		if( odometryOptions.SAVE_FILES )
		{
			inObs->imageLeft.saveToFile( format( "./%s/leftImg.jpg", odometryOptions.OUT_DIR.c_str() ) );
			inObs->imageRight.saveToFile( format( "./%s/rightImg.jpg", odometryOptions.OUT_DIR.c_str() ) );

			m_odInfo.m_leftFeats.saveToTextFile( format("./%s/lFeats_%04d.txt", odometryOptions.OUT_DIR.c_str(), m_nIter) );
			m_odInfo.m_rightFeats.saveToTextFile( format("./%s/rFeats_%04d.txt", odometryOptions.OUT_DIR.c_str(), m_nIter) );
			mfList.saveToTextFile( format("./%s/matches_%04d.txt", odometryOptions.OUT_DIR.c_str(), m_nIter) );

			m_odInfo.m_Prev_cloud.landmarks.saveToMATLABScript3D( format("./%s/cloud_%04d.m", odometryOptions.OUT_DIR.c_str(), m_nIter) );
		}

		//mrpt::system::pause();

		if( odometryOptions.VERBOSE )
		{
			std::cout << ".: Extract features :." << std::endl;
			std::cout << "\tLeft features: " << m_odInfo.m_leftFeats.size() << std::endl;
			std::cout << "\tRight features: " << m_odInfo.m_rightFeats.size() << std::endl << std::endl;
			std::cout << ".: Match features :." << std::endl;
			std::cout << "\t# of matches: " << nMatches << "\n\n";
			std::cout << ".: Project features :." << std::endl;
			std::cout << "\t# of projected features: " << m_odInfo.m_Prev_cloud.landmarks.size() << "\n\n";
		}
		// -------------------------------------

		// Store the matched features inside the inner CFeatureLists
		if( odometryOptions.TIME_INFO ) { tictac.Tic(); }
		storeMatchedFeatureList( mfList );
		if( odometryOptions.TIME_INFO ) { t4 = 1000.0f*tictac.Tac(); }

		// Insert additional data into the CLandmarksMap
		m_odInfo.m_Prev_cloud.timestamp = inObs->timestamp;
		m_odInfo.m_Prev_cloud.landmarks.changeCoordinatesReference( inObs->cameraPose );

		// Store the current images
		m_odInfo.m_obs = inObs;

		// AT THIS POINT, WE HAVE STORED A CLOUD OF 3D POINTS, TWO SETS OF KLT POINTS AND TWO IMAGES.
		if( odometryOptions.SAVE_FILES )
		{
			mrpt::system::os::fclose( fposes );
			mrpt::system::os::fclose( fcov );
		}
		if( odometryOptions.TIME_INFO )
		{
			mrpt::system::os::fprintf( ftime1, "%.3f %.3f %.3f %.3f\n", t1, t2, t3, t4 );
			mrpt::system::os::fclose( ftime1 );
		}

		return true;
	} // end if 'First observation'
	else
	{
		// Track features
		if( odometryOptions.TIME_INFO ) { tictac.Tic(); }
		vision::trackFeatures( m_odInfo.m_obs->imageLeft, inObs->imageLeft, m_odInfo.m_leftFeats );			// LEFT IMAGE
		vision::trackFeatures( m_odInfo.m_obs->imageRight, inObs->imageRight, m_odInfo.m_rightFeats );		// RIGHT IMAGE
		if( odometryOptions.TIME_INFO ) { t5 = 1000.0f*tictac.Tac(); }

		//CFeatureList auxLList, auxRList;										// Auxiliary lists for storing drawn features
		//if( odometryOptions.DEBUG_ESTIMATE_COVARIANCE )
		//{
		//	dbgGetSampledList( m_leftFeats, auxLList );
		//	dbgGetSampledList( m_rightFeats, auxRList );
		//}

		if( odometryOptions.VERBOSE )
		{
			std::cout << ".: Track features :." << std::endl;
			std::cout << format("\tLeft features: %d", (int)m_odInfo.m_leftFeats.size() ) << std::endl;
			std::cout << format("\tRight features: %d", (int)m_odInfo.m_rightFeats.size() ) << std::endl;
		}

		// Delete bad tracked features (using also crossCorrelation for comparing patches)
		//if( odometryOptions.DEBUG_ESTIMATE_COVARIANCE )
		//	vision::checkTrackedFeatures( auxLList, auxRList, matchingOptions );
		//else
		if( odometryOptions.TIME_INFO ) { tictac.Tic(); }
		//checkTrackedFeatures( m_odInfo.m_leftFeats, m_odInfo.m_rightFeats, m_odInfo.m_rightFeats
		vision::checkTrackedFeatures( m_odInfo.m_leftFeats, m_odInfo.m_rightFeats, matchingOptions );
		if( odometryOptions.TIME_INFO ) { t6 = 1000.0f*tictac.Tac(); }

		if( odometryOptions.VERBOSE )
		{
			std::cout << ".: Track checking performed :." << std::endl;
			std::cout << format("\t# good matches: %d", (int)m_odInfo.m_rightFeats.size() ) << std::endl;
		}

		// Save Files
		if( odometryOptions.SAVE_FILES )
		{
			//if( odometryOptions.DEBUG_ESTIMATE_COVARIANCE )
			//{
			//	auxLList.saveToTextFile( format("./LOG_VODOMETRY/ltFeats_%04d.txt", m_nIter) );
			//	auxRList.saveToTextFile( format("./LOG_VODOMETRY/rtFeats_%04d.txt", m_nIter) );
			//}
			//else
			//{
				m_odInfo.m_leftFeats.saveToTextFile( format("./%s/ltFeats_%04d.txt", odometryOptions.OUT_DIR.c_str(), m_nIter) );
				m_odInfo.m_rightFeats.saveToTextFile( format("./%s/rtFeats_%04d.txt", odometryOptions.OUT_DIR.c_str(), m_nIter) );
			//}
		} // end odometryOptions.SAVE_FILES

		bool dispersion = true;
		if( odometryOptions.USE_DISPERSION )
		{
			vector_float std, mean;
			//vision::getDispersion( odometryOptions.DEBUG_ESTIMATE_COVARIANCE ? auxLList : m_leftFeats, std, mean );
			vision::getDispersion( m_odInfo.m_leftFeats, std, mean );

			double currDist = sqrt( square( mean[0] - inObs->imageLeft.getWidth()/2 ) + square( mean[1] - inObs->imageLeft.getHeight()/2 ) );

			if( currDist > odometryOptions.minSetDist || std[0] < odometryOptions.minDispX || std[1] < odometryOptions.minDispY ) // If they are not scattered enough, we search for more
				dispersion = false;

		} // end USE_DISPERSION

		// Are there enough features?
		//unsigned int nCurrentFeatures = odometryOptions.DEBUG_ESTIMATE_COVARIANCE ? (unsigned int)auxLList.size() : (unsigned int)m_leftFeats.size();
		unsigned int nCurrentFeatures = (unsigned int)m_odInfo.m_leftFeats.size();

		if( nCurrentFeatures < odometryOptions.minNumFeatures || !dispersion ) // If either there are not enough features or they are not dispersed
		{
			CMatchedFeatureList auxMFList;
			CFeatureList		auxLeftList, auxRightList;

			// Find more features in the images
			//m_fExtract.findMoreFeatures( inObs->imageLeft, odometryOptions.DEBUG_ESTIMATE_COVARIANCE ? auxLList : m_leftFeats, auxLeftList );
			//m_fExtract.findMoreFeatures( inObs->imageRight, odometryOptions.DEBUG_ESTIMATE_COVARIANCE ? auxRList : m_rightFeats, auxRightList );

			m_fExtract.findMoreFeatures( inObs->imageLeft, m_odInfo.m_leftFeats, auxLeftList );
			m_fExtract.findMoreFeatures( inObs->imageRight, m_odInfo.m_rightFeats, auxRightList );

			// Match them
			vision::matchFeatures( auxLeftList, auxRightList, auxMFList, matchingOptions );

			// Add matches
			addMatches( auxMFList );

			if( odometryOptions.VERBOSE )
			{
				std::cout << ".: Find more features :." << std::endl;
				std::cout << "\t# new matches: " << auxMFList.size() << std::endl;
			}
		} // end if 'are there enough features?'

		CMatchedFeatureList matchedList;
		if( odometryOptions.TIME_INFO ) { tictac.Tic(); }
		getMatchedList( matchedList );
		if( odometryOptions.TIME_INFO ) { t7 = 1000.0f*tictac.Tac(); }

		// PLOT IMAGES
		if( odometryOptions.PLOT_IMAGES )
		{
		#if VISUALODOMETRY_USES_GUI
			if(m_wind1 == NULL)
				m_wind1 = new CDisplayWindow();
			if(m_wind2 == NULL)
				m_wind2 = new CDisplayWindow();

			CMatchedFeatureList::iterator	it;
			vector_float					xL, yL, xR, yR;
			unsigned int					k;
			size_t							tam = matchedList.size();

			xL.resize( tam );
			yL.resize( tam );
			xR.resize( tam );
			yR.resize( tam );

			for(k = 0, it = matchedList.begin(); it != matchedList.end(); it++, k++)
			{
				xL[k] = (it->first)->x;
				yL[k] = (it->first)->y;

				xR[k] = (it->second)->x;
				yR[k] = (it->second)->y;

			}

			m_wind1->setPos( 680, 600 );
			m_wind1->setWindowTitle("Left Image");
			m_wind1->showImageAndPoints( inObs->imageLeft, xL, yL );
			m_wind2->setPos( 1100, 600 );
			m_wind2->setWindowTitle("Right Image");
			m_wind2->showImageAndPoints( inObs->imageRight, xR, yR );
		#endif
		}

		// PROBABILISTIC ESTIMATION OF THE CHANGE IN POSE
		// Computation of the current cloud of points
		slam::CObservationVisualLandmarks	Cur_cloud;

		// Project matched features
		if( odometryOptions.TIME_INFO ) { tictac.Tic(); }
		vision::projectMatchedFeatures( matchedList, stereoParams, Cur_cloud.landmarks );
		if( odometryOptions.TIME_INFO ) { t8 = 1000.0f*tictac.Tac(); }

		if( odometryOptions.SAVE_FILES )
			Cur_cloud.landmarks.saveToMATLABScript3D( format("./%s/cloud_%04d.m", odometryOptions.OUT_DIR.c_str(), m_nIter) );

		Cur_cloud.timestamp = inObs->timestamp;
		Cur_cloud.landmarks.changeCoordinatesReference( inObs->cameraPose );

		// Get the list of matched 3D points which act as inputs of the algorithm
		slam::CMetricMap::TMatchingPairList		list;
		list.clear();

		// *********************************************
		// DEBUG ***************************************
		// *********************************************
		/** /
		CMatrixD	p1(15,3), p2(15,3);
		for( unsigned int i = 0; i < 15; i++)
		{
			p1(i,0) = mrpt::math::random::RandomUni( 0, 6 );
			p1(i,1) = mrpt::math::random::RandomUni( 0, 6 );
			p1(i,2) = mrpt::math::random::RandomUni( 0, 6 );

			p2(i,0) = p1(i,0)+6;
			p2(i,1) = p1(i,1);
			p2(i,2) = p1(i,2);
		} // end for

		Prev_cloud.landmarks.clear();
		Cur_cloud.landmarks.clear();

		for (unsigned int i = 0; i < 15; i++)
		{
			slam::CLandmark lm1, lm2;
			lm1.ID = lm2.ID = i;

			lm1.pose_mean.x = p1(i,0);		lm2.pose_mean.x = p2(i,0);
			lm1.pose_mean.y = p1(i,1);		lm2.pose_mean.y = p2(i,1);
			lm1.pose_mean.z = 0;			lm2.pose_mean.z = 0;

			lm1.pose_cov_11 = 1;			lm2.pose_cov_11 = 1;
			lm1.pose_cov_12 = 0;			lm2.pose_cov_12 = 0;
			lm1.pose_cov_13 = 0;			lm2.pose_cov_13 = 0;
			lm1.pose_cov_22 = 1;			lm2.pose_cov_22 = 1;
			lm1.pose_cov_23 = 0;			lm2.pose_cov_23 = 0;
			lm1.pose_cov_33 = 1;			lm2.pose_cov_33 = 1;

			Prev_cloud.landmarks.landmarks.push_back( &lm1 );
			Cur_cloud.landmarks.landmarks.push_back( &lm2 );
		}

		/ **/

		if( odometryOptions.TIME_INFO ) { tictac.Tic(); }
		vision::cloudsToMatchedList( m_odInfo.m_Prev_cloud, Cur_cloud, list );
		if( odometryOptions.TIME_INFO ) { t9 = 1000.0f*tictac.Tac(); }

		if( list.size() < 5 )
			std::cout << "List size < 5" << std::endl;

		// Compute the mean of the change in pose
		poses::CPose3D	meanPose;
		//vision::filterBadCorrsByDistance( list, 3 );

		if( odometryOptions.SAVE_FILES )
			list.dumpToFile( format("./%s/matchedList_%04d.txt", odometryOptions.OUT_DIR.c_str(), m_nIter).c_str() );

		// Use RANSAC
		double		scale;
		vector_int	inliers;

		if( odometryOptions.TIME_INFO ) { tictac.Tic(); }
		odometryOptions.USE_RANSAC ?
			scan_matching::leastSquareErrorRigidTransformation6DRANSAC( list, meanPose, scale, inliers ) :
			scan_matching::leastSquareErrorRigidTransformation6D( list, meanPose, scale );

		if( odometryOptions.TIME_INFO ) { t10 = 1000.0f*tictac.Tac(); }

		//if( odometryOptions.DEBUG_ESTIMATE_COVARIANCE )
		//{
		//	m_dbg_outEstimations.push_back( meanPose );		// Save the sampled outPut Pose estimation
		//	m_dbg_sampleCount++;
		//}

		//scan_matching::leastSquareErrorRigidTransformation6D( list, meanPose, scale ); // WITHOUT RANSAC

		// ***************************************************************
		// Delete outliers if necessary! NOT EFFICIENT!!
		// ***************************************************************
		if( odometryOptions.TIME_INFO ) { tictac.Tic(); }
		if( odometryOptions.USE_RANSAC )
		{

			if( list.size() > inliers.size() )
			{
				slam::CMetricMap::TMatchingPairList::iterator	it;
				vector_int::iterator							vIt;
				slam::CMetricMap::TMatchingPairList				auxMatchedList;
				for( vIt = inliers.begin(); vIt != inliers.end(); vIt++ )
					auxMatchedList.push_back( list[ *vIt ] );

				list.clear();
				list = auxMatchedList;
			}
		}
		if( odometryOptions.TIME_INFO ) { t11 = 1000.0f*tictac.Tac(); }
		// ***************************************************************
		// ***************************************************************

		// Compute the covariance of the change in pose
		CMatrixD RTCov;
		RTCov.zeros( 6, 6 );
		if( odometryOptions.TIME_INFO ) { tictac.Tic(); }
		computeRTCovariance( meanPose, list, m_odInfo.m_Prev_cloud, Cur_cloud, RTCov );
		if( odometryOptions.TIME_INFO ) { t12 = 1000.0f*tictac.Tac(); }

		if( odometryOptions.SAVE_FILES )
		{
			list.dumpToFile( format("./%s/RANSACmatchedList_%04d.txt", odometryOptions.OUT_DIR.c_str(), m_nIter).c_str() );

			mrpt::system::os::fprintf( fposes, "%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t%.5f\n",
				meanPose.x(), meanPose.y(), meanPose.z(),
				meanPose.yaw(), meanPose.pitch(), meanPose.roll());

			mrpt::system::os::fprintf( fcov, "%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t", RTCov(0,0), RTCov(0,1), RTCov(0,2), RTCov(0,3), RTCov(0,4), RTCov(0,5) );
			mrpt::system::os::fprintf( fcov, "%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t", RTCov(1,0), RTCov(1,1), RTCov(1,2), RTCov(1,3), RTCov(1,4), RTCov(1,5) );
			mrpt::system::os::fprintf( fcov, "%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t", RTCov(2,0), RTCov(2,1), RTCov(2,2), RTCov(2,3), RTCov(2,4), RTCov(2,5) );
			mrpt::system::os::fprintf( fcov, "%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t", RTCov(3,0), RTCov(3,1), RTCov(3,2), RTCov(3,3), RTCov(3,4), RTCov(3,5) );
			mrpt::system::os::fprintf( fcov, "%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t", RTCov(4,0), RTCov(4,1), RTCov(4,2), RTCov(4,3), RTCov(4,4), RTCov(4,5) );
			mrpt::system::os::fprintf( fcov, "%.5f\t%.5f\t%.5f\t%.5f\t%.5f\t%.5f\n", RTCov(5,0), RTCov(5,1), RTCov(5,2), RTCov(5,3), RTCov(5,4), RTCov(5,5) );
		}

		// Fill the output 'outEst'
		outEst.mean = meanPose;
		outEst.cov = RTCov;

		if( odometryOptions.VERBOSE )
			std::cout << "COV: " << std::endl << RTCov << std::endl;

		m_odInfo.m_obs->imageLeft	= inObs->imageLeft;
		m_odInfo.m_obs->imageRight	= inObs->imageRight;

		// Storage of the current cloud of points
		m_odInfo.m_Prev_cloud = Cur_cloud;
		if( odometryOptions.VERBOSE )
			std::cout << meanPose << std::endl;

		if( odometryOptions.SAVE_FILES )
		{
			mrpt::system::os::fclose( fposes );
			mrpt::system::os::fclose( fcov );
		}
		if( odometryOptions.TIME_INFO )
		{
			ftime2	= mrpt::system::os::fopen( format( "./%s/times2.txt", odometryOptions.OUT_DIR.c_str() ), "at");
			mrpt::system::os::fprintf( ftime2, "%.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f\n", t5, t6, t7, t8, t9, t10, t11, t12 );
			mrpt::system::os::fclose( ftime2 );
		}
		return true;
	} // end else

	return true;
} // end of 'process' method

const TOdometryInfo &CVisualOdometryStereo::getInfo()
{
	return m_odInfo;
} // end of 'getInfo' method
