#include <mrpt/core.h>
#include <vector>
#include <numeric>
#include <algorithm>
#include <functional>

using namespace std;
using namespace mrpt::math;
using namespace mrpt::utils;
using namespace mrpt::random;

const size_t matrixSize=10;
const size_t howMany=1000;

typedef double NumericType;

const NumericType eps=1e-7;

template<typename It1,typename It2> class DoubleIterator	{
public:
	typedef pair<typename It1::value_type,typename It2::value_type> value_type;
	typedef value_type *pointer;
	typedef value_type &reference;
	typedef ptrdiff_t difference_type;
	typedef input_iterator_tag iterator_category;
private:
	It1 base1;
	It2 base2;
public:
	inline DoubleIterator(const It1 &b1,const It2 &b2):base1(b1),base2(b2)	{}
	inline DoubleIterator<It1,It2> operator++()	{
		++base1;
		++base2;
		return *this;
	}
	inline DoubleIterator<It1,It2> operator++(int)	{
		DoubleIterator<It1,It2> res=*this;
		++(*this);
		return res;
	}
	inline bool operator==(const DoubleIterator<It1,It2> &it) const	{
		return (base1==it.base1)&&(base2==it.base2);
	}
	inline bool operator!=(const DoubleIterator<It1,It2> &it) const	{
		return (base1!=it.base1)||(base2!=it.base2);
	}
	inline value_type operator*() const	{
		return make_pair(*base1,*base2);
	}
};
template<typename It1,typename It2> inline DoubleIterator<It1,It2> itDouble(const It1 &i1,const It2 &i2)	{
	return DoubleIterator<It1,It2>(i1,i2);
}

template<typename T> class PositiveSemidefiniteGenerator	{
private:
	CRandomGenerator gen;
	mutable CMatrixTemplateNumeric<T> tmp;
	mutable CMatrixTemplateNumeric<T> res;
public:
	inline PositiveSemidefiniteGenerator(size_t s):gen(),tmp(s,s),res(s,s)	{}
	inline CMatrixTemplateNumeric<T> operator()()	{
		do gen.drawUniformMatrix(tmp,T(-3.0),T(3.0)); while (tmp.isSingular(eps));
		res.multiply_AAt(tmp);
		return res;
	}
};

template<typename T> class RandomVectorGenerator	{
private:
	CRandomGenerator gen;
	mutable vector<T> res;
public:
	inline RandomVectorGenerator(size_t s):gen(),res(s,T(0.0))	{}
	inline vector<T> operator()()	{
		gen.drawUniformVector(res,T(-3.0),T(3.0));
		return res;
	}
};

template<typename T> inline T get_xCxT_basic(const pair<vector<T>,CMatrixTemplateNumeric<T> > &p)	{
	return multiply_HCHt_scalar(p.first,p.second);
}

template<typename T> class Get_xCxT_cholesky	{
private:
	mutable CMatrixTemplateNumeric<T> R;
	//mutable vector<T> tmpV;
	const size_t N;
public:
	typedef pair<vector<T>,CMatrixTemplateNumeric<T> > argument_type;
	typedef T result_type;
	inline Get_xCxT_cholesky(size_t s):R(s,s),/*tmpV(s),*/N(s)	{}
	inline T operator()(const pair<vector<T>,CMatrixTemplateNumeric<T> > &p)	{
		const vector<T> &vec=p.first;
		p.second.chol(R);
		T res=T(0);
		for (size_t i=0;i<N-1;++i)	{
			T accum=T(0.0);
			for (size_t j=i;j<N;++j) accum+=R(i,j)*vec[j];
			//tmpV[i]=accum;
			res+=square(accum);
		}
		//tmpV[N-1]=vec[N-1]*R.get_unsafe(0,0);
		res+=square(vec[N-1]*R.get_unsafe(N-1,N-1));
		return res;
	}
};

template<typename T> inline size_t compareAndSum(size_t acc,const pair<T,T> &p)	{
	return acc+((p.first-p.second>=T(eps))?1:0);
}

int test_mahalanobis_main(int argc,char **argv)	{
	CTicTac time;
	vector<CMatrixTemplateNumeric<NumericType> > matrices(howMany,CMatrixTemplate<NumericType>(matrixSize,matrixSize));
	vector<vector<NumericType> > vectors(howMany,vector<NumericType>(matrixSize));
	vector<NumericType> res1(howMany);
	vector<NumericType> res2(howMany);
	generate(matrices.begin(),matrices.end(),PositiveSemidefiniteGenerator<NumericType>(matrixSize));
	generate(vectors.begin(),vectors.end(),RandomVectorGenerator<NumericType>(matrixSize));
	time.Tic();
	transform(itDouble(vectors.begin(),matrices.begin()),itDouble(vectors.end(),matrices.end()),res1.begin(),&get_xCxT_basic<NumericType>);
	double t=time.Tac();
	time.Tic();
	transform(itDouble(vectors.begin(),matrices.begin()),itDouble(vectors.end(),matrices.end()),res2.begin(),Get_xCxT_cholesky<NumericType>(matrixSize));
	t=time.Tac();
	return accumulate(itDouble(res1.begin(),res2.begin()),itDouble(res1.end(),res2.end()),NumericType(0.0),&compareAndSum<NumericType>);
}
