/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.

#include <mrpt/math/CMatrixFixedNumeric.h>
#include <mrpt/poses/CPose3D.h>
#include <mrpt/poses/CPose2D.h>
#include <mrpt/poses/CPoint3D.h>
#include <mrpt/poses/CPoint2D.h>

using std::string;
using std::cout;
using std::endl;
using namespace mrpt;
using namespace mrpt::math;
using namespace mrpt::utils;
using namespace mrpt::poses;


template <> CMatrixDouble21 & mrpt::math::matrixFromPoseOrPoint(CMatrixDouble21 &M, const CPoint2D &p) {
	M.m_Val[0] = p.x();
	M.m_Val[1] = p.y();
	return M;
}
template <> CMatrixDouble12 & mrpt::math::matrixFromPoseOrPoint(CMatrixDouble12 &M, const CPoint2D &p) {
	M.m_Val[0] = p.x();
	M.m_Val[1] = p.y();
	return M;
}

template <> CMatrixDouble31 & mrpt::math::matrixFromPoseOrPoint(CMatrixDouble31 &M, const CPoint3D &p) {
	M.m_Val[0] = p.x();
	M.m_Val[1] = p.y();
	M.m_Val[2] = p.z();
	return M;
}
template <> CMatrixDouble13 & mrpt::math::matrixFromPoseOrPoint(CMatrixDouble13 &M, const CPoint3D &p) {
	M.m_Val[0] = p.x();
	M.m_Val[1] = p.y();
	M.m_Val[2] = p.z();
	return M;
}

template <> CMatrixDouble31 & mrpt::math::matrixFromPoseOrPoint(CMatrixDouble31 &M, const CPose2D &p) {
	M.m_Val[0] = p.x();
	M.m_Val[1] = p.y();
	M.m_Val[2] = p.phi();
	return M;
}
template <> CMatrixDouble13 & mrpt::math::matrixFromPoseOrPoint(CMatrixDouble13 &M, const CPose2D &p) {
	M.m_Val[0] = p.x();
	M.m_Val[1] = p.y();
	M.m_Val[2] = p.phi();
	return M;
}


template <> CMatrixDouble61 & mrpt::math::matrixFromPoseOrPoint(CMatrixDouble61 &M, const CPose3D &p) {
	M.m_Val[0] = p.x();
	M.m_Val[1] = p.y();
	M.m_Val[2] = p.z();
	M.m_Val[3] = p.yaw();
	M.m_Val[4] = p.pitch();
	M.m_Val[5] = p.roll();
	return M;
}
template <> CMatrixDouble16 & mrpt::math::matrixFromPoseOrPoint(CMatrixDouble16 &M, const CPose3D &p) {
	M.m_Val[0] = p.x();
	M.m_Val[1] = p.y();
	M.m_Val[2] = p.z();
	M.m_Val[3] = p.yaw();
	M.m_Val[4] = p.pitch();
	M.m_Val[5] = p.roll();
	return M;
}

template <typename T>
void mrpt::math::eigenVectorsMatrix(
	const CMatrixFixedNumeric<T,2,2> &M,
	CMatrixFixedNumeric<T,2,2> &Z,
	CMatrixFixedNumeric<T,2,2> &D )
{
	D.zeros();

#ifdef _DEBUG
	ASSERT_(M.get_unsafe(1,0)==M.get_unsafe(0,1))
#endif
	const T a = M.get_unsafe(0,0);
	const T b = M.get_unsafe(0,1);
	const T d = M.get_unsafe(1,1);

	const T k = 0.5*(a+d);
	T discr = 4*square(b)+square(a-d);

	ASSERTMSG_(discr>=0,"Matrix is not positive-semidefinite")
	
	discr=0.5*std::sqrt(discr);

	D.get_unsafe(0,0) = k - discr;  // First, the smaller one
	D.get_unsafe(1,1) = k + discr;

	// The eigenvectors:
	if (b==0)
	{
		// [0 1] base:
		Z.get_unsafe(0,0) = Z.get_unsafe(1,1) = 1;
		Z.get_unsafe(0,1) = Z.get_unsafe(1,0) = 0;
	}
	else
	{
		T v1 = (D.get_unsafe(0,0) - d)/b;
		T v_norm = std::sqrt(1+square(v1));
		T v2 = 1/v_norm;
		v1 /= v_norm;
		Z.get_unsafe(0,0) = v1;
		Z.get_unsafe(1,0) = v2;

		v1 = (D.get_unsafe(1,1) - d)/b;
		v_norm = std::sqrt(1+square(v1));
		v2 = 1/v_norm;
		v1 /= v_norm;
		Z.get_unsafe(0,1) = v1;
		Z.get_unsafe(1,1) = v2;
	}
}

// Explicit instantation:
template void MRPTDLLIMPEXP mrpt::math::eigenVectorsMatrix(const CMatrixFixedNumeric<float,2,2> &M,CMatrixFixedNumeric<float,2,2> &Z,CMatrixFixedNumeric<float,2,2> &D );
template void MRPTDLLIMPEXP mrpt::math::eigenVectorsMatrix(const CMatrixFixedNumeric<double,2,2> &M,CMatrixFixedNumeric<double,2,2> &Z,CMatrixFixedNumeric<double,2,2> &D );

