/*
 * smtp.h
 * 
 * This file is part of msmtp, an SMTP client.
 *
 * Copyright (C) 2000, 2003, 2004
 * Martin Lambers <marlam@users.sourceforge.net>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   msmtp is released under the GPL with the additional exemption that
 *   compiling, linking, and/or using OpenSSL is allowed.
 */

#ifndef SMTP_H
#define SMTP_H

#include <stdio.h>

#include "merror.h"
#ifdef HAVE_SSL
#include "tls.h"
#endif /* HAVE_SSL */

/* SMTP errors */
#define SMTP_EIO		1	/* Input/output error */
#define SMTP_ENOMEM		2	/* Cannot allocate memory */
#define SMTP_EPROTO		3	/* Protocol violation */
#define SMTP_EINVAL		4	/* Invalid input data */
#define SMTP_EUNAVAIL		5	/* Requested service unavailable */
#define SMTP_EAUTHFAIL		6	/* Authentication failed */
#define SMTP_ELIBFAILED		7	/* An underlying library failed */
#define SMTP_EINSECURE		8	/* The requested action would be insecure */

/* SMTP capabilities */
#define SMTP_CAP_STARTTLS		1
#define SMTP_CAP_DSN			2
#define SMTP_CAP_PIPELINING		4
#define SMTP_CAP_SIZE			8
#define SMTP_CAP_AUTH			16
#define SMTP_CAP_AUTH_PLAIN		32
#define SMTP_CAP_AUTH_LOGIN		64
#define SMTP_CAP_AUTH_CRAM_MD5	 	128
#define SMTP_CAP_AUTH_DIGEST_MD5 	256
#define SMTP_CAP_AUTH_NTLM		512

/*
 * This structure describes the capabilities of an SMTP server. 
 * 'flags' is a combination of the SMTP_CAP_* values above. 
 * If (flags & SMTP_CAP_SIZE), 'size' contains the max size of a message that 
 * the SMTP server will accept.
 */
typedef struct
{
    int flags;
    long size;
} smtp_cap_t;

/*
 * This structure represents an SMTP server. Do not access it directly.
 */
typedef struct
{
    int fd;
#ifdef HAVE_SSL
    tls_t tls;
#endif /* HAVE_SSL */
    smtp_cap_t cap;
    FILE *debug;
} smtp_server_t;

/*
 * All smtp_* functions may call merror(), which uses a static buffer for error
 * strings. So if you want to preserve an error message, copy it to a private
 * buffer before calling another smtp_* function!
 */

/*
 * smtp_new()
 *
 * Create a new smtp_server_t. If 'debug' is not NULL, the complete 
 * conversation with the SMTP server will be logged to the referenced file. 
 * Beware: this log may contain user passwords.
 */
smtp_server_t smtp_new(FILE *debug);

/*
 * smtp_connect()
 *
 * Connect to a SMTP server.
 * Used error codes: NET_EHOSTNOTFOUND, NET_ESOCKET, NET_ECONNECT
 */
merror_t smtp_connect(smtp_server_t *srv, char *host, int port);

/*
 * smtp_msg_status()
 *
 * Returns the three digit status code of the SMTP server message 'msg', which
 * *must* be a valid SMTP server message.
 */
int smtp_msg_status(list_t *msg);

/*
 * smtp_get_greeting()
 *
 * Get the greeting message from the SMTP server.
 * If 'buf' is not NULL, it will contain a pointer to an allocated string
 * containing the identificatin string of the SMTP server.
 * Used error codes: SMTP_EIO, SMTP_EPROTO, SMTP_ENOMEM
 */
merror_t smtp_get_greeting(smtp_server_t *srv, list_t **errmsg, char **buf);
    
/*
 * smtp_init()
 *
 * Initialize an SMTP session with the connected SMTP server 'srv'
 * (via the SMTP EHLO/HELO command). This function must be used after
 * the server is connected and before any mail is send. It must also be used
 * (a second time) after TLS is started via the STARTTLS command. 
 * This function determines the capabilities of the SMTP server.
 * 'ehlo_domain' is the parameter for the EHLO/HELO command. If you don't know
 * what to use, use "localhost".
 * 'error_msg' contains an error message from the SMTP server or NULL.
 * Used error codes: SMTP_EIO, SMTP_EPROTO, SMTP_ENOMEM, SMTP_EINVAL
 */
merror_t smtp_init(smtp_server_t *srv, char *ehlo_domain, list_t **errmsg);

/*
 * smtp_tls_init()
 *
 * Prepare TLS encryption. See tls_init() for a description of the arguments.
 * Used error codes: TLS_ELIBFAILED, TLS_EFILE
 */
#ifdef HAVE_SSL
merror_t smtp_tls_init(smtp_server_t *srv, 
	char *tls_key_file, char *tls_ca_file, char *tls_trust_file);
#endif /* HAVE_SSL */

/*
 * smtp_tls_starttls()
 *
 * Announce the start of TLS encryption with an initialized SMTP server, 
 * using the STARTTLS command.
 * Use this function after smtp_init(). The SMTP server must have the
 * SMTP_CAP_STARTTLS capability.
 * Call smtp_tls() afterwards. Finally, call smtp_init() again (the SMTP server
 * might advertise different capabilities when TLS is active, for example plain
 * text authentication mechanisms).
 * 'error_msg' contains the error message from the SMTP server or NULL.
 * Used error codes: SMTP_EIO, SMTP_EPROTO, SMTP_EINVAL, SMTP_ENOMEM
 */
#ifdef HAVE_SSL
merror_t smtp_tls_starttls(smtp_server_t *srv, list_t **error_msg);
#endif /* HAVE_SSL */

/*
 * smtp_tls()
 *
 * Start TLS with a connected SMTP server.
 * Use this function either after smtp_connect() for SMTP servers 
 * that use TLS without the STARTTLS command (service ssmtp; default port 465),
 * or after smtp_tls_starttls() for SMTP servers that support the STARTTLS
 * command.
 * See tls_start() for a description of the arguments.
 * Used error codes: TLS_ELIBFAILED, TLS_ECERT, TLS_EHANDSHAKE
 */
#ifdef HAVE_SSL
merror_t smtp_tls(smtp_server_t *srv, char *hostname, int tls_nocertcheck);
#endif /* HAVE_SSL */

/*
 * smtp_authmech_is_supported()
 *
 * Returns 1 if the authentication mechanism is supported and 0 otherwise. The
 * argument must be one of "DIGEST-MD5", "CRAM-MD5", "PLAIN", "LOGIN", "NTLM".
 */
int smtp_authmech_is_supported(char *mech);

/*
 * smtp_auth()
 *
 * Authenticate with a username and a password.
 * Supported values for 'auth_mech' are "PLAIN", "LOGIN", "CRAM-MD5" and
 * (when GSASL support is compiled in) "DIGEST-MD5" and "NTLM".
 * The special value "" causes the function to choose the best authentication
 * method supported by the server, unless TLS is incative and the method sends
 * plain text passwords (PLAIN, LOGIN). In this case, the function fails with
 * SMTP_EINSECURE.
 * The hostname is needed for DIGEST-MD5 authentication.
 * The SMTP server must support SMTP_CAP_AUTH and one of the supported
 * authentication methods. If it does not support one of the methods above, this
 * function will return SMTP_EUNAVAIL.
 * 'error_msg' contains the error message from the SMTP server or NULL.
 * Used error codes: SMTP_EIO, SMTP_EINVAL, SMTP_EPROTO, SMTP_ENOMEM,
 * 		     SMTP_EAUTHFAIL, SMTP_ELIBFAILED, SMTP_EINSECURE,
 * 		     SMTP_EUNAVAIL
 */
merror_t smtp_auth(smtp_server_t *srv,
	char *hostname,
	char *user, 
	char *password,
	char *auth_mech,
	list_t **error_msg);

/*
 * smtp_send_mail()
 *
 * Sends a mail via the SMTP server 'srv'.
 * envelope_from:	The envelope from address
 * rcptc:		The number of recipients
 * rcptv:		The recipients
 * dsn_notify:		Delivery Status Notification request string (see man
 * 			page) or NULL. The SMTP server must support SMTP_CAP_DSN.
 * dsn_return:		Either "HDRS", "FULL" or NULL. The SMTP server must 
 * 			support SMTP_CAP_DSN.
 * keep_bcc:		When his flag is set, the Bcc header will be sent
 * mailf:		The file containing the mail
 * mailsize:		The number of bytes of the mail (as transferred to the
 *                      SMTP server) will be stored here (in case of successful 
 *                      delivery; the contents are undefined in case of failure).
 * error_msg:		If an error occurs, this will contain the SMTP server 
 * 			message (or NULL)
 * Used error codes: SMTP_EIO, SMTP_ENOMEM, SMTP_EPROTO, SMTP_EINVAL,
 *                   SMTP_EUNAVAIL
 */
merror_t smtp_send_mail(smtp_server_t *srv,
	char *envelope_from, 
	int rcptc, char *rcptv[],
	char *dsn_notify,
	char *dsn_return,
	int keep_bcc,
	FILE *mailf,
	long *mailsize,
	list_t **error_msg);

/*
 * smtp_quit()
 *
 * Sends the QUIT command to the SMTP server 'srv' to end the current session.
 * Use smtp_close() after this function.
 * Used error codes: SMTP_EIO, SMTP_ENOMEM, SMTP_EPROTO, SMTP_EINVAL
 */
merror_t smtp_quit(smtp_server_t *srv);
    
/*
 * smtp_close()
 *
 * Closes the connection to the SMTP server 'srv'.
 * 'srv' is unusable afterwards; reinitialize it with smtp_new() if you want 
 * to reuse it.
 */
void smtp_close(smtp_server_t *srv);

#endif
