/* Time-stamp: <2006-05-21 04:15:17 poser> */
/*
 * Copyright (C) 1993-2006 William J. Poser.
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or go to the web page:  http://www.gnu.org/licenses/gpl.txt.
 *
 * This program sorts text using an arbitrary sort order specified
 * in a file. Records may be either single lines or blocks of text.
 * Sorting may be on the entire record or on a specified field,.
 * where the set of field terminator characters may be specified
 * on the command line. The fields on which to sort may be specified
 * either by number or by the tags with which they begin.
 * Keys may be optional or obligatory.
 *
 * Author:  Bill Poser
 *	 
 */

#define PROGNAME PACKAGE_NAME
#define LOGNAME PROGNAME ".log"

#include "config.h"
#include "compdefs.h"

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <alloca.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/file.h>
#include <wctype.h>
#include <wchar.h>
#include <time.h>
#include <errno.h>

#include "unicode.h"
#include "exitcode.h"
#include "retcodes.h"
#include "input.h"
#include "limits.h"
#include "regex.h"
#include "key.h"
#include "record.h"
#include "ex_codes.h"
#include "dstr.h"
#include "comparisons.h"
#ifdef HAVE_GETOPT_LONG
#define _GNU_SOURCE
#include <getopt.h>
#endif
#ifdef HAVE_LANGINFO_H
#include <langinfo.h>
#endif
#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif
#ifdef LOCALE_GETTEXT
#include <libintl.h>
#define _(x) gettext(x)
#else
#define _(x) (x)
#endif
#define REPORTINTERVAL 500
#define QUICKSORTTHRESHOLD 1000L
#define INTWREPORTINTERVAL (4 * REPORTINTERVAL)
#define NOTACHARACTER (wchar_t) 0xFDDF

#ifdef HAVE_LONGDOUBLE
#define DOUBLE (long double)
#else
#define DOUBLE double
#endif

#define PRTEMPSTRLEN 128
#define MIN(x,y) (x<y?x:y) 
#define MAX(x,y) (x>y?x:y) 

#define TRUE 1
#define FALSE 0

/* How are exclusions specified? */

#define FILESPEC 1
#define STRINGSPEC 2

#ifndef HAVE_GETOPT_LONG

struct option
{
# if (defined __STDC__ && __STDC__) || defined __cplusplus
  const char *name;
# else
  char *name;
# endif
  /* has_arg can't be an enum because some compilers complain about
     type mismatches in all the code that assumes it is an int.  */
  int has_arg;
  int *flag;
  int val;
};

int
getopt_long(int ac,
	    char *const *av,
	    const char * sopts,
	    const struct option *lopts, 
	    int *lind){

  return(getopt(ac,av,sopts));
}
#endif

char progname[]=PROGNAME;
char version[]=PACKAGE_VERSION;
char *compdate= __DATE__ " " __TIME__;

/*
 *  Missing key comparison options
 *  DO NOT CHANGE THESE VALUES! The code in Compare()
 *  depends on these specific values.
 */

#define COMPARE_LESSTHAN 0
#define COMPARE_EQUAL 1
#define COMPARE_GREATERTHAN 2

/* Types of key field selections */

#define WHOLE 0
#define TAGGED 1
#define NUMBERED 2
#define CRANGE 3

#define RPT_NNBLOCK 0
#define RPT_LINE    1
#define RPT_SEPCHAR 2
#define RPT_FIXEDLENGTH 3

long RankTableSize = MAXUNICODE;
wchar_t *DefaultWhiteSpace=L"\x0020\x0009\x1361\x1680\x2000\x2001\x2002\x2003\x2004\x2005\x2006\x2007\x2008\x2009\x200A\x200B\x205F\x3000";


struct option opts[]={
  "algorithm",1,NULL,'a',
  "BMP",0,NULL,'B',
  "block",0,NULL,'b',
  "check-only",0,NULL,'Q',
  "fold-case",0,NULL,'C',
  "comparison-type",1,NULL,'c',
  "defaults",0,NULL,'D',
  "field-separators",1,NULL,'d',
  "character-range",1,NULL,'e',
  "general-options",0,NULL,'F',
  "date-format",1,NULL,'f',
  "help",0,NULL,'h',
  "invert-globally",0,NULL,'I',
  "invert-locally",0,NULL,'i',
  "key-specific-options",0,NULL,'K',
  "limits",0,NULL,'L',
  "line",0,NULL,'l',
  "initial-maximum-records",1,NULL,'M',
  "line-end-carriage-return",0,NULL,'m',
  "position",1,NULL,'n',
  "fixed-size-record-size",1,NULL,'O',
  "optional",1,NULL,'o',
  "reserve-private-use-areas",0,NULL,'p',
  "quiet",0,NULL,'q',
  "reverse-key",0,NULL,'R',
  "record-separator",1,NULL,'r',
  "substitution-file",1,NULL,'S',
  "sort-order",1,NULL,'s',
  "transformations",1,NULL,'T',
  "tag",1,NULL,'t',
  "version",0,NULL,'v',
  "sort-order-file-separators",1,NULL,'W',
  "whole-record",0,NULL,'w',
  "exclude-characters",1,NULL,'X',
  "exclusion-file",1,NULL,'x',
  "fold-case-turkic",0,NULL,'z',
    0,0,0,0
};

/* This is the array containing information about all the keys */

#define INITIALKEYS 4

struct keyinfo **KeyInfo;

/* This is the list of all of the records */

struct record **RecordList;

int VerboseP=1;		        /* Chat while we work */
int KeyCount;			/* Number of keys to sort on */
UTF8 *InputText;		/* Temporary storage for input text*/
wchar_t *wcInputText;		/* UTF-32 counterpart of above */
struct dstr TempString;

wchar_t *Terminators;		/* Characters terminating fields */
int TerminatorCnt; /* Number of terminator characters */
int InitialMaxFields;	/* Initial value of MaxFields */

short BMPOnlyP;			/* Only deal with Basic Multilingual Plane? */
short MultipleLocalesP;		/* Are we using more than one locale? */
short ReservePrivateUseAreaP;
short ReverseP;		/* Invert the sense of comparisons */	

FILE *Logfp;			/* Log file */

#ifdef HAVE_LONG_LONG
typedef long long LongLong; 
#else
typedef long LongLong;
#endif

wchar_t *DefaultMonthNames[]={
  L"January",L"Jan",
  L"February",L"Feb",
  L"March",L"Mar",
  L"April",L"Apr",
  L"May",L"May",
  L"June",L"Jun",
  L"July",L"Jul",
  L"August",L"Aug",
  L"September",L"Sep",
  L"October",L"Oct",
  L"November",L"Nov",
  L"December",L"Dec"
};

LongLong ComparisonCnt;	/* Calls to Compare */

wchar_t MG_First;
wchar_t MG_Last;
long MG_Total;

extern void putu8s(wchar_t *, FILE *);

void
SetInitialKeysMultigraphInfo () {
  int i;
  for(i=0; i< INITIALKEYS; i++) { 
    KeyInfo[i]->NextCode = MG_First;
    KeyInfo[i]->MaxMGCode = MG_Last;
    KeyInfo[i]->TotalCodes = MG_Total;
  }
}


void
SetMultigraphInfo (void) {

  if(ReservePrivateUseAreaP){
    MG_First = NPU_FIRSTMULTIGRAPHCODE;   
    MG_Last  = NPU_MAXMULTIGRAPHCODE;
  }
  else if (BMPOnlyP) {
    MG_First = BMP_FIRSTMULTIGRAPHCODE;  
    MG_Last =  BMP_MAXMULTIGRAPHCODE;
  }
  else {
    MG_First = FUL_FIRSTMULTIGRAPHCODE;
    MG_Last =  FUL_MAXMULTIGRAPHCODE; 
  }
  MG_Total = (long) (MG_Last - MG_First + 1);
  SetInitialKeysMultigraphInfo();
}


void
ValidateCRange(char *s,int fields,int keycnt) {
  if(fields != 2) {
    fprintf(stderr,_("Ill-formed character range %s\n"),s);
    exit(BADOPTIONARG);
  }
  if(KeyInfo[keycnt]->RangeFirst > KeyInfo[keycnt]->RangeLast) {
    fprintf(stderr,_("Invalid character range: %s. Beginning of character range must not follow end.\n"),s);
    exit(BADOPTIONARG);
  }
}

#ifdef LOCALE_MONTHNAMES
int mb2u32(char *src,wchar_t **tgt,struct keyinfo *ki) {
  int CharsNeeded;
  int BytesNeeded;
  int Status;
  wchar_t *new;

  /* Do I need to save and restore the previous locale? */
  CharsNeeded = mbstowcs(NULL,src,0);
  /* allocate */
  BytesNeeded =  (size_t) ((CharsNeeded + 1) * sizeof(wchar_t));
  if((new = (wchar_t *) malloc(BytesNeeded)) == NULL) {
    fprintf(stderr,"Out of memory.\n");
    exit(OUTOFMEMORY);
  }
  Status = mbstowcs(new,src,CharsNeeded);
  if(Status < 0) {
    return 1;
  } else {
    *tgt = new;
    return 0;
  }
}
#endif

void GetMonthNames(int KeyNumber) {
#ifdef LOCALE_MONTHNAMES
  char *tmp;
  int status = 0;
  wchar_t dummy;
  extern int u82u32(UTF8 *, wchar_t **,wchar_t *);
  
  if(KeyInfo[KeyNumber]->LocaleP) {
    tmp = setlocale(LC_TIME,KeyInfo[KeyNumber]->Locale);
    setlocale(LC_CTYPE,KeyInfo[KeyNumber]->Locale);
  }
  else{
    tmp = setlocale(LC_TIME,"");
    setlocale(LC_CTYPE,"");
  }
  if(tmp == NULL) {
    fprintf(stderr,_("Current locale is not supported.\n"));
    exit(OTHERERROR);
  }
  status+=mb2u32(nl_langinfo(MON_1),&(KeyInfo[KeyNumber]->MonthNames[0]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(ABMON_1),&(KeyInfo[KeyNumber]->MonthNames[1]),KeyInfo[KeyNumber]); 
  status+=mb2u32(nl_langinfo(MON_2),&(KeyInfo[KeyNumber]->MonthNames[2]),KeyInfo[KeyNumber]); 
  status+=mb2u32(nl_langinfo(ABMON_2),&(KeyInfo[KeyNumber]->MonthNames[3]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(MON_3),&(KeyInfo[KeyNumber]->MonthNames[4]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(ABMON_3),&(KeyInfo[KeyNumber]->MonthNames[5]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(MON_4),&(KeyInfo[KeyNumber]->MonthNames[6]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(ABMON_4),&(KeyInfo[KeyNumber]->MonthNames[7]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(MON_5),&(KeyInfo[KeyNumber]->MonthNames[8]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(ABMON_5),&(KeyInfo[KeyNumber]->MonthNames[9]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(MON_6),&(KeyInfo[KeyNumber]->MonthNames[10]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(ABMON_6),&(KeyInfo[KeyNumber]->MonthNames[11]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(MON_7),&(KeyInfo[KeyNumber]->MonthNames[12]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(ABMON_7),&(KeyInfo[KeyNumber]->MonthNames[13]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(MON_8),&(KeyInfo[KeyNumber]->MonthNames[14]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(ABMON_8),&(KeyInfo[KeyNumber]->MonthNames[15]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(MON_9),&(KeyInfo[KeyNumber]->MonthNames[16]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(ABMON_9),&(KeyInfo[KeyNumber]->MonthNames[17]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(MON_10),&(KeyInfo[KeyNumber]->MonthNames[18]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(ABMON_10),&(KeyInfo[KeyNumber]->MonthNames[19]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(MON_11),&(KeyInfo[KeyNumber]->MonthNames[20]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(ABMON_11),&(KeyInfo[KeyNumber]->MonthNames[21]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(MON_12),&(KeyInfo[KeyNumber]->MonthNames[22]),KeyInfo[KeyNumber]);
  status+=mb2u32(nl_langinfo(ABMON_12),&(KeyInfo[KeyNumber]->MonthNames[23]),KeyInfo[KeyNumber]);
  if(status) {
    fprintf(stderr,"Error reading month name information for locale.\n");
    exit(OTHERERROR);
  }
#else
  int i;
  for(i=0;i<24;i++){
    KeyInfo[KeyNumber]->MonthNames[i] = DefaultMonthNames[i];
  }
#endif
}

void CheckFree (void *p,int lineno) {
#ifdef DEBUGP
  fprintf(stderr,"CheckFree - line %5d:  %p\n",lineno,p);
#endif
  if(p != NULL) free(p);
  else {
    fprintf(stderr,"Attemp to free null pointer %p at line %d.\n",p,lineno);
    fflush(stderr);
  }
}

#define CheckFree(x,y) free(x)
int
main(int ac, char **av)
{
  int i;			/* Loop index */
  int InitialMaxRecords;	/* Initial value of MaxRecords */
  int MaxRecords;		/* Maximum records allocated */
  int MaxKeys = INITIALKEYS;	/* Maximum number of keys for which slots are currently available */
  int InputTextSize;		/* Size of input text buffer */
  unsigned long Records;	/* Number of well-formed records */
  unsigned long InputRecords;	/* Number of records in input, ill-formed or well-formed */
  int RecordLength;		/* Length of text in current record */
  int WholeKeyP;		/* Use whole record as key? */
  enum Algorithm {INSERTIONSORT,MERGESORT,QUICKSORT,SHELLSORT};
  enum Algorithm SortAlgorithm = QUICKSORT;
  short RecordParseType;	/* Line, block, etc. */
  int opt;			/* Option returned by getopt() */
  int TerminatorsSetP;		/* Did user override default terminators? */
  wchar_t RecordSeparator;
  wchar_t MaxCodeInInput = 0L;
  wchar_t tmpwc;
  int tmpint1;			/* Temporary integer */
  short FailedKeyP;		/* True if key extraction failed on some record */
  short CheckOnlyP= 0;
  int ExitValue = SUCCESS;

  wchar_t *TagTemp = NULL;	/* Temporary string storage */
  size_t TagTempStringLength =0; /* Current length of above */
  size_t TagTempLengthNeeded;
  char PrTempString[PRTEMPSTRLEN+1];
  char EndOfLine = 0x0A;	/* Default to linefeed */
  char *tmpptr1;		/* Temporary pointer */
  char *tmpptr2;		/* Temporary pointer */
  wchar_t *tmpwcptr;		/* Temporary pointer */
  char *infile;			/* Name of input file */
  FILE *infp;			/* Input file */
  FILE *outfp;			/* Output file */
  FILE *fp;
  int status;
  int SortOrderSpecifiedP = 0;
  char *endptr;			/* Used by strtol */
  char * tmpstr;
  char *FirstLocale = NULL;
  void *InitialBreak;
  void *FinalBreak;
  int lgoindex;			/* Unused but needed by getopt_long */
  char t;
struct record **SavedRecordList;

  UTF8 * (*GetRecord)(FILE *, UTF8 *, int *, int *, wchar_t);

  int GetExclusions(char *,struct keyinfo *);
  int GetKeys(struct record *, struct keyinfo **,int,wchar_t);
  int GetSortOrder(char *,struct keyinfo *);
  int SetDateFormat(wchar_t *, struct ymdinfo *);
  void WriteOutRecords(long, FILE *,int,wchar_t);
  void DescribeKeys(FILE *);
  void InitializeKeyInfo(struct keyinfo *);
  void CreateRankTable(struct keyinfo *);
  void ProvideDefaultRanks(void);
  void Copyright(FILE *);
  void CreateInitialKeys(void);
  void GetExclusionsString(UTF8 *,struct keyinfo *);
  void CheckKeyAllocation(int, int *);
  int GetSubstitutions(char *, struct keyinfo *);
  int GetWhiteSpaceDefinition(char *, struct keyinfo *);
  
  extern int optind;		/* Set by getopt */
  extern char *optarg;		/* Set by getopt */
  extern int optopt;
  extern int opterr;

  extern FILE *OpenFile(char *,char *,char *);
  extern char *copy_string(char *);
  extern wchar_t *wCreateString(int);
  extern int EvalEscapes(wchar_t *);
		
  extern UTF8 * GetNNBlockRAUTF8(FILE *, unsigned char *, int *, int *, wchar_t);
  extern UTF8 * GetBlockSepCharRAUTF8(FILE *, UTF8 *, int *, int *, wchar_t);
  extern UTF8 * GetFixedLengthRecord(FILE *, UTF8 *, int *, int *, wchar_t);

  extern  void InsertionSort(struct record **,long);
  extern  void MergeSort(struct record **, long, long);
  extern  void QuickSort(struct record **,long,long);
  extern  void ShellSort(struct record **,long);

  extern void CopyCommandLine(FILE *, int, char **);
  extern void IdentifySelf(FILE *);
  extern void PrintDefaults(FILE *);
  extern void PrintGeneralFlags(FILE *);
  extern void PrintKeySpecificFlags(FILE *);
  extern void PrintLimits(FILE *);
  extern void PrintUsage(void);
  extern wchar_t *WCopyU8String(UTF8 *); /* Need to write this or find it in a library */
  extern int u82u32 (UTF8 *,wchar_t **,wchar_t *);

  extern int getopt(int,char * const [],const char *);
  extern UTF8 *strcpyu8(UTF8 *, UTF8 *);

  
  /* Executable statements begin here */

  InitialBreak = sbrk(0);
#ifdef HAVE_SETLOCALE
   setlocale(LC_ALL,"");
#endif
#ifdef LOCALE_GETTEXT
   bindtextdomain (PACKAGE, LOCALEDIR);
   textdomain (PACKAGE);
#endif

  if(ac < 2){
    IdentifySelf(stderr);
    Copyright(stderr);
    PrintUsage();
    exit(INFO);
  }

  /* 
   * Initializations and defaults.
   *
   * Note that the defaults for field terminators are set
   * after the command line is processed since they depend
   * on whether records are lines or blocks.
   */

  Records = 0L;
  InputRecords = 0L;
  KeyCount = 0;
  ComparisonCnt = 0L;
   
  InitialMaxRecords = DEFMAXRECORDS;
  InitialMaxFields = DEFMAXFIELDS;
  InputTextSize = INPUTTEXTSIZEINIT;

  RecordParseType = RPT_NNBLOCK;
  GetRecord = GetNNBlockRAUTF8;
   
  ReverseP = FALSE;
  TerminatorsSetP = FALSE;
  WholeKeyP = FALSE;
  FailedKeyP = FALSE;
  BMPOnlyP = FALSE;
  MultipleLocalesP = FALSE;
  ReservePrivateUseAreaP = FALSE;

 
  /* Allocate storage */
   
  InputText = (UTF8 *) malloc((size_t) InputTextSize * sizeof(UTF8));
  if(InputText == NULL){
    fprintf(stderr,_("%s: out of memory - InputText .\n"),progname);
    exit(OUTOFMEMORY);
  }
   
  TempString.c = InputTextSize;
  TempString.l = 0;
  TempString.s = (wchar_t *) malloc((size_t) TempString.l * sizeof(wchar_t));
  if(TempString.s == NULL){
    fprintf(stderr,_("%s: out of memory - TempString.\n"),progname);
    exit(OUTOFMEMORY);
  }
  
  /* Open log file */

  Logfp = OpenFile(LOGNAME,"w",progname);
  if(Logfp == NULL) {
    fprintf(stderr,_("Warning: unable to open log file %s in current directory.\n"),LOGNAME);
    snprintf(PrTempString,PRTEMPSTRLEN,"/tmp/%s",LOGNAME);
    Logfp = OpenFile(PrTempString,"w",progname);
    if(Logfp == NULL) {
      fprintf(stderr,_("Warning: unable to open log file %s.\n"),PrTempString);
      exit(OPENERROR);
    }
  }
  IdentifySelf(Logfp);
  Copyright(Logfp);

  fprintf(Logfp,_("Invoked by command line:\n"));
  CopyCommandLine(Logfp,ac,av);
  CreateInitialKeys();

  /* Process command line */
#ifdef DEBUGP 
  fprintf(stderr,"About to process command line.\n");fflush(stderr);
#endif
   
  opterr = 0;			/* We'll handle errors ourselves */
  while( (opt = getopt_long(ac,av,":a:Bbc:Cd:De:f:FhiIlKLmM:n:O:o:pQqr:Rs:S:T:t:vwW:x:X:z",&opts[0],&lgoindex)) != EOF){
    switch(opt){
    case 'a':
      switch(tolower(optarg[0])){
      case 'i':
	SortAlgorithm = INSERTIONSORT;
	break;
      case 'm':
	SortAlgorithm = MERGESORT;
	break;
      case 'q':
	SortAlgorithm = QUICKSORT;
	break;
      case 's':
	SortAlgorithm = SHELLSORT;
	break;
      default:
	fprintf(stderr,_("Unrecognized sort algorithm %s"),optarg);
	exit(BADOPTIONARG);
      }
      break;
    case 'm':
      EndOfLine = 0x0d;		/* Carriage return, as on Mac */
      break;
    case 'p':
      if(SortOrderSpecifiedP) {
	fprintf(stderr,_("The -p flag must precede all sort order specifications.\n"));
	fprintf(stderr,_("Since this is not the case, it will be ignored.\n"));
	break;
      }
      ReservePrivateUseAreaP = 1;
      break;
    case 'B':		/* Keys will not contain characters outside the BMP */
      if(SortOrderSpecifiedP) {
	fprintf(stderr,_("The -B flag must precede all sort order specifications.\n"));
	fprintf(stderr,_("Since this is not the case, it will be ignored.\n"));
	break;
      }
      BMPOnlyP = 1;
      RankTableSize = MAXBMP;
      break;
      /* How records are delimited */
    case 'b':		/* A record is a newline-terminated block */
      RecordParseType = RPT_NNBLOCK;
      break;
    case 'l':
      RecordParseType = RPT_LINE;
      break;
    case 'r':		/* Specify record separator */
      RecordParseType = RPT_SEPCHAR;
      if ((status = u82u32((unsigned char *)optarg,&tmpwcptr,&tmpwc)) != 0) {
	fprintf(stderr,_("Fatal error while setting record separator.\n"));
	exit(status);
      }
      if(EvalEscapes(tmpwcptr) == ERROR){
	fprintf(stderr,_("EvalEscapes: out of memory.\n"));
	exit(OUTOFMEMORY);
      }
      RecordSeparator = tmpwcptr[0];
      CheckFree(tmpwcptr,__LINE__);
      break;

    case 'O':			/* Fixed-length record */
      RecordParseType = RPT_FIXEDLENGTH;
      tmpint1 = (int) strtol(optarg,&endptr,0);
      if(errno == ERANGE){
	fprintf(stderr,_("%s: value %s out of representable range\n"),
		progname,optarg);
	exit(BADOPTIONARG);
      }
      if(errno == EINVAL){
	fprintf(stderr,_("%s: value %s ill-formed\n"),
		progname,optarg);
	exit(BADOPTIONARG);
      }
      if(*endptr != '\0'){
	fprintf(stderr,_("%s: ill-formed input %s\n"),progname,optarg);
	exit(BADOPTIONARG);
      }
      if (tmpint1 < 1) {
	fprintf(stderr,_("%s: The length of a fixed length record must be positive.\n"),progname);
	exit(BADOPTIONARG);
      }
      InputTextSize = tmpint1;
      break;
	 
      /* How fields are delimited */
    case 'd':		/* Fields are terminated by the named characters */
      if((status = u82u32((UTF8 *)optarg,&tmpwcptr,&tmpwc)) != 0) {
	fprintf(stderr,_("Fatal error while setting field separator.\n"));
	exit(status);
      }
      if(EvalEscapes(tmpwcptr) == ERROR){
	fprintf(stderr,_("EvalEscapes: out of memory.\n"));
	exit(OUTOFMEMORY);
      }
      Terminators = tmpwcptr;
      tmpwcptr = NULL;
      TerminatorsSetP = TRUE;
      break;
 
      /* What to sort on */
    case 'e':		/* Character range */
      if(WholeKeyP){
	fprintf(stderr,_("Use of key field and whole record is inconsistent.\n"));
	exit(BADOPTION);
      }
      CheckKeyAllocation (KeyCount,&MaxKeys);
      KeyInfo[KeyCount]->SelectType = CRANGE;
      tmpint1 = sscanf(optarg,"%d,%d",&KeyInfo[KeyCount]->RangeFirst,&KeyInfo[KeyCount]->RangeLast);
      /* Our indices are zero-based but the user's are one-based */
      if (KeyInfo[KeyCount]->RangeFirst > 0) KeyInfo[KeyCount]->RangeFirst--;
      if (KeyInfo[KeyCount]->RangeLast > 0)  KeyInfo[KeyCount]->RangeLast--;
      ValidateCRange(optarg,tmpint1,KeyCount);
      ++KeyCount;
      break;
    case 't':		/* Next key has specified tag */
      if(WholeKeyP){
	fprintf(stderr,_("Use of key field and whole record is inconsistent.\n"));
	exit(BADOPTION);
      }
      CheckKeyAllocation(KeyCount,&MaxKeys);
      KeyInfo[KeyCount]->SelectType = TAGGED;
      if((status = u82u32((UTF8 *)optarg,&tmpwcptr,&tmpwc)) != 0) {
	fprintf(stderr,_("Fatal error while setting tag.\n"));
	exit(status);
      }
      TagTempLengthNeeded = wcslen(tmpwcptr)+wcslen(L"()(.*)")+2;
      if( TagTempLengthNeeded > TagTempStringLength){
	TagTemp = (wchar_t *) realloc((void *)TagTemp,(size_t) TagTempLengthNeeded * sizeof(wchar_t));
	if(TagTemp == NULL){
	  fprintf(stderr,_("TagTemp: out of memory.\n"));
	  exit(OUTOFMEMORY);
	}
	TagTempStringLength = TagTempLengthNeeded;
      }	   
      tmpint1 = swprintf(TagTemp,TagTempStringLength-1,L"(%ls)(.*)",tmpwcptr);
      if(EvalEscapes(TagTemp) == ERROR){
	fprintf(stderr,_("EvalEscapes: out of memory.\n"));
	exit(OUTOFMEMORY);
      }
      tmpint1 = regwcomp(&(KeyInfo[KeyCount]->cregexp),TagTemp,REG_EXTENDED);
      if(tmpint1 != 0){
	fprintf(stderr,_("Failed to compile regular expression: %s.\n"), optarg);
	fprintf(Logfp, _("Failed to compile regular expression: %s.\n"), optarg);
	exit(BADOPTIONARG);
      }
      KeyInfo[KeyCount]->Tag = (wchar_t *) malloc(sizeof(wchar_t) * wcslen(tmpwcptr)+1);
      if(KeyInfo[KeyCount]->Tag) wcscpy(KeyInfo[KeyCount]->Tag,tmpwcptr); /* For report */
      else{
	fprintf(stderr,_("Out of memory.\n"));
	exit(OUTOFMEMORY);
      }
      ++KeyCount;
      CheckFree((void *)tmpwcptr,__LINE__);
      break;
    case 'n':		/* Next key is position or range between positions */
      if(WholeKeyP){
	fprintf(stderr,_("Use of key field and whole record is inconsistent.\n"));
	exit(BADOPTION);
      }
      CheckKeyAllocation(KeyCount,&MaxKeys);
      tmpptr1 = strchr(optarg,',');
      if(tmpptr1 != NULL) {
	*tmpptr1 = '\0';
	/* First = optarg */
	tmpptr2 = strchr(optarg,'.');
	if(tmpptr2 != NULL) {
	  *tmpptr2 = '\0';
	  KeyInfo[KeyCount]->FirstOffset = atoi(tmpptr2+1)-1;
	}
	KeyInfo[KeyCount]->Number = atoi(optarg) -1;
	
	/* Second = tmpptr1+1 */
	tmpptr2 = strchr(tmpptr1+1,'.');
	if(tmpptr2 != NULL) {
	  *tmpptr2 = '\0';
	  KeyInfo[KeyCount]->LastOffset = atoi(tmpptr2+1)-1;
	}
	KeyInfo[KeyCount]->Last = atoi(tmpptr1+1) -1;
      }
      else {
	tmpptr2 = strchr(optarg,'.');
	if(tmpptr2 != NULL) {
	  *tmpptr2 = '\0';
	  KeyInfo[KeyCount]->FirstOffset = atoi(tmpptr2+1)-1;
	}
	KeyInfo[KeyCount]->Number = atoi(optarg) -1;
	KeyInfo[KeyCount]->Last = KeyInfo[KeyCount]->Number;
      }
      KeyInfo[KeyCount]->SelectType = NUMBERED;
      ++KeyCount;
      break;
    case 'Q':
      CheckOnlyP = 1;
      break;
    case 'w':		/* Sort on the entire input text */
      WholeKeyP=TRUE;
      if(KeyCount > 0){
	fprintf(stderr,_("Use of key field and whole record is inconsistent.\n"));
	exit(BADOPTION);
      }
      KeyCount = 1;
      KeyInfo[0]->SelectType = WHOLE;
      break;
    case 'i':
      if(KeyCount <1){
	fprintf(stderr,_("Key inversion specified without previous key selector.\n"));
	exit(BADOPTION);
      }
      KeyInfo[KeyCount-1]->CompType |= CINVERSE;
      break;
    case 'C':
      KeyInfo[KeyCount-1]->FoldCaseP = 1;
      break;
    case 'z':
      KeyInfo[KeyCount-1]->FoldCaseP = 1;
      KeyInfo[KeyCount-1]->TurkicFoldCaseP = 1;
      break;
    case 'R':
      if(KeyCount <1){
	fprintf(stderr,_("Key reversal specified without previous key selector.\n"));
	exit(BADOPTION);
      }
      KeyInfo[KeyCount-1]->ReverseP = TRUE;
      break;

    case 'f':
      if(KeyCount <1){
	fprintf(stderr,_("Format specified without previous key selector.\n"));
	exit(BADOPTION);
      }
      if((status = u82u32((UTF8 *)optarg,&tmpwcptr,&tmpwc)) != 0) {
	fprintf(stderr,_("Fatal error setting date format.\n"));
	exit(status);
      }
      if(KeyInfo[KeyCount-1]->CompType & CDATE) {
	if(SetDateFormat(tmpwcptr,&(KeyInfo[KeyCount-1]->ymd)) == ERROR){
	  fprintf(stderr,_("%s is not a valid date format.\n"),optarg);
	  exit(BADOPTIONARG);
	}
      } else {
	fprintf(stderr,
		_("It doesn't make sense to set a format for a key that is not a date or time.\n"));
      }
      CheckFree((void *)tmpwcptr,__LINE__);
      break;
    case 'o':
      if(KeyCount <1){
	fprintf(stderr,_("Optionality specified without previous key selector.\n"));
	exit(BADOPTION);
      }
      else{
	if(KeyInfo[0]->SelectType == WHOLE){
	  fprintf(stderr,_("If the whole record is the key, the key may not be optional.\n"));
	  exit(BADOPTION);
	}
      }
      KeyInfo[KeyCount-1]->OptionalP = TRUE;
      switch(optarg[0]){
      case 'l':
      case 'L':
      case '<':
	KeyInfo[KeyCount-1]->MissingKeyComparison = COMPARE_LESSTHAN;
	break;
      case 'g':
      case 'G':
      case '>':
	KeyInfo[KeyCount-1]->MissingKeyComparison = COMPARE_GREATERTHAN;
	break;
      case 'e':
      case 'E':
      case '=':
      default:
	KeyInfo[KeyCount-1]->MissingKeyComparison = COMPARE_EQUAL;
	break;
      }
      break;

      /* What files to use */	
    case 'S':		/* Read substitutions from the named file */
      if(KeyCount < 1){
	fprintf(stderr,_("Substitutions specified without previous key selector.\n"));
	exit(BADOPTION);
      }
      if(GetSubstitutions(optarg,KeyInfo[KeyCount-1]) != SUCCESS){
	fprintf(stderr,
		_("Attempt to read substitutions from file %s failed.\n"),optarg);
	exit(OTHERERROR);
      }
      fprintf(Logfp,_("Read %d substitutions for key %d from file %s.\n"),
	      KeyInfo[KeyCount-1]->SubListEntries,KeyCount,optarg);
      break;
    case 's':		/* Use sort order provided by locale or file */
      if(KeyCount <1){
	fprintf(stderr,_("Sort type specified without previous key selector.\n"));
	exit(BADOPTION);
      }

      if( (strcmp(optarg,"locale") == 0) || (strcmp(optarg,"current") == 0)) {
#ifdef LOCALE_SORT_ORDER
	tmpstr = setlocale(LC_COLLATE,"");
	if(tmpstr != NULL) {
	  KeyInfo[KeyCount-1]->LocaleP = TRUE;
	  KeyInfo[KeyCount-1]->Locale = copy_string(tmpstr);
	  break;
	} else {
	  fprintf(stderr,_("Current locale is not supported.\n"));
	  exit(BADOPTIONARG);
	}
#else
	fprintf(stderr,"The use of locale information is not available on this system.\n");
	exit(BADOPTIONARG);
#endif
      }

      /* First see if the argument is a filename */
      if ((fp = fopen(optarg,"r")) == NULL) {
#ifdef LOCALE_SORT_ORDER
	if( (tmpstr = setlocale(LC_COLLATE,optarg)) == NULL) {
	  fprintf(stderr,_("Could not set LC_COLLATE to %s.\n"),optarg);
	  exit(BADOPTIONARG);
	}
	else {
	  KeyInfo[KeyCount-1]->LocaleP = TRUE;
	  KeyInfo[KeyCount-1]->Locale = copy_string(tmpstr);
	  break;
	}
#else
	fprintf(stderr,"Could not open file %s.\n",optarg);
#endif
      }
      else fclose(fp);
      if(SortOrderSpecifiedP == 0) SetMultigraphInfo();
      SortOrderSpecifiedP = 1;
      if(GetSortOrder(optarg,KeyInfo[KeyCount-1]) == ERROR){
	fprintf(stderr,
		_("Attempt to read sort order from file %s failed.\n"),optarg);
	exit(OTHERERROR);
      }
      fprintf(Logfp,_("Read sort order with %d multigraphs for key %d from file %s.\n"),
	      KeyInfo[KeyCount-1]->MapTableEntries,KeyCount,optarg);
      break;
    case 'x':		/* Read exclusions from the named file */
      if(KeyCount <1){
	fprintf(stderr,_("Exclusions specified without previous key selector.\n"));
	exit(BADOPTION);
      }
      if(KeyInfo[KeyCount-1]->ExclusionSpec == STRINGSPEC){
	fprintf(stderr,_("Exclusions already specified on command line for this key.\n"));
	exit(BADOPTION);
      }
      if(GetExclusions(optarg,KeyInfo[KeyCount-1]) != SUCCESS){
	fprintf(stderr,
		_("Attempt to read exclusions from file %s failed.\n"),optarg);
	exit(OTHERERROR);
      }
      fprintf(Logfp,_("Read %d exclusions for key %d from file %s.\n"),
	      KeyInfo[KeyCount-1]->ExclusionEntries,KeyCount,optarg);
      KeyInfo[KeyCount-1]->ExclusionSpec=FILESPEC;
      break;

    case 'T':			/* Transformations */
      tmpptr1 = optarg;
      while((t = *tmpptr1)) {
	tmpptr1++;
	switch(t) {
	case 'd':
	  KeyInfo[KeyCount-1]->StripDiacriticsP = 1;
	  break;
	case 'e':
	  KeyInfo[KeyCount-1]->ConvertEnclosedP = 1;
	  break;
	case 's':
	  KeyInfo[KeyCount-1]->ConvertStylisticP = 1;
	  break;
	default:
	  fprintf(stderr,_("%c is not a recognized transformation designation.\n"),t);
	  exit(BADOPTIONARG);
	}
      }
      break;
    case 'X':		/* Read exclusions from command line */
      if(KeyCount <1){
	fprintf(stderr,_("Exclusions specified without previous key selector.\n"));
	exit(BADOPTION);
      }
      if(KeyInfo[KeyCount-1]->ExclusionSpec == FILESPEC){
	fprintf(stderr,_("Exclusions already specified in file for this key.\n"));
	exit(BADOPTION);
      }
      GetExclusionsString((unsigned char *)optarg,KeyInfo[KeyCount-1]);	
      KeyInfo[KeyCount-1]->ExclusionSpec=STRINGSPEC;
      break;
	 
      /* Sizes of things */
	 
    case 'M':
      InitialMaxRecords = atoi(optarg);
      if(InitialMaxRecords <= 2){
	fprintf(stderr,
		_("%s: invalid initial maximum record specification %s.\n"),
		progname,optarg);
	exit(BADOPTIONARG);
      }
      break;
	 
      /* Miscellaneous */
	 
    case 'I':		/* Reverse sense of comparisons globally */
      ReverseP = TRUE;
      break;
    case 'c':		/* Comparison type */
      if(KeyCount <1){
	fprintf(stderr,_("Comparison type specified without previous key selector.\n"));
	exit(BADOPTION);
      }
      switch(optarg[0]){
      case 'a':
	KeyInfo[KeyCount-1]->CompType |= (CANGLE | CNUMERIC);
	break;
      case 'd':
	KeyInfo[KeyCount-1]->CompType |= (CDATE | CNUMERIC);
	break;
      case 'h':
	KeyInfo[KeyCount-1]->CompType |= CHYBRID;
	break;
      case 'i':
	KeyInfo[KeyCount-1]->CompType |= (CISO8601 | CNUMERIC);
	break;
      case 'l':
	KeyInfo[KeyCount-1]->CompType &= ~CNUMERIC;
	break;
      case 'N':
	KeyInfo[KeyCount-1]->CompType |= CNUMSTR;
	break;
      case 'm':
	KeyInfo[KeyCount-1]->CompType |= CMONTH;
	break;
      case 'n':
	KeyInfo[KeyCount-1]->CompType |= CNUMERIC;
	break;
      case 'r':
	KeyInfo[KeyCount-1]->CompType |= CRANDOM;
	break;
      case 's':
	KeyInfo[KeyCount-1]->CompType |= CSIZE; /* Don't set numeric yet */
	break;
      case 't':
	KeyInfo[KeyCount-1]->CompType |= (CTIME | CNUMERIC);
	break;
      default:
	fprintf(stderr,
		_("Comparison type %s not recognized.\n"),optarg);
	exit(BADOPTIONARG);
      }
      break;
    case 'W':			/* Specify whitespace for this key's sort order file */
      if(KeyInfo[KeyCount-1]->RankTable != NULL) {
	fprintf(stderr,_("The sort order for key %d has already been read.\n"),
			       KeyCount);
	fprintf(stderr,_("The whitespace definition must come first.\n"));
	exit(OTHERERROR);
      }
      if(GetWhiteSpaceDefinition(optarg,KeyInfo[KeyCount-1])  == ERROR) {
	fprintf(stderr,
		_("Attempt to read whitespace list from file %s failed.\n"),optarg);
	exit(OTHERERROR);
      }
      fprintf(Logfp,_("Read whitespace list for key %d from file %s.\n"),
	      KeyCount,optarg);
      break;
    case 'q':		/* Be quiet */
      VerboseP = FALSE;
      break;
	 
      /* Provide information */
	 
    case 'D':		/* List defaults */
      PrintDefaults(stdout);
      PrintDefaults(Logfp);
      exit(INFO);
    case 'F':
      PrintGeneralFlags(stdout);
      PrintGeneralFlags(Logfp);
      exit(INFO);
    case 'K':
      PrintKeySpecificFlags(stdout);
      PrintKeySpecificFlags(Logfp);
      exit(INFO);
    case 'L':		/* List limits */
      PrintLimits(stdout);
      PrintLimits(Logfp);
      exit(INFO);
    case 'h':
      PrintUsage();
      exit(INFO);
    case 'v':		/* This info is always printed so provide flag for compatibility */
      IdentifySelf(stdout);
      exit(INFO);
    case ':':
       fprintf(stderr,_("%s: missing argument to option flag %c.\n"),progname,optopt);
       exit(BADOPTIONARG);
    case '?':
      fprintf(stderr,_("%s: invalid option flag %c\n"),progname,optopt);
      exit(BADOPTION);
    } /* End switch */
  } /* End getopt while */


#ifdef DEBUGP
  fprintf(stderr,"Processed command line.\n"); fflush(stderr);
#endif

  if(KeyCount < 1){
    fprintf(stderr,_("%s: no sort key specified.\n\n"),progname);
    exit(OTHERERROR);
  } 
  ProvideDefaultRanks();
#ifdef DEBUGP
  fprintf(stderr,"Provided default ranks.\n"); fflush(stderr);
#endif

  DescribeKeys(Logfp);
#ifdef DEBUGP
  fprintf(stderr,"Described keys.\n");fflush(stderr);
#endif
  if(VerboseP){
    DescribeKeys(stderr);
    if(ReverseP) fprintf(stderr,_("Order globally inverted.\n"));
  }

  switch(RecordParseType) {
  case RPT_LINE:
    GetRecord = GetBlockSepCharRAUTF8;
    RecordSeparator = EndOfLine;
    break;
  case RPT_NNBLOCK:
    GetRecord = GetNNBlockRAUTF8;
    RecordSeparator = EndOfLine;
    break;
  case RPT_SEPCHAR:
    GetRecord = GetBlockSepCharRAUTF8;
    break;
  case RPT_FIXEDLENGTH:
    GetRecord = GetFixedLengthRecord;
    RecordSeparator = NOTACHARACTER;	/* This is a codepoint guaranteed to be undefined */
    break;
  }

   
  /* Set field terminators here since defaults are option-dependent */
   
  if(!TerminatorsSetP){
    if(RecordParseType == RPT_LINE) {
      Terminators = wCreateString(2);
      Terminators[0] = L'\t';
      Terminators[1] = L' ';
      Terminators[2] = L'\0';
    }
    else{			/* Block */
      Terminators = wCreateString(1);
      Terminators[0] = (wchar_t) (0x00FF & EndOfLine);
      Terminators[1] = L'\0';
    }
  }

  TerminatorCnt = wcslen(Terminators);

  /* We wait to do this here so that we have the locale if it was set with the -s flag */
  for(i=0;i < KeyCount;i++) {
    if( (KeyInfo[i]->CompType & CMONTH) && (KeyInfo[i]->MapTable == NULL)) {
	GetMonthNames(i);
	KeyInfo[i]->CompType |= CNUMERIC;
    }
  }

  /* Set up input and output */
   
  if(ac > optind){
    infile = copy_string(av[optind]);
    infp = OpenFile(infile,"r",progname);
    if(infp == NULL) exit(OTHERERROR);
  }
  else{
    infile=copy_string("stdin");
    infp = stdin;
  }
  outfp = stdout;
   
  if(VerboseP) fprintf(stderr,_("Reading from %s.\n"),infile);
   
  MaxRecords = InitialMaxRecords;
  RecordList = (struct record **) malloc((size_t) MaxRecords * sizeof(struct record *));
  if(RecordList == (struct record **)0){
    fprintf(stderr,_("%s: out of memory - RecordList.\n"),progname);
    exit(OUTOFMEMORY);
  }

#ifdef LOCALE_SORT_RDER
  /* Find out if we are using more than one distinct locale */
  for(i =0; i < KeyCount; i++) {
    if(KeyInfo[i]->LocaleP) {
      if(NULL == FirstLocale) FirstLocale = KeyInfo[i]->Locale;
      else {
	if(strcmp(FirstLocale,KeyInfo[i]->Locale)) {
	  MultipleLocalesP = 1;
	  break;
	}
      }
    }
  }

  if( (MultipleLocalesP == 0) && (FirstLocale != NULL)) {
    setlocale(LC_COLLATE,FirstLocale);
  }
#endif

  /* Now we read in one record at a time and extract and store its key */
   
  if(VerboseP)fprintf(stderr,_("Records processed:            %15ld"),Records);
  while(TRUE){
    InputText = (*GetRecord)(infp,InputText,&RecordLength,&InputTextSize,RecordSeparator);
    InputRecords++;
    if(RecordLength == BUFOVERFLOW){
      fprintf(stderr,_("%s: unable to allocate memory for input buffer.\n"),progname);
      exit(OUTOFMEMORY);
    }
    else if(RecordLength == SHORTRECORD) {
      fprintf(stderr,_("%s: fixed length record too short.\n"),progname);
      exit(BADRECORD);
    }
    else if(RecordLength == ENDOFINPUT) break;
    else if(RecordLength == 0) continue;
    /* Absorb extra separators - should perhaps be done by input routines */
    else if((RecordLength == 1) && (InputText[0] == RecordSeparator)) continue;
      
    if(Records == MaxRecords){ /* Need more storage? */
      MaxRecords += InitialMaxRecords/2;
      RecordList = (struct record **) realloc((void *) RecordList,
					      (unsigned) MaxRecords * sizeof(struct record *));
      if(RecordList == (struct record **)0){
	fprintf(stderr,_("\n%s: out of memory - RecordList.\n"),progname);
	fprintf(stderr,_("Storage allocation failed on record %ld.\n"),
		InputRecords+1L);
	exit(OUTOFMEMORY);
      }
    }
      
    /* Allocate storage for record structure */
      
    RecordList[Records] = (struct record *) malloc(sizeof(struct record));
    if(RecordList[Records] == (struct record *) 0){
      fprintf(stderr,_("\n%s: out of memory - new record - #%ld.\n"),
	      progname,InputRecords+1L);
      exit(OUTOFMEMORY);
    }
    RecordList[Records]->length = (size_t) RecordLength;
      
    RecordList[Records]->klistptr = (union key *) malloc( (unsigned) KeyCount * sizeof(union key));
    if(RecordList[Records]->klistptr == (union key *) 0){
      fprintf(stderr,_("\n%s: out of memory - keylist of record -#%ld.\n"),
	      progname,InputRecords+1L);
      exit(OUTOFMEMORY);
    }

    /* Store record text */

    if(!CheckOnlyP) {
      RecordList[Records]->text = (UTF8 *) malloc((size_t) (RecordLength+1) * sizeof(UTF8));
      if(RecordList[Records]->text == NULL){
	fprintf(stderr,_("\n%s: out of memory - record text -#%ld.\n"),
		progname,InputRecords+1L);
	exit(OUTOFMEMORY);
      }
      (void)strcpyu8(RecordList[Records]->text,InputText);
    }

    /* Reallocate TempString if necessary to ensure size exceeds current record */

    if(TempString.c <= RecordLength){
      TempString.c = RecordLength+1;
      TempString.s = (wchar_t *) realloc((void *) TempString.s,
				       TempString.c * sizeof(wchar_t));
    }
    if(TempString.s == NULL){
      fprintf(stderr,_("\n%s: Failed to reallocate memory for TempString.\n"),progname);
      exit(OUTOFMEMORY); 
    }

    /* Generate the UTF32 copy of the record text. */

    if((status = u82u32(InputText,&wcInputText,&MaxCodeInInput)) != 0) {
      fprintf(stderr,_("Fatal error while making UTF32 copy of record %ld.\n"),
	      InputRecords);
      exit(status);
    }
    if(BMPOnlyP && (MaxCodeInInput > 0xFFFF)) {
      fprintf(Logfp,
	      _("The character U+%06lX, which lies outside the BMP, was detected in record %lu:\n%s\n"),
	      (unsigned long)MaxCodeInInput,InputRecords,InputText);
      fprintf(stderr,_("The character U+%06lX, which lies outside the BMP, was detected.\n"),
	      (unsigned long)MaxCodeInInput);
      fprintf(stderr,_("A copy of the offending record is in the log.\n"));
      exit(RANGEERROR);
    }

    /* Now extract and store the sort key */

    if(GetKeys(RecordList[Records++],KeyInfo,KeyCount,MaxCodeInInput) == ERROR){
      FailedKeyP=1;
      fprintf(Logfp,_("Key extraction failed on ill-formed record %ld:\n%s\n"),
	      InputRecords,InputText);
      /* Just skip the record by decrementing counter. */
      Records--;
      /* Free up the storage we already allocated */
      CheckFree((void *) (RecordList[Records]->text), __LINE__);
      CheckFree((void *) (RecordList[Records]),__LINE__);
    }
    CheckFree((void *) wcInputText, __LINE__);
      
    /* Report on progress */
      
    if(VerboseP && (Records % REPORTINTERVAL == 0)){
      fprintf(stderr,"\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b%15ld",Records);
    }
#ifdef DEBUGP
    fflush(Logfp);
    fflush(stderr);
#endif
  } /* End of record-reading loop */


  if(VerboseP) fprintf(stderr,"\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b%15ld\n",Records);
  fclose(infp);

  if(InputRecords > 2L) {
    if(Records < 1L) {
      fprintf(stderr,_("All of the input records were ill-formed.\n"));
    }
    else if(Records < 2L) {
      fprintf(stderr,_("All but one input record was ill-formed.\n"));
    }
  }
  if(Records < 2L) {
    fprintf(stderr,_("There's no point in sorting fewer than two records.\n"));
    exit(1);
  }

  /* Copy the record pointer list if we're doing a check  */
  if(CheckOnlyP) {
    SavedRecordList = (struct record **) malloc((size_t) Records * sizeof(struct record *));
    if(SavedRecordList == NULL) {
      fprintf(stderr,"Cannot allocate storage for pointer list copy.\n");
      exit(OUTOFMEMORY);
    }
    for(i = 0; i < Records; i++) SavedRecordList[i] = RecordList[i]; 
  }

  /* Make keys sorted on string length compare numerically  */
  for(i =0; i < KeyCount; i++) {
    if(KeyInfo[i]->CompType & CSIZE) KeyInfo[i]->CompType |= CNUMERIC;  
  }

  /* Seed random number generator for use in random comparisons */
  srandom((unsigned int) time(NULL));

  /* Sort the stored records */
  
  if(VerboseP) fprintf(stderr,_("Sorting..."));
  if(VerboseP) fprintf(stderr,"\n");
  if(Records > 1L){
    switch (SortAlgorithm) { 
    case INSERTIONSORT:
      InsertionSort(RecordList,Records);
      break;
    case MERGESORT:
      MergeSort(RecordList,0L,Records);
      break;
    case SHELLSORT:
      ShellSort(RecordList,Records);
      break;
    case QUICKSORT:
    default:
      QuickSort(RecordList,0L,Records-1L);
      break;
    }
  }

  if(VerboseP){
    fputs("\b\b\b\b\b\b\b\b\b\b",stderr);
  }

  /* Write the sorted records back out */

  if(!CheckOnlyP) {
    if(VerboseP) fprintf(stderr,_("Records written:              %15ld"),0L);
  }
  fflush(stdout);fflush(stderr);
  if(!CheckOnlyP) {
    WriteOutRecords(Records,outfp,ReverseP,RecordSeparator);
  }
  else {
    for (i = 0; i < Records; i++) {
      if(RecordList[i] != SavedRecordList[i]) {
	ExitValue = NOTSORTED;
	break;
      } 
    }
  }

  /* Record dynamic memory usage and number of comparisons */
#ifdef HAVE_PRINTF_THSEP
  fprintf(Logfp,_("Dynamically allocated memory: %'15d bytes\n"),
	  ((char *)sbrk(0) - (char *)InitialBreak));
  fprintf(Logfp,_("Comparisons:                  %'15llu\n"),ComparisonCnt);
  if(VerboseP) fprintf(stderr,_("Comparisons:                  %'15llu\n"),ComparisonCnt);

#else
  fprintf(Logfp,"Dynamically allocated memory: %d bytes\n",
	  ((char *)sbrk(0) - (char *)InitialBreak));
  fprintf(Logfp,_("Comparisons:      %'14llu\n"),ComparisonCnt);
  if(VerboseP) fprintf(stderr,_("Comparisons:      %'14llu\n"),ComparisonCnt);
#endif
  fflush(Logfp);
  
  if(CheckOnlyP) {
    if(ExitValue == NOTSORTED) {
      if(VerboseP) fprintf(stdout,"The input is not already sorted.\n");
    }
    else {
      if(VerboseP) fprintf(stdout,"The input is already sorted.\n");
    }
  }

  /* We're done */

  if(FailedKeyP == TRUE){
    fprintf(stderr,
	    _("Key extraction failed on one or more records.\nConsult the log for details.\n")); 
    fclose(Logfp);
    ExitValue = BADRECORD;
  }
#ifdef HAVE_PRINTF_THSEP
  fprintf(Logfp,_("Records written:              %'15ld\n"),Records);
#else
  fprintf(Logfp,_("Records written:              %15ld\n"),Records);
#endif
  fclose(Logfp);
  exit(ExitValue);
   
} /* End of main() */

int
SetDateFormat(wchar_t *str, struct ymdinfo *ymd)
{
  int i;
  int j;
	
  if(wcslen(str) != 5) return(ERROR); 
	
  ymd->sep1 = str[1];
  ymd->sep2 = str[3];
  ymd->y = ymd->m = ymd->d = -1; /* For testing if value has been set */
  for(i = 0, j=0; i < 3; i++,j+=2){
    switch(str[j]){
    case 'y':
      ymd->y = i;
      break;
    case 'm':
      ymd->m = i;
      break;
    case 'd':
      ymd->d = i;
      break;
    default:
      return(ERROR);
    }
  }
  if(ymd->y < 0 || ymd->m < 0 || ymd->d < 0) return(ERROR);
  else return(SUCCESS);
}

#define TMPSIZE 128

void	
DescribeKeys(FILE *fp)
{
  int i;
  int first;
  int last;
  wchar_t wctmp[TMPSIZE];
  char tmp[TMPSIZE];

  extern void DumpExclusions(wchar_t *,long,FILE *);

  if(KeyInfo[0]->SelectType == WHOLE){
    fprintf(fp,"Sorting on whole record.\n");
    if(KeyInfo[0]->CompType & CINVERSE)fprintf(fp,"Decreasing ");
    else fprintf(fp,"Increasing ");
    if(!(KeyInfo[0]->CompType & CNUMERIC)){
      if(KeyInfo[0]->ReverseP) fprintf(fp,"reversed ");
      if(KeyInfo[0]->CompType & CSIZE) fprintf(fp,"by size\n");
      if(KeyInfo[0]->CompType & CNUMSTR) fprintf(fp,"numeric string\n");
      if(KeyInfo[0]->CompType & CRANDOM) fprintf(fp,"random\n");
      if(KeyInfo[0]->CompType & CHYBRID) fprintf(fp,"hybrid\n");
      if(KeyInfo[0]->CompType & CMONTH) fprintf(fp,"month name\n");
      else fprintf(fp,"lexicographic\n");
      if(KeyInfo[0]->MapTableEntries) fprintf(fp," %3d multigraph%s\n",
					     KeyInfo[0]->MapTableEntries,
					     (KeyInfo[0]->MapTableEntries == 1 ? "":"s"));
      if(KeyInfo[0]->ExclusionTable != NULL){
	DumpExclusions(KeyInfo[0]->ExclusionTable,RankTableSize,fp);
      }
    }
    else{
      if(KeyInfo[0]->CompType & CDATE) fprintf(fp,"date\n");
      else if(KeyInfo[0]->CompType & CTIME) fprintf(fp,"time\n");
      else if(KeyInfo[0]->CompType & CISO8601) fprintf(fp,"iso8601 date/time\n");
      else if(KeyInfo[i]->CompType & CANGLE) fprintf(fp,"angle\n");
      else if(KeyInfo[i]->CompType & CMONTH) fprintf(fp,"month name\n");
      else fprintf(fp,"numeric\n");
    }
    return;
  }
	
  for(i = 0;i < KeyCount; i++){
    fprintf(fp,"Key %d ",i+1);
    fprintf(fp,"%-11.11s ",(KeyInfo[i]->OptionalP ? "optional":"obligatory"));
    if(KeyInfo[i]->SelectType == NUMBERED){
      first = 1+KeyInfo[i]->Number;
      if(first > 0){
	if(first == 1) sprintf(tmp,"first field");
	else if(first == 2) sprintf(tmp,"second field");
	else if(first == 3) sprintf(tmp,"third field");
	else if(first == 4) sprintf(tmp,"fourth field");
	else if(first == 5) sprintf(tmp,"fifth field");
	else if(first % 100 == 11) sprintf(tmp,"%dth field",first);
	else if(first % 100 == 12) sprintf(tmp,"%dth field",first);
	else if(first %  10 == 1) sprintf(tmp,"%dst field",first);
	else if(first %  10 == 2) sprintf(tmp,"%dnd field",first);
	else sprintf(tmp,"%dth field",first);
      }
      else{
	first *= -1;
	if(first == 1) sprintf(tmp,"last field");
	else if(first == 2) sprintf(tmp,"penultimate field");
	else if(first == 3) sprintf(tmp,"antepenultimate field");
	else if(first == 4) sprintf(tmp,"fourth field from end");
	else if(first == 5) sprintf(tmp,"fifth field from end");
	else if(first % 100 == 11) sprintf(tmp,"%dth field from end",first);
	else if(first % 100 == 12) sprintf(tmp,"%dth field from end",first);
	else if(first %  10 == 1) sprintf(tmp,"%dst field from end",first);
	else if(first %  10 == 2) sprintf(tmp,"%dnd field from end",first);
	else sprintf(tmp,"%dth field from end",first);
      }
      fprintf(fp,tmp);
      if (KeyInfo[i]->FirstOffset > 0) {
	fprintf(fp," character %d",(KeyInfo[i]->FirstOffset)+1);
      }
      last = 1+KeyInfo[i]->Last;
      if(last != first) {
	fprintf(fp," through ");
	if(last > 0){
	  if(last == 1) sprintf(tmp,"first field");
	  else if(last == 2) sprintf(tmp,"second field");
	  else if(last == 3) sprintf(tmp,"third field");
	  else if(last == 4) sprintf(tmp,"fourth field");
	  else if(last == 5) sprintf(tmp,"fifth field");
	  else if(last % 100 == 11) sprintf(tmp,"%dth field",last);
	  else if(last % 100 == 12) sprintf(tmp,"%dth field",last);
	  else if(last %  10 == 1) sprintf(tmp,"%dst field",last);
	  else if(last %  10 == 2) sprintf(tmp,"%dnd field",last);
	  else sprintf(tmp,"%dth field",last);
	}
	else{
	  last *= -1;
	  if(last == 1) sprintf(tmp,"last field");
	  else if(last == 2) sprintf(tmp,"penultimate field");
	  else if(last == 3) sprintf(tmp,"antepenultimate field");
	  else if(last == 4) sprintf(tmp,"fourth field from end");
	  else if(last == 5) sprintf(tmp,"fifth field from end");
	  else if(last % 100 == 11) sprintf(tmp,"%dth field from end",last);
	  else if(last % 100 == 12) sprintf(tmp,"%dth field from end",last);
	  else if(last %  10 == 1) sprintf(tmp,"%dst field from end",last);
	  else if(last %  10 == 2) sprintf(tmp,"%dnd field from end",last);
	  else sprintf(tmp,"%dth field from end",last);
	}
	fprintf(fp,tmp);
	if (KeyInfo[i]->LastOffset > 0) {
	  fprintf(fp," character %d",(KeyInfo[i]->LastOffset)+1);
	}
      }
      fprintf(fp,"   ");
    }
    else if (KeyInfo[i]->SelectType == CRANGE) {
      fprintf(fp,"character range %02d through %02d ",
	      KeyInfo[i]->RangeFirst,KeyInfo[i]->RangeLast);
    }
    else {
      swprintf(wctmp,TMPSIZE,L"   tag %ls     ",KeyInfo[i]->Tag);
      putu8s(wctmp,fp);
    }
    fflush(fp);
    if(KeyInfo[i]->CompType & CINVERSE)fprintf(fp,"\n\tDecreasing ");
    else fprintf(fp,"\n\tIncreasing ");
    if(!(KeyInfo[i]->CompType & CNUMERIC)){
      if(KeyInfo[i]->ReverseP) fprintf(fp,"reversed ");
      if(KeyInfo[i]->CompType & CSIZE) fprintf(fp,"by size\n");
      if(KeyInfo[i]->CompType & CNUMSTR) fprintf(fp,"numeric string\n");
      if(KeyInfo[i]->CompType & CRANDOM) fprintf(fp,"random\n");
      if(KeyInfo[i]->CompType & CMONTH) fprintf(fp,"month name\n");
      else if(KeyInfo[i]->CompType & CHYBRID) fprintf(fp,"hybrid\n");
      else fprintf(fp,"lexicographic\n");
      if(KeyInfo[i]->MapTableEntries){
	fprintf(fp," %3d multigraph%s\n",
		KeyInfo[i]->MapTableEntries,
		(KeyInfo[i]->MapTableEntries == 1 ? "":"s"));
      }
      if(KeyInfo[i]->ExclusionTable != NULL){
	DumpExclusions(KeyInfo[i]->ExclusionTable,RankTableSize,fp);
      }
    }
    else{
      if(KeyInfo[i]->CompType & CDATE) fprintf(fp,"date\n");
      else if(KeyInfo[i]->CompType & CTIME) fprintf(fp,"time\n");
      else if(KeyInfo[i]->CompType & CISO8601) fprintf(fp,"iso8601 date/time\n");
      else if(KeyInfo[i]->CompType & CANGLE) fprintf(fp,"angle\n");
      else if(KeyInfo[i]->CompType & CMONTH) fprintf(fp,"month name\n");
      else fprintf(fp,"numeric\n");
    }
  }
}


DOUBLE *
AllocateDouble()
{
  DOUBLE *new;
	
  new = (DOUBLE *) malloc(sizeof(DOUBLE));
  if(new == NULL){
    fprintf(stderr,_("%s: out of memory - numeric key\n"),progname);
    exit(OUTOFMEMORY);
  }
  return(new);
}


DOUBLE atofwc(wchar_t *s, int *status){
  char ts[MAXDOUBLELEN+1];
  char *t;
  double rval;
  char *endptr;

  if(wcslen(s) > MAXDOUBLELEN) return RANGEERROR;

  t = ts;
  while (*s != L'\0'){
    /* Strip whitespace from end of field */
    if(!iswspace(*s)) *t++ = 0x00FF & (char) *s;
    s++;
  }
  *t = '\0';
#ifdef HAVE_LONGDOUBLE
  rval = strtold(ts,&endptr);
#else
  rval = strtod(ts,&endptr);
#endif
  if(errno == ERANGE){
    *status = RANGEERROR;
    return 0;
  }
  if(*endptr != '\0'){
    *status = BADRECORD;
    return 0;
  }
  *status = SUCCESS;
  return (rval);
}

void ASCIIFoldCase (wchar_t *s) {
  while (*s != 0){
    if ( (*s < 0x5B) && (*s > 0x40) ) *s |= 0x20;
    s++;
  }
}

void TurkicFoldCase (wchar_t *s){
  wchar_t c;

  while ((c = *s) != L'\0') {
    switch(c) {
    case 0x0049:
      *s++ = 0x0131;
      break;
    case 0x0130:
      *s++ = 0x0069;
      break;
    default:
      s++;
      break;
    }
  }
}


/* Full Unicode case folding */

struct cp {
  wchar_t  i;
  wchar_t *o;
};

static struct cp CaseFoldTbl []={ 
{0x0041,L"\x0061"},
{0x0042,L"\x0062"},
{0x0043,L"\x0063"},
{0x0044,L"\x0064"},
{0x0045,L"\x0065"},
{0x0046,L"\x0066"},
{0x0047,L"\x0067"},
{0x0048,L"\x0068"},
{0x0049,L"\x0069"},
{0x004A,L"\x006A"},
{0x004B,L"\x006B"},
{0x004C,L"\x006C"},
{0x004D,L"\x006D"},
{0x004E,L"\x006E"},
{0x004F,L"\x006F"},
{0x0050,L"\x0070"},
{0x0051,L"\x0071"},
{0x0052,L"\x0072"},
{0x0053,L"\x0073"},
{0x0054,L"\x0074"},
{0x0055,L"\x0075"},
{0x0056,L"\x0076"},
{0x0057,L"\x0077"},
{0x0058,L"\x0078"},
{0x0059,L"\x0079"},
{0x005A,L"\x007A"},
{0x00B5,L"\x03BC"},
{0x00C0,L"\x00E0"},
{0x00C1,L"\x00E1"},
{0x00C2,L"\x00E2"},
{0x00C3,L"\x00E3"},
{0x00C4,L"\x00E4"},
{0x00C5,L"\x00E5"},
{0x00C6,L"\x00E6"},
{0x00C7,L"\x00E7"},
{0x00C8,L"\x00E8"},
{0x00C9,L"\x00E9"},
{0x00CA,L"\x00EA"},
{0x00CB,L"\x00EB"},
{0x00CC,L"\x00EC"},
{0x00CD,L"\x00ED"},
{0x00CE,L"\x00EE"},
{0x00CF,L"\x00EF"},
{0x00D0,L"\x00F0"},
{0x00D1,L"\x00F1"},
{0x00D2,L"\x00F2"},
{0x00D3,L"\x00F3"},
{0x00D4,L"\x00F4"},
{0x00D5,L"\x00F5"},
{0x00D6,L"\x00F6"},
{0x00D8,L"\x00F8"},
{0x00D9,L"\x00F9"},
{0x00DA,L"\x00FA"},
{0x00DB,L"\x00FB"},
{0x00DC,L"\x00FC"},
{0x00DD,L"\x00FD"},
{0x00DE,L"\x00FE"},
{0x00DF,L"\x0073\x0073"},
{0x0100,L"\x0101"},
{0x0102,L"\x0103"},
{0x0104,L"\x0105"},
{0x0106,L"\x0107"},
{0x0108,L"\x0109"},
{0x010A,L"\x010B"},
{0x010C,L"\x010D"},
{0x010E,L"\x010F"},
{0x0110,L"\x0111"},
{0x0112,L"\x0113"},
{0x0114,L"\x0115"},
{0x0116,L"\x0117"},
{0x0118,L"\x0119"},
{0x011A,L"\x011B"},
{0x011C,L"\x011D"},
{0x011E,L"\x011F"},
{0x0120,L"\x0121"},
{0x0122,L"\x0123"},
{0x0124,L"\x0125"},
{0x0126,L"\x0127"},
{0x0128,L"\x0129"},
{0x012A,L"\x012B"},
{0x012C,L"\x012D"},
{0x012E,L"\x012F"},
{0x0130,L"\x0069\x0307"},
{0x0132,L"\x0133"},
{0x0134,L"\x0135"},
{0x0136,L"\x0137"},
{0x0139,L"\x013A"},
{0x013B,L"\x013C"},
{0x013D,L"\x013E"},
{0x013F,L"\x0140"},
{0x0141,L"\x0142"},
{0x0143,L"\x0144"},
{0x0145,L"\x0146"},
{0x0147,L"\x0148"},
{0x0149,L"\x02BC\x006E"},
{0x014A,L"\x014B"},
{0x014C,L"\x014D"},
{0x014E,L"\x014F"},
{0x0150,L"\x0151"},
{0x0152,L"\x0153"},
{0x0154,L"\x0155"},
{0x0156,L"\x0157"},
{0x0158,L"\x0159"},
{0x015A,L"\x015B"},
{0x015C,L"\x015D"},
{0x015E,L"\x015F"},
{0x0160,L"\x0161"},
{0x0162,L"\x0163"},
{0x0164,L"\x0165"},
{0x0166,L"\x0167"},
{0x0168,L"\x0169"},
{0x016A,L"\x016B"},
{0x016C,L"\x016D"},
{0x016E,L"\x016F"},
{0x0170,L"\x0171"},
{0x0172,L"\x0173"},
{0x0174,L"\x0175"},
{0x0176,L"\x0177"},
{0x0178,L"\x00FF"},
{0x0179,L"\x017A"},
{0x017B,L"\x017C"},
{0x017D,L"\x017E"},
{0x017F,L"\x0073"},
{0x0181,L"\x0253"},
{0x0182,L"\x0183"},
{0x0184,L"\x0185"},
{0x0186,L"\x0254"},
{0x0187,L"\x0188"},
{0x0189,L"\x0256"},
{0x018A,L"\x0257"},
{0x018B,L"\x018C"},
{0x018E,L"\x01DD"},
{0x018F,L"\x0259"},
{0x0190,L"\x025B"},
{0x0191,L"\x0192"},
{0x0193,L"\x0260"},
{0x0194,L"\x0263"},
{0x0196,L"\x0269"},
{0x0197,L"\x0268"},
{0x0198,L"\x0199"},
{0x019C,L"\x026F"},
{0x019D,L"\x0272"},
{0x019F,L"\x0275"},
{0x01A0,L"\x01A1"},
{0x01A2,L"\x01A3"},
{0x01A4,L"\x01A5"},
{0x01A6,L"\x0280"},
{0x01A7,L"\x01A8"},
{0x01A9,L"\x0283"},
{0x01AC,L"\x01AD"},
{0x01AE,L"\x0288"},
{0x01AF,L"\x01B0"},
{0x01B1,L"\x028A"},
{0x01B2,L"\x028B"},
{0x01B3,L"\x01B4"},
{0x01B5,L"\x01B6"},
{0x01B7,L"\x0292"},
{0x01B8,L"\x01B9"},
{0x01BC,L"\x01BD"},
{0x01C4,L"\x01C6"},
{0x01C5,L"\x01C6"},
{0x01C7,L"\x01C9"},
{0x01C8,L"\x01C9"},
{0x01CA,L"\x01CC"},
{0x01CB,L"\x01CC"},
{0x01CD,L"\x01CE"},
{0x01CF,L"\x01D0"},
{0x01D1,L"\x01D2"},
{0x01D3,L"\x01D4"},
{0x01D5,L"\x01D6"},
{0x01D7,L"\x01D8"},
{0x01D9,L"\x01DA"},
{0x01DB,L"\x01DC"},
{0x01DE,L"\x01DF"},
{0x01E0,L"\x01E1"},
{0x01E2,L"\x01E3"},
{0x01E4,L"\x01E5"},
{0x01E6,L"\x01E7"},
{0x01E8,L"\x01E9"},
{0x01EA,L"\x01EB"},
{0x01EC,L"\x01ED"},
{0x01EE,L"\x01EF"},
{0x01F0,L"\x006A\x030C"},
{0x01F1,L"\x01F3"},
{0x01F2,L"\x01F3"},
{0x01F4,L"\x01F5"},
{0x01F6,L"\x0195"},
{0x01F7,L"\x01BF"},
{0x01F8,L"\x01F9"},
{0x01FA,L"\x01FB"},
{0x01FC,L"\x01FD"},
{0x01FE,L"\x01FF"},
{0x0200,L"\x0201"},
{0x0202,L"\x0203"},
{0x0204,L"\x0205"},
{0x0206,L"\x0207"},
{0x0208,L"\x0209"},
{0x020A,L"\x020B"},
{0x020C,L"\x020D"},
{0x020E,L"\x020F"},
{0x0210,L"\x0211"},
{0x0212,L"\x0213"},
{0x0214,L"\x0215"},
{0x0216,L"\x0217"},
{0x0218,L"\x0219"},
{0x021A,L"\x021B"},
{0x021C,L"\x021D"},
{0x021E,L"\x021F"},
{0x0220,L"\x019E"},
{0x0222,L"\x0223"},
{0x0224,L"\x0225"},
{0x0226,L"\x0227"},
{0x0228,L"\x0229"},
{0x022A,L"\x022B"},
{0x022C,L"\x022D"},
{0x022E,L"\x022F"},
{0x0230,L"\x0231"},
{0x0232,L"\x0233"},
{0x023B,L"\x023C"},
{0x023D,L"\x019A"},
{0x0241,L"\x0294"},
{0x0345,L"\x03B9"},
{0x0386,L"\x03AC"},
{0x0388,L"\x03AD"},
{0x0389,L"\x03AE"},
{0x038A,L"\x03AF"},
{0x038C,L"\x03CC"},
{0x038E,L"\x03CD"},
{0x038F,L"\x03CE"},
{0x0390,L"\x03B9\x0308\x0301"},
{0x0391,L"\x03B1"},
{0x0392,L"\x03B2"},
{0x0393,L"\x03B3"},
{0x0394,L"\x03B4"},
{0x0395,L"\x03B5"},
{0x0396,L"\x03B6"},
{0x0397,L"\x03B7"},
{0x0398,L"\x03B8"},
{0x0399,L"\x03B9"},
{0x039A,L"\x03BA"},
{0x039B,L"\x03BB"},
{0x039C,L"\x03BC"},
{0x039D,L"\x03BD"},
{0x039E,L"\x03BE"},
{0x039F,L"\x03BF"},
{0x03A0,L"\x03C0"},
{0x03A1,L"\x03C1"},
{0x03A3,L"\x03C3"},
{0x03A4,L"\x03C4"},
{0x03A5,L"\x03C5"},
{0x03A6,L"\x03C6"},
{0x03A7,L"\x03C7"},
{0x03A8,L"\x03C8"},
{0x03A9,L"\x03C9"},
{0x03AA,L"\x03CA"},
{0x03AB,L"\x03CB"},
{0x03B0,L"\x03C5\x0308\x0301"},
{0x03C2,L"\x03C3"},
{0x03D0,L"\x03B2"},
{0x03D1,L"\x03B8"},
{0x03D5,L"\x03C6"},
{0x03D6,L"\x03C0"},
{0x03D8,L"\x03D9"},
{0x03DA,L"\x03DB"},
{0x03DC,L"\x03DD"},
{0x03DE,L"\x03DF"},
{0x03E0,L"\x03E1"},
{0x03E2,L"\x03E3"},
{0x03E4,L"\x03E5"},
{0x03E6,L"\x03E7"},
{0x03E8,L"\x03E9"},
{0x03EA,L"\x03EB"},
{0x03EC,L"\x03ED"},
{0x03EE,L"\x03EF"},
{0x03F0,L"\x03BA"},
{0x03F1,L"\x03C1"},
{0x03F4,L"\x03B8"},
{0x03F5,L"\x03B5"},
{0x03F7,L"\x03F8"},
{0x03F9,L"\x03F2"},
{0x03FA,L"\x03FB"},
{0x0400,L"\x0450"},
{0x0401,L"\x0451"},
{0x0402,L"\x0452"},
{0x0403,L"\x0453"},
{0x0404,L"\x0454"},
{0x0405,L"\x0455"},
{0x0406,L"\x0456"},
{0x0407,L"\x0457"},
{0x0408,L"\x0458"},
{0x0409,L"\x0459"},
{0x040A,L"\x045A"},
{0x040B,L"\x045B"},
{0x040C,L"\x045C"},
{0x040D,L"\x045D"},
{0x040E,L"\x045E"},
{0x040F,L"\x045F"},
{0x0410,L"\x0430"},
{0x0411,L"\x0431"},
{0x0412,L"\x0432"},
{0x0413,L"\x0433"},
{0x0414,L"\x0434"},
{0x0415,L"\x0435"},
{0x0416,L"\x0436"},
{0x0417,L"\x0437"},
{0x0418,L"\x0438"},
{0x0419,L"\x0439"},
{0x041A,L"\x043A"},
{0x041B,L"\x043B"},
{0x041C,L"\x043C"},
{0x041D,L"\x043D"},
{0x041E,L"\x043E"},
{0x041F,L"\x043F"},
{0x0420,L"\x0440"},
{0x0421,L"\x0441"},
{0x0422,L"\x0442"},
{0x0423,L"\x0443"},
{0x0424,L"\x0444"},
{0x0425,L"\x0445"},
{0x0426,L"\x0446"},
{0x0427,L"\x0447"},
{0x0428,L"\x0448"},
{0x0429,L"\x0449"},
{0x042A,L"\x044A"},
{0x042B,L"\x044B"},
{0x042C,L"\x044C"},
{0x042D,L"\x044D"},
{0x042E,L"\x044E"},
{0x042F,L"\x044F"},
{0x0460,L"\x0461"},
{0x0462,L"\x0463"},
{0x0464,L"\x0465"},
{0x0466,L"\x0467"},
{0x0468,L"\x0469"},
{0x046A,L"\x046B"},
{0x046C,L"\x046D"},
{0x046E,L"\x046F"},
{0x0470,L"\x0471"},
{0x0472,L"\x0473"},
{0x0474,L"\x0475"},
{0x0476,L"\x0477"},
{0x0478,L"\x0479"},
{0x047A,L"\x047B"},
{0x047C,L"\x047D"},
{0x047E,L"\x047F"},
{0x0480,L"\x0481"},
{0x048A,L"\x048B"},
{0x048C,L"\x048D"},
{0x048E,L"\x048F"},
{0x0490,L"\x0491"},
{0x0492,L"\x0493"},
{0x0494,L"\x0495"},
{0x0496,L"\x0497"},
{0x0498,L"\x0499"},
{0x049A,L"\x049B"},
{0x049C,L"\x049D"},
{0x049E,L"\x049F"},
{0x04A0,L"\x04A1"},
{0x04A2,L"\x04A3"},
{0x04A4,L"\x04A5"},
{0x04A6,L"\x04A7"},
{0x04A8,L"\x04A9"},
{0x04AA,L"\x04AB"},
{0x04AC,L"\x04AD"},
{0x04AE,L"\x04AF"},
{0x04B0,L"\x04B1"},
{0x04B2,L"\x04B3"},
{0x04B4,L"\x04B5"},
{0x04B6,L"\x04B7"},
{0x04B8,L"\x04B9"},
{0x04BA,L"\x04BB"},
{0x04BC,L"\x04BD"},
{0x04BE,L"\x04BF"},
{0x04C1,L"\x04C2"},
{0x04C3,L"\x04C4"},
{0x04C5,L"\x04C6"},
{0x04C7,L"\x04C8"},
{0x04C9,L"\x04CA"},
{0x04CB,L"\x04CC"},
{0x04CD,L"\x04CE"},
{0x04D0,L"\x04D1"},
{0x04D2,L"\x04D3"},
{0x04D4,L"\x04D5"},
{0x04D6,L"\x04D7"},
{0x04D8,L"\x04D9"},
{0x04DA,L"\x04DB"},
{0x04DC,L"\x04DD"},
{0x04DE,L"\x04DF"},
{0x04E0,L"\x04E1"},
{0x04E2,L"\x04E3"},
{0x04E4,L"\x04E5"},
{0x04E6,L"\x04E7"},
{0x04E8,L"\x04E9"},
{0x04EA,L"\x04EB"},
{0x04EC,L"\x04ED"},
{0x04EE,L"\x04EF"},
{0x04F0,L"\x04F1"},
{0x04F2,L"\x04F3"},
{0x04F4,L"\x04F5"},
{0x04F6,L"\x04F7"},
{0x04F8,L"\x04F9"},
{0x0500,L"\x0501"},
{0x0502,L"\x0503"},
{0x0504,L"\x0505"},
{0x0506,L"\x0507"},
{0x0508,L"\x0509"},
{0x050A,L"\x050B"},
{0x050C,L"\x050D"},
{0x050E,L"\x050F"},
{0x0531,L"\x0561"},
{0x0532,L"\x0562"},
{0x0533,L"\x0563"},
{0x0534,L"\x0564"},
{0x0535,L"\x0565"},
{0x0536,L"\x0566"},
{0x0537,L"\x0567"},
{0x0538,L"\x0568"},
{0x0539,L"\x0569"},
{0x053A,L"\x056A"},
{0x053B,L"\x056B"},
{0x053C,L"\x056C"},
{0x053D,L"\x056D"},
{0x053E,L"\x056E"},
{0x053F,L"\x056F"},
{0x0540,L"\x0570"},
{0x0541,L"\x0571"},
{0x0542,L"\x0572"},
{0x0543,L"\x0573"},
{0x0544,L"\x0574"},
{0x0545,L"\x0575"},
{0x0546,L"\x0576"},
{0x0547,L"\x0577"},
{0x0548,L"\x0578"},
{0x0549,L"\x0579"},
{0x054A,L"\x057A"},
{0x054B,L"\x057B"},
{0x054C,L"\x057C"},
{0x054D,L"\x057D"},
{0x054E,L"\x057E"},
{0x054F,L"\x057F"},
{0x0550,L"\x0580"},
{0x0551,L"\x0581"},
{0x0552,L"\x0582"},
{0x0553,L"\x0583"},
{0x0554,L"\x0584"},
{0x0555,L"\x0585"},
{0x0556,L"\x0586"},
{0x0587,L"\x0565\x0582"},
{0x10A0,L"\x2D00"},
{0x10A1,L"\x2D01"},
{0x10A2,L"\x2D02"},
{0x10A3,L"\x2D03"},
{0x10A4,L"\x2D04"},
{0x10A5,L"\x2D05"},
{0x10A6,L"\x2D06"},
{0x10A7,L"\x2D07"},
{0x10A8,L"\x2D08"},
{0x10A9,L"\x2D09"},
{0x10AA,L"\x2D0A"},
{0x10AB,L"\x2D0B"},
{0x10AC,L"\x2D0C"},
{0x10AD,L"\x2D0D"},
{0x10AE,L"\x2D0E"},
{0x10AF,L"\x2D0F"},
{0x10B0,L"\x2D10"},
{0x10B1,L"\x2D11"},
{0x10B2,L"\x2D12"},
{0x10B3,L"\x2D13"},
{0x10B4,L"\x2D14"},
{0x10B5,L"\x2D15"},
{0x10B6,L"\x2D16"},
{0x10B7,L"\x2D17"},
{0x10B8,L"\x2D18"},
{0x10B9,L"\x2D19"},
{0x10BA,L"\x2D1A"},
{0x10BB,L"\x2D1B"},
{0x10BC,L"\x2D1C"},
{0x10BD,L"\x2D1D"},
{0x10BE,L"\x2D1E"},
{0x10BF,L"\x2D1F"},
{0x10C0,L"\x2D20"},
{0x10C1,L"\x2D21"},
{0x10C2,L"\x2D22"},
{0x10C3,L"\x2D23"},
{0x10C4,L"\x2D24"},
{0x10C5,L"\x2D25"},
{0x1E00,L"\x1E01"},
{0x1E02,L"\x1E03"},
{0x1E04,L"\x1E05"},
{0x1E06,L"\x1E07"},
{0x1E08,L"\x1E09"},
{0x1E0A,L"\x1E0B"},
{0x1E0C,L"\x1E0D"},
{0x1E0E,L"\x1E0F"},
{0x1E10,L"\x1E11"},
{0x1E12,L"\x1E13"},
{0x1E14,L"\x1E15"},
{0x1E16,L"\x1E17"},
{0x1E18,L"\x1E19"},
{0x1E1A,L"\x1E1B"},
{0x1E1C,L"\x1E1D"},
{0x1E1E,L"\x1E1F"},
{0x1E20,L"\x1E21"},
{0x1E22,L"\x1E23"},
{0x1E24,L"\x1E25"},
{0x1E26,L"\x1E27"},
{0x1E28,L"\x1E29"},
{0x1E2A,L"\x1E2B"},
{0x1E2C,L"\x1E2D"},
{0x1E2E,L"\x1E2F"},
{0x1E30,L"\x1E31"},
{0x1E32,L"\x1E33"},
{0x1E34,L"\x1E35"},
{0x1E36,L"\x1E37"},
{0x1E38,L"\x1E39"},
{0x1E3A,L"\x1E3B"},
{0x1E3C,L"\x1E3D"},
{0x1E3E,L"\x1E3F"},
{0x1E40,L"\x1E41"},
{0x1E42,L"\x1E43"},
{0x1E44,L"\x1E45"},
{0x1E46,L"\x1E47"},
{0x1E48,L"\x1E49"},
{0x1E4A,L"\x1E4B"},
{0x1E4C,L"\x1E4D"},
{0x1E4E,L"\x1E4F"},
{0x1E50,L"\x1E51"},
{0x1E52,L"\x1E53"},
{0x1E54,L"\x1E55"},
{0x1E56,L"\x1E57"},
{0x1E58,L"\x1E59"},
{0x1E5A,L"\x1E5B"},
{0x1E5C,L"\x1E5D"},
{0x1E5E,L"\x1E5F"},
{0x1E60,L"\x1E61"},
{0x1E62,L"\x1E63"},
{0x1E64,L"\x1E65"},
{0x1E66,L"\x1E67"},
{0x1E68,L"\x1E69"},
{0x1E6A,L"\x1E6B"},
{0x1E6C,L"\x1E6D"},
{0x1E6E,L"\x1E6F"},
{0x1E70,L"\x1E71"},
{0x1E72,L"\x1E73"},
{0x1E74,L"\x1E75"},
{0x1E76,L"\x1E77"},
{0x1E78,L"\x1E79"},
{0x1E7A,L"\x1E7B"},
{0x1E7C,L"\x1E7D"},
{0x1E7E,L"\x1E7F"},
{0x1E80,L"\x1E81"},
{0x1E82,L"\x1E83"},
{0x1E84,L"\x1E85"},
{0x1E86,L"\x1E87"},
{0x1E88,L"\x1E89"},
{0x1E8A,L"\x1E8B"},
{0x1E8C,L"\x1E8D"},
{0x1E8E,L"\x1E8F"},
{0x1E90,L"\x1E91"},
{0x1E92,L"\x1E93"},
{0x1E94,L"\x1E95"},
{0x1E96,L"\x0068\x0331"},
{0x1E97,L"\x0074\x0308"},
{0x1E98,L"\x0077\x030A"},
{0x1E99,L"\x0079\x030A"},
{0x1E9A,L"\x0061\x02BE"},
{0x1E9B,L"\x1E61"},
{0x1EA0,L"\x1EA1"},
{0x1EA2,L"\x1EA3"},
{0x1EA4,L"\x1EA5"},
{0x1EA6,L"\x1EA7"},
{0x1EA8,L"\x1EA9"},
{0x1EAA,L"\x1EAB"},
{0x1EAC,L"\x1EAD"},
{0x1EAE,L"\x1EAF"},
{0x1EB0,L"\x1EB1"},
{0x1EB2,L"\x1EB3"},
{0x1EB4,L"\x1EB5"},
{0x1EB6,L"\x1EB7"},
{0x1EB8,L"\x1EB9"},
{0x1EBA,L"\x1EBB"},
{0x1EBC,L"\x1EBD"},
{0x1EBE,L"\x1EBF"},
{0x1EC0,L"\x1EC1"},
{0x1EC2,L"\x1EC3"},
{0x1EC4,L"\x1EC5"},
{0x1EC6,L"\x1EC7"},
{0x1EC8,L"\x1EC9"},
{0x1ECA,L"\x1ECB"},
{0x1ECC,L"\x1ECD"},
{0x1ECE,L"\x1ECF"},
{0x1ED0,L"\x1ED1"},
{0x1ED2,L"\x1ED3"},
{0x1ED4,L"\x1ED5"},
{0x1ED6,L"\x1ED7"},
{0x1ED8,L"\x1ED9"},
{0x1EDA,L"\x1EDB"},
{0x1EDC,L"\x1EDD"},
{0x1EDE,L"\x1EDF"},
{0x1EE0,L"\x1EE1"},
{0x1EE2,L"\x1EE3"},
{0x1EE4,L"\x1EE5"},
{0x1EE6,L"\x1EE7"},
{0x1EE8,L"\x1EE9"},
{0x1EEA,L"\x1EEB"},
{0x1EEC,L"\x1EED"},
{0x1EEE,L"\x1EEF"},
{0x1EF0,L"\x1EF1"},
{0x1EF2,L"\x1EF3"},
{0x1EF4,L"\x1EF5"},
{0x1EF6,L"\x1EF7"},
{0x1EF8,L"\x1EF9"},
{0x1F08,L"\x1F00"},
{0x1F09,L"\x1F01"},
{0x1F0A,L"\x1F02"},
{0x1F0B,L"\x1F03"},
{0x1F0C,L"\x1F04"},
{0x1F0D,L"\x1F05"},
{0x1F0E,L"\x1F06"},
{0x1F0F,L"\x1F07"},
{0x1F18,L"\x1F10"},
{0x1F19,L"\x1F11"},
{0x1F1A,L"\x1F12"},
{0x1F1B,L"\x1F13"},
{0x1F1C,L"\x1F14"},
{0x1F1D,L"\x1F15"},
{0x1F28,L"\x1F20"},
{0x1F29,L"\x1F21"},
{0x1F2A,L"\x1F22"},
{0x1F2B,L"\x1F23"},
{0x1F2C,L"\x1F24"},
{0x1F2D,L"\x1F25"},
{0x1F2E,L"\x1F26"},
{0x1F2F,L"\x1F27"},
{0x1F38,L"\x1F30"},
{0x1F39,L"\x1F31"},
{0x1F3A,L"\x1F32"},
{0x1F3B,L"\x1F33"},
{0x1F3C,L"\x1F34"},
{0x1F3D,L"\x1F35"},
{0x1F3E,L"\x1F36"},
{0x1F3F,L"\x1F37"},
{0x1F48,L"\x1F40"},
{0x1F49,L"\x1F41"},
{0x1F4A,L"\x1F42"},
{0x1F4B,L"\x1F43"},
{0x1F4C,L"\x1F44"},
{0x1F4D,L"\x1F45"},
{0x1F50,L"\x03C5\x0313"},
{0x1F52,L"\x03C5\x0313\x0300"},
{0x1F54,L"\x03C5\x0313\x0301"},
{0x1F56,L"\x03C5\x0313\x0342"},
{0x1F59,L"\x1F51"},
{0x1F5B,L"\x1F53"},
{0x1F5D,L"\x1F55"},
{0x1F5F,L"\x1F57"},
{0x1F68,L"\x1F60"},
{0x1F69,L"\x1F61"},
{0x1F6A,L"\x1F62"},
{0x1F6B,L"\x1F63"},
{0x1F6C,L"\x1F64"},
{0x1F6D,L"\x1F65"},
{0x1F6E,L"\x1F66"},
{0x1F6F,L"\x1F67"},
{0x1F80,L"\x1F00\x03B9"},
{0x1F81,L"\x1F01\x03B9"},
{0x1F82,L"\x1F02\x03B9"},
{0x1F83,L"\x1F03\x03B9"},
{0x1F84,L"\x1F04\x03B9"},
{0x1F85,L"\x1F05\x03B9"},
{0x1F86,L"\x1F06\x03B9"},
{0x1F87,L"\x1F07\x03B9"},
{0x1F88,L"\x1F00\x03B9"},
{0x1F89,L"\x1F01\x03B9"},
{0x1F8A,L"\x1F02\x03B9"},
{0x1F8B,L"\x1F03\x03B9"},
{0x1F8C,L"\x1F04\x03B9"},
{0x1F8D,L"\x1F05\x03B9"},
{0x1F8E,L"\x1F06\x03B9"},
{0x1F8F,L"\x1F07\x03B9"},
{0x1F90,L"\x1F20\x03B9"},
{0x1F91,L"\x1F21\x03B9"},
{0x1F92,L"\x1F22\x03B9"},
{0x1F93,L"\x1F23\x03B9"},
{0x1F94,L"\x1F24\x03B9"},
{0x1F95,L"\x1F25\x03B9"},
{0x1F96,L"\x1F26\x03B9"},
{0x1F97,L"\x1F27\x03B9"},
{0x1F98,L"\x1F20\x03B9"},
{0x1F99,L"\x1F21\x03B9"},
{0x1F9A,L"\x1F22\x03B9"},
{0x1F9B,L"\x1F23\x03B9"},
{0x1F9C,L"\x1F24\x03B9"},
{0x1F9D,L"\x1F25\x03B9"},
{0x1F9E,L"\x1F26\x03B9"},
{0x1F9F,L"\x1F27\x03B9"},
{0x1FA0,L"\x1F60\x03B9"},
{0x1FA1,L"\x1F61\x03B9"},
{0x1FA2,L"\x1F62\x03B9"},
{0x1FA3,L"\x1F63\x03B9"},
{0x1FA4,L"\x1F64\x03B9"},
{0x1FA5,L"\x1F65\x03B9"},
{0x1FA6,L"\x1F66\x03B9"},
{0x1FA7,L"\x1F67\x03B9"},
{0x1FA8,L"\x1F60\x03B9"},
{0x1FA9,L"\x1F61\x03B9"},
{0x1FAA,L"\x1F62\x03B9"},
{0x1FAB,L"\x1F63\x03B9"},
{0x1FAC,L"\x1F64\x03B9"},
{0x1FAD,L"\x1F65\x03B9"},
{0x1FAE,L"\x1F66\x03B9"},
{0x1FAF,L"\x1F67\x03B9"},
{0x1FB2,L"\x1F70\x03B9"},
{0x1FB3,L"\x03B1\x03B9"},
{0x1FB4,L"\x03AC\x03B9"},
{0x1FB6,L"\x03B1\x0342"},
{0x1FB7,L"\x03B1\x0342\x03B9"},
{0x1FB8,L"\x1FB0"},
{0x1FB9,L"\x1FB1"},
{0x1FBA,L"\x1F70"},
{0x1FBB,L"\x1F71"},
{0x1FBC,L"\x03B1\x03B9"},
{0x1FBE,L"\x03B9"},
{0x1FC2,L"\x1F74\x03B9"},
{0x1FC3,L"\x03B7\x03B9"},
{0x1FC4,L"\x03AE\x03B9"},
{0x1FC6,L"\x03B7\x0342"},
{0x1FC7,L"\x03B7\x0342\x03B9"},
{0x1FC8,L"\x1F72"},
{0x1FC9,L"\x1F73"},
{0x1FCA,L"\x1F74"},
{0x1FCB,L"\x1F75"},
{0x1FCC,L"\x03B7\x03B9"},
{0x1FD2,L"\x03B9\x0308\x0300"},
{0x1FD3,L"\x03B9\x0308\x0301"},
{0x1FD6,L"\x03B9\x0342"},
{0x1FD7,L"\x03B9\x0308\x0342"},
{0x1FD8,L"\x1FD0"},
{0x1FD9,L"\x1FD1"},
{0x1FDA,L"\x1F76"},
{0x1FDB,L"\x1F77"},
{0x1FE2,L"\x03C5\x0308\x0300"},
{0x1FE3,L"\x03C5\x0308\x0301"},
{0x1FE4,L"\x03C1\x0313"},
{0x1FE6,L"\x03C5\x0342"},
{0x1FE7,L"\x03C5\x0308\x0342"},
{0x1FE8,L"\x1FE0"},
{0x1FE9,L"\x1FE1"},
{0x1FEA,L"\x1F7A"},
{0x1FEB,L"\x1F7B"},
{0x1FEC,L"\x1FE5"},
{0x1FF2,L"\x1F7C\x03B9"},
{0x1FF3,L"\x03C9\x03B9"},
{0x1FF4,L"\x03CE\x03B9"},
{0x1FF6,L"\x03C9\x0342"},
{0x1FF7,L"\x03C9\x0342\x03B9"},
{0x1FF8,L"\x1F78"},
{0x1FF9,L"\x1F79"},
{0x1FFA,L"\x1F7C"},
{0x1FFB,L"\x1F7D"},
{0x1FFC,L"\x03C9\x03B9"},
{0x2126,L"\x03C9"},
{0x212A,L"\x006B"},
{0x212B,L"\x00E5"},
{0x2160,L"\x2170"},
{0x2161,L"\x2171"},
{0x2162,L"\x2172"},
{0x2163,L"\x2173"},
{0x2164,L"\x2174"},
{0x2165,L"\x2175"},
{0x2166,L"\x2176"},
{0x2167,L"\x2177"},
{0x2168,L"\x2178"},
{0x2169,L"\x2179"},
{0x216A,L"\x217A"},
{0x216B,L"\x217B"},
{0x216C,L"\x217C"},
{0x216D,L"\x217D"},
{0x216E,L"\x217E"},
{0x216F,L"\x217F"},
{0x24B6,L"\x24D0"},
{0x24B7,L"\x24D1"},
{0x24B8,L"\x24D2"},
{0x24B9,L"\x24D3"},
{0x24BA,L"\x24D4"},
{0x24BB,L"\x24D5"},
{0x24BC,L"\x24D6"},
{0x24BD,L"\x24D7"},
{0x24BE,L"\x24D8"},
{0x24BF,L"\x24D9"},
{0x24C0,L"\x24DA"},
{0x24C1,L"\x24DB"},
{0x24C2,L"\x24DC"},
{0x24C3,L"\x24DD"},
{0x24C4,L"\x24DE"},
{0x24C5,L"\x24DF"},
{0x24C6,L"\x24E0"},
{0x24C7,L"\x24E1"},
{0x24C8,L"\x24E2"},
{0x24C9,L"\x24E3"},
{0x24CA,L"\x24E4"},
{0x24CB,L"\x24E5"},
{0x24CC,L"\x24E6"},
{0x24CD,L"\x24E7"},
{0x24CE,L"\x24E8"},
{0x24CF,L"\x24E9"},
{0x2C00,L"\x2C30"},
{0x2C01,L"\x2C31"},
{0x2C02,L"\x2C32"},
{0x2C03,L"\x2C33"},
{0x2C04,L"\x2C34"},
{0x2C05,L"\x2C35"},
{0x2C06,L"\x2C36"},
{0x2C07,L"\x2C37"},
{0x2C08,L"\x2C38"},
{0x2C09,L"\x2C39"},
{0x2C0A,L"\x2C3A"},
{0x2C0B,L"\x2C3B"},
{0x2C0C,L"\x2C3C"},
{0x2C0D,L"\x2C3D"},
{0x2C0E,L"\x2C3E"},
{0x2C0F,L"\x2C3F"},
{0x2C10,L"\x2C40"},
{0x2C11,L"\x2C41"},
{0x2C12,L"\x2C42"},
{0x2C13,L"\x2C43"},
{0x2C14,L"\x2C44"},
{0x2C15,L"\x2C45"},
{0x2C16,L"\x2C46"},
{0x2C17,L"\x2C47"},
{0x2C18,L"\x2C48"},
{0x2C19,L"\x2C49"},
{0x2C1A,L"\x2C4A"},
{0x2C1B,L"\x2C4B"},
{0x2C1C,L"\x2C4C"},
{0x2C1D,L"\x2C4D"},
{0x2C1E,L"\x2C4E"},
{0x2C1F,L"\x2C4F"},
{0x2C20,L"\x2C50"},
{0x2C21,L"\x2C51"},
{0x2C22,L"\x2C52"},
{0x2C23,L"\x2C53"},
{0x2C24,L"\x2C54"},
{0x2C25,L"\x2C55"},
{0x2C26,L"\x2C56"},
{0x2C27,L"\x2C57"},
{0x2C28,L"\x2C58"},
{0x2C29,L"\x2C59"},
{0x2C2A,L"\x2C5A"},
{0x2C2B,L"\x2C5B"},
{0x2C2C,L"\x2C5C"},
{0x2C2D,L"\x2C5D"},
{0x2C2E,L"\x2C5E"},
{0x2C80,L"\x2C81"},
{0x2C82,L"\x2C83"},
{0x2C84,L"\x2C85"},
{0x2C86,L"\x2C87"},
{0x2C88,L"\x2C89"},
{0x2C8A,L"\x2C8B"},
{0x2C8C,L"\x2C8D"},
{0x2C8E,L"\x2C8F"},
{0x2C90,L"\x2C91"},
{0x2C92,L"\x2C93"},
{0x2C94,L"\x2C95"},
{0x2C96,L"\x2C97"},
{0x2C98,L"\x2C99"},
{0x2C9A,L"\x2C9B"},
{0x2C9C,L"\x2C9D"},
{0x2C9E,L"\x2C9F"},
{0x2CA0,L"\x2CA1"},
{0x2CA2,L"\x2CA3"},
{0x2CA4,L"\x2CA5"},
{0x2CA6,L"\x2CA7"},
{0x2CA8,L"\x2CA9"},
{0x2CAA,L"\x2CAB"},
{0x2CAC,L"\x2CAD"},
{0x2CAE,L"\x2CAF"},
{0x2CB0,L"\x2CB1"},
{0x2CB2,L"\x2CB3"},
{0x2CB4,L"\x2CB5"},
{0x2CB6,L"\x2CB7"},
{0x2CB8,L"\x2CB9"},
{0x2CBA,L"\x2CBB"},
{0x2CBC,L"\x2CBD"},
{0x2CBE,L"\x2CBF"},
{0x2CC0,L"\x2CC1"},
{0x2CC2,L"\x2CC3"},
{0x2CC4,L"\x2CC5"},
{0x2CC6,L"\x2CC7"},
{0x2CC8,L"\x2CC9"},
{0x2CCA,L"\x2CCB"},
{0x2CCC,L"\x2CCD"},
{0x2CCE,L"\x2CCF"},
{0x2CD0,L"\x2CD1"},
{0x2CD2,L"\x2CD3"},
{0x2CD4,L"\x2CD5"},
{0x2CD6,L"\x2CD7"},
{0x2CD8,L"\x2CD9"},
{0x2CDA,L"\x2CDB"},
{0x2CDC,L"\x2CDD"},
{0x2CDE,L"\x2CDF"},
{0x2CE0,L"\x2CE1"},
{0x2CE2,L"\x2CE3"},
{0xFB00,L"\x0066\x0066"},
{0xFB01,L"\x0066\x0069"},
{0xFB02,L"\x0066\x006C"},
{0xFB03,L"\x0066\x0066\x0069"},
{0xFB04,L"\x0066\x0066\x006C"},
{0xFB05,L"\x0073\x0074"},
{0xFB06,L"\x0073\x0074"},
{0xFB13,L"\x0574\x0576"},
{0xFB14,L"\x0574\x0565"},
{0xFB15,L"\x0574\x056B"},
{0xFB16,L"\x057E\x0576"},
{0xFB17,L"\x0574\x056D"},
{0xFF21,L"\xFF41"},
{0xFF22,L"\xFF42"},
{0xFF23,L"\xFF43"},
{0xFF24,L"\xFF44"},
{0xFF25,L"\xFF45"},
{0xFF26,L"\xFF46"},
{0xFF27,L"\xFF47"},
{0xFF28,L"\xFF48"},
{0xFF29,L"\xFF49"},
{0xFF2A,L"\xFF4A"},
{0xFF2B,L"\xFF4B"},
{0xFF2C,L"\xFF4C"},
{0xFF2D,L"\xFF4D"},
{0xFF2E,L"\xFF4E"},
{0xFF2F,L"\xFF4F"},
{0xFF30,L"\xFF50"},
{0xFF31,L"\xFF51"},
{0xFF32,L"\xFF52"},
{0xFF33,L"\xFF53"},
{0xFF34,L"\xFF54"},
{0xFF35,L"\xFF55"},
{0xFF36,L"\xFF56"},
{0xFF37,L"\xFF57"},
{0xFF38,L"\xFF58"},
{0xFF39,L"\xFF59"},
{0xFF3A,L"\xFF5A"},
{0x10400,L"\x10428"},
{0x10401,L"\x10429"},
{0x10402,L"\x1042A"},
{0x10403,L"\x1042B"},
{0x10404,L"\x1042C"},
{0x10405,L"\x1042D"},
{0x10406,L"\x1042E"},
{0x10407,L"\x1042F"},
{0x10408,L"\x10430"},
{0x10409,L"\x10431"},
{0x1040A,L"\x10432"},
{0x1040B,L"\x10433"},
{0x1040C,L"\x10434"},
{0x1040D,L"\x10435"},
{0x1040E,L"\x10436"},
{0x1040F,L"\x10437"},
{0x10410,L"\x10438"},
{0x10411,L"\x10439"},
{0x10412,L"\x1043A"},
{0x10413,L"\x1043B"},
{0x10414,L"\x1043C"},
{0x10415,L"\x1043D"},
{0x10416,L"\x1043E"},
{0x10417,L"\x1043F"},
{0x10418,L"\x10440"},
{0x10419,L"\x10441"},
{0x1041A,L"\x10442"},
{0x1041B,L"\x10443"},
{0x1041C,L"\x10444"},
{0x1041D,L"\x10445"},
{0x1041E,L"\x10446"},
{0x1041F,L"\x10447"},
{0x10420,L"\x10448"},
{0x10421,L"\x10449"},
{0x10422,L"\x1044A"},
{0x10423,L"\x1044B"},
{0x10424,L"\x1044C"},
{0x10425,L"\x1044D"},
{0x10426,L"\x1044E"},
{0x10427,L"\x1044F"}
};


static int Count = sizeof(CaseFoldTbl)/sizeof(struct cp);

/*
 * Since Unicode case folding can translate a single character
 * into as many as three, we reallocate storage here.
 */

void
UnicodeFoldCase(struct dstr *ds) {
   int FoundP;
   int MaximumCharsNeeded;
   long l;			/* Lower bound of range */
   long u;			/* Upper bound of range */
   long m;			/* Midpoint of range */
   wchar_t c;
   wchar_t *t;
   wchar_t *t0; 
   wchar_t *s;
   wchar_t *s0; 
   wchar_t *cf;
   wchar_t *new;
   size_t NewCharCnt;
   size_t OriginalCharCnt;

   /* Allocate a working buffer guaranteed to be large enough */
   /* The output can be no longer than three times the length of the input */
   s0 = s = ds->s;
   MaximumCharsNeeded = (wcslen(s) * 3) + 1;
   t = (wchar_t *) malloc(MaximumCharsNeeded * sizeof(wchar_t));
   if(t == NULL){
     fprintf(stderr,_("Out of memory.\n"));
     exit(OUTOFMEMORY);
   }

   t0 = t;
   while ((c = *s) != L'\0'){
     l = 0;
     u = Count - 1;
     FoundP = 0;
     while(l <= u){   /* Textbook binary search */
       m = (l + u) / 2;
       if (c == CaseFoldTbl[m].i){
	 cf = CaseFoldTbl[m].o;
	 while (*cf != L'\0') {
	   *t++ = *cf++;
	 }
	 FoundP = 1;
	 break;
       }
       else if(c > CaseFoldTbl[m].i) l = m + 1;
       else u = m - 1;
     }
     if(!FoundP) *t++ = c;
     s++;
   }
   *t = L'\0';
   OriginalCharCnt = s - s0;
   NewCharCnt = t - t0; 
   if(NewCharCnt > OriginalCharCnt) {
     free( (void *) s0);
     new = (wchar_t *) malloc((NewCharCnt +1) * sizeof(wchar_t));
     if(new == NULL){
       fprintf(stderr,_("Out of memory.\n"));
       exit(OUTOFMEMORY);
     }
     ds->s = wcscpy(new,t0);
   } else {
     ds->s = wcscpy(s0,t0);
   }
   ds->l = ds->c = NewCharCnt;
   free( (void *) t0);
}

inline
int NumStrKeyCopy(char * tgt,wchar_t *src) {
  int cnt= 0;			/* Characters preceding decimal point */
  int GotDecimalP = 0;

  if(*src == 0x002e) cnt = 1;
  while (*src != L'\0') {
    *tgt = (char) *src++;
    if (*tgt == '.') GotDecimalP = 1;
    if (!GotDecimalP) cnt++;
    tgt++;
  }
  *tgt = '\0';
  return cnt;
}

void
MapPerLocale(struct dstr *sptr) {
  size_t WCharsNeeded;
  wchar_t *new;

  errno = 0;
  WCharsNeeded =  wcsxfrm(NULL,sptr->s,0) + 1;
  if(errno == EINVAL) {
    fprintf(stderr,
	    "Key contains characters outside the collation sequence for the locale.\n");
    exit(OTHERERROR);
  }
  new = (wchar_t *) malloc(WCharsNeeded * sizeof(wchar_t));
  if(new == NULL) {fprintf(stderr,_("Out of memory.\n"));exit(OUTOFMEMORY);}
  (void)wcsxfrm(new,sptr->s,WCharsNeeded);
  free((void *) sptr->s);
  sptr->s = new;
  sptr->l = sptr->c = WCharsNeeded;
}

#ifdef LOCALE_SORT_ORDER
long ExtractNum(wchar_t *str,int f, int n) {
  wchar_t *tmp;
  size_t Length;
  int i;
  wchar_t *s;
  long num;

  Length = n-f;
#ifdef ALLOCAOK
  tmp = (wchar_t *) alloca(Length+1);
#else
  tmp = (wchar_t *) malloc(Length+1);
#endif

  s = str+f;
  for(i=0; i < Length; i++) *(tmp+i) = *s++;
  *(tmp+i) = L'\0';
  errno = 0;
  num = wcstol(tmp,NULL,10);
#ifndef ALLOCAOK
  free((void *) tmp);
#endif
  if((errno == EINVAL)||(errno == ERANGE)){
    return -1L;
  }
  return num;
}

wchar_t * ExtractTxt(wchar_t *str,int f, int n) {
  wchar_t *tmp;
  size_t Length;
  int i;
  wchar_t *s;
  wchar_t *xstr;
  int WCharsNeeded;

  Length = n-f;
#ifdef ALLOCAOK
  tmp = (wchar_t *) alloca(Length+1);
#else
  tmp = (wchar_t *) malloc(Length+1);
#endif

  s = str+f;
  for(i=0; i < Length; i++) *(tmp+i) = *s++;
  *(tmp+i) = L'\0';

  WCharsNeeded =  wcsxfrm(NULL,tmp,0);
  if(errno == EINVAL) {
    fprintf(stderr,"Key contains characters outside the collation sequence for the locale.\n");
    return NULL;
  }
  xstr = (wchar_t *) malloc((WCharsNeeded+1) * sizeof(wchar_t));
  if(xstr == NULL) return NULL;
  (void)wcsxfrm(xstr,tmp,WCharsNeeded);
  
#ifndef ALLOCAOK
  free((void *) tmp);
#endif

  return xstr;
}

/* We have to avoid transforming the digits so parse and only xfrm non-digits */

/* Or with the following */
#define SETHIGHBITMASK   0x80
/* And with the following */
#define UNSETHIGHBITMASK 0x7F
#define MAXHYBRIDSUBPARTS 128

struct HybridList *
MapHybridPerLocale (struct dstr *sptr) {
  wchar_t *s;
  short FirstPartIsNumericP = 0;
  int i,k;
  int Offsets[MAXHYBRIDSUBPARTS];
  size_t SubParts = 0;
  size_t TxtCnt = 0;
  size_t NumCnt = 0;
  enum {START,NUM,TXT} State;
  struct HybridList *ptr;

  s = sptr->s;
  if(iswdigit((wint_t) (*s))) FirstPartIsNumericP = 1;
  State = START;
  i = 0;
  /* Record the offsets of the transition points */
  while(*s != L'\0') {
    if(iswdigit(*s++)) {
      if(State != NUM) {
	Offsets[SubParts] = i;
	SubParts++;
	if(SubParts > MAXHYBRIDSUBPARTS ) {
	  fprintf(stderr,"A hybrid string cannot have more than %d subparts.\n",MAXHYBRIDSUBPARTS);
	  exit(LIMITEXCEEDED);
	}
	State = NUM;
      }
    }
    else {
      if(State != TXT) {
	Offsets[SubParts] = i;
	SubParts++;
	if(SubParts > MAXHYBRIDSUBPARTS ) {
	  fprintf(stderr,"A hybrid string cannot have more than %d subparts.\n",MAXHYBRIDSUBPARTS);
	  exit(LIMITEXCEEDED);
	}
	State = TXT;
      }
    }
    i++;
  }
  Offsets[SubParts] =i;

  ptr = (struct HybridList *) malloc(sizeof(struct HybridList));
  if(ptr == NULL) return NULL;

  /* Record cnt and start */
  ptr->cnt = ((unsigned char) SubParts) & UNSETHIGHBITMASK;
  if(FirstPartIsNumericP) ptr->cnt |= SETHIGHBITMASK;
  /* We can now compute how many subparts there are of each type */
  NumCnt = SubParts/2;
  if( (SubParts%2 == 1) && FirstPartIsNumericP) NumCnt+=1;
  TxtCnt = SubParts - NumCnt;
  /* Allocate storage for txt list and num list */
  ptr->ilist = (long *) malloc(NumCnt * sizeof(int));
  ptr->tlist = (wchar_t **) malloc(TxtCnt * sizeof(wchar_t *));
  if((ptr->ilist == NULL) || (ptr->tlist == NULL)) {
    fprintf(stderr,_("Out of memory.\n"),progname);
    exit(OUTOFMEMORY);
  }

  /* Convert num strings to ints and store */
  for(i = 0; i < NumCnt; i++) {
    k = 2*i - FirstPartIsNumericP +1;
    ptr->ilist[i] = ExtractNum(sptr->s,Offsets[k],Offsets[k+1]);
    if(ptr->ilist[i] < 0L) return NULL;
  }

  /* Transform txt strings and store */
  for(i = 0; i < TxtCnt; i++) {
    k = 2*i + FirstPartIsNumericP;
    ptr->tlist[i] = ExtractTxt(sptr->s,Offsets[k],Offsets[k+1]);
    if(ptr->tlist[i] == NULL) return NULL;
  }
  return ptr;
}

#endif

DOUBLE GetMonthKey(wchar_t *mn,wchar_t **mtbl) {
  int i;
  for(i=0;i<MONTHNAMES;i++) {
    if(wcscmp(mn,mtbl[i]) ==0) return((DOUBLE)((i/2)+1));
  }
  return((DOUBLE)(-1));
}

/*
 * Extract the keys from the record, map them into internal form
 * and store them with the record.
 */	

#define CAPTURED_GROUPS 10
int
GetKeys(struct record *recptr, struct keyinfo **info,int keys,wchar_t MaxInInput)
{
  wchar_t *Next;
  wchar_t *FStart;
  wchar_t **FieldPtrs;
  wchar_t *wcstokStore;
  int i, j,k;
  int FieldCnt;
  int FieldsNeeded;
  int MaxFields;
  int KeyLength;
  int FieldLength;
  int FirstIndex;
  int LastIndex;
  int SrcStart;
  int TgtStart;
  long first;			/* Index of first character in range */
  long last;		        /* Index of last character in range */
  DOUBLE TempNumKey;
  short FoundP;
  short KeyMissingP;
  int stat;
  regmatch_t pm[CAPTURED_GROUPS];
  wchar_t *OutPtr;
  wchar_t *Key;
  wchar_t *ConstKey;
  struct dstr TempKey;
  short FreeKeyP;		/* Does Key point at memory that should be freed? */
  size_t BytesNeeded;
  size_t WCharsNeeded;
   
  void KeyCopy(wchar_t *, wchar_t *, int);
  void RevKeyCopy(wchar_t *, wchar_t *, int);
  int MapKey(struct dstr *,struct mapentry **,int);
  void UnicodeFoldCase (struct dstr *);
  int SubKey(struct dstr *, struct keyinfo *);

  extern void StripDiacritics (wchar_t *);
  extern void ConvertEnclosed (wchar_t *);
  extern void ConvertStylistic (struct dstr *);

  extern int Exclude(wchar_t *,int, wchar_t *);
  extern int ExcludeChar(wchar_t *, wchar_t **, int, wchar_t *);
  extern int GetAngleKey(wchar_t *, DOUBLE *);
  extern int GetDateKey(wchar_t *, DOUBLE *,struct ymdinfo *);
  extern int GetISO8601Key(wchar_t *, DOUBLE *);
  extern int GetTimeKey(wchar_t *, DOUBLE *);
  extern wchar_t *wcCopyRange(wchar_t *, long, long);

  InitializeDynamicString(&TempKey);

  /*
   * If the whole record is to be used as key, skip the parse and
   * treat the whole record as the first key.
   */

  if(info[0]->SelectType == WHOLE) MaxFields = 1;
  else MaxFields = InitialMaxFields;
  FieldPtrs = (wchar_t **) malloc((size_t) MaxFields * sizeof(wchar_t *));
  if(FieldPtrs == NULL){
    fprintf(stderr,"%s: out of memory - FieldPtrs.\n",progname);
    exit(OUTOFMEMORY);
  }

  if(info[0]->SelectType == WHOLE){
    FieldCnt = 1;
    FieldPtrs[0] = wcInputText;
  } else {
    /*
     * Parse the record into fields, leaving the number of fields
     * found in Fieldcnt and pointers to the beginnings in FieldPtrs.
     */	    

    FieldCnt = 0;
    Next = &(wcInputText[0]);
    for (FStart = wcstok(Next,Terminators,&wcstokStore);
	 FStart != NULL;
	 FStart = wcstok(NULL,Terminators,&wcstokStore)) {
      if(FieldCnt == MaxFields){/* Reallocate pointers here */
	MaxFields += InitialMaxFields/2;
	FieldPtrs = (wchar_t **) realloc((void *) FieldPtrs,
					 (unsigned) MaxFields * sizeof(wchar_t *));
	if(FieldPtrs == (wchar_t **)0){
	  fprintf(stderr,"\n%s: out of memory - FieldPtrs.\n",progname);
	  exit(OUTOFMEMORY);
	}
      }
      FieldPtrs[FieldCnt++] = FStart;
      Next = NULL;
    }
  }
  /* We have now got all of the fields into FieldPtrs except for any ranges */

  /* For each key, find the required field and process. */    
  
  FreeKeyP = 0;
  for(i = 0; i < keys; i++){
    FoundP = FALSE;
    if(FreeKeyP) free( (void *) ConstKey);
    FreeKeyP = 0;
    KeyMissingP = FALSE;
      /* Key fields identified by character range */
    if(info[i]->SelectType == CRANGE) {
      if(info[i]->RangeFirst < 0) first = recptr->length + info[i]->RangeFirst;
      else first = info[i]->RangeFirst;
      if(info[i]->RangeLast  < 0) last  = recptr->length + info[i]->RangeLast; 
      else last = info[i]->RangeLast;
      if( (first > recptr->length - 1) ||
	  (last  > recptr->length - 1) ||
	  (first < 0) ||
	  (last < 0)) {
	if(info[i]->OptionalP == FALSE){ /* If not optional, gripe */
	  fprintf(stderr,
		  _("The specified range falls outside the bounds of this record.\n"));
	  fprintf(Logfp,
		  _("The specified range falls outside the bounds of this record.\n"));
	  return(ERROR);
	}
	else KeyMissingP=TRUE;
      } else {
	Key = wcCopyRange(wcInputText,first,last);
	FreeKeyP = 1;
      }
      /* Key fields identified by numerical position */
    } else if( (info[i]->SelectType == NUMBERED) || (info[i]->SelectType == WHOLE)){
      if(info[i]->Number < 0) FirstIndex = FieldCnt + info[i]->Number +1;
      else FirstIndex = info[i]->Number;
      if(info[i]->Last < 0) LastIndex = FieldCnt + info[i]->Last+1;
      else LastIndex = info[i]->Last;
      if(FirstIndex > LastIndex) {
	fprintf(Logfp,"The start field is greater than the end field.\n");
	return(ERROR);
      }
      if((FirstIndex >= 0) && (FirstIndex < FieldCnt) && (LastIndex >= 0) &&
	 (LastIndex < FieldCnt)) {
	if (FirstIndex == LastIndex) {
	  FieldLength = wcslen(FieldPtrs[FirstIndex]);
	  if (FieldLength > info[i]->FirstOffset) {
	    Key = FieldPtrs[FirstIndex] + info[i]->FirstOffset;
	  }
	  else {
	    fprintf(Logfp,"Field %d is not as long as the offset.\n",i+1);
	    return(ERROR);
	  }
	}
	else {
	  FreeKeyP = 1;
	  KeyLength = 0;
	  for(j = FirstIndex; j <= LastIndex; j++) KeyLength += wcslen(FieldPtrs[j]);
	  Key = (wchar_t *) malloc ( (KeyLength+1+(LastIndex-FirstIndex)) * sizeof(wchar_t));
	  if (Key == NULL) { 
	    fprintf(stderr,"\n%s: out of memory - FieldPtrs.\n",progname);
	    exit(OUTOFMEMORY);
	  }
	  TgtStart =0;
	  for(j = FirstIndex; j <= LastIndex; j++) {
	    SrcStart = 0; /* Default */
	    if(j == FirstIndex) {
	      if (wcslen(FieldPtrs[j]) > info[i]->FirstOffset) {	
		SrcStart = info[i]->FirstOffset;
	      }
	      else {
		fprintf(Logfp,"Field %d is not as long as the offset.\n",j+1);
		return(ERROR);
	      }
	    }
	    if(j == LastIndex) {
	      if(info[i]->LastOffset < 0) { /* No last offset specified -> use whole field */	
		wcscpy(Key+TgtStart,FieldPtrs[j]);
	      }
	      else {
		if (wcslen(FieldPtrs[j]) <= info[i]->LastOffset) {	
		  fprintf(Logfp,"Field %d is not as long as the offset.\n",j+1);
		  return(ERROR);
		}
		for (k = 0; k <= info[i]->LastOffset; k++) {
		  Key[TgtStart+k] = FieldPtrs[j][k];
		}
	      }
	    }
	    else {
	      wcscpy(Key+TgtStart,FieldPtrs[j]+SrcStart);
	      TgtStart+= (wcslen(FieldPtrs[j]) - SrcStart);
	      Key[TgtStart++] = Terminators[0];
	    }
          }
	}
      }
      else{		/* Key field not present */
	if( (FirstIndex < 0) || (LastIndex < 0) ) {
	  FieldsNeeded = (-1 * MIN(info[i]->Number,info[i]->Last)) - 1;
	}
	if( (FirstIndex >= FieldCnt) || (LastIndex >= FieldCnt)) {
	  FieldsNeeded = MAX(FirstIndex,LastIndex) + 1;
	}
	if(info[i]->OptionalP == FALSE){ /* If not optional, gripe */
	  fprintf(stderr, "\n%s: record does not have %d fields.\n", progname,FieldsNeeded);
	  fprintf(Logfp, "\n%s: record does not have %d fields.\n", progname,FieldsNeeded);
	  return(ERROR);
	}
	else KeyMissingP=TRUE;
      }
    } else{
      /* Key fields  identified by tag */
      for(j = 0;j < FieldCnt; j++){
	stat = regwexec(&(info[i]->cregexp),FieldPtrs[j],CAPTURED_GROUPS,pm,0);
	if(stat == 0) {
	  FoundP = TRUE;
	  Key = FieldPtrs[j] + pm[1].rm_eo;
	}
      }
      if(FoundP == FALSE){
	if(info[i]->OptionalP == FALSE){ /* If not optional, gripe */
	  fwprintf(Logfp,
		  L"\nRecord does not have a field with tag matching \"%s\" on which to sort.\n",
		  info[i]->Tag);
	  return(ERROR);
	}
	else KeyMissingP=TRUE;
      }
    }	/* End of search for tagged key */
    if(KeyMissingP){
      if(info[i]->CompType & CNUMERIC){
	recptr->klistptr[i].n = NULL;
      } else {
	recptr->klistptr[i].t = NULL;
      }
      continue;
    }

    /* If we get here, we have found the key, so we process it */
    ConstKey = Key;		/* Save since we may skip leading chars in Key  */
    if(info[i]->CompType & CNUMERIC){
      if(info[i]->ExclusionTable != NULL){
	KeyLength = wcslen(Key);
	KeyLength = ExcludeChar(Key,&OutPtr,KeyLength,info[i]->ExclusionTable);
	if(KeyLength == 0){
	  fprintf(stderr,"After exclusions record is empty.\n");
	  return(ERROR);
	}
	Key = OutPtr;
	ConstKey = Key;
      }
      if(info[i]->CompType & CTIME){
	if(GetTimeKey(Key,&TempNumKey) == ERROR){
	  fwprintf(Logfp,L"\n%s is not a well-formed time\n",Key);
	  return(ERROR);
	}
      }
      else if(info[i]->CompType & CANGLE){
	if(GetAngleKey(Key,&TempNumKey) == ERROR){
	  fwprintf(Logfp,L"\n%s is not a well-formed angle\n",Key);
	  return(ERROR);
	}
      }
      else if(info[i]->CompType & CISO8601){
	if(GetISO8601Key(Key,&TempNumKey) == ERROR){
	  fwprintf(Logfp,L"\n%s is not a well-formed ISO8601 date/time combination.\n",
		  Key);
	  return(ERROR);
	}
      }
      else if(info[i]->CompType & CDATE){
	if(GetDateKey(Key,
		      &TempNumKey,&(info[i]->ymd)) == ERROR){
	  fwprintf(Logfp,L"\n%s is not a well-formed date.\n",Key);
	  return(ERROR);
	}
      }
      else if ((info[i]->CompType & CMONTH) && (info[i]->CompType & CNUMERIC)) {
	TempNumKey = GetMonthKey(Key,info[i]->MonthNames);
	if(TempNumKey < 1.0) {
	  fwprintf(Logfp,L"\n%s is not a recognized month name.\n",Key);
	  return(ERROR);
	}
      }
      else {
	TempNumKey = atofwc(Key,&stat);
	if(stat == BADRECORD) {
	  fwprintf(Logfp,L"\n%s is not a well-formed number.\n",Key);
	  return(ERROR);
	}
	if(stat == RANGEERROR) {
	  fwprintf(Logfp,L"\n%s is outside the range of representable numbers.\n",Key);
	  return(ERROR);
	}
      }
    } /* End of plain numeric key */
    else if(info[i]->CompType & CNUMSTR){
      if(!KeyMissingP){
	if (Key[0] == 0x002d) {	/* negative sign */
	  Key++;
	  recptr->klistptr[i].N.sign = -1;
	}
	else if (Key[0] == 0x002b) { /* positive sign */
	  Key++;
	  recptr->klistptr[i].N.sign = 1;
	}
	else recptr->klistptr[i].N.sign = 1;
	while(*Key == 0x0030) Key++; /* skip leading zeroes */
	KeyLength = wcslen(Key);
	recptr->klistptr[i].N.txt = (char *) malloc(sizeof(char) * (KeyLength +1));
	if(recptr->klistptr[i].N.txt == NULL) {
	  fprintf(stderr,"\n%s: out of memory - new key.\n", progname);
	  exit(OUTOFMEMORY);
	}
	recptr->klistptr[i].N.cnt = NumStrKeyCopy(recptr->klistptr[i].N.txt,Key);
      }
      continue;
    }
    else{		/* Lexicographic key: pure lexicographic, hybrid, and string length */
      if(FillDynamicString(&TempKey,Key) == ERROR) {
	fprintf(stderr,_("Out of memory.\n"));
	exit(OUTOFMEMORY);
      }

      /* Perform substitutions */
      if(info[i]->SubListEntries > 0) {
	(void)SubKey(&TempKey,info[i]);
      }

      /* Handle exclusions */
      if( (info[i]->ExclusionTable != NULL) && !KeyMissingP){
	KeyLength = Exclude(TempKey.s,TempKey.l,info[i]->ExclusionTable);
	if(KeyLength == 0){
	  if(!info[i]->OptionalP){
	    fprintf(stderr,"After exclusions non-optional field is empty.\n");
	    return(ERROR);
	  }
	  KeyMissingP=TRUE;
	}
      }

      if (info[i]->StripDiacriticsP) {
	StripDiacritics(TempKey.s);
      }
      if (info[i]->ConvertEnclosedP) {
	ConvertEnclosed(TempKey.s);
      }
      if (info[i]->ConvertStylisticP) {
	ConvertStylistic(&TempKey);
      }

      /* Fold case */
      if(info[i]->FoldCaseP) {
	if (MaxInInput <= 0x7F) ASCIIFoldCase(TempKey.s);
	else {		/* Don't recorder these - it is important to do Turkic first */
	  if(info[i]->TurkicFoldCaseP) TurkicFoldCase(TempKey.s);
	  UnicodeFoldCase(&TempKey);

	}
      }

      /* Map multigraphs */

#ifdef LOCALE_SORT_ORDER
      if(info[i]->LocaleP) {
	if(MultipleLocalesP) setlocale(LC_COLLATE,info[i]->Locale);
	if(info[i]->CompType & CHYBRID) {
	  recptr->klistptr[i].H = MapHybridPerLocale(&TempKey);
	  if(recptr->klistptr[i].H == NULL) {
	    fprintf(stderr,"\n%s: out of memory - new key.\n", progname);
	    exit(OUTOFMEMORY);
	  }
	  continue;
	}
	else MapPerLocale(&TempKey);
      }
      else {
#endif
	(void)MapKey(&TempKey,
		     info[i]->MapTable,
		     info[i]->MapTableEntries);
#ifdef LOCALE_SORT_ORDER
      }
#endif
    }

    /* Store the key */

    if(info[i]->CompType & CNUMERIC){
      if(!KeyMissingP){
	recptr->klistptr[i].n = AllocateDouble();
	*(recptr->klistptr[i].n) = TempNumKey;
      }
      else recptr->klistptr[i].n = NULL;
    }
    else{			/* Lexicographic key */
      if(!KeyMissingP){
	if(info[i]->CompType & CSIZE){
	  recptr->klistptr[i].n = AllocateDouble();
	  *(recptr->klistptr[i].n) = (DOUBLE) TempKey.l;
	}
	else{
	  WCharsNeeded = 1 + TempKey.l;
	  BytesNeeded = WCharsNeeded * sizeof(wchar_t);
	  recptr->klistptr[i].t = (wchar_t *) malloc(BytesNeeded);
	  if(recptr->klistptr[i].t == (wchar_t *) 0){
	    fprintf(stderr,"\n%s: out of memory - new key.\n", progname);
	    exit(OUTOFMEMORY);
	  }
	  if(info[i]->ReverseP) RevKeyCopy(recptr->klistptr[i].t,
					   TempKey.s,WCharsNeeded);
	  else KeyCopy(recptr->klistptr[i].t,
		       TempKey.s,WCharsNeeded);
	}
      }
      else{
	if(info[i]->CompType & CSIZE) recptr->klistptr[i].n = NULL;  
	else recptr->klistptr[i].t = NULL;   
      }
    }
  } /* End of loop over keys */

  if(FieldPtrs != NULL) {free((void *) FieldPtrs);FieldPtrs = NULL;}
  if(TempKey.s != NULL) {free((void *) TempKey.s); TempKey.s = NULL;}
  return (SUCCESS);
} /* End of GetKeys */

int
GetSortOrder(char *file,struct keyinfo *ki)
{
  FILE *fp;
   
  extern FILE *OpenFile(char *, char *, char *);
  void SortMapTable(struct keyinfo *);
  void CreateRankTable(struct keyinfo *);
  int ConstructSortOrder(FILE *, struct keyinfo *);

  fp = OpenFile(file,"r",progname);
  if(fp == NULL) return(ERROR);
  CreateRankTable(ki);
  if(ConstructSortOrder(fp,ki) == ERROR) return(ERROR);
  fclose(fp);
  SortMapTable(ki);
  return(SUCCESS);
}

/*
 * Initialize all characters into numerical order but so that,
 * if left unset, they will be ranked after all characters whose
 * order is set.	 
 */	 

void
CreateRankTable(struct keyinfo *ki)
{
  long i;

  ki->RankTable = (wchar_t *) malloc(sizeof(wchar_t) * RankTableSize);
  if(ki->RankTable == NULL){
    fprintf(stderr,_("Out of memory.\n"));
    exit(OUTOFMEMORY);
  }

  /*
   * Code 0 must rank first so that a string will sort before
   *	a string of which it is a prefix.
   */
  ki->RankTable[0] = 0;
	
  /* This yields machine collating order but after all user-specified codes */
  for(i=1;i<RankTableSize;i++) ki->RankTable[i] = RankTableSize + i +1;
}

/* Multigraph info is initialized separately, after reading the command line */
void
InitializeKeyInfo(struct keyinfo *ptr)
{
  ptr->SelectType = NUMBERED;
  ptr->Number = 0;
  ptr->Last = 0;
  ptr->CompType = (unsigned char) (0 & ~CNUMERIC & ~CINVERSE);
  ptr->RangeFirst = 0;
  ptr->RangeLast  = 0;
  ptr->FirstOffset = 0;
  ptr->LastOffset = -1;
  ptr->OptionalP = FALSE;
  ptr->MissingKeyComparison = COMPARE_EQUAL;
  ptr->ReverseP = FALSE;
  ptr->FoldCaseP = FALSE;
  ptr->TurkicFoldCaseP = FALSE;
  ptr->StripDiacriticsP = FALSE;
  ptr->ConvertEnclosedP = FALSE;
  ptr->ConvertStylisticP = FALSE;
  ptr->LocaleP = FALSE;
  ptr->Locale = NULL;
  ptr->MapTable = NULL;
  ptr->MapTableEntries = 0;
  ptr->SubList = NULL;
  ptr->SubListEntries = 0;
  ptr->RankTable = NULL;
  ptr->ExclusionTable = NULL;
  ptr->ExclusionSpec=0;
  ptr->ExclusionEntries = 0;
  ptr->ymd.y = 0;
  ptr->ymd.m = 1;
  ptr->ymd.d = 2;
  ptr->ymd.sep1 = L'-';
  ptr->ymd.sep2 = L'-';
  ptr->WhiteSpace = DefaultWhiteSpace;
  ptr->NextCode = MG_First;
  ptr->MaxMGCode = MG_Last;
  ptr->TotalCodes = MG_Total;
}

/* Allocate storage for a key and initialize it */
struct keyinfo *
SetupKey(void) {
  struct keyinfo *ptr;

  ptr = (struct keyinfo *) malloc(sizeof(struct keyinfo));
  if(ptr == NULL) {
    fprintf(stderr,"Could not allocate initial key struct.\n");
    exit(OUTOFMEMORY);
  }
  InitializeKeyInfo(ptr);
  return ptr;
}

void
CreateInitialKeys (void) {
  int i;

  KeyInfo  = (struct keyinfo **) malloc (INITIALKEYS * sizeof(struct keyinfo *));
  if(NULL == KeyInfo) {
    fprintf(stderr,"Could not allocate initial key array.\n");
    exit(OUTOFMEMORY);
  }

  for(i = 0; i < INITIALKEYS; i++) {
    KeyInfo[i] = SetupKey();
  }
}

void
CheckKeyAllocation (int keycount, int *maxkeys) {

  if (keycount < *maxkeys) return;
  *maxkeys = *maxkeys + 1;
  KeyInfo = (struct keyinfo **) realloc(KeyInfo,sizeof(struct keyinfo *) * *maxkeys);
  if(KeyInfo == NULL) {
    fprintf(stderr,_("Out of memory: cannot reallocate KeyInfo.\n"));
    exit(OUTOFMEMORY); 
  }
  KeyInfo[(*maxkeys)-1] = SetupKey();
}

/* Write the records out in sorted order */

void
WriteOutRecords(long Records, FILE *fp,int ReverseP,wchar_t sep)
{
  long i;
  int fd;
  int seplen;
  short FixedLengthP;
  UTF8 utf8sep[6];
  char PrTemp[128];
  struct record *ptr;

  extern int wc2utf8(UTF8 *, wchar_t);

  if(sep == NOTACHARACTER) {
    FixedLengthP = 1;
  } else {
    FixedLengthP = 0;
    seplen = wc2utf8(utf8sep,sep);
  }
  fd = fileno(fp);

  for(i=1L; i <= Records; i++){
    ptr = RecordList[(ReverseP ? Records-i : i-1)];
    write(fd,ptr->text,ptr->length);
    if(!FixedLengthP) write(fd,utf8sep,seplen);
    if(VerboseP && (i % INTWREPORTINTERVAL == 0)){
      fprintf(stderr,"\b\b\b\b\b\b\b\b\b%9ld",i);
    }
  }
  
  if(VerboseP) {
#ifdef HAVE_PRINTF_THSEP
    fprintf(stderr,_("\b\b\b\b\b\b\b\b\b%'9ld\n"),i-1);
#else
    fprintf(stderr,_("\b\b\b\b\b\b\b\b\b%9ld\n"),i-1);
#endif
  }
}

#define INITIPSIZE 128

int
ConstructSortOrder(FILE *sfp, struct keyinfo *ki)
{
  int CurrentRank = 0;
  wchar_t *Field;
  wchar_t *str;
  UTF8 *InputLine;
  wchar_t *wcstokStore;
  wchar_t Index;
  int status;
  int InputLineSize;
  int LineNumber;
  wchar_t tmpwc;
  wchar_t AddStringToMapTable(wchar_t *,struct keyinfo *);
  extern int u82u32 (UTF8 *,wchar_t **,wchar_t *);
  extern UTF8 * GetBlockSepCharRAUTF8(FILE *, UTF8 *, int *, int *, wchar_t);
  extern int EvalEscapes(wchar_t *);

  InputLineSize = INITIPSIZE;
  InputLine = (UTF8 *) malloc(INITIPSIZE * sizeof(UTF8));
  if(InputLine == NULL){
    fprintf(stderr,_("ConstructSortOrder: out of memory\n"));
    exit(OUTOFMEMORY);
  }
  LineNumber = 0;
  while(1){
    InputLine = GetBlockSepCharRAUTF8(sfp,InputLine,&status,&InputLineSize,L'\n');
    LineNumber++;
    if(status == ENDOFINPUT) break;
    if(status == 0) continue;
    if(status == BUFOVERFLOW) {
      fprintf(stderr,_("ConstructSortOrder: out of memory\n"));
      return ERROR;
    }
    ++CurrentRank;
    if ((status = u82u32(InputLine,&str,&tmpwc)) != 0) {
      fprintf(stderr,_("Fatal error in ConstructSortOrder.\n"));
      exit(status);
    }
    for (Field = wcstok(str,ki->WhiteSpace,&wcstokStore);
	Field != NULL;
	Field = wcstok(NULL,ki->WhiteSpace,&wcstokStore)){
      if(EvalEscapes(Field) == ERROR){
	fprintf(stderr,_("EvalEscapes: out of memory.\n"));
	exit(OUTOFMEMORY);
      }
      if(wcslen(Field) > 1){
	Index = AddStringToMapTable(Field,ki);
	ki->RankTable[Index] = CurrentRank;
      }
      else {
	Index = Field[0];
	if(BMPOnlyP) {
	  if(Index > MAXBMP) {
	    fprintf(stderr,"Character in sort order at line %d lies outside of BMP.\n",LineNumber);
	    exit(LIMITEXCEEDED);
	  }
	}
	ki->RankTable[Index] = CurrentRank;
      }

    }
  }
  free((void *)InputLine);
  free((void *)str);
  return(SUCCESS);
}

struct mapentry **
AllocateMapTable(long TotalCodes)
{
  struct mapentry **new;
  int bytes;
	
  bytes = sizeof(struct mapentry *) * TotalCodes;
  new = malloc((size_t) bytes);
  if(new == NULL){
    fprintf(stderr,"%s: out of memory - map table\n",progname);
    exit(OUTOFMEMORY);
  }
  return(new);
}

/*
 * Add a string to the mapping table and return the code allocated to it.
 */

wchar_t
AddStringToMapTable(wchar_t *s, struct keyinfo *ki)
{
  struct mapentry *new;

  if(ki->MapTable == NULL) ki->MapTable = AllocateMapTable(ki->TotalCodes);
  if(ki->NextCode == ki->MaxMGCode){
    fprintf(stderr,"%s:Too many multigraphs defined.\n",progname);
    fprintf(stderr,
	    "NextCode = %lX  MaxMGCode == %lX\n",
	    (unsigned long)(ki->NextCode),
	    (unsigned long)(ki->MaxMGCode));
    exit(LIMITEXCEEDED);
  }
   
  /* Allocate a new map table entry */

  new = (struct mapentry *) malloc(sizeof(struct mapentry));
  if(new == (struct mapentry *) 0){
    fprintf(stderr,"%s: out of memory - map table entry\n",progname);
    exit(OUTOFMEMORY);
  }
  new->len = wcslen(s);

  new->str = (wchar_t *) malloc((new->len + 1) * sizeof(wchar_t));
  if(new->str == NULL){
    fprintf(stderr,"%s:out of memory - map table string\n",progname);
    exit(OUTOFMEMORY);
  }
  wcscpy(new->str,s);
  new->code = ki->NextCode;

  ki->MapTable[ki->MapTableEntries]=new;
  ki->MapTableEntries+=1;
   
  return((ki->NextCode)++);
}


/* Sort the map table so that sequence length is decreasing */

void
SortMapTable(struct keyinfo *ki)
{
  int gap;
  int i;
  int j;
   
  struct mapentry *temp;
   
  for(gap = ki->MapTableEntries/2; gap > 0; gap /= 2){
    for(i = gap; i < ki->MapTableEntries; i++){
      for(j = i-gap; j>= 0; j -= gap){
	if(ki->MapTable[j]->len >= ki->MapTable[j+gap]->len) break;
	else{
	  temp = ki->MapTable[j];
	  ki->MapTable[j] = ki->MapTable[j+gap];
	  ki->MapTable[j+gap] = temp;
	}
      }
    }
  }
}

#define GEBUFSIZE 16
int
GetExclusions(char *file,struct keyinfo *ki)
{
  FILE *fp;
  wchar_t buf[GEBUFSIZE];
  int i;
  unsigned char flag;
  int poslen;
  int ExCount;
  int LineNo;
  int cnt;
  wchar_t *Field1;
  wchar_t *Field2;
  wchar_t *wcstokStore;

  extern FILE *OpenFile(char *, char *, char *);
  extern int ugetline(FILE *,wchar_t *, int);
  extern int EvalEscapes(wchar_t *);
  wchar_t *CreateExclusionTable(void);
   
  ExCount = 0;
  LineNo = 0;
  fp = OpenFile(file,"r",progname);
  ki->ExclusionTable = CreateExclusionTable();
  while( (cnt = ugetline(fp,buf,GEBUFSIZE)) >= 0) {
    LineNo++;
    if(cnt == 0) continue;
    Field1 = wcstok(buf,L" \t",&wcstokStore);
    Field2 = wcstok(NULL,L" \t",&wcstokStore);
    if( (Field1 == NULL) || (Field2 == NULL)) {
      fprintf(stderr,"Format error in exclusion file %s at line %d.\n",file,LineNo);
      exit(OTHERERROR);
    }
    if(EvalEscapes(Field1) == ERROR){
      fprintf(stderr,"EvalEscapes: out of memory.\n");
      exit(OUTOFMEMORY);
    }
    if(*Field1 > RankTableSize) {
      fprintf(stderr,"Character 0x%X out of range in exclusion file %s at line %d.\n",
	      (unsigned)*Field1,file,LineNo);
      exit(LIMITEXCEEDED);
    }
    flag = 0;
    poslen = wcslen(Field2);
    for(i = 0; i < poslen; i++){
      switch(Field2[i]){
      case 'F':
      case 'f':
	flag |= EX_FINAL;
	break;
      case 'I':
      case 'i':
	flag |= EX_INITIAL;
	break;
      case 'M':
      case 'm':
	flag |= EX_MEDIAL;
	break;
      default:
	fprintf(stderr,
		"Invalid exclusion position specifier at line %d ofn file %s\n",
		LineNo,file);
	exit(OTHERERROR);
      }
    }
    ki->ExclusionTable[*Field1] = flag;
    ++ExCount;
  }
  fclose(fp);
  if(ExCount < 1){
    free((void *) ki->ExclusionTable);
    ki->ExclusionTable = NULL;
  }
  ki->ExclusionEntries = ExCount;
  return(SUCCESS);
}

wchar_t *
CreateExclusionTable(void)
{
  wchar_t *new;
  long i;
	
  new = (wchar_t *) malloc(sizeof(wchar_t) * RankTableSize);
  if(new == NULL){
    fprintf(stderr,"Out of memory.\n");
    exit(OUTOFMEMORY);
  }
  for(i = 0L; i < RankTableSize; i++) new[i] = 0;
  return(new);
}

void
GetExclusionsString(UTF8 *string,struct keyinfo *ki)
{
  wchar_t *CreateExclusionTable(void);
  int len;
  int ExCount;
  wchar_t *exstr;
  int i; 
  int status;
  wchar_t max;
  wchar_t c;

  extern int u82u32(UTF8 *, wchar_t **,wchar_t *);
  extern int EvalEscapes(wchar_t *);

  ExCount = 0;
  ki->ExclusionTable = CreateExclusionTable();
  if ((status = u82u32(string,&exstr,&max)) != 0) {
    fprintf(stderr,_("Fatal error in GetExclusionsString.\n"));
    exit(status);
  }
  if(EvalEscapes(exstr) == ERROR){
    fprintf(stderr,"EvalEscapes: out of memory.\n");
    exit(OUTOFMEMORY);
  }
  len= wcslen(exstr);
  for(i=0;i<len;i++){
    c = exstr[i];
    if(c > RankTableSize) {
      fprintf(stderr,"Character 0x%X out of range in command-line exclusion.\n",(unsigned)c);
      exit(LIMITEXCEEDED);
    }
    ki->ExclusionTable[c] = (EX_FINAL | EX_INITIAL | EX_MEDIAL);
    ++ExCount;
  }
  free((void *) exstr);
}


/*
 * Map key string characters and sequences to internal sort codes.
 * This presupposes that the map table has been sorted by SortMapTable()
 * into decreasing order of length.
 *	 
 *	Start at the beginning of the input string.
 * While there are still characters in the input string
 * 	Set the index into the MapTable to zero
 *		Work through MapTable, looking for prefixes of the input string.
 * 	If we find a prefix
 *			copy the corresponding code to the output
 *			advance the input pointer by the length of the prefix
 *			try again
 *		If we don't find a prefix
 *			copy the current character to the output
 *			advance the input pointer one character
 *			try again
 * Else we're done mapping this string
 */	 

int
MapKey(struct dstr *ds,struct mapentry **table,int entries)
{
  int mte;
  int FoundP;
  wchar_t *ip;
  wchar_t *op;
   
  if(entries == 0) return(ds->l);
  
  ip = op = ds->s;
  FoundP = FALSE;
  while(*ip != L'\0'){
    mte = 0;
    FoundP = FALSE;
    while(mte < entries){
      if(wcsncmp(ip,table[mte]->str,table[mte]->len) == 0){
	FoundP = TRUE;
	*op++ = table[mte]->code;
	ip+=table[mte]->len;
	break;
      }
      ++mte;
    }
    if(!FoundP) {
      ip++;
      op++;
    }
  }
  *op = (wchar_t) L'\0';
  return(ds->l = op-(ds->s)+1);
}

int
AddSubEntry (struct keyinfo *ki,wchar_t *restr,wchar_t *se)
{
  int tmpint1;
  struct subentry *new;
  extern wchar_t *WCopyString(wchar_t *);

  new = (struct subentry *) malloc(sizeof(struct subentry));
  if (new == NULL) {
    fprintf(stderr,_("Out of memory.\n"));
    exit(OUTOFMEMORY);
  }
  tmpint1 = regwcomp(&(new->cregexp),restr,REG_EXTENDED);
  if(tmpint1 != 0) return -1;
  if(se == NULL) {
    new->se = NULL;
    new->len = 0;
  }
  else {
    new->se = WCopyString(se);
    new->len = wcslen(se);
  }

  /* Prepend new entry to linked list */
  new->next = ki->SubList;
  ki->SubList = new;
  (ki->SubListEntries)++;

  return 0;
} 

#define GSBUFSIZE 128
int
GetSubstitutions(char *file, struct keyinfo *ki)
{
  FILE *fp;
  wchar_t buf[GSBUFSIZE];
  int LineNo;
  int cnt;
  wchar_t CommentCharacter = L'#';
  wchar_t *Field1;
  wchar_t *Field2;
  wchar_t *wcstokStore;

  extern FILE *OpenFile(char *, char *, char *);
  extern int ugetline(FILE *,wchar_t *, int);
  extern int EvalEscapes(wchar_t *);
   
  LineNo = 0;
  fp = OpenFile(file,"r",progname);
  while ( (cnt = ugetline(fp,buf,GSBUFSIZE)) >= 0) {
    LineNo++;
    if(cnt == 0) continue;
    if(buf[0] == CommentCharacter) continue;
    Field1 = wcstok(buf,L" \t",&wcstokStore);
    Field2 = wcstok(NULL,L" \t",&wcstokStore);
    if(Field1 == NULL) {
      fprintf(stderr,_("Format error in substitution file %s at line %d.\n"),file,LineNo);
      exit(OTHERERROR);
    }
    if((EvalEscapes(Field1) == ERROR) || (EvalEscapes(Field2) == ERROR)){
      fprintf(stderr,_("EvalEscapes: out of memory.\n"));
      exit(OUTOFMEMORY);
    }
    if(AddSubEntry(ki,Field1,Field2) < 0) {
      fprintf(stderr,
	      _(
	      "Failed to compile regular expression in substitution file %s at line %d.\n"),
	      file,LineNo);
      exit(OTHERERROR);
    }
  }
  fclose(fp);
  return(SUCCESS);
}

/*
 * Substitute se for each match to re in str.
 * Return the result in newly allocated memory.
 * Record the number of substitutions in matches.
 */

wchar_t *
regwsub(regex_t *cre,wchar_t *se,wchar_t *str,int *matches)
{
  regmatch_t pm[2];
  wchar_t *s;			/* Subject portion of string */
  wchar_t *new;
  int Substitutions = 0;
  int TotalChars = 0;
  int NewCharsAvailable;
  int CharsAvailable;
  extern wchar_t *wcCatRange(wchar_t *,wchar_t *, long, long);

  CharsAvailable = 3+wcslen(str);
  new = (wchar_t *) malloc (sizeof(wchar_t) * CharsAvailable);
  if(new == NULL) {
    fprintf(stderr,_("Out of memory.\n"));
    exit(OUTOFMEMORY);
  }
  new[0] = L'\0';
  s = str;

  while (*s != L'\0') {
    if(regwexec(cre,s,2,pm,0) != 0) {
      TotalChars += wcslen(s);
      wcscat(new,s);
      break;
    }
    TotalChars += pm[0].rm_so;
    (void)wcCatRange(new,s,0,pm[0].rm_so-1); /* Append the region preceding the match */
    TotalChars += (pm[0].rm_eo - pm[0].rm_so);
    if (se != NULL) {
      TotalChars += wcslen(se);
    }
    if (TotalChars >= CharsAvailable) {
      NewCharsAvailable = TotalChars+3;
      new = (wchar_t *) realloc(new,sizeof(wchar_t) * NewCharsAvailable);
      if (new == NULL) {
	fprintf(stderr,_("Out of memory.\n"));
	exit(OUTOFMEMORY); 
      }
      else CharsAvailable = NewCharsAvailable;
    }
    if (se != NULL) {
      (void)wcscat(new,se);		       /* Append the substitution for the match */
    }
    Substitutions++;
    s = s + pm[0].rm_eo;
  }
  *matches = Substitutions;
  return new;
}

int
SubKey(struct dstr *in, struct keyinfo *ki) {
  int len;
  struct subentry *sp;
  wchar_t *substr;
  int matches;

  sp = ki->SubList;
  substr = in->s;
  while (sp != NULL) {
    substr = regwsub(&(sp->cregexp), sp->se, substr, &matches);
    sp = sp->next;
  }
  free ( (void *) in->s);
  in->s = substr;
  /* This assumes that regwsub allocates just enough storage,
     which is a conservative assumption. */
  in->l = in->c = len = wcslen(substr);
  return len;
}


void
ProvideDefaultRanks(void)
{
  int i;

  for(i = 0; i < KeyCount;i++){
    if((!(KeyInfo[i]->CompType & CNUMERIC)) && (KeyInfo[i]->RankTable == NULL)){
      CreateRankTable(KeyInfo[i]);
    }
  }
}

void
Copyright(FILE *fp)
{
  fprintf(fp,"Copyright 1993-2006 William J. Poser\n");
  fprintf(fp,_("This program is free software; you can redistribute\n\
it and/or modify it under the terms of version 2 of\n\
the GNU General Public License as published by the\n\
Free Software Foundation.\n"));
  putc('\n',fp);
  fprintf(fp,_("Report bugs to: billposer@alum.mit.edu\n"));
}



#define LBUFSIZE 64
#define WSBUFSIZE 128
int
GetWhiteSpaceDefinition(char *fname, struct keyinfo *ki) {
  FILE *fp;
  int cnt;
  wchar_t lbuf[LBUFSIZE];
  wchar_t buf[WSBUFSIZE];

  extern FILE *OpenFile(char *, char *, char *);
  extern int ugetline(FILE *,wchar_t *, int);
  extern wchar_t *WCopyString(wchar_t *);
  extern int EvalEscapes(wchar_t *);
   
  fp = OpenFile(fname,"r",progname);
  while ( (cnt = ugetline(fp,lbuf,LBUFSIZE)) != ENDOFINPUT) {
    fprintf(stderr,"cnt = %X\n",cnt);
    if(cnt == 0) continue;
    if(EvalEscapes(lbuf) == ERROR){
      fprintf(stderr,"EvalEscapes: out of memory.\n");
      exit(OUTOFMEMORY);
    }
    wcsncat(buf,lbuf,WSBUFSIZE-1);
  }
  ki->WhiteSpace = WCopyString(buf);
  return(1);
}

