#include <ctype.h>
#include <sys/types.h>
#include <regex.h>
#include <string.h>
#include <stdlib.h>

#include "mt.h"
#include "error.h"
#include "my_pty.h"
#include "utils.h"
#include "term.h"
#include "colors.h"

extern proginfo *pi;
extern buffer *lb;
extern int path_max;
extern int nfd;
extern int max_x, max_y;
extern int min_n_bufferlines;

int find_string(int window, char *find, int offset)
{
	int loop, index = -1;
        regex_t regex;

	/* compile the searchstring (which can be a regular expression) */
	if (regcomp(&regex, find, REG_EXTENDED))
		return -1;	/* failed -> not found */

	for(loop=offset; loop<lb[window].curpos; loop++)
	{
		if (regexec(&regex, lb[window].Blines[loop], 0, NULL, 0) == 0)
		{
			index = loop;
			break;
		}
	}

       	regfree(&regex);

	return index;
}

void scrollback_help(void)
{
	NEWWIN *mywin = create_popup(9, 44);

	escape_print(mywin -> win, 1, 2, "^x^/^q^    exit");
	escape_print(mywin -> win, 2, 2, "^f^/^/^    search for string");
	escape_print(mywin -> win, 3, 2, "^n^      repeat search");
	escape_print(mywin -> win, 4, 2, "^c^      set colors");
	escape_print(mywin -> win, 5, 2, "^s^      save to file");
	mydoupdate(mywin -> win);

	wait_for_keypress();

	delete_popup(mywin);
}

void scrollback_savefile(int window)
{
	char *file = NULL;
	char sel;
	NEWWIN *mywin = create_popup(8, 40);

	win_header(mywin -> win, "Save buffer to file");

	if (lb[window].markset == 'a')
	{
		escape_print(mywin -> win, 3, 2, "Write all (^a^) or filtered (^m^)?");
		mydoupdate(mywin -> win);

		for(;;)
		{
			sel = tolower(wait_for_keypress());

			if (sel == 'a' || sel == 'm' || sel == 'q')
				break;

			wrong_key();
		}
	}
	else
	{
		sel = 'a'; /* no need to filter again, it is already filtered */
	}

	if (sel != 'q')
	{
		mvwprintw(mywin -> win, 4, 2, "Select file");
		file = edit_string(mywin -> win, 5, 40, path_max, 0, NULL);
	}
	if (file)
	{
		FILE *fh = fopen(file, "w");
		if (fh)
		{
			int loop;

			for(loop=0; loop<lb[window].curpos; loop++)
			{
				if (lb[window].Blines[loop])
				{
					char *error;
					int dummy = -1;

					/* check filter */
					if (sel == 'a' || check_filter(lb[window].pi[loop], lb[window].Blines[loop], NULL, &error, &dummy) || check_no_suppress_lines_filter(lb[window].pi[loop]) == 0)
					{
						fprintf(fh, "%s", lb[window].Blines[loop]);
					}
					else if (error)
					{
						fprintf(fh, "---> Error: %s\n", error);
						free(error);
					}
				}
			}

			fclose(fh);
		}
		else
		{
			mvwprintw(mywin -> win, 6, 2, "Cannot write to file!");
			mydoupdate(mywin -> win);

			wait_for_keypress();
		}
	}

	delete_popup(mywin);
}

int get_lines_needed(char *string, int terminal_width)
{
	if (string)
		return (strlen(string) + terminal_width - 1) / terminal_width;
	else
		return 1;
}

void scrollback_displayline(WINDOW *win, int window, int offset, int terminal_offset)
{
	proginfo *cur_line_meta = lb[window].pi[offset];
	char *cur_line = lb[window].Blines[offset];
	regmatch_t *pmatch = NULL;
	int matching_regex = -1;
	char *error = NULL;
	char ok;

	wmove(win, terminal_offset, 0);

	if (cur_line)
	{
		if (!cur_line_meta) /* markerline? */
		{
			color_print(win, NULL, NULL, NULL, -1);
		}
		else /* just a buffered line */
		{
			ok = check_filter(cur_line_meta, cur_line, &pmatch, &error, &matching_regex);
			if (ok)
			{
				color_print(win, cur_line_meta, cur_line, pmatch, matching_regex);
			}
			else if (error)
			{
				color_print(win, cur_line_meta, error, NULL, -1);
				free(error);
			}
		}
	}
	else /* an empty line */
	{
		/* do nothing */
	}

	free(pmatch);
}

void scrollback(void)
{
	int window = 0;
	NEWWIN *mywin1 = create_popup(7, 22);

	win_header(mywin1 -> win, "Scrollback");

	if (nfd > 1)
	{
		window = select_window();
	}

	if (window != -1)
	{
		if (lb[window].markset == 0)
		{
			mvwprintw(mywin1 -> win, 3, 2, "Cannot scrollback:");
			mvwprintw(mywin1 -> win, 4, 2, "no mark was set!");
			mydoupdate(mywin1 -> win);
			wait_for_keypress();
		}
	}

	delete_popup(mywin1);

	if (window != -1 && lb[window].markset != 0)
	{
		char *find = NULL;
		NEWWIN *mywin2;
		int nlines = max_y - 6, ncols = max_x - 6;
		int offset = max(0, lb[window].curpos - nlines);
		char redraw = 2;
		int *line_lengths = (int *)mymalloc(lb[window].curpos * sizeof(int), "line lengths");
		int loop;

		for(loop=0; loop<lb[window].curpos; loop++)
			line_lengths[loop] = get_lines_needed(lb[window].Blines[loop], ncols);

		mywin1 = create_popup(max_y - 4, max_x - 4);
		wattron(mywin1 -> win, A_REVERSE);
		mvwprintw(mywin1 -> win, max_y - 5, 1, "%02d] %s - %d buffered lines", window, pi[window].filename, lb[window].curpos);
		wattroff(mywin1 -> win, A_REVERSE);
		wnoutrefresh(mywin1 -> win);

		mywin2 = create_popup(nlines, ncols);
		scrollok(mywin2 -> win, FALSE); /* supposed to always return OK, according to the manpage */

		for(;;)
		{
			int c;

			if (redraw == 2)
			{
				int index = 0;
				int lines_used = 0;

				werase(mywin2 -> win);

				for(;(offset + index) < lb[window].curpos;)
				{
					int prev_lines_used = lines_used;

					lines_used += line_lengths[offset + index];
					if (lines_used > nlines)
						break;

					scrollback_displayline(mywin2 -> win, window, offset + index, prev_lines_used);
					index++;
				}

				redraw = 1;
			}

			if (redraw == 1)
			{
				mydoupdate(mywin2 -> win);

				redraw = 0;
			}

			c = wait_for_keypress();

			if (toupper(c) == 'Q' || toupper(c) == 'X')
			{
				break;
			}
			else if (c == KEY_UP && offset > 0)
			{
				int n_to_move;

				offset--;

                                wmove(mywin2 -> win, 0, 0);
				n_to_move = line_lengths[offset];
				winsdelln(mywin2 -> win, n_to_move);

				scrollback_displayline(mywin2 -> win, window, offset, 0);

				redraw = 1;
			}
			else if ((c == KEY_DOWN || c == KEY_NPAGE) && offset < (lb[window].curpos - 1))
			{
				int dummy = 0;
				while(offset < (lb[window].curpos - 1))
				{
					offset++;
					dummy += line_lengths[offset];
					if ((dummy > nlines && c == KEY_NPAGE) ||
					    (dummy > (nlines/4) && c == KEY_DOWN))
					{
						offset--;
						break;
					}
				}
				redraw = 2;
			}
			else if (c == KEY_PPAGE && offset > 0)
			{
				int dummy = 0;
				while(offset > 0)
				{
					offset--;
					dummy += line_lengths[offset];
					if (dummy > nlines)
					{
						offset++;
						break;
					}
				}
				redraw = 2;
			}
			else if (c == KEY_HOME && offset > 0)
			{
				offset = 0;
			}
#ifdef N_CURSES
			else if (c == KEY_END && offset < (lb[window].curpos - 1))
			{
				offset = lb[window].curpos - 1;
			}
#endif
			else if (toupper(c) == 'F' || c == '/')
			{
				char *dummy;
				int new_f_index;
				NEWWIN *mywin = create_popup(5, 40);

				win_header(mywin -> win, "Find");

				dummy = edit_string(mywin -> win, 3, 40, 80, 0, find);
				free(find);
				find = dummy;

				redraw = 2; /* force redraw */

				new_f_index = find_string(window, find, 0);
				if (new_f_index == -1)
					wrong_key();
				else
					offset = new_f_index;

				delete_popup(mywin);
			}
			else if (toupper(c) == 'N')
			{
				if (find)
				{
					int new_f_index = find_string(window, find, offset + 1);
					if (new_f_index == -1)
						wrong_key();
					else
						offset = new_f_index;
				}
				else
					wrong_key();
			}
			else if (toupper(c) == 'S')
			{
				scrollback_savefile(window);
				redraw = 2;	/* force redraw */
			}
			else if (toupper(c) == 'H')
			{
				scrollback_help();
				redraw = 2;	/* force redraw */
			}
			else if (toupper(c) == 'C')
			{
				toggle_colors();
				redraw = 2;	/* force redraw */
			}
			else
			{
				wrong_key();
			}
		}

		delete_popup(mywin2);
		delete_popup(mywin1);

		free(find);
		free(line_lengths);
	}
}

void delete_mark(void)
{
	int window = 0;
	NEWWIN *mywin = create_popup(13, 40);

	win_header(mywin -> win, "Delete mark");

	if (nfd > 1)
	{
		window = select_window();
	}

	if (window != -1)
	{
		if (lb[window].markset == 0)
		{
			mvwprintw(mywin -> win, 3, 2, "No mark was set!");
			mydoupdate(mywin -> win);
			wait_for_keypress();
		}
		else
		{
			lb[window].markset = 0;
			lb[window].maxnlines = min_n_bufferlines;

			delete_array(lb[window].Blines, lb[window].curpos);
			lb[window].Blines = NULL;
			free(lb[window].pi);

			lb[window].curpos = 0;
		}
	}

	delete_popup(mywin);
}
