!
! robfit - fitting absolute deviations
!
! Copyright © 2011-7 F.Hroch (hroch@physics.muni.cz)
!
! This file is part of Munipack.
!
! Munipack is free software: you can redistribute it and/or modify
! it under the terms of the GNU General Public License as published by
! the Free Software Foundation, either version 3 of the License, or
! (at your option) any later version.
!
! Munipack is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU General Public License for more details.
!
! You should have received a copy of the GNU General Public License
! along with Munipack.  If not, see <http://www.gnu.org/licenses/>.
!

module robfit

  implicit none

  logical, private :: debug = .false.
  logical, private :: analytic = .true.
  integer, parameter, private :: dbl = selected_real_kind(15)
  real(dbl), parameter, private :: rad = 57.295779513082322865_dbl
  real(dbl), dimension(:), allocatable, private :: uref,vref
  real(dbl), dimension(:), pointer, private :: xstar,ystar
  real(dbl), private :: refl, xcen, ycen, sig
  integer, private :: ndat

  private :: dres, minfun, fundif, funder, difjac, hessian_min

contains

  subroutine robmin(type,a,d,xc,yc,rf,xx,yy,mad,acen,dacen,dcen,ddcen,sc,dsc,pa,dpa,rms,verbose,reliable)

    ! find proper transformation by robust mean squares

    use rfun
    use astrotrafo
    use minpacks
    use neldermead
    use entropyscale

    ! parameters
    character(len=*),intent(in) :: type
    real(dbl),dimension(:), intent(in) :: a,d
    real(dbl),dimension(:), target, intent(in) :: xx,yy
    real(dbl),intent(in) :: mad,xc,yc,rf
    real(dbl),intent(inout) :: acen,dcen,sc,pa
    real(dbl),intent(out) :: dacen,ddcen,dsc,dpa,rms
    logical, intent(in) :: verbose
    logical, intent(out) :: reliable

    real(dbl), parameter :: eps = 10*epsilon(1.0_dbl)
    integer, parameter :: npar = 4

    real(dbl),dimension(npar) :: p,dp,pmin
    real(dbl),dimension(npar,npar) :: fjac,cov
    real(dbl),dimension(:), allocatable :: du,dv,res
    integer :: iter,i,j,info,nprint,ifault,nra,ndec
    real(dbl) :: s0,sum1,sum2,wx,wy,s,smin
    type(AstroTrafoProj) :: t,ti
    character(len=80), parameter :: fmt = '(2i2,es10.2,2f11.5,2g12.3,f10.1,f7.1)'
    character(len=20) :: fmt1
    logical :: reli

    debug = verbose
    reliable = .false.

    if(debug) write(*,*) '=== Robust fitting ==='

    rms = huge(rms)
    refl = rf
    xcen = xc
    ycen = yc
    ndat = size(a)
    xstar => xx
    ystar => yy

    if( .not. (ndat > npar) ) stop 'Robust fitting requires five stars > 5.'
    if( mad < epsilon(mad) ) stop 'Robust fit needs mad > 0.'

    ! normalization
    allocate(uref(ndat),vref(ndat),du(ndat),dv(ndat),res(2*ndat))

    call trafo_init(t,type,acen,dcen)
    call proj(t,a,d,uref,vref)

    ! setup
    p(1:2) = 0
    p(3) = 1 / sc
    p(4) = pa / rad
    sig = mad / 0.6745
    dp(1:2) = 0.1 / p(3)
    dp(3) = 0.001*p(3)
    dp(4) = 0.1 / rad

    do iter = 1, precision(p)
       nprint = 0 ! or 1

       ! Update estimate of parameters by method without derivation
       ! Because there are no methods reliable for finding of roots
       ! in multidimensional case, we replaced root finding by minimizing.
       pmin = p
       call nelmin1(minfun,pmin,dp,smin,ifault)
       if( debug ) write(*,fmt) iter,ifault,sig,acen,dcen,p(1:2)/p(3),1/p(3),rad*p(4)

       if( ifault == 0 ) then

          p = pmin

          ! update estimate of dispersion by entropy
          call dres(p,res(1:ndat),res(ndat+1:2*ndat))
          s = sig
          call escale(res,s,reli)
          if( reli ) sig = s

       else
          ! what to do when nelmin fails?
          continue

       end if

       if( analytic ) then
          call lmder2(funder,p,eps,nprint,info)
       else
          call lmdif2(fundif,p,eps,nprint,info)
       end if
       if( info == 0 ) stop 'Warning: robfit: Improper input parameters.'
       ! testing errors:
       ! if( info == 4 ? ) write(error_unit,*) 'No konvergence.'

       ! residual sum
       call dres(p,du,dv)
       sum1 = 0.0_dbl
       sum2 = 0.0_dbl
       do i = 1, ndat
          wx = du(i)/sig
          wy = dv(i)/sig
          sum1 = sum1 + tukey(wx)**2 + tukey(wy)**2
          sum2 = sum2 + dtukey(wx) + dtukey(wy)
       enddo

       ! robust estimate of sum of squares in minimum
       if( sum1 > 0 .and. sum2 > 0 ) then
          s0 = sig**2*sum1/sum2**2*ndat**2
          rms = sqrt(s0 / (ndat - npar))
       else
          s0 = -1
          rms = -1
       end if

       if( debug ) write(*,fmt) iter,info,sig,acen,dcen,p(1:2)/p(3),1/p(3),rad*p(4)

       ! finish when location parameters are appropriate small
       ! and iterations couldn't made further progress
       !       if( iter > 1 .and. all(abs(p(1:2)) < eps)  ) exit
       ! the precision is better than 1/1000 of a pixel
       if( iter > 1 .and. all(abs(p(1:2)/p(3)) < 0.001)  ) exit

       ! update center of projection and projected coordinates
       call trafo_init(ti,type,acen=acen,dcen=dcen, &
            xcen=xcen,ycen=ycen,scale=p(3),rot=rad*p(4),refl=refl)
       call invproj(ti,p(1),p(2),acen,dcen)
       call trafo_init(t,type,acen,dcen)
       call proj(t,a,d,uref,vref)
       p(1:2) = 0.0_dbl

    end do

    call hessian_min(p,fjac)
    call qrinv(fjac,cov)

    ! statistical errors of parameters
    ! Hessian is recomputed analyticaly in view of the fact that
    ! we needs errors of acen, dcen (but p(1:2) are X0,Y0) and
    ! Jacobian by lmdif2 is scaled by errors (multiplication of s**2
    ! is satisfactory only for p(3:4).

    if( debug ) then
       write(fmt1,'(a,i0,a)') '(a,',npar,'g15.5)'
       write(*,'(a)') '# Hessian at minimum:'
       write(*,fmt1) ('#',fjac(i,:),i=1,npar)
       write(*,'(a)') '# Covariance matrix (no regularisation):'
       write(*,fmt1) ('#',cov(i,:),i=1,npar)
       write(*,'(a)') '# Correlation matrix:'
       do i = 1,npar
          write(*,'(a)',advance="no")  '#'
          do j = 1,npar
             write(*,'(f9.3)',advance="no") cov(i,j)/sqrt(cov(i,i)*cov(j,j))
          end do
          write(*,*)
       end do

    end if

    dp = -1
    do i = 1,npar
       if( s0 > 0 .and. cov(i,i) > 0 ) &
            dp(i) = sqrt(s0*cov(i,i)/(ndat - npar))
    end do

    sc = 1.0_dbl/p(3)
    pa = rad*p(4)

    ! deviations of output parameters
    dacen = dp(1) / cos(dcen / rad)  !*sc???  ! cos() sqeezes metric (distance) on sphere
    ddcen = dp(2) !*sc???
    dsc = sc*(dp(3)/p(3))
    dpa = rad*dp(4)

    if( debug ) then
       nra = count(du > 0)
       ndec = count(dv > 0)

       write(*,'(a,4es13.3)') '# solution:  ',p(1:npar)
       write(*,'(a,4es13.2)') '# deviations:',dp(1:npar)
       write(*,'(a,2es10.2,a,g0.3,a)') '# s0,rms: ',s0,rms,' [deg]    ', &
            3600*rms,' [arcsec]'
       write(*,'(a,es10.2,a,g0.3,a)') '# s: ',sig,' [deg]    ', &
            3600*sig,' [arcsec]'
       write(*,'(a,i7,"/",f0.1,2(i5,"+-",f0.1))') &
            '# sign test (total/expected, RA+, Dec+): ',&
            ndat,ndat/2.0,nra,sqrt(nra*0.25),ndec,sqrt(ndec*0.25)
       ! simple version of sign test, var = n*p*(1-p), where p = 0.5
    end if

    deallocate(uref,vref,du,dv,res)

  end subroutine robmin


  subroutine dres(p,du,dv,u,v)

    use astrotrafo

    real(dbl), dimension(:), intent(in) :: p
    real(dbl), dimension(:), intent(out) :: du,dv
    real(dbl), dimension(:), intent(out), optional :: u,v

    real(dbl) :: uu,vv,x,y
    integer :: i
    type(AstroTrafoProj) :: t

    call trafo_init(t,xcen=xcen,ycen=ycen, &
         ucen=p(1),vcen=p(2),scale=p(3),rot=rad*p(4),refl=refl)

    do i = 1, ndat
       call invaffine(t,xstar(i),ystar(i),uu,vv,x,y)
       du(i) = uref(i) - uu
       dv(i) = vref(i) - vv

       ! u,v - projected coordinates of refence stars, refreshed in main loop
       ! x,y - coordinates of stars on CCD
       ! xx,yy - rotated (and reflexed), but not scaled, x,y coordinates
       ! du,dv - residuals
       ! uu,vv - rotated, sometimes reflexed, and scaled x,y coordinates

       if( present(u) .and. present(v) ) then
          u(i) = x
          v(i) = y
       end if
!       write(*,'(9f12.5)') u(i),v(i),x(i),y(i),3600*du,3600*dv
    end do

  end subroutine dres


  subroutine fundif(m,np,p,fvec,iflag)

    use rfun

    integer, intent(in) :: m,np
    integer, intent(inout) :: iflag
    real(dbl), dimension(np), intent(in) :: p
    real(dbl), dimension(m), intent(out) :: fvec
    real(dbl), dimension(:), allocatable :: dx,dy,xx,yy
    real(dbl) :: sf,sc,sx,sy,fx,fy
    integer :: i

    if( iflag == 0 .and. debug ) write(*,*) '#robfit t:',real(p)

    sf = 0.0_dbl
    sc = 0.0_dbl
    sx = 0.0_dbl
    sy = 0.0_dbl

    allocate(dx(ndat),dy(ndat),xx(ndat),yy(ndat))
    call dres(p,dx,dy,xx,yy)
    do i = 1, ndat
       fx = tukey(dx(i)/sig)
       fy = tukey(dy(i)/sig)
       sx = sx + fx
       sy = sy + fy
       sc = sc + (fx*xx(i) + fy*yy(i))
       sf = sf + (fx*yy(i) - fy*xx(i))
    end do
    fvec = (/-sx,-sy,-sc,sf*p(3)/)/sig
!    write(*,'(4g15.5)') fvec
    deallocate(dx,dy,xx,yy)

  end subroutine fundif


  subroutine funder(m,np,p,fvec,fjac,ldfjac,iflag)

    use rfun

    integer, intent(in) :: m,np,ldfjac
    integer, intent(inout) :: iflag
    real(dbl), dimension(np), intent(in) :: p
    real(dbl), dimension(m), intent(out) :: fvec
    real(dbl), dimension(ldfjac,np), intent(out) :: fjac
    real(dbl), dimension(:), allocatable :: fu,fv,dfu,dfv,du,dv,u,v,ru,rv
    real(dbl), dimension(m) :: fc
    real(dbl), dimension(4,4) :: dfjac
    real(dbl) :: r
    integer :: i,j

    if( iflag == 0 ) then
       write(*,'(4f15.5)') p
       return
    end if

    allocate(fu(ndat),fv(ndat),du(ndat),dv(ndat),u(ndat),v(ndat),ru(ndat),rv(ndat))
    call dres(p,du,dv,u,v)

    r = p(3)
    ru = du / sig
    rv = dv / sig
    call tukeys(ru,fu)
    call tukeys(rv,fv)

    fc(1) = sum(fu)
    fc(2) = sum(fv)
    fc(3) = sum(fu*u + fv*v)
    fc(4) = sum(fu*v - fv*u) * r

    if( iflag == 1 ) then

       fvec(1:3) = - fc(1:3)
       fvec(4) = fc(4)
       fvec = fvec / sig

    else if( iflag == 2 ) then

       allocate(dfu(ndat),dfv(ndat))

       call dtukeys(ru,dfu)
       call dtukeys(rv,dfv)

       fjac = 0.0_dbl
       fjac(1,1) = sum(dfu)
       fjac(1,3) = sum(dfu*u)
       fjac(1,4) =-sum(dfu*v)*r
       fjac(2,2) = sum(dfv)
       fjac(2,3) = sum(dfv*v)
       fjac(2,4) = sum(dfv*u)*r
       fjac(3,3) = sum(dfu*u**2 + dfv*v**2)
       fjac(3,4) =-sum((dfu - dfv)*u*v)*r
       fjac(4,4) = sum(dfu*v**2 + dfv*u**2)*r**2

       fjac(3,4) = fjac(3,4) + fc(4) * sig / r
       fjac(4,4) = fjac(4,4) - fc(3) * sig * r
       fjac = fjac / sig**2

       do i = 1,np
          do j = 1,i-1
             fjac(i,j) = fjac(j,i)
          end do
       end do

       if( debug .and. .false. ) then
          do i = 1,np
             write(*,'(4g15.5)') fjac(i,:)*sig**2
          end do
          call difjac(p,dfjac)
          do i = 1,np
             write(*,'(a,4g15.5)') 'dif: ',dfjac(i,:) * sig**2
          end do
       end if

       deallocate(dfu,dfv)

    end if

    deallocate(fu,fv,du,dv,u,v,ru,rv)

  end subroutine funder

  subroutine hessian_min(p,hess)

    real(dbl), dimension(:), intent(in) :: p
    real(dbl), dimension(:,:), intent(out) :: hess
    real(dbl), dimension(size(p),size(p)) :: fjac
    real(dbl), dimension(size(p)) :: fvec
    integer :: iflag,m

    iflag = 2
    m = size(p)
    call funder(m,m,p,fvec,fjac,m,iflag)
    hess = fjac*sig**2

  end subroutine hessian_min


  subroutine difjac(p,jac)

    ! numerical approximation of jacobian

    real(dbl), dimension(:), intent(in) :: p
    real(dbl), dimension(:,:), intent(out) :: jac
    real(dbl), dimension(size(p)) :: fv1, fv2, p1
    integer :: i,n,iflag
    real(dbl) :: d

    iflag = 1
    n = size(p)

    do i = 1, n
       p1 = p
       if( abs(p(i)) > epsilon(p) ) then
          d = sqrt(epsilon(d))*abs(p(i))
       else
          d = sqrt(epsilon(d))
       end if
       p1(i) = p(i) + d
       call fundif(n,n,p1,fv1,iflag)
       call fundif(n,n,p ,fv2,iflag)
       jac(i,:) = (fv1 - fv2)/d
    end do

  end subroutine difjac

  function minfun(p)

    use rfun

    real(dbl) :: minfun
    real(dbl), dimension(:), intent(in) :: p
    real(dbl), dimension(:), allocatable :: du,dv,fu,fv

    allocate(du(ndat),dv(ndat),fu(ndat),fv(ndat))
    call dres(p,du,dv)
    call itukeys(du/sig,fu)
    call itukeys(dv/sig,fv)

    minfun = sum(fu) + sum(fv)

    deallocate(du,dv,fu,fv)

  end function minfun

end module robfit
