//network.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010
 *
 *  This file is part of RoarD,
 *  a sound server daemon for using the RoarAudio protocol.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  or (at your option) any later version as published by
 *  the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "muroard.h"

int network_init(void) {
#ifdef __WIN32
 WSADATA wsadata;

 WSAStartup(MAKEWORD(1,1) , &wsadata);
#endif
 return 0;
}

int network_free(void) {
 return network_prefree();
}

int network_prefree(void) {
#ifdef MUROAR_FEATURE_SOCKET_LISTEN
 if ( g_listen_socket != -1 )
  network_close(g_listen_socket);

  g_listen_socket = -1;
#endif

 return 0;
}

#ifdef MUROAR_FEATURE_SOCKET_UNIX
static inline int network_test_unix (char * addr) {
 struct sockaddr_un   un;
 int fh;
 int ret;

 if ( (fh = socket(AF_UNIX, SOCK_STREAM, 0)) == -1 )
  return -1;

 un.sun_family = AF_UNIX;
 strncpy(un.sun_path, addr, sizeof(un.sun_path) - 1);

 if ( connect(fh, (struct sockaddr *)&un, sizeof(struct sockaddr_un)) == -1 ) {
  if ( errno == ECONNREFUSED ) {
   ret =  0;
  } else {
   ret = -1;
  }
 } else {
  ret = 1;
 }

 network_close(fh);

 return ret;
}
#endif

#ifdef MUROAR_FEATURE_SOCKET_LISTEN
int network_listen(int type, char * addr, int port) {
#ifdef MUROAR_FEATURE_SOCKET_UNIX
 struct sockaddr_un   un;
#endif
#ifdef MUROAR_FEATURE_SOCKET_INET
 struct hostent     * he;
 struct sockaddr_in   in;
#endif
 int fh;
#ifdef MUROAR_DEFAULT_LISTEN_CHMOD
 mode_t defmask;
#endif

 switch (type) {
#ifdef MUROAR_FEATURE_SOCKET_UNIX
  case NETWORK_TYPE_UNIX:
    if ( addr == NULL )
     addr = MUROAR_DEFAULT_LISTEN_ADDR_UNIX;

    if ( (fh = socket(AF_UNIX, SOCK_STREAM, 0)) == -1 )
     return -1;

    un.sun_family = PF_UNIX;
    strncpy(un.sun_path, addr, sizeof(un.sun_path) - 1);

#ifdef MUROAR_DEFAULT_LISTEN_CHMOD
    defmask = umask(0000);

    if ( fchmod(fh, MUROAR_DEFAULT_LISTEN_CHMOD) == -1 ) {
     network_close(fh);
     return -1;
    }
#endif

    if ( bind(fh, (struct sockaddr *)&un, sizeof(struct sockaddr_un)) == -1 ) {
     if ( network_test_unix(addr) != 0 ) {
      network_close(fh);
      return -1;
     }

     if ( unlink(addr) == -1 ) {
      network_close(fh);
      return -1;
     }

     if ( bind(fh, (struct sockaddr *)&un, sizeof(struct sockaddr_un)) == -1 ) {
      network_close(fh);
      return -1;
     }
    }

#ifdef MUROAR_DEFAULT_LISTEN_CHMOD
    umask(defmask);
#endif

   break;
#endif
#ifdef MUROAR_FEATURE_SOCKET_INET
  case NETWORK_TYPE_INET:
    if ( addr == NULL )
     addr = MUROAR_DEFAULT_LISTEN_ADDR_INET;

    if ( (he = gethostbyname(addr)) == NULL ) {
     return -1;
    }

    if ( (fh = socket(AF_INET, SOCK_STREAM, 0)) == -1 )
     return -1;

    memcpy((struct in_addr *)&in.sin_addr, he->h_addr, sizeof(struct in_addr));

    in.sin_family = AF_INET;
    in.sin_port   = htons(port);

    if ( bind(fh, (const struct sockaddr *)&in, sizeof(in)) == -1 ) {
     network_close(fh);
     return -1;
    }
   break;
#endif
  default:
    return -1;
   break;
 }

 if ( listen(fh, 8) == -1 ) {
  network_close(fh);
 }

 g_listen_socket = fh;

 return 0;
}
#endif

int network_nonblock(int fh) {
#ifndef __WIN32
 int flags;

 if ( (flags = fcntl(fh, F_GETFL, 0)) == -1 )
  return -1;

 flags |= O_NONBLOCK;

 return fcntl(fh, F_SETFL, flags);
#else
 return 0;
#endif
}

int network_check(void) {
 struct timeval tv;
 fd_set rfds;
 void * data;
 int ret;
 int fh;
 int i;
 int max_fh = -1;

 tv.tv_sec  = 0;
 tv.tv_usec = 1;

 FD_ZERO(&rfds);

#ifdef MUROAR_FEATURE_SOCKET_LISTEN
 if ( g_listen_socket != -1 ) {
  FD_SET(g_listen_socket, &rfds);
  max_fh = g_listen_socket;
 }
#endif

 // TODO: should we maybe only check input (PLAY,...) streams?

 for (i = 0; i < MUROAR_MAX_STREAMS; i++) {
  if ( stream_get_datadir(i) == STREAM_DATADIR_IN ){
   if ( (fh = stream_get_sock(i)) != -1 ) {
    FD_SET(fh, &rfds);
    if ( fh > max_fh )
     max_fh = fh;
   }
  }
 }

 for (i = 0; i < MUROAR_MAX_CLIENTS; i++) {
  if ( (fh = client_get_sock(i)) != -1 ) {
   FD_SET(fh, &rfds);
   if ( fh > max_fh )
    max_fh = fh;
  }
 }

 if ( max_fh == -1 ) { // all IOs are gone, terminate the daemon
  g_alive = 0;
  return 0;
 }

 ret = select(max_fh + 1, &rfds, NULL, NULL, &tv);

 if ( ret == -1 )
  return -1;

 if ( ret == 0 )
  return 0;

#ifdef MUROAR_FEATURE_SOCKET_LISTEN
 if ( g_listen_socket != -1 ) {
  if ( FD_ISSET(g_listen_socket, &rfds) ) {
   fh = accept(g_listen_socket, NULL, NULL);
   client_new(fh);
  }
 }
#endif

 for (i = 0; i < MUROAR_MAX_STREAMS; i++) {
  if ( stream_get_datadir(i) == STREAM_DATADIR_IN ){
   if ( (fh = stream_get_sock(i)) != -1 ) {
    if ( FD_ISSET(fh, &rfds) ) {
     stream_read(i);
    } else {
     if ( (data = stream_get_iobuf(i)) != NULL )
      memset(data, 0, g_abuffer_size);
    }
   }
  }
 }

 for (i = 0; i < MUROAR_MAX_CLIENTS; i++) {
  if ( (fh = client_get_sock(i)) != -1 ) {
   if ( FD_ISSET(fh, &rfds) ) {
    client_handle(i);
   }
  }
 }

 return 0;
}

#ifdef MUROAR_FEATURE_CMD_PASSFH
#define _SCMR_CONTROLLEN (sizeof(struct cmsghdr) + sizeof(int))
int network_recvfh(int fh) {
 struct iovec     iov[1];
 struct msghdr    msg;
 char             cmptr_buf[_SCMR_CONTROLLEN];
 struct cmsghdr * cmptr = (struct cmsghdr *) cmptr_buf;
 char             localmes[1];

 iov[0].iov_base = localmes;
 iov[0].iov_len  = 1;
 msg.msg_iov     = iov;
 msg.msg_iovlen  = 1;
 msg.msg_name    = NULL;
 msg.msg_namelen = 0;

 msg.msg_control    = (caddr_t) cmptr;
 msg.msg_controllen = _SCMR_CONTROLLEN;

 if ( recvmsg(fh, &msg, 0) == -1 )
  return -1;

 if ( msg.msg_controllen != _SCMR_CONTROLLEN )
  return -1;

 return *(int *)CMSG_DATA(cmptr);
}
#endif

//ll
