# tags.py - Keeps a representation of the tags the program knows about.
#
#   Copyright (C) 2003 Daniel Burrows <dburrows@debian.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import gtk

# Information about known tags.
class Tag:
    # tag: the string associated with the tag (in dictionaries and so forth)
    # title: the string to be displayed for the tag.
    # validator: a function to check if the given string is valid for
    #            the given list item.  It should return None if the
    #            string is *VALID* and a string describing the problem
    #            otherwise.  (an alternative is to throw an exception
    #            for invalid tags, but that has other issues and is
    #            hard to get right)
    def __init__(self, tag, title, validator=lambda li,s:None):
        self.tag=tag
        self.title=title
        self.validator=validator

    # returns True if the input is ok.
    def validate(self, li, s):
        err=self.validator(li, s)
        if err==None:
            return True
        else:
            # FIXME: set the parent properly.
            dlg=gtk.MessageDialog(None,
                                  gtk.DIALOG_DESTROY_WITH_PARENT,
                                  gtk.MESSAGE_ERROR,
                                  gtk.BUTTONS_OK,
                                  err)
            dlg.connect('response', lambda *args:dlg.destroy())
            dlg.show_all()
            return False

def validate_tracknum(li, s):
    try:
        int(s)
        return None
    except ValueError:
        return 'The track number must be a whole number'

def validate_genre(li, s):
    if li.valid_genre(s):
        return None
    else:
        return 'The genre "%s" is not valid for all of the selected files.\nPerhaps it is not a known MP3 genre?'%s

# Tags that are known.  Indexed by strings to provide a
# reasonably simple way of referring to them in config files
# (eventually).  "title" is special, since it doubles as the
# label column (it can never be activated or deactivated)
#
# FIXME: uses lower-case when most stuff uses upper-case
known_tags={
    'album' : Tag('album', 'Album'),
    'artist' : Tag('artist', 'Artist'),
    'comment' : Tag('comment', 'Comment'),
    'genre' : Tag('genre', 'Genre', validate_genre),
    'title' : Tag('title', 'Title'),
    'tracknumber' : Tag('tracknumber', 'Track Number', validate_tracknum),
    'year' : Tag('year', 'Year')
}

