/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 fakepoll.h  -  Poll implementation using select() for those without poll()

 Original source provided by Greg Parker <gparker-darwindev@sealiesoftware.com>
 Warning: a call to this poll() takes about 4K of stack space.
 
    begin                : Sat Sep 1 2001
    copyright            : (C) 2001 by
    email:               : gblock@ctoforaday.com

 ***************************************************************************/

#ifndef _FAKE_POLL_H
#define _FAKE_POLL_H

#include <limits.h>

#ifndef FD_SETSIZE
#define FD_SETSIZE OPEN_MAX
#endif

#include <sys/types.h>
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

typedef struct pollfd {
    int fd;                         /* file desc to poll */
    short events;                   /* events of interest on fd */
    short revents;                  /* events that occurred on fd */
} pollfd_t;


// poll flags
#define POLLIN  0x0001
#define POLLOUT 0x0004
#define POLLERR 0x0008

// synonyms
#define POLLNORM POLLIN
#define POLLPRI POLLIN
#define POLLRDNORM POLLIN
#define POLLRDBAND POLLIN
#define POLLWRNORM POLLOUT
#define POLLWRBAND POLLOUT

// ignored
#define POLLHUP 0x0010
#define POLLNVAL 0x0020

inline int poll(struct pollfd *pollSet, int pollCount, int pollTimeout)
{
    struct timeval tv;
    struct timeval *tvp;
    fd_set readFDs, writeFDs, exceptFDs;
    fd_set *readp, *writep, *exceptp;
    struct pollfd *pollEnd, *p;
    int selected;
    int result;
    int maxFD;

    if (!pollSet) {
        pollEnd = NULL;
        readp = NULL;
        writep = NULL;
        exceptp = NULL;
        maxFD = 0;
    } 
    else {
        pollEnd = pollSet + pollCount;
        readp = &readFDs;
        writep = &writeFDs;
        exceptp = &exceptFDs;

        FD_ZERO(readp);
        FD_ZERO(writep);
        FD_ZERO(exceptp);
        
        // Find the biggest fd in the poll set
        maxFD = 0;
        for (p = pollSet; p < pollEnd; p++) {
            if (p->fd > maxFD) maxFD = p->fd;
        }
        
        // Transcribe flags from the poll set to the fd sets
        for (p = pollSet; p < pollEnd; p++) {
            if (p->events & POLLIN)  FD_SET(p->fd, readp);
            if (p->events & POLLOUT) FD_SET(p->fd, writep);
            if (p->events != 0)      FD_SET(p->fd, exceptp);
            // POLLERR is never set coming in; poll() always reports errors.
            // But don't report if we're not listening to anything at all.
        }
    }
        
    // poll timeout is in milliseconds. Convert to struct timeval.
    // poll timeout == -1 : wait forever : select timeout of NULL
    // poll timeout == 0  : return immediately : select timeout of zero
    if (pollTimeout >= 0) {
        tv.tv_sec = pollTimeout / 1000;
        tv.tv_usec = (pollTimeout % 1000) * 1000;
        tvp = &tv;
    } else {
        tvp = NULL;
    }
    
    
    selected = select(maxFD+1, readp, writep, exceptp, tvp);


    if (selected < 0) {
        // Error during select
        result = -1;
    } 
    else if (selected > 0) {
        // Select found something
        // Transcribe result from fd sets to poll set.
        // Also count the number of selected fds. poll returns the 
        // number of ready fds; select returns the number of bits set.
        int polled = 0;
        for (p = pollSet; p < pollEnd; p++) {
	    p->revents = 0;
	    if (FD_ISSET(p->fd, readp))   p->revents |= POLLIN;
	    if (FD_ISSET(p->fd, writep))  p->revents |= POLLOUT;
	    if (FD_ISSET(p->fd, exceptp)) p->revents |= POLLERR;
	    if (p->revents) polled++;
	}
        result = polled;
    }
    else {
	// selected == 0, select timed out before anything happened
        // Clear all result bits and return zero.
        for (p = pollSet; p < pollEnd; p++) {
            p->revents = 0;
        }
        result = 0;
    }

    return result;
}


#endif
