/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 gnusock.cpp  -  Socket handler for the Gnutella network protocol
 
 the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)
 
    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "structures.h"

#include "asyncsocket.h"
#include "packet.h"

#include "rcobject.h"
#include "gnudirector.h"
#include "gnuupload.h"
#include "gnunode.h"
#include "gnusock.h"
#include "conversions.h"
#include "property.h"
#include "preferences.h"

#include <unistd.h>
#include <sys/ioctl.h>

/////////////////////////////////////////////////////////////////////////////
// MGnuSock

MGnuSock::MGnuSock(MGnuDirector* pComm)
{	
	m_pDirector  = pComm;
	m_pPrefs = m_pDirector->GetPrefs();

	m_nSecsAlive = 0;
	m_bDestroy   = false;
	//printf("sock created\n");
	m_dwBytesReceived = 0;
}

MGnuSock::~MGnuSock()
{
	//printf("sock destroed\n");
}

/////////////////////////////////////////////////////////////////////////////
// MGnuSock member functions

void MGnuSock::OnClose(int nErrorCode)
{
	//printf("MGnuSock::OnClose; errno=%d\n", errno);
	ForceDisconnect();
}

#define DEF_BUF_SIZE  4096
#define MAX_BUF_SIZE  16384
#define MAX_HANDSHAKE 16384

void MGnuSock::OnReceive(int nErrorCode) 
{
	int nBufferRequired;
	#ifdef FIONREAD
	if ( ::ioctl(m_hSocket, FIONREAD, (char*)&nBufferRequired) >=0  && nBufferRequired >0 )
	{
		if (nBufferRequired>MAX_BUF_SIZE)
			nBufferRequired>MAX_BUF_SIZE;
	}
	else
		nBufferRequired=DEF_BUF_SIZE;
	#else
	nBufferRequired=DEF_BUF_SIZE;
	#endif
	//
	char* pBuf = (char*) alloca(nBufferRequired+1); //MZ: need this for the trailing '\0' to make strstr() work properly
	ASSERT(pBuf);
	
	//printf("sock on receive\n");
	DWORD dwReceived = Receive(pBuf, nBufferRequired);
	
	switch (dwReceived)
	{
	case 0:
		//printf("MGnuSock::OnReceive: zero-size; errno=%d\n", errno);
		ForceDisconnect();
		return;
	case SOCKET_ERROR:
		ForceDisconnect();
		return;
	}
	// add a trailing '\0'
	ASSERT(dwReceived>0);
	ASSERT(dwReceived<=nBufferRequired);
	pBuf[dwReceived]='\0';
	// keep trace of the size of handshake
	m_dwBytesReceived += dwReceived;
	//
	bool bTerminated = false;
	char* pRest;
	char* pRNRN = NULL;
	char* pNN = NULL;
	// ckeck for the ternimation combination which is either "\r\n\r\n" or "\n\n"
	int nHlen = m_sHandshake.length();
	// complicated situation when there is "\r\n" sent now and "\r\n" in the tail of the handshake already
	if ( dwReceived>=2 && nHlen>=2 && pBuf[0]=='\r' && pBuf[1]=='\n' && m_sHandshake[nHlen-2]=='\r' && m_sHandshake[nHlen-1]=='\n' )
	{
		m_sHandshake += "\r\n";
		pRest = pBuf + 2;
		bTerminated = true;
	}
	else
	// same complicated sitiation when there is "\n" in the buffer and handshake tails with either "\n" or "\r\n\r"
	if ( dwReceived>=1 && pBuf[0]=='\n' &&
		 ( (nHlen>=3 && m_sHandshake[nHlen-3]=='\r' && m_sHandshake[nHlen-2]=='\n' && m_sHandshake[nHlen-1]=='\r') ||
		   (nHlen>=1 && m_sHandshake[nHlen-1]=='\n') ) )
	{
		m_sHandshake += "\n";
		pRest = pBuf + 1;
		bTerminated = true;
	}
	else
	// less complicated case of "\r\n\r\n" in the buffer
	if (pRNRN=strstr(pBuf, "\r\n\r\n"))
	{
		m_sHandshake += CString(pBuf, pRNRN-pBuf+4);
		pRest = pRNRN+4;
		bTerminated = true;
	}
	else
	// even less complicated case of "\n\n" in the buffer
	if (pNN=strstr(pBuf, "\n\n"))
	{
		m_sHandshake += CString(pBuf, pNN-pBuf+2);
		pRest = pNN+2;
		bTerminated = true;
	}
	// looks like this is it
	if (bTerminated)
	{
		DWORD dwRestSize = dwReceived - (pRest-pBuf);
		if ((pRest-pBuf) >= dwReceived)
		{
			ASSERT(dwRestSize == 0); // '>' should never happen
			pRest = NULL;
			dwRestSize = 0;
		}
		// we are preared to parse m_sHandshake
		ParseHandshake(pRest, dwRestSize);
	}
	else
	{
		// check if we are not receiving something suspicios
		if (m_dwBytesReceived >= MAX_HANDSHAKE)
		{
			ForceDisconnect();
			return;
		}
		// save received bytes for the next time
		m_sHandshake += CString(pBuf, dwReceived);
	}
	
	MAsyncSocket::OnReceive(nErrorCode);
}

void MGnuSock::ParseHandshake(char* pBuffRest, DWORD dwRestSize)
{
	// Get Connected's info -- we will need it in multiple places
	CString sHost;
	UINT    nPort = 0;
	IP      ipHost = MakeIP(0,0,0,0);
	GetPeerName(sHost, nPort);
	Str2Ip(sHost, ipHost);
	//////////////////
	// New Connection
	if(m_sHandshake.find("CONNECT/0.") != -1)
	{
		m_pDirector->SetReceivedIncomingConnections();
		// Check for duplicate connections
		if(m_pDirector->FindNode(ipHost, 0) != NULL)
		{
			ForceDisconnect();
			return;
		}
		// Create a new node connection
		MGnuNode* pNodeSock		= new MGnuNode(m_pDirector, ipHost, 0, true, false);
		ASSERT(pNodeSock);
		pNodeSock->m_nStatus	= SOCK_NEGOTIATING;

		// Convert this socket to a Node Socket
		SOCKET hFreeSock = Detach();
		pNodeSock->Attach(hFreeSock, FD_READ | FD_CLOSE);

		m_pDirector->AddNode(pNodeSock);

		#warning by this we break compatibility with 0.4 clients
		if(!pNodeSock->ParseIncomingHandshake06(m_sHandshake, (BYTE*)pBuffRest, dwRestSize, false))
			pNodeSock->ForceDisconnect();

		ForceDisconnect();
		return;
	}
	/////////////////
	// Upload Request
	else if(m_sHandshake.find("GET /get/") == 0                  ||
			m_sHandshake.find("GET /uri-res/N2R?urn:sha1:") == 0 )
	{
		// Create a new upload connection
		MGnuUpload* pUploadSock = new MGnuUpload(m_pDirector);
		ASSERT(pUploadSock);
		//
		m_pDirector->AddUpload(pUploadSock);
		//
		pUploadSock->m_mutex.lock();
		// Set Variables
		pUploadSock->m_ipHost      = ipHost;
		pUploadSock->m_nPort       = nPort;
		pUploadSock->SetHandshake(m_sHandshake);
		pUploadSock->m_nStatus     = TRANSFER_CONNECTED;
		// Convert this socket to a Node Socket
		SOCKET hFreeSock = Detach();
		pUploadSock->Attach(hFreeSock, FD_READ | FD_CLOSE);
		//
		pUploadSock->UploadFile(m_sHandshake, false);
		//
		pUploadSock->m_mutex.unlock();
		//
		
		ForceDisconnect();
		return;
	}
	////////////////
	// Incoming Push
	else if(m_sHandshake.find("GIV ") == 0)
	{
		DWORD	nIndex;
		CString sFileName;

		// TODO: Get GUID and check also
		sscanf( m_sHandshake.c_str(), "GIV %ld:*/*\r\n", &nIndex);

		int nFront = m_sHandshake.find("/") + 1;
		int nEnd   = m_sHandshake.find("\n\n"); // MZ: ???? IS THIS "\n\n" right? but it seem to work!
		if (nEnd < 0)
			nEnd   = m_sHandshake.find("\r\n\r\n"); // give it another try
		sFileName  = m_sHandshake.substr(nFront, nEnd - nFront);
		MakeLower(sFileName);
		//
		SOCKET hPushSock = Detach();
		// inform Director about the push
		if (!m_pDirector->OnIncomingPush(sFileName, nIndex, hPushSock))
		{
			// nobody seem to be interested in this connection
			::close(hPushSock);
		}
		ForceDisconnect();
	}
	///////////////////
	// HTML GET request
	else if(m_sHandshake.find("GET /") == 0 || m_sHandshake.find("POST /") == 0)
	{
		// extract the path
		int nPos = m_sHandshake.find("\r\n");
		CString sLine;
		if (nPos>=0)
			sLine = m_sHandshake.substr(0, nPos);
		else
			sLine = m_sHandshake;
		nPos = sLine.find("HTTP/1");
		CString sPath;
		if (nPos>=0)
			sPath = StripWhite(sLine.substr(4, nPos-4));
		else
			sPath = StripWhite(sLine.substr(4));
		// now we are going to make somewhat stupid manipulation:
		// first we detach the file descriptor, but
		// if MGnuDirector says there is nobody to process this
		// request we will attach it back to sent the page from below
		SOCKET hSocket = Detach();
		//
		if (m_pDirector->OnExternalHttpRequest(ipHost, sPath.c_str(), m_sHandshake.c_str(), pBuffRest,  hSocket))
		{
			ForceDisconnect();
			return;
		}
		//
		Attach(hSocket, FD_READ | FD_CLOSE);
		//
		CString Http200 =  "HTTP/1.0 200\r\n";
				Http200 += "Server: Mutella\r\n";
				Http200 += "Content-type:text/html\r\n\r\n";
				Http200 += "<HTML>\r\n";
				Http200 += "<HEAD><TITLE>Gnutella Network</TITLE></HEAD>\r\n";
				Http200 += "<BODY>\r\n";
				Http200 += "<H1>Gnutella</H1>\r\n";
				Http200 += "This is not a web server, it's a Gnutella-network node.\r\n";
				Http200 += "<br><smaller>Requested URL: " + sPath + "</smaller>\r\n";
				Http200 += "<HR NOSHADE SIZE=1>\r\n";
				Http200 += "Powered by <B><A HREF=\"http://mutella.sourceforge.net\">Mutella ";
				Http200 += VERSION;
				Http200 += "</A></B>.\r\n";
				Http200 += "</BODY></HTML>\r\n\r\n";
		
		Send(Http200.c_str(), Http200.length());
		Close();
	}
	// seem to be something we dont know about
	ForceDisconnect();
}

void MGnuSock::ForceDisconnect()
{
	// Tag for cleanup
	m_bDestroy = true;
	//
	if(m_hSocket != INVALID_SOCKET)
	{
		AsyncSelect(FD_CLOSE);
		ShutDown(2);
	}

	Close();
}

// Called every once a second
void MGnuSock::OnTimer()
{
	m_nSecsAlive++;
}

