// datamodifier.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "localdefs.h"
#include "controller.h"
#include "datamodifier.h"
#include "data.h"
#include "editor.h"
#include "crossfader.h"
#include "formatrequester.h"
#include "lpcdata.h"
#include "processfun.h"
#include "range.h"
#include "request.h"
#include "sound.h"
#include "valuerequester.h"

#undef debug

DataModifier::DataModifier(Data* data, bool needUndo)
		: myData(data), myUndoBuffer(nil), _needUndoBuffer(needUndo), 
		  myEditor(nil)
{
	myData->ref();
}

DataModifier::DataModifier(Data* data, DataEditor* editor, bool needUndo)
		: myData(data), myUndoBuffer(nil), _needUndoBuffer(needUndo), 
		  myEditor(editor)
{
	myData->ref();
}

DataModifier::~DataModifier() {
	Resource::unref(myData);
	Resource::unref(myUndoBuffer);
}

int
DataModifier::saveUndoBuffer(Data *data)
{
	int status = true;
	// Dont go to all this work if we are not saving undo.
	if ((int) Application::getGlobalResourceValue("UndoDepth") != 0) {
		boolean was = Data::deferRescan(true);
		data->ref();
#ifdef debug
		printf("datamodifier with message '%s' is creating undo buffer\n", message());
#endif
		// This can fail due to memory limitations
		myUndoBuffer = data->copyOf();
		Data::deferRescan(was);
		if (myUndoBuffer)
			myUndoBuffer->ref();
		else if (Application::confirm("Unable to save undo buffer.", 
									  "This operation will not be reversable.", 
									  "Continue?",
									  Cancel) == false)
		{
			status = false;
		}
		data->unref();
	}
	return status;
}

// This is for use by derived class which need to override the default value
//	for needing an undo buffer.  Default is no-op.

Status
DataModifier::doConfigure(Controller *)
{
	return Succeed;
}

Status 
DataModifier::configure(Controller *c) {
	Status status = Super::configure(c) && doConfigure(c);
	if (status && _needUndoBuffer) {
		Application::inform("Saving undo buffer...");
		status = saveUndoBuffer(myData);
	}
	return status;
}

// The default undo operation for all DataModifiers is to replace the
//	modified section with a saved copy of the original data.

Modifier *
DataModifier::createUndo() {
	return (getUndoBuffer() != nil) ? 
		new Replacer(target(), getUndoBuffer())  : nil;
}

int
DataModifier::apply() {
	int status = false;
	if ((status = ok())) {
		Application::inform(message());
		status = doApply(myData);
		if (status)
			doPostModifyCommands();
	}
	return status;
}

void
DataModifier::doPostModifyCommands() {
	showProgress(1.0);	// This makes sure the progress bar dissappears	
	if(myEditor != nil)
		doEditorCommands(myEditor);
}

int
DataModifier::sampRate() { return myData->sRate(); }

//********

Modifier *
Reverser::create(DataEditor* ed) {
	return new Reverser(ed->currentSelection());
}

int
Reverser::doApply(Data* data) {
	data->reverse();
	return true;
}

Modifier *
Reverser::createUndo() {
	return new Reverser(target());
}

//********

Modifier *
ByteSwapper::create(DataEditor* ed) {
	return new ByteSwapper(ed->currentSelection());
}

int
ByteSwapper::doApply(Data* data) {
	data->swapBytes();
	return true;
}

Modifier *
ByteSwapper::createUndo() {
	return new ByteSwapper(target());
}

//********

class TimeInsertRequester : public TitledRequester {
	friend class SpaceInsert;
protected:
	TimeInsertRequester(SpaceInsert *);
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	SpaceInsert* client;
	TimeVal timeToInsert;
};

TimeInsertRequester::TimeInsertRequester(SpaceInsert* s)
	: TitledRequester("Insert Blank Space:"),
	  client(s), timeToInsert(client->_nFrames / client->sampRate()) {}

void
TimeInsertRequester::configureRequest(Request* request) {
	request->appendValue("Amount To Insert (MM:SS):", &timeToInsert);
}

boolean
TimeInsertRequester::confirmValues() {
	client->_nFrames = iround(timeToInsert * client->sampRate());
	return true;
}

//********

int SpaceInsert::_savedFrames = 1;

SpaceInsert::SpaceInsert(Data* d, int nframes)
		: DataModifier(d, false), _nFrames(nframes), _insertTime(false) {
	initialize();
}

SpaceInsert::SpaceInsert(Data* d, boolean asTime)
	: DataModifier(d, false), _nFrames(d->length()), _insertTime(asTime) {}

Modifier *
SpaceInsert::create(DataEditor* ed) {
	return new SpaceInsert(ed->currentSelection());
}

Requester *
SpaceInsert::createRequester() {
	if(_insertTime)
		return new TimeInsertRequester(this);
	else
		return new RangedValueRequester<int>(
			"Insert Blank Space:", "Amount To Insert (frames):",
			this->_nFrames, NonNegativeIntegers);
}

void
SpaceInsert::saveConfig() {
    _savedFrames = _nFrames;
}

int
SpaceInsert::doApply(Data *data) {
	Range edit(0, _nFrames);
	boolean was = Data::deferRescan(true);	// inserting zeros so no rescan
	int status = data->spliceIn(edit);
	Data::deferRescan(was);
	return status;
}

Modifier *
SpaceInsert::createUndo() {
	return new OutSplicer(target(), Range(0, _nFrames));
}

//********

OutSplicer::OutSplicer(Data* d, const Range& r)
		: DataModifier(d, false), _spliceRegion(r) {
	initialize();
}

Modifier *
OutSplicer::create(DataEditor* de) {
	return nil;
}

// Overridden so we can save our special undo buffer

Status 
OutSplicer::configure(Controller *c) {
	Status status = Super::configure(c);
	if (status) {
		Application::inform("Saving undo buffer...");
		status = saveUndoBuffer(target()->clone(_spliceRegion));
	}
	return status;
}

int
OutSplicer::doApply(Data* data) {
	if(_spliceRegion == data->frameRange()) {
		Application::alert("Entire file may not be spliced out.",
						   "Select only a portion of the file.");
		return false;
	}
	else
		return data->spliceOut(_spliceRegion);
}

Modifier *
OutSplicer::createUndo() {
	return (getUndoBuffer() != nil) ? 
		new Splicer(target(),
					getUndoBuffer(), 
					Range(_spliceRegion.intMin(), _spliceRegion.intMax()+1))
		: nil;
}

//********

class LengthChangeRequester : public TitledRequester {
	friend class LengthChanger;
protected:
	LengthChangeRequester(const char* title, LengthChanger* l)
		: TitledRequester(title), client(l) {}
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
protected:
	LengthChanger* client;
};

void
LengthChangeRequester::configureRequest(Request* request) {
	Range lengths = PositiveIntegers;
	lengths.set(8, lengths.intMax());	// min length 8
	request->appendValue("New Length (in frames):",
	                     &client->newLength, lengths);
	request->appendLabel("(Minimum duration is 8 frames)");
}

boolean
LengthChangeRequester::confirmValues() {
	boolean status = false;
	status = (client->newLength >= client->oldLength ||
		Application::confirm(
			"This operation will destroy any data past",
			"the new file endpoint, and is not reversible.",
			"Continue?",
			Cancel)
	);
	return status;
}

//********

class DurChangeRequester : public LengthChangeRequester {
	friend class LengthChanger;
protected:
	DurChangeRequester(const char* title, LengthChanger *);
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	TimeVal newDuration;
};

DurChangeRequester::DurChangeRequester(const char* title, LengthChanger* client)
		: LengthChangeRequester(title, client) {
	newDuration = double(client->newLength)/client->sampRate();
}

void
DurChangeRequester::configureRequest(Request* request) {
	Range durations = AllNumbers;
	durations.set(8.0/client->sampRate(), durations.max());
	request->appendValue("New Duration (MM:SS):", &newDuration,
			     durations);
	char label[64];
	sprintf(label, "(Minimum duration is %g seconds)", durations.min());
	request->appendLabel(label);
}

boolean
DurChangeRequester::confirmValues() {
	client->newLength = iround(newDuration * client->sampRate());
	return LengthChangeRequester::confirmValues();
}

//********

int LengthChanger::_savedNewLength = 0;

LengthChanger::LengthChanger(Data* d, boolean asDur)
	: DataModifier(d, false),
	  oldLength(d->length()),
	  newLength(_savedNewLength == 0 ? oldLength : _savedNewLength),
	  asDuration(asDur) {}

LengthChanger::LengthChanger(Data* d, int newlen)
		: DataModifier(d, false),
		  oldLength(d->length()), newLength(newlen), asDuration(false) {
	initialize();
}

Modifier *
LengthChanger::create(DataEditor* ed) {
	return new LengthChanger(ed->model());
}

Requester *
LengthChanger::createRequester() {
	if(asDuration)
		return new DurChangeRequester("Change File Duration:", this);
	else
		return new LengthChangeRequester("Change File Length:", this);
}

void
LengthChanger::saveConfig() {
    _savedNewLength = newLength;
}

Status
LengthChanger::doConfigure(Controller *) {
	setNeedUndo(newLength < oldLength);
	return Succeed;
}

int
LengthChanger::doApply(Data *data) {
	int status = data->changeLength(newLength);
	if(status == true) {
		// defer rescan if increasing length since extending with zeros
		boolean was = Data::deferRescan(newLength >= oldLength);
		data->Notify();		// data does not automatically do this 
		Data::deferRescan(was);
	}
	return status;
}

Modifier *
LengthChanger::createUndo() {
	return (newLength > oldLength) ?
		new LengthChanger(target(), oldLength) : Super::createUndo();
}

//********

class SampRateRequester : public TitledRequester {
	friend class SampRateChanger;
protected:
	SampRateRequester(SampRateChanger *s)
		: TitledRequester("Change Sampling Rate:"), client(s) {}
	redefined void configureRequest(Request *);
private:
	SampRateChanger* client;
};

void
SampRateRequester::configureRequest(Request* request) {
	request->appendValue("New Sampling Rate:",
	                     &client->newSrate, PositiveIntegers);
	if(client->target()->isA(Sound_Data))
		request->appendChoice(
			"Interpolate sound to preserve pitch level:",
			"|No|Yes|", &client->doInterp, true);
	else client->doInterp = False;
}

//********

int SampRateChanger::_savedNewSrate = 0;
ChoiceValue SampRateChanger::_savedDoInterp = Requester::True;

SampRateChanger::SampRateChanger(Data* d)
	: DataModifier(d, false),
	  newSrate(_savedNewSrate == 0 ? sampRate() : _savedNewSrate),
	  oldSrate(sampRate()),
	  doInterp(_savedDoInterp) {}
		
SampRateChanger::SampRateChanger(Data* d, int newsrate, boolean interp)
		: DataModifier(d, false), newSrate(newsrate), oldSrate(sampRate()),
		doInterp(interp ? Requester::True : Requester::False) {
	initialize();
}

Modifier *
SampRateChanger::create(DataEditor* ed) {
	return new SampRateChanger(ed->model());
}

Requester *
SampRateChanger::createRequester() {
	return new SampRateRequester(this);
}

void
SampRateChanger::saveConfig() {
    _savedNewSrate = newSrate;
    _savedDoInterp = doInterp;
}

Status 
SampRateChanger::doConfigure(Controller *) {
	setNeedUndo(doInterp == Requester::True);
	return Succeed;
}

int
SampRateChanger::doApply(Data *data) {
	return data->changeSRate(newSrate, doInterp == Requester::True);
}

Modifier *
SampRateChanger::createUndo() {
	// First we must convert back to old rate, then replace contents
	Modifier *undoRateChange = new SampRateChanger(target(), oldSrate, false);
	return (doInterp == Requester::True) ? 
			new ModifierChain(undoRateChange, Super::createUndo()) :
			undoRateChange;
}

//********

class FormatChangeRequester : public FormatRequester {
	friend class FormatChanger;
protected:
	FormatChangeRequester(FormatChanger *);
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	FormatChanger* client;
};

FormatChangeRequester::FormatChangeRequester(FormatChanger* f)
	: FormatRequester(nil, "Convert format to:", f->newType),
	  TitledRequester("Change Sound Sample Format:"),
	  client(f) {}

void
FormatChangeRequester::configureRequest(Request* request) {
	FormatRequester::configureRequest(request);
	request->appendChoice(
		"Rescale source to fit target range:",
		"|No|Yes|", &client->_rescaleFirst, true);
}

boolean
FormatChangeRequester::confirmValues() {
	boolean status = true;
	if(client->newType == client->oldType) {
		Application::alert("Samples are already in this format.");
		status = false;
	}
	return status;
}

//********

ChoiceValue FormatChanger::_savedRescaleFirst = Requester::True;

FormatChanger::FormatChanger(Data* d)
		: DataModifier(d, true),
		oldType(d->dataType()), _rescaleFirst(_savedRescaleFirst)
{
	newType = (oldType == ShortData) ? FloatData : ShortData;
}
		
FormatChanger::FormatChanger(Data* d, DataType newtype)
		: DataModifier(d, false),
		  oldType(d->dataType()), newType(newtype),
		  _rescaleFirst(_savedRescaleFirst)
{
	initialize();
}

Modifier *
FormatChanger::create(DataEditor* ed) {
	return new FormatChanger(ed->model());
}

Requester *
FormatChanger::createRequester() {
	return new FormatChangeRequester(this);
}

int
FormatChanger::doApply(Data *data) {
	return ((Sound *) data)->changeDataType(DataType(newType), _rescaleFirst);
}

Modifier *
FormatChanger::createUndo() {
	// First we must convert back to old format, then replace contents
	return new ModifierChain(new FormatChanger(target(), DataType(oldType)),
							 Super::createUndo());
}

void
FormatChanger::saveConfig() {
    _savedRescaleFirst = _rescaleFirst;
}

//********

class AutoCorrelateRequester : public TitledRequester {
	friend class AutoCorrelator;
protected:
	AutoCorrelateRequester(AutoCorrelator* a)
		: TitledRequester("Auto-correlate Data (remove glitches):"), _client(a) {}
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
protected:
	AutoCorrelator* _client;
};

void
AutoCorrelateRequester::configureRequest(Request* request) {
	request->appendValue("Deviation threshold:",
	                     &_client->_slopeThreshold, NonNegativeNumbers);
	request->appendValue("Number of frames to look ahead:",
	                     &_client->_lookAhead, Range(4, 64), True);
}

boolean
AutoCorrelateRequester::confirmValues() {
	boolean status = true;
	if (_client->_slopeThreshold == 0.0) {
		Application::alert("Threshold must be factor greater than zero.");
		status = false;
	}
	if (_client->_lookAhead >= _client->target()->length() / 2) {
		Application::alert("Selected area too short for this look-ahead.",
						   "Extend your selected area to the right");
		status = false;
	}
	return status;
}

//********

float AutoCorrelator::_savedSlopeThreshold = 5.0;
int AutoCorrelator::_savedLookAhead = 4;

AutoCorrelator::AutoCorrelator(Data* d, float slopeThreshold, int lookAhead)
	: DataModifier(d, false),
	  _slopeThreshold(slopeThreshold), _lookAhead(lookAhead)
{
	initialize();
}

AutoCorrelator::AutoCorrelator(Data* d)
	: DataModifier(d, true),
	_slopeThreshold(_savedSlopeThreshold), _lookAhead(_savedLookAhead)
{
}

Modifier *
AutoCorrelator::create(DataEditor *ed)
{
	return new AutoCorrelator(ed->model());
}

Modifier *
AutoCorrelator::createUndo()
{
	return new Replacer(target(), getUndoBuffer());
}

Requester *
AutoCorrelator::createRequester()
{
	return new AutoCorrelateRequester(this);
}

void
AutoCorrelator::saveConfig() {
    _savedSlopeThreshold = _slopeThreshold;
    _savedLookAhead = _lookAhead;
}

int
AutoCorrelator::doApply(Data *data)
{
	const int frameSize = _lookAhead + 1;
	double *frame = new double[frameSize * 2];
	Data *dataFrame;
	double sum = 0.0;
	int count = 0;
	int endFrame = data->length() - 1;
	Range segment(0, 2 * frameSize - 1);
	while (segment.intMax() <= endFrame)
	{
//		printf("cloning segment %d, %d\n", segment.intMin(), segment.intMax());
		dataFrame = data->clone(segment);
		int got = dataFrame->getArray(frame, frameSize * 2);
//		printf("getArray returned %d out of %d frames\n", got, frameSize * 2);
		int toScan = min(frameSize, got);
		int lookEnd = got;
//		printf("scanning segment %d, %d\n", segment.intMin(), segment.intMin() + toScan - 1);
		for (int n = 0; n < toScan - 1; ++n, count)
		{
			++count;
			double slopeAverage = sum / (count - 1);
			double cur = frame[n];
			double next = frame[n+1];
			double absSlope = fabs(next - cur) / cur;
//			printf("\tpair %d: [%g, %g] slopeAverage: %g  current slope: %g\n",
//				   n, cur, next, slopeAverage, absSlope);
			if (slopeAverage > 0.0 && absSlope / slopeAverage > _slopeThreshold)
			{
				int lastGoodIndex = n;
				int idx;
//				printf("\tlooking ahead from slot %d.  Last good value: %g\n",
//					  lastGoodIndex + 1, cur);
				// We hit a glitch -- search ahead for a good sample.
				for (idx = n+2; idx < lookEnd; ++idx)
				{
					// The threshold against which we test gets wider the farther 
					// we have to look ahead.
					double lookThreshold = _slopeThreshold * double(idx - n) / 2.0;
					double lookValue = frame[idx];
					double testSlope = fabs(lookValue - cur) / cur;
//					printf("\t\tpair %d: [%g, %g] testSlope: %g  threshold: %g\n",
//					       idx, cur, lookValue, testSlope, lookThreshold);
					if (testSlope / slopeAverage < lookThreshold)
					{
						// OK, found a good one.  Now adjust all samples
						// between lastGoodIndex and where we are now.
//						printf("\t\tfilling between slot %d [%g] and slot %d [%g]\n",
//							   n, cur, idx, lookValue);
						for (int x = n+1; x < idx; ++x, ++count)
						{
							double fraction = double(x - n) / (idx - n);
							double newVal = cur + fraction * (lookValue - cur);
//							printf("\t\t\tresetting slot %d with %g\n",
//								   x, newVal);
							dataFrame->set(newVal, x);
						}
						sum += testSlope;
//						printf("\t\tbreaking to outer loop.\n");
						break;
					}
				}
				if (idx == lookEnd)
				{
					Application::alert("Lookahead failed to find target within slope range.",
									   "Run again with larger lookahead or slope factor");
					delete [] frame;
					data->Notify();
					return Succeed;
				}
				n = idx - 1;	// Overlap the return to the pair search by one
			}
			else
			{
				sum += absSlope;
			}
		}
		Resource::unref(dataFrame);
		segment += (frameSize - 1);		// each scan overlaps last by one frame
	}
	delete [] frame;
	data->Notify();
	return Succeed;
}

//********

Modifier *
FrameStabilizer::create(DataEditor* ed) {
	return new FrameStabilizer(ed->model());
}

int
FrameStabilizer::applyToLPCData(LPCData* lpc) {
	lpc->stabilizeFrames();
	return true;
}
