/* Copyright (C) 2003 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include "myx_admin_library.h"

#define USER_INFO_TABLENAME "user_info"

/*
 * public functions definitions
 */

MYX_USER_NAMES * myx_get_user_names(MYSQL *mysql)
{
  MYSQL_RES* res;

  //Get Total number of users and hosts
  //to allocate one block of memory
  const char *query= "SELECT DISTINCT User FROM mysql.user ORDER BY User";
  if (myx_mysql_query(mysql,query) || !(res= mysql_store_result(mysql)))
  {
    return NULL;
  }
  else
  {
    MYSQL_ROW row;
    MYX_USER_NAMES *user_names= g_malloc0(sizeof(MYX_USER_NAMES));
    char ** user_name;

    user_names->user_names_num= (int)mysql_num_rows(res);
    user_name= user_names->user_names=
                        g_malloc0(sizeof(char *)*user_names->user_names_num);

    for ( ; (row= mysql_fetch_row(res)); user_name++)
      *user_name= myx_convert_dbstr_utf8(mysql, row[0]);
    mysql_free_result(res);
    return user_names;
  }
}

int myx_free_user_names(MYX_USER_NAMES *user_names)
{
  unsigned int i;

  for(i= 0;i<user_names->user_names_num;i++)
  {
    g_free(user_names->user_names[i]);
  }
  g_free(user_names->user_names);
  g_free(user_names);

  return 0;
}

char * q_create_user_info_base=
  "CREATE TABLE mysql." USER_INFO_TABLENAME " ("
  " User VARCHAR(16) NOT NULL,"
  " Full_name VARCHAR(60),"
  " Description VARCHAR(255),"
  " Email VARCHAR(80),"
  " Contact_information TEXT,"
  " Icon BLOB,"
  " PRIMARY KEY(User),"
  " INDEX user_info_Full_name(Full_name)"
  ")"
  " TYPE=MYISAM"
  " COMMENT='Stores additional user information'";

char * q_create_user_info_4_1=
  "CREATE TABLE mysql." USER_INFO_TABLENAME " ("
  " User VARCHAR(16) NOT NULL,"
  " Full_name VARCHAR(60),"
  " Description VARCHAR(255),"
  " Email VARCHAR(80),"
  " Contact_information TEXT,"
  " Icon BLOB,"
  " PRIMARY KEY(User),"
  " INDEX user_info_Full_name(Full_name)"
  ")"
  " TYPE=MYISAM"
  " COMMENT='Stores additional user information'"
  " CHARACTER SET UTF8";

int myx_check_mysql_user_info_table(MYSQL *mysql)
{
  MYSQL_RES* res;
  int result;
  char * old_schema;

  if (!use_schema_store_old_one(mysql,"mysql",&old_schema))
    return 0;

  if (!(res= mysql_list_tables(mysql, USER_INFO_TABLENAME)))
  {
    result= 0;
  }
  else if (mysql_num_rows(res))
  { 
    result= 1;
  }
  else //If the USER_INFO_TABLE does not exist, create it
  {
    //Create table
    const char *query= mysql_version_is_later_or_equal_than(mysql,4,1)
                       ? q_create_user_info_base : q_create_user_info_4_1;
    mysql_free_result(res);
    result= myx_mysql_query(mysql,query) ? 0 : 1;
  }

  restore_old_schema(mysql,old_schema);

  return result;
}

typedef enum query_variants
{
  q_v_base= 0,
  q_v_4_1= 1,
  q_v_5_0= 2,
  q_v_count
} QUERY_VARIANTS;

char * q_select_user_info[q_v_count]=
{
  "SELECT Full_name, Description, Email, Contact_information, Icon "
  "FROM mysql." USER_INFO_TABLENAME " "
  "WHERE User=@goal_user"
  ,
  "SELECT Full_name, Description, Email, Contact_information, Icon "
  "FROM mysql." USER_INFO_TABLENAME " "
  "WHERE cast(cast(User AS BINARY) AS CHAR CHARACTER SET utf8)=cast(cast(@goal_user AS BINARY) AS CHAR CHARACTER SET utf8)"
  ,
  "SELECT Full_name, Description, Email, Contact_information, Icon "
  "FROM mysql." USER_INFO_TABLENAME " "
  "WHERE cast(cast(User AS BINARY) AS CHAR CHARACTER SET utf8)=cast(cast(@goal_user AS BINARY) AS CHAR CHARACTER SET utf8)"
};

#define WHERE_V_40()\
 " WHERE User=@goal_user"

#define WHERE_V_41()\
 " WHERE cast(cast(User AS BINARY) AS CHAR CHARACTER SET utf8)=cast(cast(@goal_user AS BINARY) AS CHAR CHARACTER SET utf8)"

#define SELECT_USER_PRIV_40(name)\
 "select host as h, NULL as o, '" name "' as pn, "\
                                         name "  as pv from mysql.user"

#define SELECT_USER_PRIV_41(name)\
 "select "\
    "cast(cast(host AS BINARY) AS CHAR CHARACTER SET utf8) as h, "\
    "cast(cast(NULL AS BINARY) AS CHAR CHARACTER SET utf8) as o, "\
    "_utf8'" name "' as pn, "\
    "cast(cast("name " AS BINARY) AS CHAR CHARACTER SET utf8) as pv"\
    " from mysql.user"

#define SELECT_DB_PRIV_40(name)\
 "select host as h, db as o, '" name "' as pn, "\
                                      name "  as pv from mysql.db"

#define SELECT_DB_PRIV_41(name)\
 "select "\
        "cast(cast(host AS BINARY) AS CHAR CHARACTER SET utf8) as h, "\
        "cast(cast(db AS BINARY) AS CHAR CHARACTER SET utf8) as o, "\
        "_utf8'" name "' as pn, "\
        "cast(cast(" name " AS BINARY) AS CHAR CHARACTER SET utf8) as pv"\
        " from mysql.db"

#define SELECT_TABLE_PRIV_40(name)\
 "select host as h, concat(db,'.',table_name) as o, "\
        "'Table_priv_" name "' as pn, "\
        "if(find_in_set('" name "',Table_priv),'Y','N') as pv "\
        "from mysql.tables_priv"

#define SELECT_TABLE_PRIV_41(name)\
 "select "\
    "cast(cast(host AS BINARY) AS CHAR CHARACTER SET utf8) as h, "\
    "concat(cast(cast(db AS BINARY) AS CHAR CHARACTER SET utf8),"\
            "_utf8'.',"\
         "cast(cast(table_name AS BINARY) AS CHAR CHARACTER SET utf8)) as o, "\
    "_utf8'Table_priv_" name "' as pn, "\
    "if(find_in_set(_utf8'" name "', "\
        "cast(cast(Table_priv AS BINARY) AS CHAR CHARACTER SET utf8)),"\
          "_utf8'Y',_utf8'N') as pv "\
    "from mysql.tables_priv"

#define SELECT_TABLE_COLUMNS_PRIV_40(name)\
 "select host as h, concat(db,'.',table_name) as o, "\
                "'Column_priv_" name "' as pn, "\
                "if(find_in_set('" name "',Column_priv),'Y','N') as pv "\
                "from mysql.tables_priv"

#define SELECT_TABLE_COLUMNS_PRIV_41(name)\
 "select "\
        "cast(cast(host AS BINARY) AS CHAR CHARACTER SET utf8) as h, "\
        "concat(cast(cast(db AS BINARY) AS CHAR CHARACTER SET utf8),"\
                "_utf8'.',"\
         "cast(cast(table_name AS BINARY) AS CHAR CHARACTER SET utf8)) as o, "\
        "_utf8'Column_priv_" name "' as pn, "\
        "if(find_in_set(_utf8'" name "',"\
               "cast(cast(Column_priv AS BINARY) AS CHAR CHARACTER SET utf8))"\
               ",_utf8'Y',_utf8'N') as pv "\
     "from mysql.tables_priv"

#define SELECT_TABLE_COLUMN_PRIV_40(name)\
  "select host as h, concat(db,'.',table_name,'.',column_name) as o, "\
                "'Column_priv_" name "' as pn, "\
                "if(find_in_set('" name "',Column_priv),'Y','N') as pv "\
                "from mysql.columns_priv"

#define SELECT_TABLE_COLUMN_PRIV_41(name)\
  "select "\
         "cast(cast(host AS BINARY) AS CHAR CHARACTER SET utf8) as h, "\
         "concat(cast(cast(db AS BINARY) AS CHAR CHARACTER SET utf8),"\
            "_utf8'.',"\
            "cast(cast(table_name AS BINARY) AS CHAR CHARACTER SET utf8),"\
            "_utf8'.',"\
        "cast(cast(column_name AS BINARY) AS CHAR CHARACTER SET utf8)) as o, "\
         "_utf8'Column_priv_" name "' as pn, "\
         "if(find_in_set(_utf8'" name "',"\
             "cast(cast(Column_priv AS BINARY) AS CHAR CHARACTER SET utf8)),"\
             "_utf8'Y',_utf8'N') as pv "\
      "from mysql.columns_priv"

char * q_select_user_40[]=
{
  SELECT_USER_PRIV_40(           "Select_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(           "Insert_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(           "Update_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(           "Delete_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(           "Create_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(             "Drop_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(           "Reload_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(         "Shutdown_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(          "Process_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(             "File_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(            "Grant_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(       "References_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(            "Index_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(            "Alter_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(          "Show_db_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(            "Super_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40( "Create_tmp_table_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(      "Lock_tables_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(          "Execute_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(       "Repl_slave_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40(      "Repl_client_priv" ) WHERE_V_40(),
  SELECT_USER_PRIV_40( "max_questions"         ) WHERE_V_40(),
  SELECT_USER_PRIV_40( "max_updates"           ) WHERE_V_40(),
  SELECT_USER_PRIV_40( "max_connections"       ) WHERE_V_40()
};

char * q_select_user_41[]=
{
  SELECT_USER_PRIV_41(           "Select_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(           "Insert_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(           "Update_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(           "Delete_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(           "Create_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(             "Drop_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(           "Reload_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(         "Shutdown_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(          "Process_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(             "File_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(            "Grant_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(       "References_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(            "Index_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(            "Alter_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(          "Show_db_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(            "Super_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41( "Create_tmp_table_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(      "Lock_tables_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(          "Execute_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(       "Repl_slave_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(      "Repl_client_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41( "max_questions"         ) WHERE_V_41(),
  SELECT_USER_PRIV_41( "max_updates"           ) WHERE_V_41(),
  SELECT_USER_PRIV_41( "max_connections"       ) WHERE_V_41()
};

char * q_select_user_50[]=
{
  SELECT_USER_PRIV_41(           "Select_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(           "Insert_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(           "Update_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(           "Delete_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(           "Create_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(             "Drop_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(           "Reload_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(         "Shutdown_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(          "Process_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(             "File_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(            "Grant_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(       "References_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(            "Index_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(            "Alter_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(          "Show_db_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(            "Super_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41( "Create_tmp_table_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(      "Lock_tables_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(          "Execute_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(       "Repl_slave_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(      "Repl_client_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(      "Create_view_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41(        "Show_view_priv" ) WHERE_V_41(),
  SELECT_USER_PRIV_41( "max_questions"         ) WHERE_V_41(),
  SELECT_USER_PRIV_41( "max_updates"           ) WHERE_V_41(),
  SELECT_USER_PRIV_41( "max_connections"       ) WHERE_V_41()
};

typedef struct 
{
  char ** first_query;
  char ** end_query;
}Query_privs;

Query_privs q_select_user_privs[q_v_count]=
{
  {
    q_select_user_40,
    q_select_user_40 + sizeof(q_select_user_40)/sizeof(*q_select_user_40)
  },
  {
    q_select_user_41,
    q_select_user_41 + sizeof(q_select_user_41)/sizeof(*q_select_user_41)
  },
  {
    q_select_user_50,
    q_select_user_50 + sizeof(q_select_user_50)/sizeof(*q_select_user_50)
  }
};

char * q_select_db_40[]=
{
  SELECT_DB_PRIV_40(           "Select_priv" )   WHERE_V_40(),
  SELECT_DB_PRIV_40(           "Insert_priv" )   WHERE_V_40(),
  SELECT_DB_PRIV_40(           "Update_priv" )   WHERE_V_40(),
  SELECT_DB_PRIV_40(           "Delete_priv" )   WHERE_V_40(),
  SELECT_DB_PRIV_40(           "Create_priv" )   WHERE_V_40(),
  SELECT_DB_PRIV_40(             "Drop_priv" )   WHERE_V_40(),
  SELECT_DB_PRIV_40(            "Grant_priv" )   WHERE_V_40(),
  SELECT_DB_PRIV_40(       "References_priv" )   WHERE_V_40(),
  SELECT_DB_PRIV_40(            "Index_priv" )   WHERE_V_40(),
  SELECT_DB_PRIV_40(            "Alter_priv" )   WHERE_V_40(),
  SELECT_DB_PRIV_40( "Create_tmp_table_priv" )   WHERE_V_40(),
  SELECT_DB_PRIV_40(      "Lock_tables_priv" )   WHERE_V_40()
};

char * q_select_db_41[]=
{
  SELECT_DB_PRIV_41(           "Select_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(           "Insert_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(           "Update_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(           "Delete_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(           "Create_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(             "Drop_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(            "Grant_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(       "References_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(            "Index_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(            "Alter_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41( "Create_tmp_table_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(      "Lock_tables_priv" )   WHERE_V_41()
};

char * q_select_db_50[]=
{
  SELECT_DB_PRIV_41(           "Select_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(           "Insert_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(           "Update_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(           "Delete_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(           "Create_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(             "Drop_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(            "Grant_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(       "References_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(            "Index_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(            "Alter_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41( "Create_tmp_table_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(      "Lock_tables_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(      "Create_view_priv" )   WHERE_V_41(),
  SELECT_DB_PRIV_41(        "Show_view_priv" )   WHERE_V_41()
};

Query_privs q_select_db_privs[q_v_count]=
{
  {
    q_select_db_40,
    q_select_db_40 + sizeof(q_select_db_40)/sizeof(*q_select_db_40)
  },
  {
    q_select_db_41,
    q_select_db_41 + sizeof(q_select_db_41)/sizeof(*q_select_db_41)
  },
  {
    q_select_db_50,
    q_select_db_50 + sizeof(q_select_db_50)/sizeof(*q_select_db_50)
  }
};

char * q_select_tables_40[]=
{
  SELECT_TABLE_COLUMNS_PRIV_40( "Select"     )   WHERE_V_40(),
  SELECT_TABLE_COLUMNS_PRIV_40( "Insert"     )   WHERE_V_40(),
  SELECT_TABLE_COLUMNS_PRIV_40( "Update"     )   WHERE_V_40(),
  SELECT_TABLE_COLUMNS_PRIV_40( "References" )   WHERE_V_40(),

  SELECT_TABLE_PRIV_40( "Select"     )           WHERE_V_40(),
  SELECT_TABLE_PRIV_40( "Insert"     )           WHERE_V_40(),
  SELECT_TABLE_PRIV_40( "Update"     )           WHERE_V_40(),
  SELECT_TABLE_PRIV_40( "Delete"     )           WHERE_V_40(),
  SELECT_TABLE_PRIV_40( "Create"     )           WHERE_V_40(),
  SELECT_TABLE_PRIV_40( "Drop"       )           WHERE_V_40(),
  SELECT_TABLE_PRIV_40( "Grant"      )           WHERE_V_40(),
  SELECT_TABLE_PRIV_40( "References" )           WHERE_V_40(),
  SELECT_TABLE_PRIV_40( "Index"      )           WHERE_V_40(),
  SELECT_TABLE_PRIV_40( "Alter"      )           WHERE_V_40()
};

char * q_select_tables_41[]=
{
  SELECT_TABLE_COLUMNS_PRIV_41( "Select"     )   WHERE_V_41(),
  SELECT_TABLE_COLUMNS_PRIV_41( "Insert"     )   WHERE_V_41(),
  SELECT_TABLE_COLUMNS_PRIV_41( "Update"     )   WHERE_V_41(),
  SELECT_TABLE_COLUMNS_PRIV_41( "References" )   WHERE_V_41(),

  SELECT_TABLE_PRIV_41( "Select"     )           WHERE_V_41(),
  SELECT_TABLE_PRIV_41( "Insert"     )           WHERE_V_41(),
  SELECT_TABLE_PRIV_41( "Update"     )           WHERE_V_41(),
  SELECT_TABLE_PRIV_41( "Delete"     )           WHERE_V_41(),
  SELECT_TABLE_PRIV_41( "Create"     )           WHERE_V_41(),
  SELECT_TABLE_PRIV_41( "Drop"       )           WHERE_V_41(),
  SELECT_TABLE_PRIV_41( "Grant"      )           WHERE_V_41(),
  SELECT_TABLE_PRIV_41( "References" )           WHERE_V_41(),
  SELECT_TABLE_PRIV_41( "Index"      )           WHERE_V_41(),
  SELECT_TABLE_PRIV_41( "Alter"      )           WHERE_V_41()
};

Query_privs q_select_tables_privs[q_v_count]=
{
  {
    q_select_tables_40,
    q_select_tables_40 + sizeof(q_select_tables_40)/sizeof(*q_select_tables_40)
  },
  {
    q_select_tables_41,
    q_select_tables_41 + sizeof(q_select_tables_41)/sizeof(*q_select_tables_41)
  },
  {
    q_select_tables_41,
    q_select_tables_41 + sizeof(q_select_tables_41)/sizeof(*q_select_tables_41)
  }
};

char * q_select_columns_40[]=
{
  SELECT_TABLE_COLUMN_PRIV_40( "Select"     )    WHERE_V_40(),
  SELECT_TABLE_COLUMN_PRIV_40( "Insert"     )    WHERE_V_40(),
  SELECT_TABLE_COLUMN_PRIV_40( "Update"     )    WHERE_V_40(),
  SELECT_TABLE_COLUMN_PRIV_40( "References" )    WHERE_V_40()
};

char * q_select_columns_41[]=
{
  SELECT_TABLE_COLUMN_PRIV_41( "Select"     )    WHERE_V_41(),
  SELECT_TABLE_COLUMN_PRIV_41( "Insert"     )    WHERE_V_41(),
  SELECT_TABLE_COLUMN_PRIV_41( "Update"     )    WHERE_V_41(),
  SELECT_TABLE_COLUMN_PRIV_41( "References" )    WHERE_V_41()
};

Query_privs q_select_columns_privs[q_v_count]=
{
  {
    q_select_columns_40,
    q_select_columns_40 +
                       sizeof(q_select_columns_40)/sizeof(*q_select_columns_40)
  },
  {
    q_select_columns_41,
    q_select_columns_41 +
                       sizeof(q_select_columns_41)/sizeof(*q_select_columns_41)
  },
  {
    q_select_columns_41,
    q_select_columns_41 +
                       sizeof(q_select_columns_41)/sizeof(*q_select_columns_41)
  }
};

char * myx_resize_vector_block(char * ptr,
                               size_t size_of_block,
                               size_t count_of_blocks)
{
  size_t * reserved_ptr;
  if (!ptr)
  {
    reserved_ptr= g_malloc0((gulong)(sizeof(size_t) +
                                    count_of_blocks * size_of_block));
    if (!reserved_ptr)
      return 0;
    *reserved_ptr= count_of_blocks;
    return (char*)(reserved_ptr + 1);
  }
  else
  {
    reserved_ptr= ((size_t*)ptr)-1;
    if (count_of_blocks <= *reserved_ptr)
    {
      return ptr;
    }
    else
    {
      count_of_blocks+= count_of_blocks/3;
      reserved_ptr= g_realloc(reserved_ptr,
                              (gulong)(sizeof(size_t) +
                                       count_of_blocks * size_of_block));
    }
  }
  if (!reserved_ptr)
    return 0;
  *reserved_ptr= count_of_blocks;
  return (char*)(reserved_ptr + 1);
}

void myx_free_vector_block(char * ptr)
{
  if (ptr)
  {
    size_t * reserved_size_ptr= ((size_t*)ptr)-1;
    g_free(reserved_size_ptr);
  }
}

#define MYX_VECTOR_RESERVE(ptr,size)\
  (((char*)(ptr))= myx_resize_vector_block((char*)(ptr),sizeof(*(ptr)),\
                                           (size_t)(size)))

#define MYX_VECTOR_PUSH_BACK(ptr,num,ptr_to_top)\
{\
  (num)++;\
  ((char*)(ptr))= myx_resize_vector_block((char*)(ptr),sizeof(*(ptr)),(num));\
  (ptr_to_top)= (ptr) + (num) - 1;\
}

#define MYX_VECTOR_FREE(ptr) myx_free_vector_block((char*)ptr)

static const char * select_privileges_names[]=
{
  "h",  // 0 host
  "o",  // 1 object_name
  "pn", // 2 privilege_name
  "pv", // 3 privilege_value
};
static const char ** select_privileges_names_end=
       select_privileges_names + sizeof(select_privileges_names)/sizeof(char*);

MYX_USER_OBJECT_PRIVILEGES * look_for_obj_name(MYX_USER * user, 
                                               char * object_name,
                                               char * host)
{
  MYX_USER_OBJECT_PRIVILEGES * priv= user->user_object_privileges;
  MYX_USER_OBJECT_PRIVILEGES * priv_end=
                                       priv + user->user_object_privileges_num;
  for (; priv!=priv_end; priv++)
  {
    if (!strcmp(priv->object_name,object_name) && 
        !strcmp(priv->host,host))
      return priv;
  }
  return 0;
}

char ** look_for_host(MYX_USER * user, char * host_name)
{
  unsigned char ** host= user->hosts;
  unsigned char ** hosts_end= host + user->hosts_num;
  for ( ; host!=hosts_end; host++)
  {
    if (!strcmp(*host,host_name))
      return (char**)host;
  }
  return 0;
}

MYX_USER_OBJECT_PRIVILEGES * 
      safe_add_object(MYSQL *mysql, MYX_USER * user,
                      char * object_name, char * host, int get_hosts)
{
  char * chost=   myx_convert_dbstr_utf8(mysql,host);
  char * coname=  myx_convert_dbstr_utf8(mysql,object_name);

  MYX_USER_OBJECT_PRIVILEGES * o_priv= look_for_obj_name(user,coname,chost);
  if (o_priv)
  {
    g_free(chost);
    g_free(coname);
  }
  else
  {
    MYX_VECTOR_PUSH_BACK(user->user_object_privileges,
                         user->user_object_privileges_num,o_priv);
    o_priv->object_name=          coname;
    o_priv->host=                 chost;
    o_priv->user_privileges_num=  0;
    o_priv->user_privileges=      0;
    if (!look_for_host(user,host) && get_hosts)
    {
      unsigned char ** added_host;
      MYX_VECTOR_PUSH_BACK(user->hosts,user->hosts_num,added_host);
      *added_host= g_strdup(o_priv->host);
    }
  }
  return o_priv;
}

int get_privileges_from_query(MYSQL *mysql, MYX_USER * user,
                                       const char * query, int get_hosts)
{
  MYSQL_RES * res;
  MYSQL_ROW row;
  my_ulonglong num_rows;
  MYSQL_FIELD * fields;
  int fi[4];

  size_t len= strlen(query);
  if (myx_mysql_real_query(mysql, query, (unsigned int)len) ||
      !(res= mysql_store_result(mysql)))
  {
    return -1;
  }

  num_rows= mysql_num_rows(res);
  if (num_rows==0)
    return 0;

  fields= mysql_fetch_fields(res);
  build_field_subst(select_privileges_names,select_privileges_names_end,
                    fields,fields+mysql_num_fields(res),fi);

  while ((row= mysql_fetch_row(res)))
  {
    char * host=        fi[0]==-1 ? "" : row[fi[0]];
    char * object_name= fi[1]==-1 ? "" : row[fi[1]];
    char * priv_name=   fi[2]==-1 ? "" :
                                     myx_convert_dbstr_utf8(mysql, row[fi[2]]);
    if (!object_name)
      object_name= "";
    if (priv_name[0]!='+')
    {
      char * priv_value=  fi[3]==-1 ? "" :
                                     myx_convert_dbstr_utf8(mysql, row[fi[3]]);
      MYX_USER_OBJECT_PRIVILEGES *o_priv= safe_add_object(mysql,user,
                                                          object_name,
                                                          host,get_hosts);
      unsigned char ** priv;
      MYX_VECTOR_PUSH_BACK(o_priv->user_privileges,
                           o_priv->user_privileges_num,priv);
      *priv= g_strdup_printf("%s=%s", priv_name, priv_value);
      g_free(priv_value);
    }
    g_free(priv_name);
  }
  mysql_free_result(res);
  return num_rows==0 ? 0 : 1;
}

int get_privileges_from_queries(MYSQL *mysql, MYX_USER * user,
                                Query_privs * queries,
                                int get_hosts)
{
  int res= 0;
  const char ** query;
  for (query= (const char**)queries->first_query; query!=(const char**)queries->end_query; query++)
  {
    int query_res= get_privileges_from_query(mysql,user,*query,get_hosts);
    if (query_res<0)
      return -1;
    if (query_res>0)
      res= 1;
  }
  return res;
}

MYX_USER * myx_get_user_opt_privileges(MYSQL *mysql,
                                       const char *user_name,
                                       int get_privileges)
{
  MYX_USER *user= 0;
  unsigned char sqlcmd[255];
  unsigned char * sqlcmd_end;
  unsigned long * field_lengths;
  MYSQL_RES* res;
  MYSQL_ROW row;
  QUERY_VARIANTS v;

  sqlcmd_end= strmov(sqlcmd, "set @goal_user=\'");
  sqlcmd_end+= mysql_real_escape_string(mysql,sqlcmd_end,
                                        user_name,
                                        (unsigned long)strlen(user_name));
  sqlcmd_end= strmov(sqlcmd_end, "\'");
  if (myx_mysql_query(mysql, sqlcmd))
    return NULL;

  v= mysql_version_is_later_or_equal_than(mysql,5,0) ? q_v_5_0 :
     mysql_version_is_later_or_equal_than(mysql,4,1) ? q_v_4_1 : q_v_base;

  // --------------------------------------------------------------------------
  //Get additional user infos
  if (myx_mysql_query(mysql, q_select_user_info[v]))
  { // error
    if (!(myx_check_mysql_user_info_table(mysql)))
      return NULL; // Check if user_info tables exists. if not, create it
  }
  else
  {
    if ( !(res= mysql_store_result(mysql)) )
      return NULL; // error
    if ( (row= mysql_fetch_row(res)))
    {
      user= g_malloc0(sizeof(MYX_USER));
      user->full_name=           myx_convert_dbstr_utf8(mysql, row[0]);
      user->description=         myx_convert_dbstr_utf8(mysql, row[1]);
      user->email=               myx_convert_dbstr_utf8(mysql, row[2]);
      user->contact_information= myx_convert_dbstr_utf8(mysql, row[3]);

      field_lengths= mysql_fetch_lengths(res); // Get field lengths
      user->icon_length= field_lengths[4]; // Copy Icon data
      user->icon= !user->icon_length ? 0 : g_memdup(row[4], user->icon_length);
    }
    mysql_free_result(res);
  }

  if (!user)
  {
    user= g_malloc0(sizeof(MYX_USER));
    user->full_name= g_strdup("");
    user->description= g_strdup("");
    user->email= g_strdup("");
    user->contact_information= g_strdup("");
    user->icon_length= 0;
    user->icon= 0;
  }
  user->user_name= g_strdup(user_name);
  user->password= g_strdup("");

  user->hosts_num= 0;
  user->hosts= 0;

  user->user_object_privileges_num= 0;
  user->user_object_privileges= 0;

  if (get_privileges_from_queries(mysql,user,  q_select_user_privs+v,1)> 0 &&
      get_privileges_from_queries(mysql,user,  q_select_db_privs+v,0)>=0 &&
      (!get_privileges || 
       (get_privileges_from_queries(mysql,user,q_select_tables_privs+v,0)>=0 &&
        get_privileges_from_queries(mysql,user,q_select_columns_privs+v,0)>=0))
     )
  {
    return user;
  }

  myx_free_user(user);
  return NULL;
}

MYX_USER * myx_get_user(MYSQL *mysql, const char *user_name)
{
  return myx_get_user_opt_privileges(mysql, user_name, 0);
}

MYX_USER * myx_get_user_with_privileges(MYSQL *mysql, const char *user_name)
{
  return myx_get_user_opt_privileges(mysql, user_name, 1);
}

int myx_free_user(MYX_USER *user)
{
  unsigned int i;

  g_free(user->user_name);
  g_free(user->password);
  g_free(user->full_name);
  g_free(user->description);
  g_free(user->email);
  g_free(user->contact_information);
  g_free(user->icon);

  for(i=0;i<user->hosts_num;i++)
  {
    g_free(user->hosts[i]);
  }
  MYX_VECTOR_FREE(user->hosts);

  for(i=0;i<user->user_object_privileges_num;i++)
  {
    myx_free_user_priv(user->user_object_privileges+i); 
  }
  MYX_VECTOR_FREE(user->user_object_privileges);

  g_free(user);
  return 0;
}

int myx_free_user_priv(MYX_USER_OBJECT_PRIVILEGES *user_priv)
{
  unsigned int i;

  g_free(user_priv->host);
  g_free(user_priv->object_name);

  for(i=0;i<user_priv->user_privileges_num;i++)
  {
    g_free(user_priv->user_privileges[i]);
  }

  MYX_VECTOR_FREE(user_priv->user_privileges);

  return 0;
}

const char * user_info_update_parts[6]=
{
  "UPDATE mysql." USER_INFO_TABLENAME " SET User=@new_user, Full_name='",
  "', Description='",
  "', Email='",
  "', Contact_information='",
  "', Icon='",
  "' WHERE User=@cur_user"
};

const char * user_info_insert_parts[6]=
{
  "INSERT INTO mysql." USER_INFO_TABLENAME 
  "(User, Full_name, Description, Email, Contact_information, Icon) "
  "VALUES(@new_user, '",
  "', '",
  "', '",
  "', '",
  "', '",
  "')"
};

const char ** user_info_replace_parts[2]=
{
  user_info_update_parts,
  user_info_insert_parts
};

int replace_user_info(MYSQL *mysql, MYX_USER *user, unsigned long name_len,
                      const char * replace_query_parts[6])
{
  int res;
  unsigned long full_name_len= (unsigned long)strlen(user->full_name);
  unsigned long descr_len= (unsigned long)strlen(user->description);
  unsigned long email_len= (unsigned long)strlen(user->email);
  unsigned long contact_len= (unsigned long)strlen(user->contact_information);

  char* sqlcmd= g_malloc0(name_len+full_name_len+descr_len+
                          email_len+contact_len+user->icon_length*2+200);

  char * sqlcmd_end= strmov(sqlcmd, replace_query_parts[0]);
  sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end,
                                        user->full_name, full_name_len);
  sqlcmd_end= strmov(sqlcmd_end,replace_query_parts[1]);
  sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end,
                                        user->description, descr_len);
  sqlcmd_end= strmov(sqlcmd_end,replace_query_parts[2]);
  sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end,
                                        user->email, email_len);
  sqlcmd_end= strmov(sqlcmd_end,replace_query_parts[3]);
  sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end,
                                        user->contact_information,
                                        contact_len);
  sqlcmd_end= strmov(sqlcmd_end,replace_query_parts[4]);
  sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end,
                                        user->icon, user->icon_length);
  sqlcmd_end= strmov(sqlcmd_end,replace_query_parts[5]);

  res= myx_mysql_real_query(mysql, sqlcmd,(unsigned int)(sqlcmd_end-sqlcmd));
  g_free(sqlcmd);
  return res;
}

int myx_set_local_variable(MYSQL *mysql, const char * name, 
                           const char * value, unsigned long value_len)
{
  unsigned char sqlcmd[500];
  unsigned char * sqlcmd_end;
  sqlcmd_end= strmov(sqlcmd, "set @");
  sqlcmd_end= strmov(sqlcmd_end, name);
  sqlcmd_end= strmov(sqlcmd_end, "=\'");
  sqlcmd_end+= mysql_real_escape_string(mysql,sqlcmd_end,
                                        value,value_len);
  sqlcmd_end= strmov(sqlcmd_end, "\'");
  return myx_mysql_query(mysql, sqlcmd);
}

int myx_set_user(MYSQL *mysql, MYX_USER *user,
                 const char *previous_user_name, int new_user)
{
  MYX_USER *current_user;
  unsigned char sqlcmd[500];
  unsigned char field_buf[30];
  unsigned char value_buf[30];
  unsigned char *schema_name, *table_name, *column_name;
  const char * begin;
  unsigned char *s;
  unsigned int i, j;
  int dot_count, u_variant, num_host;
  int icon_changed = 0;
  unsigned char *long_sqlcmd, *sqlcmd_end;
  MYSQL_RES* res;
  unsigned long cur_name_len, name_len= (unsigned long)strlen(user->user_name);
  int cur_user_has_been_set= 0;

  if (myx_set_local_variable(mysql,"new_user",user->user_name,name_len))
    return -1;

  if (!new_user) // Update existing User
  {
    if (!(current_user= myx_get_user(mysql, previous_user_name)))
      return -1;
    cur_name_len= (unsigned long)strlen(current_user->user_name);

    // check if name has changed
    if (cur_name_len!=name_len ||
        strncmp(user->user_name, current_user->user_name,cur_name_len))
    {
      //Update user name in mysql.user table
      if (myx_set_local_variable(mysql, "cur_user", 
                                 current_user->user_name, cur_name_len) ||
          myx_mysql_query(mysql,"UPDATE mysql.user SET User=@new_user"
                                " WHERE User=@cur_user") ||
          myx_mysql_query(mysql,"UPDATE mysql.db SET User=@new_user"
                                " WHERE User=@cur_user") ||
          myx_mysql_query(mysql,"UPDATE mysql.tables_priv SET User=@new_user"
                                " WHERE User=@cur_user") ||
          myx_mysql_query(mysql,"UPDATE mysql.columns_priv SET User=@new_user"
                                " WHERE User=@cur_user"))
      {
        return -1;
      }
      cur_user_has_been_set= 1;
    }

    //Check if all user/host combination has already been created
    for(i= 0; i<user->hosts_num; i++)
    {
      MYX_STRINGLIST hosts;
      hosts.strings_num= current_user->hosts_num;
      hosts.strings= (char**)current_user->hosts;
      num_host= myx_str_in_stringlist(&hosts,user->hosts[i]);

      // If not, insert a new user/host combination,
      if (num_host==-1) // take password from existing Host
      {
        unsigned long host_len= (unsigned long)strlen(user->hosts[i]);
        sqlcmd_end= strmov(sqlcmd,
                           "INSERT INTO mysql.user(User, Host, Password)"
                           " SELECT @new_user,'");
        sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end, 
                                              user->hosts[i],host_len);
        sqlcmd_end= strmov(sqlcmd_end,
                           "', Password FROM mysql.user"
                           " WHERE User=@new_user ORDER BY Host LIMIT 1");
        if (myx_mysql_real_query(mysql, sqlcmd,
                                 (unsigned int)(sqlcmd_end-sqlcmd)))
        {
          return -1;
        }
      }
    }

    //Check for obsolete user/host combination
    for(i=0;i<current_user->hosts_num;i++)
    {
      MYX_STRINGLIST hosts;
      hosts.strings_num= user->hosts_num;
      hosts.strings= (char**)user->hosts;
      num_host= myx_str_in_stringlist(&hosts,current_user->hosts[i]);

      //If there is a obsolete row, delete it
      if (num_host==-1)
      {
        unsigned long host_len= (unsigned long)strlen(current_user->hosts[i]);
        sqlcmd_end= strmov(sqlcmd,
                           "DELETE FROM mysql.user"
                           " WHERE User=@new_user AND Host='");
        sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end,
                                              current_user->hosts[i],host_len);
        sqlcmd_end= strmov(sqlcmd_end, "'");

        if (myx_mysql_real_query(mysql, sqlcmd,
                                 (unsigned int)(sqlcmd_end-sqlcmd)))
        {
          return -1;
        }
      }
    }

    //Check if the icon has changed
    icon_changed= (user->icon_length!=current_user->icon_length ||
                   (user->icon_length > 0 && 
                    memcmp(user->icon,current_user->icon,user->icon_length)));

    //check if user information has changed
    if (strcmp(user->user_name,           current_user->user_name)           ||
        strcmp(user->full_name,           current_user->full_name)           ||
        strcmp(user->description,         current_user->description)         ||
        strcmp(user->email,               current_user->email)               ||
        strcmp(user->contact_information, current_user->contact_information) ||
        icon_changed)
    {
      //Check if there already is an entry in the mysql.user_info table
      if ((cur_user_has_been_set || 
           !myx_set_local_variable(mysql, "cur_user", 
                                   current_user->user_name, cur_name_len)) &&
          !myx_mysql_query(mysql,"SELECT User FROM mysql." USER_INFO_TABLENAME
                                " WHERE User=@cur_user") &&
          (res= mysql_store_result(mysql)))
      {
        u_variant= mysql_num_rows(res) ? 0 : 1;
        mysql_free_result(res);
      }
      else
      {
        g_message("Error executing query %s:\n%s",sqlcmd,
                  myx_mysql_error(mysql));
        return -1;
      }

      if (replace_user_info(mysql,user,name_len,
                            user_info_replace_parts[u_variant]))
      {
        return -1;
      }
    }

    if (strcmp(user->password, "________"))
    {
      unsigned long pas_len= (unsigned long)strlen(user->password);
      //Update password in mysql.user table
      sqlcmd_end= strmov(sqlcmd, "UPDATE mysql.user SET Password=Password('");
      sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end,
                                            user->password,pas_len);
      sqlcmd_end= strmov(sqlcmd_end, "') WHERE User=@new_user");

      if (myx_mysql_real_query(mysql, sqlcmd,
                               (unsigned int)(sqlcmd_end-sqlcmd))) 
      {
        return -1; 
      }
    }
  }
  else //Create new user
  {
    // Insert into mysql.user table, one insert for each host
    unsigned long pass_len= (unsigned long)strlen(user->password);
    for(i= 0; i<user->hosts_num; i++)
    {
      unsigned long host_len= (unsigned long)strlen(user->hosts[i]);
      sqlcmd_end= strmov(sqlcmd, "INSERT INTO mysql.user(User, Host, Password)"
                                 " VALUES(@new_user, '");
      sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end,
                                            user->hosts[i], host_len);
      sqlcmd_end= strmov(sqlcmd_end, "', Password('");
      sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end, user->password,
                                            pass_len);
      sqlcmd_end= strmov(sqlcmd_end, "'))");

      if (myx_mysql_real_query(mysql, sqlcmd,
                               (unsigned int)(sqlcmd_end-sqlcmd)))
      {
        return -1;
      }
    }

    //Insert into mysql.user_info table
    if (user->full_name[0] || user->description[0] || user->email[0] ||
        user->contact_information[0] || user->icon_length>0)
    {
      if (replace_user_info(mysql,user,name_len,
                            user_info_insert_parts))
      {
        return -1;
      }
    }
  }

  // Username/host and user_info have been updated or create
  // now update/insert privileges
  if (user->user_object_privileges_num>0)
  {
    //Delete previous settings from mysql.db table
    if (myx_mysql_query(mysql, "DELETE FROM mysql.db WHERE User=@new_user") ||
        myx_mysql_query(mysql,
                        "DELETE FROM mysql.tables_priv WHERE User=@new_user")||
        myx_mysql_query(mysql,
                        "DELETE FROM mysql.columns_priv WHERE User=@new_user"))
    {
      return -1;
    }

    long_sqlcmd= g_malloc0(sizeof(char)*700);

    for(i= 0; i<user->user_object_privileges_num; i++)
    {
      MYX_USER_OBJECT_PRIVILEGES *privs= user->user_object_privileges+i;
      unsigned long host_len= (unsigned long)strlen(privs->host);

      dot_count= !privs->object_name ? -1
                                     : sub_str_count(".", privs->object_name);

      if (dot_count < 0 || !privs->object_name[0]) // Global privileges
      {
        sqlcmd_end= strmov(long_sqlcmd, "UPDATE mysql.user SET ");
        for(j= 0; j<privs->user_privileges_num; j++)
        {
          char * name= name_of_str(field_buf, privs->user_privileges[j]);
          char * value= value_of_str(value_buf, privs->user_privileges[j]);
          sqlcmd_end= strmov(sqlcmd_end, name);
          sqlcmd_end= strmov(sqlcmd_end, "='");
          sqlcmd_end= strmov(sqlcmd_end, value);
          sqlcmd_end= strmov(sqlcmd_end,
                             j<privs->user_privileges_num-1 ? "', " : "' ");
        }
        sqlcmd_end= strmov(sqlcmd_end, "WHERE User=@new_user AND Host='");
        sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end, 
                                              privs->host,host_len);
        sqlcmd_end= strmov(sqlcmd_end, "'");

        if (myx_mysql_real_query(mysql, long_sqlcmd,
                                 (unsigned int)(sqlcmd_end-long_sqlcmd)))
        {
          g_free(long_sqlcmd);
          return -1; 
        }
      }
      else if (dot_count==0) // Schema Privileges
      {
        unsigned long obj_len= (unsigned long)strlen(privs->object_name);
        //Insert new settings
        sqlcmd_end= strmov(long_sqlcmd,
                           "INSERT INTO mysql.db(User, Host, Db, ");
        for(j= 0; j<privs->user_privileges_num; j++)
        {
          sqlcmd_end= strmov(sqlcmd_end,
                             name_of_str(field_buf,privs->user_privileges[j]));
          if (j<privs->user_privileges_num-1)
            sqlcmd_end= strmov(sqlcmd_end, ", ");
        }
        sqlcmd_end= strmov(sqlcmd_end, ") VALUES(@new_user,'");
        sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end,
                                              privs->host, host_len);
        sqlcmd_end= strmov(sqlcmd_end, "', '");
        sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end,
                                              privs->object_name, obj_len);
        sqlcmd_end= strmov(sqlcmd_end, "', '");

        for(j= 0; j<privs->user_privileges_num; j++)
        {
          sqlcmd_end= strmov(sqlcmd_end,
                             value_of_str(value_buf,
                                          privs->user_privileges[j]));
          sqlcmd_end= strmov(sqlcmd_end, j<privs->user_privileges_num-1 
                                          ? "', '" : "')");
        }

        if (myx_mysql_real_query(mysql, long_sqlcmd,
                                 (unsigned int)(sqlcmd_end-long_sqlcmd)))
        {
          g_free(long_sqlcmd);
          return -1; 
        }
      }
      else // Schema Table privileges
      {
        unsigned long schema_len, table_len, column_len;
        s= g_malloc0(sizeof(char)*(64+64+64+2));
        s= strcpy(s, privs->object_name);
        schema_name= strtok(s, ".");
        schema_len= (unsigned long)strlen(schema_name);
        table_name= strtok(NULL, ".");
        table_len= (unsigned long)strlen(table_name);

        if (dot_count==1)
        {
          sqlcmd_end= strmov(long_sqlcmd,
                             "INSERT INTO mysql.tables_priv"
                             "(Host, Db, User, Table_name, Timestamp, Grantor,"
                              "Table_priv, Column_priv) VALUES('");
        }
        else
        {
          column_name= strtok(NULL, ".");
          column_len= (unsigned long)strlen(column_name);
          sqlcmd_end= strmov(long_sqlcmd,
                             "INSERT INTO mysql.columns_priv"
                             "(Host,Db,User,Table_name,Timestamp,"
                             "Column_name,Column_priv) VALUES('");
        }

        sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end, privs->host,
                                              host_len);
        sqlcmd_end= strmov(sqlcmd_end, "', '");
        sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end, schema_name,
                                              schema_len);
        sqlcmd_end= strmov(sqlcmd_end, "', @new_user, '");
        sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end, table_name,
                                              table_len);
        sqlcmd_end= strmov(sqlcmd_end, "', now(), '");

        if (dot_count!=1)
        {
          sqlcmd_end+= mysql_real_escape_string(mysql, sqlcmd_end, column_name,
                                                column_len);
        }
        else
        {
          // No grantor at this time
          begin= sqlcmd_end= strmov(sqlcmd_end, "', '");

          // Get selected Table_privs
          for(j= 0; j<privs->user_privileges_num; j++)
          {
            if ( strstr(privs->user_privileges[j], "Table_priv") &&
                !strcmp(value_of_str(value_buf, privs->user_privileges[j]),"Y"))
            {
              sqlcmd_end= strmov(sqlcmd_end,
                                 strstr(name_of_str(field_buf,
                                                    privs->user_privileges[j]),
                                        "_priv_")+6);
              sqlcmd_end= strmov(sqlcmd_end, ",");
            }
          }
          if (sqlcmd_end!=(unsigned char*)begin) // Remove last ,
            sqlcmd_end--;
        }

        begin= sqlcmd_end= strmov(sqlcmd_end, "', '");

        // Get selected Column_privs
        for(j=0;j<privs->user_privileges_num;j++)
        {
          if( strstr(privs->user_privileges[j], "Column_priv")&&
             !strcmp(value_of_str(value_buf, privs->user_privileges[j]),"Y"))
          {
            sqlcmd_end= strmov(sqlcmd_end,
                               strstr(name_of_str(field_buf,
                                                  privs->user_privileges[j]),
                                      "_priv_")+6);
            sqlcmd_end= strmov(sqlcmd_end, ",");
          }
        }

        if(sqlcmd_end!=(unsigned char*)begin) // Remove last ,
          sqlcmd_end--;

        sqlcmd_end= strmov(sqlcmd_end, "')");

        g_free(s);

        if (myx_mysql_real_query(mysql, long_sqlcmd,
                                 (unsigned int)(sqlcmd_end-long_sqlcmd)))
        {
          g_free(long_sqlcmd);
          return -1; 
        }
      }
    }

    g_free(long_sqlcmd);
  }

  //Flush Privileges
  if (myx_mysql_query(mysql, "FLUSH PRIVILEGES"))
    return -1;

  return 0;
}

MYX_USER_OBJECT_PRIVILEGES * myx_get_privilege_struct(MYSQL *mysql, 
                                                      const char *object_name)
{
  MYSQL_RES* res;
  MYSQL_FIELD *fields;
  MYSQL_ROW row;
  unsigned int num_fields;

  MYX_USER_OBJECT_PRIVILEGES *user_priv=
                                 g_malloc0(sizeof(MYX_USER_OBJECT_PRIVILEGES));
  unsigned int point_count;
  char * sqlcmd;
  unsigned char s[255];
  unsigned int j;
  char *token;
  char seps[]= "(',)";
  unsigned char **upriv_text;

  user_priv->host= NULL;
  user_priv->object_name= g_strdup(object_name);

  user_priv->user_privileges_num= 0;
  user_priv->user_privileges= NULL;

  point_count= sub_str_count(".", object_name);

  sqlcmd= 
    !*object_name  ? "SELECT * FROM mysql.user WHERE 1=2"  : // Global
    point_count==0 ? "SELECT * FROM mysql.db   WHERE 1=2"  : // db
    point_count==1 ? "SHOW COLUMNS FROM mysql.tables_priv" : // db.Table
                     "SHOW COLUMNS FROM mysql.columns_priv"; // db.table.column

  if (myx_mysql_query(mysql, sqlcmd) || !(res= mysql_store_result(mysql)))
  {
    g_free(user_priv);
    return NULL;
  }
  else
  {
    if (!point_count)
    {
      // Depending on the version of the server there might be
      num_fields= mysql_num_fields(res); // different columns
      fields= mysql_fetch_fields(res);

      MYX_VECTOR_RESERVE(user_priv->user_privileges,num_fields-2);
      for(j= 0; j<num_fields; j++)
      {
        const char * max_text= "%s=0";
        const char * priv_text= "%s=N";
        char *field_name= myx_convert_dbstr_utf8(mysql, fields[j].name);
        const char * text=
                !strcmp(str_right(s, field_name, 5), "_priv") ? priv_text :
                !strcmp(str_left (s, field_name, 4), "max_" ) ? max_text  : 0;
        if (text)
        {
          MYX_VECTOR_PUSH_BACK(user_priv->user_privileges,
                               user_priv->user_privileges_num,upriv_text)
          *upriv_text= g_strdup_printf(text, field_name);
        }
        g_free(field_name);
      }
    }
    else if (point_count==1 || point_count==2)
    {
      MYX_VECTOR_RESERVE(user_priv->user_privileges,mysql_num_rows(res)-4);
      while ((row= mysql_fetch_row(res)))
      {
        if (strcmp(str_right(s, row[0], 5), "_priv")==0)
        {
          char *row0= myx_convert_dbstr_utf8(mysql, row[0]);
          char *row1= myx_convert_dbstr_utf8(mysql, row[1]);
          strcpy(s, row1+3);
          for (token= strtok(s, seps); token; token= strtok(NULL, seps))
          {
            MYX_VECTOR_PUSH_BACK(user_priv->user_privileges,
                                 user_priv->user_privileges_num,upriv_text)
            *upriv_text= g_strdup_printf("%s_%s=N", row0, token);
          }
          g_free(row0);
          g_free(row1);
        }
      }
    }
    mysql_free_result(res);
  }

  return user_priv;
}

const char * user_tables_names[]=
{
  "columns_priv",
  "tables_priv",
  "db",
  USER_INFO_TABLENAME,
  "user"
};
const char ** user_tables_names_end=
                   user_tables_names + sizeof(user_tables_names)/sizeof(char*);

int myx_del_user(MYSQL *mysql, const char *user_name)
{
  unsigned char sqlcmd[500], *sqlcmd_end;
  unsigned char esc_user_name[17];
  const char ** user_table;

  mysql_real_escape_string(mysql, esc_user_name, user_name,
                           (unsigned long)strlen(user_name));

  for (user_table= user_tables_names;
       user_table!=user_tables_names_end; user_table++)
  {
    sqlcmd_end= strmov(sqlcmd,      "DELETE FROM mysql.");
    sqlcmd_end= strmov(sqlcmd_end,  *user_table);
    sqlcmd_end= strmov(sqlcmd_end,  " WHERE User='");
    sqlcmd_end= strmov(sqlcmd_end,  esc_user_name);
    sqlcmd_end= strmov(sqlcmd_end,  "'");

    if (myx_mysql_real_query(mysql, sqlcmd, (unsigned int)(sqlcmd_end-sqlcmd)))
      return -1;
  }

  //Flush Privileges
  if (myx_mysql_query(mysql, "FLUSH PRIVILEGES"))
    return -1;

  return 0;
}
