unit AuxAdminBackupRestore;

interface

uses
  gnugettext, ComCtrls, Classes, SysUtils,
  myx_public_interface, myx_admin_public_interface,
  AuxFuncs, SchemataTreeView, ApplicationDataModule,
  MyxError, TntClasses, TntComCtrls, IniFiles;

type
  //Definition of BackupObj Types and Nodes
  TBackupObjType = (BOTCatalog, BOTSchema, BOTTable);
  TBackupCheckType = (BCTNone, BCTSome, BCTAll);

  TBackupNode = class(TObject)
    constructor Create(ObjName: WideString; ObjType: TBackupObjType);
    destructor Destroy; override;

    function GetObjName: WideString;
    procedure SetObjName(ObjName: WideString);

    function GetObjType: TBackupObjType;
    procedure SetObjType(ObjType: TBackupObjType);

    function GetSelectState: TBackupCheckType;
    procedure SetSelectState(theState: TBackupCheckType);
  protected
    ObjName: WideString;
    ObjType: TBackupObjType;
    Selected: TBackupCheckType;
  public
    Substrings: TTntStringList;
  end;

  function GetBackupErrorMsg(BackupError: MYX_BACKUP_ERROR): WideString;

  procedure BuildContentTreeFromBackupContent(PMySQL: Pointer;
    BackupTreeView: TTntTreeView;
    BackupContent: TMYX_BACKUP_CONTENT;
    SchemataFrame: TSchemataFrame;
    CompleteSchematas: Boolean);
  procedure AddSchemaToBackup(PMySQL: Pointer;
    BackupTreeView: TTntTreeView;
    Schema: TMYX_SCHEMA; Selected: Boolean);
  function GetBackupContent(BackupTreeView: TTntTreeView): TMYX_BACKUP_CONTENT;
  procedure BuildContentTreeFromRestoreContent(BackupTreeView: TTntTreeView;
    BackupContent: TMYX_BACKUP_CONTENT);


  procedure SetNodeSelectState(Node: TTntTreeNode; Selected: TBackupCheckType);
  procedure SetChildrenSelectState(Node: TTntTreeNode);
  procedure CheckParentSelectState(Node: TTntTreeNode);
  function AllChildNodeSelected(Node: TTntTreeNode): TBackupCheckType;

implementation

function GetBackupErrorMsg(BackupError: MYX_BACKUP_ERROR): WideString;
var ErrorTxt: WideString;
begin
  case BackupError of
    MYX_BACKUP_MYSQL_ERROR:
      ErrorTxt:=_('The MySQL Server returned this Error:'#13#10+
        'MySQL Error Nr.%d'+'-'+
        '%s');
    MYX_BACKUP_CANT_OPEN_FILE:
      ErrorTxt:=_('The output file %s cannot be opened.');
    MYX_BACKUP_ILLEGAL_OPTION:
      ErrorTxt:=_('Illegal option.');
    MYX_BACKUP_PCRE_ERROR:
      ErrorTxt:=_('Regulare expression Error.');
    MYX_BACKUP_MALLOC_FAILED:
      ErrorTxt:=_('Memory allocation failed.');
    MYX_BACKUP_OUTPUTDEVICE_FULL:
      ErrorTxt:=_('The device containing the backup file %s is full.');
    MYX_BACKUP_CANNOT_FLUSH_TABLES_WITH_READ_LOCK:
      ErrorTxt:=_('The command FLUSH TABLES WITH READ LOCK cannot be '+
        'executed. You may want to choose a different Backup Execution Method '+
        'on the Advanced Options sheet.');
    MYX_BACKUP_CANNOT_START_TRANSACTION:
      ErrorTxt:=_('The command START TRANSACTION cannot be '+
        'executed. You may want to choose a different Backup Execution Method '+
        'on the Advanced Options sheet.');
    MYX_BACKUP_CANNOT_SET_ANSI_QUOTES:
      ErrorTxt:=_('The ANSI quotes cannot be set. You may want to choose '+
        'disable this option on the Advanced Options sheet.');
  else
    ErrorTxt:='';
  end;

  Result:=ErrorTxt;
end;

procedure BuildContentTreeFromBackupContent(
  PMySQL: Pointer;
  BackupTreeView: TTntTreeView;
  BackupContent: TMYX_BACKUP_CONTENT;
  SchemataFrame: TSchemataFrame;
  CompleteSchematas: Boolean);
var
  i, j: integer;
  Schemas: TList;
  PCatalogs: PMYX_CATALOGS;
  Catalogs: TMYX_CATALOGS;
  Catalog: TMYX_CATALOG;
  Schema: TMYX_SCHEMA;
  TableNode, SchemaNode, CatalogNode: TTntTreeNode;
begin
  ClearTreeView(BackupTreeView, myx_ndt_tobject);

  if(SchemataFrame<>nil)then
  begin
    Catalogs:=SchemataFrame.CatalogList;
  end
  else
  begin
    PCatalogs:=myx_get_catalogs(PMySQL);
    if(PCatalogs=nil)then
      raise EMyxSQLError.Create('Error while fetching Catalog information',
        myx_mysql_errno(PMySQL), myx_mysql_error(PMySQL));

    try
      Catalogs:=TMYX_CATALOGS.create(PCatalogs);
    finally
      myx_free_catalogs(PCatalogs);
    end;
  end;

  try
    //Add all schemas
    Schemas:=TList.Create;
    try
      Catalog:=nil;
      Schema:=nil;
      for i:=0 to BackupContent.tables.Count-1 do
      begin
        if(Catalog<>nil)and(Schema<>nil)then
        begin
          if(CompareText(
            BackupContent.tables[i].catalog,
            Catalog.catalog_name)=0)and
            (CompareText(
            BackupContent.tables[i].schema,
            Schema.schema_name)=0)then
          begin
            Continue;
          end;
        end;

        //Find Catalog

        Catalog:=nil;
        for j:=0 to Catalogs.catalogs.Count-1 do
          if(CompareText(
            Catalogs.catalogs[j].catalog_name,
            BackupContent.tables[i].catalog)=0)or
            ((CompareText(BackupContent.tables[i].catalog,
              'DEFAULT_CATALOG')=0)and(
              CompareText(Catalogs.catalogs[j].catalog_name,
              'def')=0))then
          begin
            Catalog:=Catalogs.catalogs[j];
            break;
          end;

        if(Catalog<>nil)then
        begin
          //Find Schema
          Schema:=nil;
          for j:=0 to Catalog.schemata.Count-1 do
            if(CompareText(
              Catalog.schemata[j].schema_name,
              BackupContent.tables[i].schema)=0)then
            begin
              Schema:=Catalog.schemata[j];
              break;
            end;

          //Add Schema if found and select it when CompleteSchematas is selected
          if(Schema<>nil)then
            AddSchemaToBackup(PMySQL, BackupTreeView, Schema, CompleteSchematas);
        end;
      end;
    finally
      Schemas.Free;
    end;
  finally
    if(SchemataFrame=nil)then
      Catalogs.Free;
  end;

    if(Not(CompleteSchematas))then
    begin
      //Select tables stored in profile
      for i:=0 to BackupTreeView.Items.Count-1 do
      begin
        if(BackupTreeView.Items[i].Data<>nil)then
          if(TObject(BackupTreeView.Items[i].Data) is TBackupNode)then
            if(TBackupNode(BackupTreeView.Items[i].Data).GetObjType=BOTTable)then
            begin
              TableNode:=BackupTreeView.Items[i];

              SchemaNode:=nil;
              if(TableNode.Parent<>nil)then
                if(TableNode.Parent.Data<>nil)then
                  if(TObject(TableNode.Parent.Data) is TBackupNode)then
                    if(TBackupNode(TableNode.Parent.Data).GetObjType=BOTSchema)then
                      SchemaNode:=TableNode.Parent;

              CatalogNode:=nil;
              if(SchemaNode<>nil)then
              begin
                if(SchemaNode.Parent<>nil)then
                  if(SchemaNode.Parent.Data<>nil)then
                    if(TObject(SchemaNode.Parent.Data) is TBackupNode)then
                      if(TBackupNode(SchemaNode.Parent.Data).GetObjType=BOTCatalog)then
                        CatalogNode:=SchemaNode.Parent;
              end;

              if(SchemaNode<>nil)and(CatalogNode<>nil)then
              begin
                for j:=0 to BackupContent.tables.Count-1 do
                begin
                  if(CompareText(
                    BackupContent.tables[j].table,
                    TBackupNode(TableNode.Data).GetObjName)=0)and
                    (CompareText(
                    BackupContent.tables[j].schema,
                    TBackupNode(SchemaNode.Data).GetObjName)=0)and
                    (CompareText(
                    BackupContent.tables[j].catalog,
                    TBackupNode(CatalogNode.Data).GetObjName)=0)then
                  begin
                    SetNodeSelectState(TableNode, BCTAll);

                    break;
                  end;
                end;
              end;
            end
            else if TBackupNode(BackupTreeView.Items[i].Data).GetObjType=BOTSchema then
            begin
              if BackupTreeView.Items[i].getFirstChild=nil then
              begin
                SetNodeSelectState(BackupTreeView.Items[i], BCTAll);
              end; 
            end;
      end;
    end;
end;

procedure BuildContentTreeFromRestoreContent(BackupTreeView: TTntTreeView;
  BackupContent: TMYX_BACKUP_CONTENT);
var i, pos: integer;
  Catalogs, Schematas: TTntStringList;
  CatalogNode, SchemaNode: TTntTreeNode;
  backup_node: TBackupNode;
begin
  ClearTreeView(BackupTreeView, myx_ndt_tobject);

  if(BackupTreeView.Items.Count=0)then
    AddTreeViewChildNode(BackupTreeView, nil, 'BackupRoot', -1, nil);

  Catalogs:=TTntStringList.Create;
  Schematas:=TTntStringList.Create;
  try
    for i:=0 to BackupContent.tables.Count-1 do
    begin
      pos:=Catalogs.IndexOf(BackupContent.tables[i].catalog);

      CatalogNode:=nil;
      if(pos=-1)then
      begin
        CatalogNode:=AddTreeViewChildNode(BackupTreeView, nil, //BackupTreeView.Items[0],
          '      '+BackupContent.tables[i].catalog, 2,
          TBackupNode.Create(BackupContent.tables[i].catalog, BOTCatalog));

        Catalogs.AddObject(BackupContent.tables[i].catalog, CatalogNode);
      end
      else
      begin
        if(Catalogs.Objects[pos]<>nil)then
          CatalogNode:=TTntTreeNode(Catalogs.Objects[pos]);
      end;

      if(CatalogNode<>nil)then
      begin
        pos:=Schematas.IndexOf(BackupContent.tables[i].schema);

        SchemaNode:=nil;

        if(pos=-1)then
        begin
          SchemaNode:=AddTreeViewChildNode(BackupTreeView, CatalogNode,
            '      '+BackupContent.tables[i].schema, 9,
            TBackupNode.Create(BackupContent.tables[i].schema, BOTSchema));

          Schematas.AddObject(BackupContent.tables[i].schema, SchemaNode);
        end
        else
        begin
          if(Schematas.Objects[pos]<>nil)then
            SchemaNode:=TTntTreeNode(Schematas.Objects[pos]);
        end;

        if(SchemaNode<>nil)then
        begin
          backup_node:=TBackupNode.Create(
            BackupContent.tables[i].table, BOTTable);

          AddTreeViewChildNode(BackupTreeView, SchemaNode, '      '+
            BackupContent.tables[i].table, 10,
            backup_node);
        end;
      end;
    end;

    for i:=0 to Catalogs.Count-1 do
      if(Catalogs.Objects[i]<>nil)then
      begin
        SetNodeSelectState(TTntTreeNode(Catalogs.Objects[i]), BCTAll);
        TTntTreeNode(Catalogs.Objects[i]).Expand(False);
      end;
  finally
    Schematas.Free;
    Catalogs.Free;
  end;
end;

procedure AddSchemaToBackup(PMySQL: Pointer;
  BackupTreeView: TTntTreeView;
  Schema: TMYX_SCHEMA; Selected: Boolean);
var CatalogNode, SchemaNode: TTntTreeNode;
  i: integer;
  ptable_status: PMYX_SCHEMA_TABLE_STATUS;
  table_status: TMYX_SCHEMA_TABLE_STATUS;
  backup_node: TBackupNode;
begin
  CatalogNode:=nil;

  if(BackupTreeView.Items.Count=0)then
    AddTreeViewChildNode(BackupTreeView, nil, 'BackupRoot', -1, nil);

  for i:=0 to BackupTreeView.Items.Count-1 do
    if(BackupTreeView.Items[i].Data<>nil)then
      if(TObject(BackupTreeView.Items[i].Data) is TBackupNode)then
        if(CompareText(TBackupNode(BackupTreeView.Items[i].Data).GetObjName,
          Schema.catalog_name)=0)then
        begin
          CatalogNode:=BackupTreeView.Items[i];
          break;
        end;


  if(CatalogNode=nil)then
  begin
    CatalogNode:=AddTreeViewChildNode(BackupTreeView, nil, //BackupTreeView.Items[0],
      '      '+Schema.catalog_name, 2,
      TBackupNode.Create(Schema.catalog_name, BOTCatalog));
  end;

  SchemaNode:=AddTreeViewChildNode(BackupTreeView, CatalogNode,
    '      '+Schema.schema_name, 9,
    TBackupNode.Create(Schema.schema_name, BOTSchema));

  //----------------------------------------------------------
  //Get Tables
  ptable_status:=myx_get_schema_table_status(PMySQL, '', Schema.schema_name);
  try
    table_status:=TMYX_SCHEMA_TABLE_STATUS.Create(ptable_status);

    for i:=0 to table_status.schema_tables.Count-1 do
    begin
      backup_node:=TBackupNode.Create(
        table_status.schema_tables[i].table_name, BOTTable);
      backup_node.Substrings:=TTntStringList.Create;
      backup_node.Substrings.Add(
        table_status.schema_tables[i].table_type);
      backup_node.Substrings.Add(
        table_status.schema_tables[i].rows);
      backup_node.Substrings.Add(
        table_status.schema_tables[i].data_length);
      backup_node.Substrings.Add(
        table_status.schema_tables[i].update_time);

      AddTreeViewChildNode(BackupTreeView, SchemaNode, '      '+
        table_status.schema_tables[i].table_name, 10,
        backup_node);
    end;

    //Select new Schema and all subnodes (tables)
    if(Selected)then
      SetNodeSelectState(SchemaNode, BCTAll);

    table_status.Free;
  finally
    myx_free_schema_table_status(ptable_status);
  end;

  //Expand nodes
  BackupTreeView.Items[0].Expand(False);
  CatalogNode.Expand(False);
end;

function GetBackupContent(BackupTreeView: TTntTreeView): TMYX_BACKUP_CONTENT;
var backup_content: TMYX_BACKUP_CONTENT;
  table: TMYX_BACKUP_TABLE;
  TableNode, SchemaNode, CatalogNode: TTntTreeNode;
  CatalogName, SchemaName: WideString;
  i: integer;
begin
  backup_content:=TMYX_BACKUP_CONTENT.create;
  try
    for i:=0 to BackupTreeView.Items.Count-1 do
    begin
      if(BackupTreeView.Items[i].Data<>nil)then
        if(TObject(BackupTreeView.Items[i].Data) is TBackupNode)then
          if(TBackupNode(BackupTreeView.Items[i].Data).GetObjType=BOTTable)and
            (TBackupNode(BackupTreeView.Items[i].Data).GetSelectState=BCTAll)then
          begin
            TableNode:=BackupTreeView.Items[i];

            SchemaName:='';
            CatalogName:='';

            SchemaNode:=TableNode.Parent;

            if(SchemaNode<>nil)then
              if(SchemaNode.Data<>nil)then
                if(TObject(SchemaNode.Data) is TBackupNode)then
                  if(TBackupNode(SchemaNode.Data).GetObjType=BOTSchema)then
                  begin
                    SchemaName:=Trim(SchemaNode.Text);

                    CatalogNode:=SchemaNode.Parent;
                    if(CatalogNode<>nil)then
                      if(CatalogNode.Data<>nil)then
                        if(TObject(CatalogNode.Data) is TBackupNode)then
                          if(TBackupNode(CatalogNode.Data).GetObjType=BOTCatalog)then
                            CatalogName:=Trim(CatalogNode.Text);
                  end;

            table:=TMYX_BACKUP_TABLE.create(CatalogName, SchemaName,
              Trim(TableNode.Text));
            //Add Table to backup_content
            backup_content.tables.Add(table);
          end
          else if(TBackupNode(BackupTreeView.Items[i].Data).GetObjType=BOTSchema)and
            (TBackupNode(BackupTreeView.Items[i].Data).GetSelectState=BCTAll)and
            (BackupTreeView.Items[i].getFirstChild=nil) then
          begin
            SchemaName:=Trim(BackupTreeView.Items[i].Text);
            CatalogName:=Trim(BackupTreeView.Items[i].Parent.Text);
            table:=TMYX_BACKUP_TABLE.create(CatalogName, SchemaName, '');
            //Add Table to backup_content
            backup_content.tables.Add(table);
          end;
    end;


  except
    backup_content.Free;
    raise;
  end;

  Result:=backup_content;
end;

procedure SetNodeSelectState(Node: TTntTreeNode; Selected: TBackupCheckType);
begin
  TBackupNode(Node.Data).SetSelectState(Selected);

  //Check/Uncheck all childs as well
  SetChildrenSelectState(Node);

  //See, if the current select state of the parent is valid
  CheckParentSelectState(Node);

  Node.TreeView.Invalidate;
end;

procedure SetChildrenSelectState(Node: TTntTreeNode);
var ChildNode: TTntTreeNode;
begin
  //Set Select state for all childs
  ChildNode:=Node.getFirstChild;
  while(ChildNode<>nil)do
  begin
    if(ChildNode.Data<>nil)then
      TBackupNode(ChildNode.Data).SetSelectState(TBackupNode(Node.Data).GetSelectState);

    //Call recursive if the child has another child
    if(ChildNode.getFirstChild<>nil)then
      SetChildrenSelectState(ChildNode);

    ChildNode:=Node.GetNextChild(ChildNode);
  end;
end;

procedure CheckParentSelectState(Node: TTntTreeNode);
begin
  if(Node.Parent<>nil)then
    if(Node.Parent.Data<>nil)then
    begin
      //Check, if all nodes under the parent are selected. If so, select the parent as well
      TBackupNode(Node.Parent.Data).SetSelectState(AllChildNodeSelected(Node.Parent));

      if(Node.Parent.Parent<>nil)then
        CheckParentSelectState(Node.Parent);
    end;
end;

function AllChildNodeSelected(Node: TTntTreeNode): TBackupCheckType;
var ChildNode: TTntTreeNode;
  AllSelected, AtLeastOneSelected: Boolean;
  SubChildsSelectState: TBackupCheckType;
begin
  AllSelected:=True;
  AtLeastOneSelected:=False;

  //check if all child nodes are selected
  ChildNode:=Node.getFirstChild;
  while(ChildNode<>nil)do
  begin
    if(ChildNode.Data<>nil)then
    begin
      AllSelected:=AllSelected and
        (TBackupNode(ChildNode.Data).GetSelectState=BCTAll);

      AtLeastOneSelected:=AtLeastOneSelected or
        (TBackupNode(ChildNode.Data).GetSelectState=BCTAll) or
        (TBackupNode(ChildNode.Data).GetSelectState=BCTSome);
    end;

    //Call recursive if the child has another child
    if(ChildNode.getFirstChild<>nil)then
    begin
      SubChildsSelectState:=AllChildNodeSelected(ChildNode);
      AllSelected:=AllSelected and (SubChildsSelectState=BCTAll);
      AtLeastOneSelected:=AtLeastOneSelected or
        (SubChildsSelectState=BCTAll) or
        (SubChildsSelectState=BCTSome);
    end;

    ChildNode:=Node.GetNextChild(ChildNode);
  end;

  if(AllSelected)then
    AllChildNodeSelected:=BCTAll
  else if(AtLeastOneSelected)then
    AllChildNodeSelected:=BCTSome
  else
    AllChildNodeSelected:=BCTNone;
end;

//------------------------------------------------------------------------------

constructor TBackupNode.Create(ObjName: WideString; ObjType: TBackupObjType);
begin
  inherited Create;

  self.ObjName:=ObjName;
  self.ObjType:=ObjType;
  self.Selected:=BCTNone;

  self.Substrings:=nil;
end;

destructor TBackupNode.Destroy;
begin
  if(Substrings<>nil)then
    Substrings.Free;

  inherited Destroy;
end;

function TBackupNode.GetObjName: WideString;
begin
  GetObjName:=ObjName;
end;

procedure TBackupNode.SetObjName(ObjName: WideString);
begin
  self.ObjName:=ObjName;
end;

function TBackupNode.GetObjType: TBackupObjType;
begin
  GetObjType:=ObjType;
end;

procedure TBackupNode.SetObjType(ObjType: TBackupObjType);
begin
  self.ObjType:=ObjType;
end;

function TBackupNode.GetSelectState: TBackupCheckType;
begin
  GetSelectState:=Selected;
end;

procedure TBackupNode.SetSelectState(theState: TBackupCheckType);
begin
  self.Selected:=theState;
end;


end.
