/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_gl_helper.cpp
 * @brief Helper functions for creating OpenGL data and structures out of XML data.
 */

#include "myx_gc.h"

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

#include "myx_gc_datatypes.h"
#include "myx_gc_gl_helper.h"
#include "myx_gc_utilities.h"
#include "myx_gc_font_manager.h"
#include "myx_gc_texture.h"
#include "myx_gc_const.h"
#include "myx_gc_utilities.h"

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve a float attribute.
bool getFloatAttribute(xmlNodePtr Element, const char* Name, float& Value)
{
  xmlChar* Attribute = xmlGetProp(Element, (xmlChar*) Name);
  if (Attribute == NULL)
    return false;
  Value = (float) atof((const char*) Attribute);
  xmlFree(Attribute);
  return true;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Helper method to retrieve an integer attribute. If it cannot be found a default value will be used instead.
 */
float getFloatAttributeDef(xmlNodePtr Element, const char* Name, float Default)
{
  float Result;
  if (!getFloatAttribute(Element, Name, Result))
    Result = Default;

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve an integer attribute.
bool getIntAttribute(xmlNodePtr Element, const char* Name, int& Value)
{
  xmlChar* Attribute = xmlGetProp(Element, (xmlChar*) Name);
  if (Attribute == NULL)
    return false;
  Value = atoi((const char*) Attribute);
  xmlFree(Attribute);
  return true;
}

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve an integer attribute. If it cannot be found a default value will be used instead.
int getIntAttributeDef(xmlNodePtr Element, const char* Name, int Default)
{
  int Result;
  if (!getIntAttribute(Element, Name, Result))
    Result = Default;

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve a string attribute.
// If the attribute could be found then true is returned and Value is set to the value of the attribute.
// Otherwise false is returned and Value is not touched.
bool getStringAttribute(xmlNodePtr Element, const char* Name, string& Value)
{
  xmlChar* Attribute = xmlGetProp(Element, (xmlChar*) Name);
  if (Attribute == NULL)
    return false;
  else
  {
    Value.clear();
    Value.append((char*) Attribute);
    return true;
  }
}

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve a string attribute. If the attribute is empty or cannot be found then a default value is returned.
string getStringAttributeDef(xmlNodePtr Element, const char* Name, const string Default)
{
  string Result;
  xmlChar* Value = xmlGetProp(Element, (xmlChar*) Name);
  if ((Value == NULL) || (*Value == '\0'))
    Result += Default;
  else
    Result += (char*) Value;

  if (Value != NULL)
    xmlFree(Value);

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Reads attribute Name from Element and tries to treat the string as a color.
 * The allowed syntax for colors is (as given by the SVG specification) either an HTML like
 * value (e.g. #FFFFFF, #FFF) or a function like form (e.g. rgb(100, 255, 255), rgb(10%, 100%, 0%)).
 *
 * @param Element The XML element to parse.
 * @param Name The name of the color attribute.
 * @param Color [out] The converted color.
 * @return 
 *   0 - If a color could be found and converted.
 *   1 - If a color could be found but a conversion error occured.
 *   2 - No color was given.
 *   3 - The special color "none" was found.
 */

int convertColor(xmlNodePtr Element, const char* Name, GLubyte* Color)
{
  int Result = 2;
  
  xmlChar* Attribute = xmlGetProp(Element, (xmlChar*) Name);
  if (Attribute != NULL)
  {
    Result = stringToColor((char*) Attribute, Color);
    xmlFree(Attribute);
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

// Parses the given XML node for texture information and creates a new entry in the texture manager.
void parseTextureEntry(xmlNodePtr XML)
{
  // Collect font information.
  string ID;
  string Filename;
  if (getStringAttribute(XML, "id", ID))
  {
    string WrapModeH  = getStringAttributeDef(XML, "wrap-mode-horizontal", DefaultTextureWrapMode);
    string WrapModeV  = getStringAttributeDef(XML, "wrap-mode-vertical", DefaultTextureWrapMode);
    string MinFilter = getStringAttributeDef(XML, "min-filter", DefaultTextureMinFilter);
    string MaxFilter = getStringAttributeDef(XML, "mag-filter", DefaultTextureMagFilter);
    int Dimensions = getIntAttributeDef(XML, "dimensions", DefaultTextureDimensions);
    string Mode = getStringAttributeDef(XML, "mode", DefaultTextureMode);

    // See if there are any LOD sub elements and if there is a LOD 0 entry as parameter.
    TLODList LODs;
    if (getStringAttribute(XML, "location", Filename))
    {
      // The location paremter in the texture tag always gives the top level of detail.
      LODs.push_back(Filename);
    };
    xmlNodePtr Run = XML->children;
    while (Run != NULL)
    {
      if (xmlStrcmp(Run->name, (const xmlChar *) "texture-lod") == 0)
      {
        TLODList::size_type Level = getIntAttributeDef(Run, "lod", 0);
        if (LODs.size() <= Level)
          LODs.resize(Level + 1);
        LODs[Level] = getStringAttributeDef(Run, "location", "");
      };
      Run = Run->next;
    };

    textureManager()->CreateTextureEntry(LODs, ID, WrapModeH, WrapModeV, MinFilter, MaxFilter, Dimensions, Mode);
  };
}

//----------------------------------------------------------------------------------------------------------------------

