/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_texture.cpp 
 * @brief Implementation of a texture class.
 * 
 */

#include "myx_gc.h"

#include "myx_gc_utilities.h"
#include "myx_gc_canvas.h"
#include "myx_gc_texture.h"

//----------------- CGCTexture -----------------------------------------------------------------------------------------

CGCTexture::CGCTexture(CTextureManager* Controller, const TLODList& LODData, const string& ID, GLenum WrapModeS, GLenum WrapModeT, 
                       GLenum MinFilter, GLenum MagFilter, int Dimensions, GLenum TextureMode)
{
  FManager = Controller;
  FLoaded = false;
  FName = ID;
  FLODList = LODData;
  FWrapModeS = WrapModeS;
  FWrapModeT = WrapModeT;
  FMinFilter = MinFilter;
  FMagFilter = MagFilter;
  FTarget = Dimensions == 1 ? GL_TEXTURE_1D : GL_TEXTURE_2D;
  FMode = TextureMode;

  glGenTextures(1, &FHandle);
  glBindTexture(FTarget, FHandle); 
  LoadTexture();
  glBindTexture(FTarget, 0); 
}

//----------------------------------------------------------------------------------------------------------------------

CGCTexture::~CGCTexture(void)
{
  glDeleteTextures(1, &FHandle);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 *  Rounds the given value up to the next power of two boundary.
 *
 *  @param Value The value to round up.
 *  @return Returns the rounded value.
 */
int roundUpToPowerOf2(int Value)
{
#define LOG2(value) (log(value) / log(2.0))

  double LogTwo = LOG2((double) Value);
  if (floor(LogTwo) < LogTwo)
    return (int) floor(pow(2.0, floor(LogTwo) + 1));
  else
    return Value;

#undef LOG2
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Delay loads texture data. Called from ActivateTexture, that is, when the texture is used the first time.
 * All level-of-detail texture data is loaded here and uploaded to OpenGL depending on the number of images given and
 * what is set for the minification filter.
 */
void CGCTexture::LoadTexture(void)
{
  FLoaded = true;
  TImage* image;
  unsigned char* Buffer;

  if ((FMinFilter == GL_NEAREST) || (FMinFilter == GL_LINEAR) || (FLODList.size() < 2))
  {
    // No mip-mapping.
    if (FLODList.size() > 0)
    {
      image = LoadTextureImage(FLODList[0], Buffer);
      if (image != NULL)
      {
        if (FTarget == GL_TEXTURE_1D)
        {
          if ((FMinFilter == GL_NEAREST) || (FMinFilter == GL_LINEAR))
            glTexImage1D(GL_TEXTURE_1D, 0, image->Format, image->Width, 0, image->Format, GL_UNSIGNED_BYTE, Buffer);
          else
            gluBuild1DMipmaps(GL_TEXTURE_1D, image->Format, image->Width, image->Format, GL_UNSIGNED_BYTE, Buffer);
        }
        else
        {
          if ((FMinFilter == GL_NEAREST) || (FMinFilter == GL_LINEAR))
            glTexImage2D(GL_TEXTURE_2D, 0, image->Format, image->Width, image->Height, 0, image->Format, 
              GL_UNSIGNED_BYTE, Buffer);
          else
            gluBuild2DMipmaps(GL_TEXTURE_2D, image->Format, image->Width, image->Height, image->Format, 
              GL_UNSIGNED_BYTE, Buffer);
        };

        if (Buffer != image->Data)
          free(Buffer);
        freeImage(image);
      };
    };
  }
  else
  {
    TLODList::size_type I;

    // Prepare mip-mapping pyramid using the size of the LOD 0 texture. All files must have the same color format!
    image = LoadTextureImage(FLODList[0], Buffer);
    if (image != NULL)
    {
      GLenum Format = image->Format;
      gluBuild2DMipmaps(GL_TEXTURE_2D, Format, image->Width, image->Height, Format, GL_UNSIGNED_BYTE, Buffer);
      freeImage(image);

      for (I = 0; I < FLODList.size(); I++)
      {
        image = LoadTextureImage(FLODList[I], Buffer);
        if (image != NULL && image->Format == Format)
        {
          if (FTarget == GL_TEXTURE_1D)
            glTexImage1D(GL_TEXTURE_1D, I, Format, image->Width, 0, Format, GL_UNSIGNED_BYTE, Buffer);
          else
            glTexImage2D(GL_TEXTURE_2D, I, Format, image->Width, image->Height, 0, Format, GL_UNSIGNED_BYTE, Buffer);
          if (Buffer != image->Data)
            free(Buffer);
          freeImage(image);
        };
      };
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Loads the image data referenced by Name and returns it. If the either size of the image is not a power of 2 then
 * the image is scaled up so that it becomes this size.
 *
 * @param Name The name of the file to load.
 * @param Buffer A variable that gets either the the address of the actual image data (TImage->Data) or a new memory
 *               reference if the image must be scaled. The caller is responsible to free this buffer if it differs 
 *               from TImage->Data.
 * @return An image structure containing the actual image data. The caller is responsible to free this stucture via 
 *         freeImage if it is non null. If the image could not be loaded then the result is NULL and the content of 
 *         Buffer is not touched.
 */
TImage* CGCTexture::LoadTextureImage(const string& Name, unsigned char*& Buffer)
{
  TImage* image = loadPNG(Name);

  if (image != NULL)
  {
    unsigned int actualWidth = roundUpToPowerOf2(image->Width);
    unsigned int actualHeight = roundUpToPowerOf2(image->Height);

    switch (image->ColorType)
    {
      case COLOR_TYPE_PALETTE:
        {
          image->Format = GL_COLOR_INDEX;
          break;
        };
      case COLOR_TYPE_GRAY:
        {
          image->Format = GL_LUMINANCE;
          break;
        };
      case COLOR_TYPE_GRAY_ALPHA:
        {
          image->Format = GL_LUMINANCE_ALPHA;
          break;
        };
      case COLOR_TYPE_RGB:
        {
          image->Format = GL_RGB;
          break;
        };
      case COLOR_TYPE_RGB_ALPHA:
        {
          image->Format = GL_RGBA;
          break;
        };
    };

    if (actualWidth != image->Width || actualHeight != image->Height)
    {
      Buffer = (unsigned char*) malloc(actualWidth * actualHeight * image->Channels);
      gluScaleImage(image->Format, image->Width, image->Height, GL_UNSIGNED_BYTE, image->Data, actualWidth, actualHeight, 
        GL_UNSIGNED_BYTE, Buffer);
      image->Width = actualWidth;
      image->Height = actualHeight;
    }
    else
      Buffer = image->Data;
  };

  return image;
}

//----------------------------------------------------------------------------------------------------------------------

// Activates this texture in OpenGL so all following vertex definitions are textured using this texture.
// If the texture has not been loaded yet it will be done now. Additionally, texture mode is enabled in OpenGL.
void CGCTexture::ActivateTexture(void)
{                                                          
  glEnable(FTarget);
  glBindTexture(FTarget, FHandle); 
  
  glTexParameteri(FTarget, GL_TEXTURE_WRAP_S, FWrapModeS);
  if (FTarget == GL_TEXTURE_2D)
    glTexParameteri(FTarget, GL_TEXTURE_WRAP_T, FWrapModeT);
	glTexParameteri(FTarget, GL_TEXTURE_MIN_FILTER, FMinFilter);
	glTexParameteri(FTarget, GL_TEXTURE_MAG_FILTER, FMagFilter);

  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, FMode);
}

//----------------------------------------------------------------------------------------------------------------------

// Deactivates this texture and the texture mode in OpenGL.
void CGCTexture::DeactivateTexture(void)
{
  glDisable(FTarget);
  glBindTexture(FTarget, 0);
}

//----------------- CTextureManager ------------------------------------------------------------------------------------

CTextureManager InternalTextureManager; // Singleton texture manager instance.

CTextureManager* textureManager()
{
  return &InternalTextureManager;
}

//----------------------------------------------------------------------------------------------------------------------

CTextureManager::~CTextureManager(void)
{
  ClearTextures();
}

//----------------------------------------------------------------------------------------------------------------------

// Looks throught the textures and attempts to find one with the given name.
CGCTexture* CTextureManager::FindTexture(const string& Name)
{
  CTextureIterator Iterator = FTextures.find(Name);
  if (Iterator == FTextures.end())
    return NULL;
  else
    return Iterator->second;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a new texture entry and adds the entry to the texture list. No image data is loaded yet as this will happen 
 * when the texture is used the first time.
 *
 * @param LODData A list of level identifiers and names of files, which contain the image data for that level.
 * @param ID The identifier (name) of the texture.
 * @param WrapH Horizontal wrap mode as string.
 * @param WrapV Vertical wrap mode as string.
 * @param MinificationFilter The filter mode for minification as string.
 * @param MagnificationFilter The filter mode for magnification as string.
 * @param Dimensions The number of dimensions of the image data (must be 1 or 2).
 * @param Mode The mode how the texture must be applied as string.
 * return The newly created texture object.
 */
CGCTexture* CTextureManager::CreateTextureEntry(const TLODList& LODData, const string& ID, const string& WrapH, const string& WrapV, 
                                               const string& MinificationFilterStr, const string& MagnificationFilterStr,
                                               int Dimensions, const string& Mode)
{
  // These both constants are only available with OpenGL 1.2 or up and are not defined in the standard gl.h header file.
  #define GL_CLAMP_TO_BORDER 0x812D
  #define GL_CLAMP_TO_EDGE   0x812F

  static map<string, GLenum> Mapper;
  map<string, GLenum>::const_iterator Iterator;

  // Fill our lookup table first if not yet done.
  if (Mapper.size() == 0)
  {
    Mapper["clamp"] = GL_CLAMP;
    Mapper["clamp-to-border"] = GL_CLAMP_TO_BORDER;
    Mapper["clamp-to-edge"] = GL_CLAMP_TO_EDGE;
    Mapper["repeat"] = GL_REPEAT;
    Mapper["nearest"] = GL_NEAREST;
    Mapper["linear"] = GL_LINEAR;
    Mapper["nearest-mipmap-nearest"] = GL_NEAREST_MIPMAP_NEAREST;
    Mapper["linear-mipmap-nearest"] = GL_LINEAR_MIPMAP_NEAREST;
    Mapper["nearest-mipmap-linear"] = GL_NEAREST_MIPMAP_LINEAR;
    Mapper["linear-mipmap-linear"] = GL_LINEAR_MIPMAP_LINEAR;
    Mapper["decal"] = GL_DECAL;
    Mapper["modulate"] = GL_MODULATE;
    Mapper["blend"] = GL_BLEND;
    Mapper["replace"] = GL_REPLACE;
  };

  GLenum WrapModeS = GL_CLAMP;
  Iterator = Mapper.find(WrapH);
  if (Iterator != Mapper.end())
    WrapModeS = Iterator->second;

  GLenum WrapModeT = GL_CLAMP;
  Iterator = Mapper.find(WrapV);
  if (Iterator != Mapper.end())
    WrapModeT = Iterator->second;

  GLenum MinFilter = GL_NEAREST;
  Iterator = Mapper.find(MinificationFilterStr);
  if (Iterator != Mapper.end())
    MinFilter = Iterator->second;

  GLenum MagFilter = GL_NEAREST;
  Iterator = Mapper.find(MagnificationFilterStr);
  if (Iterator != Mapper.end())
    MagFilter = Iterator->second;

  GLenum TextureMode = GL_DECAL;
  Iterator = Mapper.find(Mode);
  if (Iterator != Mapper.end())
    TextureMode = Iterator->second;

  CGCTexture* Result = new CGCTexture(this, LODData, ID, WrapModeS, WrapModeT, MinFilter, MagFilter, Dimensions, TextureMode);
  FTextures[ID] = Result;

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

void CTextureManager::ClearTextures(void)
{
  for (CTextureIterator Iterator = FTextures.begin(); Iterator != FTextures.end(); Iterator++)
    delete Iterator->second;

  FTextures.clear();
}

//----------------------------------------------------------------------------------------------------------------------

