#import "MQResultSetView.h"
#import "MQResultSetDataSource.h"
#import "MQFieldViewer.h"
#import "MQTableView.h"
#import "MQIndicatorCell.h"
#import "MQResultSetCell.h"

#include <myx_public_interface.h>

#import <MySQLToolsCommon/mxUtils.h>

#import "MAccessoryScrollView.h"
#import "MSplitView.h"


static int editBarTags[]= {
  49, 50, 51, 52, 53, 54, 55
};

typedef struct {
  int error;
  MYX_QUERY_ERROR_LEVEL level;
  NSString *message;
  unsigned int row;
  unsigned int column;
} MessageItem;


@interface MessageDS : NSObject
{
@public
  MessageItem *items;
  int itemCount;
  NSImage *images[3];
}

- (int)numberOfRowsInTableView:(NSTableView *)aTableView;
- (id)tableView:(NSTableView *)aTableView 
objectValueForTableColumn:(NSTableColumn *)aTableColumn
			row:(int)rowIndex;
- (void)addError:(int)error
          ofType:(MYX_QUERY_ERROR_LEVEL)level
		 message:(NSString*)message
           atRow:(unsigned int)row
          column:(unsigned int)column;
- (void)reset;

@end

@implementation MessageDS

- (id)init
{
  self= [super init];
  if (self)
  {
	items= NULL;
	itemCount= 0;
    images[0]= [[NSImage imageNamed:@"rs_notice.png"] retain];
    images[1]= [[NSImage imageNamed:@"rs_warning.png"] retain];
    images[2]= [[NSImage imageNamed:@"rs_error.png"] retain];
  }
  return self;
}


- (void)dealloc
{
  int i;  
  for (i= 0; i < itemCount; i++)
	[items[i].message release];
  [images[0] release];
  [images[1] release];
  [images[2] release];
  g_free(items);
  [super dealloc];
}


- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
  return itemCount;
}


- (id)tableView:(NSTableView *)aTableView 
objectValueForTableColumn:(NSTableColumn *)aTableColumn
			row:(int)rowIndex
{
  if ([[aTableColumn identifier] isEqualTo:@"error"])
	return [NSNumber numberWithInt:items[rowIndex].error];
  else
	return items[rowIndex].message;
}


- (void)addError:(int)error
          ofType:(MYX_QUERY_ERROR_LEVEL)level
		 message:(NSString*)message
           atRow:(unsigned int)row
          column:(unsigned int)column
{
  itemCount++;
  items= (MessageItem*)g_realloc(items, sizeof(MessageItem)*itemCount);
  items[itemCount-1].error= error;
  items[itemCount-1].level= level;
  items[itemCount-1].message= [message retain];
  items[itemCount-1].row= row;
  items[itemCount-1].column= column;
}


- (void)tableView:(NSTableView *)aTableView
  willDisplayCell:(id)aCell
   forTableColumn:(NSTableColumn *)aTableColumn
              row:(int)rowIndex
{
  if ([[aTableColumn identifier] isEqualTo:@"error"])
  {
    switch (items[rowIndex].level)
    {
      case MYX_QEL_NOTE:
        [aCell setImage:images[0]];
        break;
      case MYX_QEL_WARNING:
        [aCell setImage:images[1]];
        break;
      case MYX_QEL_ERROR:
        [aCell setImage:images[2]];
        break;
    }
  }
}

- (void)reset
{
  int i;
  for (i= 0; i < itemCount; i++)
	[items[i].message release];
  g_free(items);
  itemCount= 0;
  items= NULL;
}

@end


//=============================================================================


@implementation MQResultSetView

- (void)updateButtonBar
{
  if (_dataSource)
  {
	if ([_dataSource editing])
	{
	  [[view viewWithTag:50] setEnabled:NO];
	  [[view viewWithTag:51] setEnabled:YES];
	  [[view viewWithTag:52] setEnabled:YES];
	}
	else
	{
	  if ([_dataSource isEditable])
		[[view viewWithTag:50] setEnabled:YES];
	  else
		[[view viewWithTag:50] setEnabled:NO];
	  [[view viewWithTag:51] setEnabled:NO];
	  [[view viewWithTag:52] setEnabled:NO];
	}
	[[view viewWithTag:53] setEnabled:YES];
	[[view viewWithTag:54] setEnabled:YES];
	[[view viewWithTag:55] setEnabled:NO]; // SEARCH not supported yet
  }
  else
  {
	[[view viewWithTag:50] setEnabled:NO];
	[[view viewWithTag:51] setEnabled:NO];
	[[view viewWithTag:52] setEnabled:NO];
	
	[[view viewWithTag:53] setEnabled:NO];
	[[view viewWithTag:54] setEnabled:NO];
	[[view viewWithTag:55] setEnabled:NO];	
  }
}

- (void)setMessagesShown:(BOOL)flag
{
  if (flag)
    [splitView resizeSubview:[[messageTable superview] superview] toSize:70];
  else
    [splitView resizeSubview:[[messageTable superview] superview] toSize:0];
}

- (id)initWithConnectionTo:(MYX_USER_CONNECTION*)info
{
  self= [super init];
  if (self)
  {
    int i;
    
	if (![NSBundle loadNibNamed:@"ResultSet" owner:self])
	{
	  NSLog(@"Could not load nib ResultSet.nib");
	  [self release];
	  return nil;
	}
    
	_mysql= myx_mysql_init();
	if (!_mysql)
	{
	  NSLog(@"could not create MySQL connection");
	  [self release];
	  return nil;
	}
	if (myx_connect_to_instance(info, _mysql) < 0)
	{
	  MXRunAlertPanelWithMySQLError(@"Error",@"Could not connect to MySQL instance.",_mysql);
	  [self release];
      myx_mysql_close(_mysql);
	  return nil;
	}
    
	_messageDS= [[MessageDS alloc] init];
	[messageTable setDataSource:_messageDS];
    [messageTable setDelegate:_messageDS];

    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(selectedError:) 
                                                 name:NSTableViewSelectionDidChangeNotification
                                               object:messageTable];

	[self updateButtonBar];

	_statusColors[MESUnchanged] = [[NSColor colorWithDeviceRed:0.92 green:0.92 blue:1.0 alpha:1.0] retain];
	_statusColors[MESChanged] = [[NSColor colorWithDeviceRed:0.8 green:0.8 blue:1.0 alpha:1.0] retain];
    _statusColors[MESPlaceHolder] = [[NSColor colorWithDeviceRed:0.95 green:1.0 blue:0.95 alpha:1.0] retain];
	_statusColors[MESAdded] = [[NSColor colorWithDeviceRed:0.8 green:1.0 blue:0.8 alpha:1.0] retain];
	_statusColors[MESDeleted] = [[NSColor colorWithDeviceRed:1.0 green:0.8 blue:0.8 alpha:1.0] retain];
    
    _viewers= [[NSMutableArray alloc] init];
    
    [splitView setInactiveColor:[table gridColor]];
    [splitView setDividerThickness:2.0];

    [self setMessagesShown:NO];
    [self changeSplitDirection:MQDNone];
    
    [table setIntercellSpacing:NSMakeSize(0,0)];
    
    for (i= 0; i < 7; i++)
      _editBarSizes[i]= [[view viewWithTag:editBarTags[i]] frame].size;
  
//#ifdef MAC_OS_X_VERSION_10_4
//    [table setColumnAutoresizingStyle:NSTableViewNoColumnAutoresizing];
//#else
//    [table setAutoresizesAllColumnsToFit:NO];
//#endif
  }
  return self;
}


- (void)dealloc
{
  int i;
  for (i= 0; i < 4; i++)
	[_statusColors[i] release];
  [_query release];
  if (_mysql)
	myx_mysql_close(_mysql);
  [_dataSource release];
  
  [[NSNotificationCenter defaultCenter] removeObserver:self];
  
  [_viewers makeObjectsPerformSelector:@selector(close)];
  [_viewers release];
  
  [_messageDS release];
  
  [super dealloc];
}


- (void)changeSplitDirection:(MQRSSplitDirection)newDir
{
  NSButton *button;
  float size= [NSScroller scrollerWidth];
  
  [sview removeAccessoryViews];
  
  if (newDir == MQDNone || newDir == MQDHorizontal)
  {
    button= [[[NSButton alloc] initWithFrame:NSMakeRect(0,0,size,size)] autorelease];
    [button setButtonType: NSMomentaryChangeButton];
    [button setBordered:NO];
    [[button cell] setImagePosition:NSImageOnly];
    [button setImage: [NSImage imageNamed: @"sbar_splitv.png"]];
    
    [button setTarget:self];
    [button setAction:@selector(splitH:)];

    [sview addSubview:button];
    [sview addVerticalAccessoryView:button];
  }

  if (newDir == MQDNone || newDir == MQDVertical)
  {
    button= [[[NSButton alloc] initWithFrame:NSMakeRect(0,0,size,size)] autorelease];
    [button setButtonType: NSMomentaryChangeButton];
    [button setBordered:NO];
    [[button cell] setImagePosition:NSImageOnly];
    [button setImage: [NSImage imageNamed: @"sbar_splith.png"]];

    [button setTarget:self];
    [button setAction:@selector(splitV:)];

    [sview addSubview:button];
    [sview addHorizontalAccessoryView:button];
  }

  if (_closeable)
  {
    button= [[[NSButton alloc] initWithFrame:NSMakeRect(0,0,size,size)] autorelease];
    [button setButtonType: NSMomentaryChangeButton];
    [button setBordered:NO];
    [[button cell] setImagePosition:NSImageOnly];
    if (newDir == MQDHorizontal)
      [button setImage: [NSImage imageNamed: @"sbar_unsplitv.png"]];
    else
      [button setImage: [NSImage imageNamed: @"sbar_unsplith.png"]];
    
    [button setTarget:self];
    [button setAction:@selector(unsplit:)];

    [sview addSubview:button];
    if (newDir == MQDHorizontal)
      [sview addVerticalAccessoryView:button];
    else
      [sview addHorizontalAccessoryView:button];
  }

  if (newDir == MQDNone || newDir == MQDHorizontal)
    [self setCompact:NO];
  else
    [self setCompact:YES];

  _direction= newDir;
}

- (void)setCloseable:(BOOL)flag
{
  _closeable= flag;
  [self changeSplitDirection:_direction];
}

- (BOOL)closeable
{
  return _closeable;
}

- (void)splitV:(id)sender
{
  [_delegate performSelector:@selector(splitV:) withObject:self];
}

- (void)splitH:(id)sender
{
  [_delegate performSelector:@selector(splitH:) withObject:self];
}

- (void)unsplit:(id)sender
{
  [_delegate performSelector:@selector(unsplit:) withObject:self];
}

- (IBAction)addRow:(id)sender
{
  [_dataSource addRow];
  [table reloadData];
}

- (IBAction)deleteRow:(id)sender
{
  [_dataSource deleteRow:[table selectedRow]];
  [table reloadData];
}

- (IBAction)copyField:(id)sender
{
  unsigned int row= [table selectedRow];
  int column= [table lastClickedColumn]-1;
  MYX_RESULTSET *rs= [_dataSource resultset];
  NSPasteboard *pboard;
  
  if (column < 0)
    return;
  
  pboard= [NSPasteboard generalPasteboard];
    
  if (rs->columns[column].column_type == MYX_RSCT_BLOB)
  {
    NSData *data= [NSData dataWithBytes:rs->rows[row].fields[column].value
                                 length:rs->rows[row].fields[column].value_length];
    
    if ([NSImageRep imageRepClassForData:data])
    {
      NSImage *image= [[NSImage alloc] initWithData:data];
      
      [pboard declareTypes:[NSArray arrayWithObject:NSTIFFPboardType] owner:nil];
      
      [pboard setData:[image TIFFRepresentationUsingCompression:NSTIFFCompressionLZW
                                                         factor:1.0]
              forType:NSTIFFPboardType];
      [image release];
    }
  }
  else
  {
    [pboard declareTypes:[NSArray arrayWithObject:NSStringPboardType] owner:nil];
    
    [pboard setString:[[[NSString alloc] initWithCStringNoCopy:rs->rows[row].fields[column].value
                                                        length:rs->rows[row].fields[column].value_length
                                                  freeWhenDone:NO] autorelease]
              forType:NSStringPboardType];
  }
}

- (IBAction)saveField:(id)sender
{
  NSSavePanel *panel= [NSSavePanel savePanel];
  unsigned int row= [table selectedRow];
  int column= [table lastClickedColumn]-1;
  MYX_RESULTSET *rs= [_dataSource resultset];  
  NSData *data= [NSData dataWithBytes:rs->rows[row].fields[column].value
                               length:rs->rows[row].fields[column].value_length];
  
  [panel setTitle:@"Save Value to File"];
  if ([panel runModal] == NSFileHandlingPanelOKButton)
  {
    [data writeToFile:[panel filename] atomically:NO];
  }  
}

- (IBAction)loadField:(id)sender
{
  NSOpenPanel *panel= [NSOpenPanel openPanel];
  
  [panel setTitle:@"Load Value from File"];
  if ([panel runModal] == NSFileHandlingPanelOKButton)
  {
    id data;
    unsigned int row= [table selectedRow];
    int column= [table lastClickedColumn];
    MYX_RESULTSET *rs= [_dataSource resultset];  
    
    if (rs->columns[column-1].column_type == MYX_RSCT_BLOB)
      data= [NSData dataWithContentsOfFile:[panel filename]];
    else
      data= [NSString stringWithContentsOfFile:[panel filename]];
    
    [_dataSource tableView:table
            setObjectValue:data
            forTableColumn:[table tableColumnWithIdentifier:[NSNumber numberWithInt:column-1]]
                       row:row];
    
    [table setNeedsDisplay];

    [statusLabel setStringValue:@"Loaded file into field."];
  }
}

- (IBAction)clearField:(id)sender
{
  unsigned int row= [table selectedRow];
  int column= [table lastClickedColumn];
  
  [_dataSource tableView:table
          setObjectValue:nil
          forTableColumn:[table tableColumnWithIdentifier:[NSNumber numberWithInt:column-1]]
                     row:row];
  [table setNeedsDisplay];
}

- (void)editSelectedField:(BOOL)editable
{
  MQFieldViewer *viewer= [[MQFieldViewer alloc] init];
  unsigned int row= [table selectedRow];
  int column= [table lastClickedColumn]-1;
  MYX_RESULTSET *rs= [_dataSource resultset];
  NSData *data;
  
  if (column < 0)
    return;
  
  data= [NSData dataWithBytes:rs->rows[row].fields[column].value
                       length:rs->rows[row].fields[column].value_length];
  
  if (rs->columns[column].column_type == MYX_RSCT_BLOB)
    [viewer showData:data editable:editable];
  else    
    [viewer showTextData:data editable:editable];
  [viewer setInfo:[NSDictionary dictionaryWithObjectsAndKeys:
    [NSString stringWithUTF8String:rs->columns[column].name], @"columnName",
    [NSNumber numberWithInt:column], @"column",
    [NSNumber numberWithInt:row], @"row",
    nil]];
  [viewer setDelegate:self];
  [viewer showWindow:nil];
  
  [_viewers addObject:viewer];
  [viewer release];  
}

- (IBAction)editField:(id)sender
{
  [self editSelectedField:YES];
}

- (IBAction)viewField:(id)sender
{
  [self editSelectedField:NO];
}

- (IBAction)editRS:(id)sender
{
  [_dataSource setEditing:YES];
  [self updateButtonBar];
  [statusLabel setStringValue:@"Editing enabled."];
  [table reloadData];
}

- (IBAction)commitRS:(id)sender
{
  [_dataSource setEditing:NO];
  [self updateButtonBar];

  [statusLabel setStringValue:@"Saving changes..."];
  
  [_dataSource preCommit];
  
  [self performApply];
}

- (IBAction)discardRS:(id)sender
{
  [_dataSource setEditing:NO];
  [self updateButtonBar];
  [_dataSource discardChanges];
  [table reloadData];
  [statusLabel setStringValue:@"Discarded changes."];
}

- (IBAction)goFirst:(id)sender
{
  if ([_dataSource numberOfRowsInTableView:table] > 0)
	[table selectRow:0 byExtendingSelection:NO];
}

- (IBAction)goLast:(id)sender
{
  if ([_dataSource numberOfRowsInTableView:table] > 0)
	[table selectRow:[_dataSource numberOfRowsInTableView:table]-1
byExtendingSelection:NO];
}

- (MYSQL*)mysql
{
  return _mysql;
}

- (NSString*)defaultCatalog
{
  return @"def";
}

- (void)setDefaultSchema:(NSString*)schema
{
  if (_defaultSchema != schema)
  {
    [_defaultSchema release];
    _defaultSchema= [schema retain];
  }
  myx_use_schema(_mysql, [schema UTF8String]);
}

- (NSString*)defaultSchema
{
  return _defaultSchema;
}

- (id)view
{
  return view;
}

- (MQTableView*)tableView
{
  return table;
}

- (void)setActive:(BOOL)flag
{
  _active= flag;
  if (flag)
    [table setBackgroundColor:[NSColor whiteColor]];
  else
    [table setBackgroundColor:[table gridColor]];
  [splitView setActive:flag];
  [splitView setNeedsDisplay:YES];

  if (flag)
    [_delegate performSelector:@selector(resultSetViewDidGetFocus:) withObject:self];
}


- (BOOL)active
{
  return _active;
}


- (void)setDelegate:(id)delegate
{
  _delegate= delegate;
}


- (void)queryProgress:(NSDictionary*)info
{
  
}

static int fetchProgress(unsigned long current_row_count, 
						 unsigned long previous_row_count, 
						 MYX_RESULTSET *result_set, void *user_data)
{
  NSDictionary *dict;
  
  dict= [NSDictionary dictionaryWithObjectsAndKeys:
	[NSNumber numberWithLong:current_row_count], @"current",
	[NSNumber numberWithLong:previous_row_count], @"previous",
	[NSValue valueWithPointer:result_set], @"resultset",
	nil];
  
  [(id)user_data performSelectorOnMainThread:@selector(queryProgress:)
								  withObject:dict
							   waitUntilDone:NO];
  
  return 0;
}


- (void)displayResults:(NSDictionary*)info
{
  MYX_RESULTSET *rs= (MYX_RESULTSET*)[[info valueForKey:@"resultset"] pointerValue];
  MYX_LIB_ERROR error= (MYX_LIB_ERROR)[[info valueForKey:@"error"] intValue];
  unsigned int i;
  NSArray *columns= [table tableColumns];
  NSTableColumn *column;
  NSFont *font;
  
  _busy= NO;

  if (!rs)
  {
    MYX_MYSQL_ERROR_MSGS *errors;
    errors= myx_mysql_error_msgs_fetch(_mysql);	
    
    if (error != MYX_NO_ERROR && !errors)
    {
      [statusLabel setStringValue:[NSString stringWithFormat:@"Empty resultset."]];
      
      [table setDataSource:nil];
      [_dataSource release];
      _dataSource= nil;
    }
    else
    {
      unsigned int i;
      
      [statusLabel setStringValue:[NSString stringWithFormat:@"Error executing query."]];
      if (errors)
      {
        for (i= 0; i < errors->errors_num; i++)
        {
          [_messageDS addError:errors->errors[i].error
                        ofType:errors->errors[i].level
                       message:[NSString stringWithUTF8String:errors->errors[i].text]
                         atRow:(unsigned int)-1
                        column:(unsigned int)-1];
        }
        myx_mysql_error_msgs_free(errors);
      }
      else
        [_messageDS addError:-1 
                      ofType:MYX_QEL_ERROR
                     message:MXGetErrorString(error) 
                       atRow:(unsigned int)-1
                      column:(unsigned int)-1];
      
      [messageTable reloadData];
      [self setMessagesShown:YES];
    }
    [_delegate performSelector:@selector(resultSetViewFinishedQuery:error:) withObject:self withObject:(id)YES];
	return;
  }
  [_dataSource release];

  _dataSource= [[MQResultSetDataSource alloc] initWithResultSet:rs];
  while ([columns count] > 0)
	[table removeTableColumn:[columns objectAtIndex:0]];
  
  column=  [[NSTableColumn alloc] initWithIdentifier:@"indicator"];
  [column setDataCell:[[[MQIndicatorCell alloc] init] autorelease]];
  [column setWidth:40];
  [column setMinWidth:40];
  [column setMaxWidth:40];
  [column setEditable:NO];
  [[column headerCell] setTitle:@""];
  [table addTableColumn:column];
  
  font= [NSFont systemFontOfSize:11];
  
  for (i= 0; i < rs->columns_num; i++)
  {
	column= [[NSTableColumn alloc] initWithIdentifier:[NSNumber numberWithInt:i]];
	NSString *title= [NSString stringWithUTF8String:rs->columns[i].name];
	float width= 0.0;
	float twidth;
    unsigned int j;
    
    [column setDataCell:[[[MQResultSetCell alloc] init] autorelease]];
    [[column dataCell] setEditable:YES];
  
	switch (rs->columns[i].column_type)
	{
	  case MYX_RSCT_INTEGER:
		width= [font widthOfString:@"0000000000"];
		break;
	  case MYX_RSCT_FLOAT:
		width= [font widthOfString:@"0000000000"];
		break;
	  case MYX_RSCT_STRING:
        width= 0;
        for (j= 0; j < rs->rows_num; j++)
        {
          float w= [font widthOfString:[NSString stringWithUTF8String:rs->rows[j].fields[i].value]];
          if (width < w)
            width= w;
        }
        if (width == 0.0)
        {
          width= [font widthOfString:@"XYZW"] * rs->columns[i].type_size / 4;
        }
        else
          width+= 4.0;
        if (width < 40.0)
          width= 40.0;
		break;
	  case MYX_RSCT_DATE:
		width= [font widthOfString:@"00/00/0000"];
		break;
	  case MYX_RSCT_TIME:
		width= [font widthOfString:@"00:00:00.0000"];
		break;
	  case MYX_RSCT_DATETIME:
		width= [font widthOfString:@"00/00/0000  00:00:00.0000"];
		break;
	  case MYX_RSCT_BLOB:
        [[column dataCell] setIsBlob:YES];
		width= 50;
		break;
	  case MYX_RSCT_TEXT:
		width= 250;
		break;
	  case MYX_RSCT_ENUM:
		width= 100;
		break;
	  case MYX_RSCT_SET:
		width= 100;
		break;
	}

    [[column dataCell] setDrawsBackground:YES];
    [[column dataCell] setFont: font];

	twidth= [[[column headerCell] font] widthOfString:title];
	width= MIN(MAX(width, twidth), 250);
    [[column headerCell] setWraps:NO];

	if (rs->columns[i].table_column && rs->columns[i].table_column->is_pk)
	  [[column headerCell] setStringValue:[NSString stringWithFormat:@"%@%@",
		[NSString stringWithUTF8String:"\xe2\x80\xbb"], title]];
	else
	  [[column headerCell] setStringValue:title];

	[table addTableColumn:column];
    [column setWidth:width];
  }

  [table setDataSource:_dataSource];
  [table reloadData];
  
  [statusLabel setStringValue:[NSString stringWithFormat:@"%i rows fetched.", 
	rs->rows_num]];

  [_delegate performSelector:@selector(resultSetViewFinishedQuery:error:)
                  withObject:self
                  withObject:(id)NO];
  
  [self updateButtonBar];
}

- (void)queryFinished:(NSDictionary*)info
{
  // we're being called from a performSelectorOnMainThread, so
  // set up to display the results later, when we're already running normally
  [[NSRunLoop currentRunLoop] performSelector:@selector(displayResults:)
									   target:self
									 argument:info
										order:100
										modes:[NSArray arrayWithObject:NSDefaultRunLoopMode]];
}

- (void)doQuery:(NSDictionary*)args
{
  MYSQL *mysql= (MYSQL*)[[args objectForKey:@"mysql"] pointerValue];
  NSString *query= [args objectForKey:@"query"];
  MYX_STRINGLIST *params= (MYX_STRINGLIST*)[[args objectForKey:@"params"] pointerValue];
  MYX_LIB_ERROR error_code;
  MYX_RESULTSET *result;
  NSDictionary *info;
  
  result= myx_query_execute(mysql, [query UTF8String],
							TRUE,
							params, &error_code, self,
							fetchProgress,
							NULL,
							NULL);

  info= [NSDictionary dictionaryWithObjectsAndKeys:
	[NSValue valueWithPointer:result], @"resultset",
	[NSNumber numberWithInt:error_code], @"error", nil];

  [self performSelectorOnMainThread:@selector(queryFinished:)
						 withObject:info
					  waitUntilDone:YES];
}


- (void)performQuery:(NSString*)query
{
  [self setQuery:query];
  
  [statusLabel setStringValue:@"Performing query..."];
  
  [_delegate performSelector:@selector(resultSetView:willStartQuery:)
                  withObject:self
                  withObject:query];
  
  [NSApplication detachDrawingThread:@selector(doQuery:)
							toTarget:self
						  withObject:[NSDictionary dictionaryWithObjectsAndKeys:
							[NSValue valueWithPointer:_mysql], @"mysql",
							query, @"query",
							nil]];
  _busy= YES;
}


- (void)applyFinished:(id)errorsVal
{
  MYX_RS_ACTION_ERRORS *errors= (MYX_RS_ACTION_ERRORS*)[errorsVal pointerValue];

  if (errors)
  {
    unsigned int i;
    int errcount= 0;
    [_messageDS reset];
    for (i= 0; i < errors->errors_num; i++)
    {
      if (errors->errors[i].level == MYX_QEL_ERROR)
        errcount++;
      [_messageDS addError:errors->errors[i].error
                    ofType:errors->errors[i].level
                   message:[NSString stringWithUTF8String:errors->errors[i].error_text]
                     atRow:errors->errors[i].action->row
                    column:[_dataSource resultset]->columns - errors->errors[i].action->column];
    }
    myx_query_free_action_errors(errors);

    [_dataSource postCommit:YES];
    [messageTable noteNumberOfRowsChanged];
    [self setMessagesShown:YES];

    if (errcount > 0)
      [statusLabel setStringValue:@"Error while applying actions."];
    else
      [statusLabel setStringValue:@"Warnings while applying actions."];

    [_dataSource setEditing:YES];
    [self updateButtonBar];
  }
  else
  {
    [_dataSource postCommit:NO];
    [_messageDS reset];
    [messageTable reloadData];
	[statusLabel setStringValue:@"Applied actions."];    
  }
  [table reloadData];

  _busy= NO;
}

- (void)doApply:(NSDictionary*)args
{
  MYX_RESULTSET *rset= (MYX_RESULTSET*)[[args objectForKey:@"resultset"] pointerValue];
  MYX_RS_ACTION_ERRORS *errors;
  
  errors= myx_query_apply_actions(rset);
  
  [self performSelectorOnMainThread:@selector(applyFinished:)
						 withObject:[NSValue valueWithPointer:errors]
					  waitUntilDone:YES];
}


- (void)setCompact:(BOOL)flag
{
  if (flag != _compact)
  {
    NSRect rect;
    int i;
    id part;
    float x= 5.0;
    float width= NSWidth([view frame]);

    x= width;
    for (i= 6; i >= 1; i--)
    {
      part= [view viewWithTag:editBarTags[i]];
      rect= [part frame];
      if (flag)
        rect.size.width= 25;
      else
        rect.size.width= _editBarSizes[i].width;
      x-= rect.size.width;
      
      if (i == 3)
        x-= 2;
      rect.origin.x= x;
      [part setFrame:rect];
    }
    part= [view viewWithTag:editBarTags[0]];
    rect= [part frame];
    rect.size.width= width-5.0-x;
    [part setFrame:rect];
    _compact= flag;
  }
}


- (void)performApply
{
  [statusLabel setStringValue:@"Applying actions..."];

//  [_delegate resultSetView:self willStartQuery:query];
  
  [NSApplication detachDrawingThread:@selector(doApply:)
							toTarget:self
						  withObject:[NSDictionary dictionaryWithObjectsAndKeys:
							[NSValue valueWithPointer:_mysql], @"mysql",
							[NSValue valueWithPointer:[_dataSource resultset]], @"resultset",
							nil]];
  _busy= YES;
}


- (BOOL)isBusy
{
  return _busy;
}

- (void)setQuery:(NSString*)query
{
  if (_query != query)
  {
	[_query release];
    if ([query isMemberOfClass:[NSMutableString class]])
      _query= [[NSString stringWithString:query] retain];
    else
      _query= [query retain];
  }
}

- (NSString*)query
{
  return _query ? : @"";
}


- (BOOL)tableView:(NSTableView *)aTableView 
shouldEditTableColumn:(NSTableColumn *)aTableColumn 
			  row:(int)rowIndex
{
  int column= [[aTableColumn identifier] intValue];
  if ([_dataSource editing] && [_dataSource statusOfRow:rowIndex
												 column:column] != MESDeleted
      && [_dataSource resultset]->columns[column].column_type!=MYX_RSCT_BLOB)
	return YES;
  else
	return NO;
}

- (int)numberOfItemsInMenu:(NSMenu *)menu
{
  return [menu numberOfItems];
}

- (BOOL)menu:(NSMenu *)menu
  updateItem:(NSMenuItem *)item
	 atIndex:(int)index
shouldCancel:(BOOL)shouldCancel
{
  BOOL flag= NO;
  BOOL hasSelection= [table selectedRow]>=0;

  switch (index)
  {
	case 0: //add
	  flag= [_dataSource editing];
	  break;
	case 1: //delete
	  flag= [_dataSource editing] && hasSelection;
	  break;
	case 3: //load
	  flag= [_dataSource editing] && hasSelection;
	  break;
	case 4: //save
	  flag= hasSelection;
	  break;
	case 5: //copy
	  flag= hasSelection;
	  break;
	case 6: //clear
	  flag= [_dataSource editing] && hasSelection;
	  break;	  
	case 8: //view
	  flag= hasSelection;
	  break;
	case 9: //edit
	  flag= [_dataSource editing] && hasSelection;
	  break;
  }
  [item setEnabled:flag];
  [menu itemChanged:item];
  return YES;
}


- (void)tableViewWasClicked:(NSTableView *)aTableView
{
  [_delegate performSelector:@selector(resultSetViewWantsFocus:) withObject:self];
}


- (void)tableView:(NSTableView *)aTableView 
  willDisplayCell:(id)aCell
   forTableColumn:(NSTableColumn *)aTableColumn
			  row:(int)rowIndex
{
  if ([[aTableColumn identifier] isEqualTo:@"indicator"])
  {
    if ([aTableView selectedRow] == rowIndex)
      [aCell setSelected:YES];
    else
      [aCell setSelected:NO];

    if ([_dataSource statusOfRow:rowIndex column:0] == MESPlaceHolder)
      [aCell setPlaceholder:YES];
    else
      [aCell setPlaceholder:NO];
  }
  else if ([_dataSource editing])
  {
	int column= [[aTableColumn identifier] intValue];

    [aCell setPlaceholder:NO];
    
    if (column == [(MQTableView*)aTableView lastClickedColumn]-1 && 
        rowIndex == [aTableView selectedRow])
    {
      [aCell setBackgroundColor:[NSColor alternateSelectedControlColor]];
      [aCell setTextColor:[NSColor alternateSelectedControlTextColor]];
      return;
    }

    [aCell setTextColor:[NSColor blackColor]];
	switch ([_dataSource statusOfRow:rowIndex column:column])
	{
	  case MESUnchanged:
        if (rowIndex & 1)
          [aCell setBackgroundColor:_statusColors[MESUnchanged]];
        else
          [aCell setBackgroundColor:[NSColor whiteColor]];
		break;
	  case MESChanged:
		[aCell setBackgroundColor:_statusColors[MESChanged]];
		break;
      case MESPlaceHolder:
        [aCell setBackgroundColor:_statusColors[MESPlaceHolder]];
        [aCell setPlaceholder:YES];
        break;
	  case MESAdded:
		[aCell setBackgroundColor:_statusColors[MESAdded]];
        break;
	  case MESDeleted:
		[aCell setBackgroundColor:_statusColors[MESDeleted]];
		break;
	}
  }
  else
  {
    int column= [[aTableColumn identifier] intValue];

    [aCell setPlaceholder:NO];
    
    if (column == [(MQTableView*)aTableView lastClickedColumn]-1 && 
        rowIndex == [aTableView selectedRow])
    {
      //XXX find out how to change the color depending on whether control has focus
      [aCell setBackgroundColor:[NSColor alternateSelectedControlColor]];
      [aCell setTextColor:[NSColor alternateSelectedControlTextColor]];
    }
    else
    {
      [aCell setTextColor:[NSColor blackColor]];
      if (rowIndex & 1)
        [aCell setBackgroundColor:_statusColors[MESUnchanged]];
      else
        [aCell setBackgroundColor:[NSColor whiteColor]];
    }
  }
}

- (void)splitView:(NSSplitView *)sender resizeSubviewsWithOldSize:(NSSize)oldSize
{
  NSArray *children= [sender subviews];
  NSView *top= [children objectAtIndex:0];
  NSView *bottom= [children lastObject];
  NSRect rect;

  rect= [sender frame];
  rect.size.height-= [sender dividerThickness] + NSHeight([bottom frame]);  
  [top setFrame:rect];

  rect.origin.y= rect.size.height + [sender dividerThickness];
  rect.size.height= NSHeight([bottom frame]);
  [bottom setFrame:rect];
}

- (float)splitView:(NSSplitView *)sender
 constrainMinCoordinate:(float)proposedMin
       ofSubviewAt:(int)offset
{
  return NSHeight([sender frame])-80;
}


- (void)fieldViewerDidClose:(MQFieldViewer*)viewer
{
  [_viewers removeObject:viewer];
}


- (void)fieldViewer:(MQFieldViewer*)viewer saveData:(NSData*)data
{
  //XXX
}

- (void)selectedError:(NSNotification*)notif
{
  MessageItem *item= ((MessageDS*)_messageDS)->items+[messageTable selectedRow];
  
  if (item->row != (unsigned int)-1)
  {
    [table selectCellAtRow:item->row
                    column:item->column];
  }
}


@end

