/* Copyright (C) 2003 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <libxml/tree.h>

#include <myx_library.h>
#include <myx_xml_util_functions.h>

typedef enum _kind { 
  APPLICATION_SPECIFIC_FILE=  1,
  GLOBAL_FILE=                2
} FILE_KIND;

/* forward declarations */
static int parse_translation_file(MYX_TRANS *trans, const char *filename,
                                  const char *lang, MYX_LIB_ERROR *error_code,
                                  FILE_KIND kind);
static void read_in_text(const xmlNodePtr text_node, MYX_TRANS_TEXT *text,
                         const char *lang);
static void read_in_textgroup(const xmlNodePtr textgroup_node,
                              MYX_TRANS_TEXTGROUP *group, const char *lang);
static void free_textgroup_content(MYX_TRANS_TEXTGROUP *group);
static void free_text_content(MYX_TRANS_TEXT *text);
static void sort_alphabetically(MYX_TRANS *trans);
static int compare_textgroups(const void *a, const void *b);
static int compare_transtexts(const void *a, const void *b);

/*
 *public functions definitions
 */

///////////////////////////////////////////////////////////////////////////////
/** @brief collect translations from xml-files with the given names for 
           given language

    @param translation_file_name              path to file with general
                                                                   translations
    @param application_translation_file_name  path to file with
                                              application-specific translations
    @param language                           goal language
    @param error_code returned error code, <BR>
           possible values are:
             - \link MYX_LIB_ERROR::MYX_ERROR_CANT_OPEN_FILE 
                                             MYX_ERROR_CANT_OPEN_FILE \endlink
             - \link MYX_LIB_ERROR::MYX_XML_PARSE_ERROR
                                                  MYX_XML_PARSE_ERROR \endlink
             - \link MYX_LIB_ERROR::MYX_XML_EMPTY_DOCUMENT 
                                               MYX_XML_EMPTY_DOCUMENT \endlink
             - \link MYX_LIB_ERROR::MYX_XML_NO_VALID_DOCUMENT
                  MYX_XML_NO_VALID_DOCUMENT \endlink 
                  if the root of the document wasn't equal to
                  "messages"
             - initial value (value that was before calling the function)
                  on success

    @return prepared MYX_TRANS struct on success,
            NULL on error, for details check error_code
*//////////////////////////////////////////////////////////////////////////////
MYX_TRANS* myx_init_trans(const char*             translation_file_name,
                          const char* application_translation_file_name,
                          const char *language, MYX_LIB_ERROR *error_code)
{
  MYX_TRANS *trans;

  *error_code= MYX_NO_ERROR;

  trans= calloc(1, sizeof(MYX_TRANS) );

  if (parse_translation_file(trans, translation_file_name,
                             language, error_code, GLOBAL_FILE) )
  {
    free(trans);
    return NULL;
  }

  if (parse_translation_file(trans, application_translation_file_name,
                             language, error_code, APPLICATION_SPECIFIC_FILE) )
  {
    myx_free_trans(trans);
    return NULL;
  }

  sort_alphabetically(trans);

  return trans;
}

///////////////////////////////////////////////////////////////////////////////
/** @brief free memory for the MYX_TRANS struct
    @param trans MYX_TRANS struct to free
    @return 0 always
*//////////////////////////////////////////////////////////////////////////////
int myx_free_trans(MYX_TRANS *trans)
{
  unsigned int i;

  if (trans)
  {
    for (i= 0; i< trans->textgroups_general_num; i++)
      free_textgroup_content(trans->textgroups_general+i);
    free(trans->textgroups_general);

    for (i= 0; i< trans->textgroups_num; i++)
      free_textgroup_content(trans->textgroups+i);
    free(trans->textgroups);

    free(trans);
  }

  return 0;
}

///////////////////////////////////////////////////////////////////////////////
/** @brief find the translation of the text

    @param textgroups_num    the number of translations groups to look through
    @param textgroups        array of translations groups to look through
    @param current_textgroup this text_groups is checked at first.
                             found text_groups is returned in this variable.
    @param id_textgroup      identifier of translation group
    @param id_text           identifier of text

    @return found MYX_TRANS_TEXT struct
*//////////////////////////////////////////////////////////////////////////////
MYX_TRANS_TEXT * find_translation(unsigned int textgroups_num,
                                  MYX_TRANS_TEXTGROUP *textgroups,
                                  MYX_TRANS_TEXTGROUP **current_textgroup,
                                  const char *id_textgroup,
                                  const char *id_text)
{
  MYX_TRANS_TEXTGROUP *tg;
  MYX_TRANS_TEXTGROUP tmp_tg= {(unsigned char *)id_textgroup,0,NULL};
  MYX_TRANS_TEXT tmp_text= {(unsigned char *)id_text, NULL};

  if (*current_textgroup != NULL &&
      !xmlStrcmp((*current_textgroup)->id, (xmlChar*)id_textgroup) )
  {
    tg= *current_textgroup;
  }
  else
  {
    tg= (MYX_TRANS_TEXTGROUP*) bsearch(&tmp_tg,textgroups, textgroups_num,
                                        sizeof(MYX_TRANS_TEXTGROUP),
                                        compare_textgroups);
    if (tg == NULL)
      return 0;
    *current_textgroup= tg;
  }

  /*now find the text with id_text in the group*/
  return (MYX_TRANS_TEXT *) bsearch(&tmp_text,tg->texts,tg->texts_num,
                                    sizeof(MYX_TRANS_TEXT),compare_transtexts);
}

///////////////////////////////////////////////////////////////////////////////
/** @brief find the application-specific translation of the text
    @param trans         translations desription
    @param id_textgroup  identifier of translation group
    @param id_text       identifier of text
    @param original_text result by default (isn't used in the search process)
    @return translated text

    gets a translation of a string with id_text/id_group from the 
    application-specific translation file returns the orginial_text if not 
    found.

    As a side effect of function MYX_TRANS::current_textgroup is changed to
    text group with the given id.
*//////////////////////////////////////////////////////////////////////////////
const char * myx_t(MYX_TRANS *trans, const char *id_textgroup,
                   const char *id_text, const char *original_text)
{
  MYX_TRANS_TEXT *text= find_translation(trans->textgroups_num,
                                         trans->textgroups,
                                         &trans->current_textgroup,
                                         id_textgroup,id_text);
  //hack: g_strdup the return-value because the libinterfacemapper
  //      automatically generates function wrappers where the return
  //      value is free'ed
  return !text || !text->trans ? g_strdup(original_text) : g_strdup(text->trans);
}

///////////////////////////////////////////////////////////////////////////////
/** @brief find the "general" translation of the text
    @param trans         translations desription
    @param id_textgroup  identifier of translation group
    @param id_text       identifier of text
    @param original_text result by default (isn't used in the search process)
    @return translated text

    gets a translation of a string with id_text/id_group from the general
    translation file

    As a side effect of function MYX_TRANS::current_textgroup_general is 
    changed to text group with the given id.
*//////////////////////////////////////////////////////////////////////////////
const char * myx_tg(MYX_TRANS *trans, const char *id_textgroup,
                    const char *id_text, const char *original_text)
{
  MYX_TRANS_TEXT *text= find_translation(trans->textgroups_general_num,
                                         trans->textgroups_general,
                                         &trans->current_textgroup_general,
                                         id_textgroup,id_text);

  //hack: g_strdup the return-value because the libinterfacemapper
  //      automatically generates function wrappers where the return
  //      value is free'ed
  return !text || !text->trans ? g_strdup(original_text) : g_strdup(text->trans);
}

/**********************/
/*   PRIVATE FUNCTIONS*/
/**********************/

///////////////////////////////////////////////////////////////////////////////
/** @brief compare two MYX_TRANS_TEXT struct by their id
    @param a first MYX_TRANS_TEXT struct to compare
    @param b first MYX_TRANS_TEXT struct to compare
    @return the integer result of the comparison 

    function returns result of xmlStrcmp. <br>
    function is for using in qsort and in bsearch.
*//////////////////////////////////////////////////////////////////////////////
static int compare_transtexts(const void *a, const void *b)
{
  MYX_TRANS_TEXT *t1= (MYX_TRANS_TEXT *) a;
  MYX_TRANS_TEXT *t2= (MYX_TRANS_TEXT *) b;

  return xmlStrcmp(t1->id, t2->id);
}

///////////////////////////////////////////////////////////////////////////////
/** @brief compare two MYX_TRANS_TEXTGROUP struct by their id
    @param a first MYX_TRANS_TEXTGROUP struct to compare
    @param b first MYX_TRANS_TEXTGROUP struct to compare
    @return the integer result of the comparison 

    function returns result of xmlStrcmp. <br>
    function is for using in qsort and in bsearch.
*//////////////////////////////////////////////////////////////////////////////
static int compare_textgroups(const void *a, const void *b)
{
  MYX_TRANS_TEXTGROUP *g1= (MYX_TRANS_TEXTGROUP *) a;
  MYX_TRANS_TEXTGROUP *g2= (MYX_TRANS_TEXTGROUP *) b;

  return xmlStrcmp(g1->id,g2->id);
}

///////////////////////////////////////////////////////////////////////////////
/** @brief sort groups and translations in the MYX_TRANS alphabetically by id
    @param trans MYX_TRANS struct to sort members of

    Next members are sorted:
      MYX_TRANS::textgroups, MYX_TRANS::textgroups_general,
      MYX_TRANS_TEXTGROUP::texts
*//////////////////////////////////////////////////////////////////////////////
static void sort_alphabetically(MYX_TRANS *trans)
{
  unsigned int i;

  /*sort the groups*/
  if (trans->textgroups)
  {
    qsort(trans->textgroups, trans->textgroups_num,
          sizeof(MYX_TRANS_TEXTGROUP), compare_textgroups);
  }

  if (trans->textgroups_general)
  {
    qsort(trans->textgroups_general,
          trans->textgroups_general_num,
          sizeof(MYX_TRANS_TEXTGROUP), compare_textgroups);
  }

  /*sort the text-elements in their groups*/
  for (i= 0; i < trans->textgroups_num; i++)
  {
    if (trans->textgroups[i].texts)
    {
      qsort(trans->textgroups[i].texts, trans->textgroups[i].texts_num,
            sizeof(MYX_TRANS_TEXT), compare_transtexts);
    }
  }

  for (i= 0; i < trans->textgroups_general_num; i++)
  {
    if (trans->textgroups_general[i].texts)
    {
      qsort(trans->textgroups_general[i].texts,
            trans->textgroups_general[i].texts_num,
            sizeof(MYX_TRANS_TEXT), compare_transtexts);
    }
  }
}

///////////////////////////////////////////////////////////////////////////////
/** @brief parse translation file to a specified textgroups
           in the MYX_TRANS struct

    @param filename path to xml-file with stored translations
    @param trans MYX_TRANS struct to fill
    @param lang name of language to read
    @param error_code returned error code, <BR>
           possible values are:
             - \link MYX_LIB_ERROR::MYX_ERROR_CANT_OPEN_FILE 
                                             MYX_ERROR_CANT_OPEN_FILE \endlink
             - \link MYX_LIB_ERROR::MYX_XML_PARSE_ERROR
                                                  MYX_XML_PARSE_ERROR \endlink
             - \link MYX_LIB_ERROR::MYX_XML_EMPTY_DOCUMENT 
                                               MYX_XML_EMPTY_DOCUMENT \endlink
             - \link MYX_LIB_ERROR::MYX_XML_NO_VALID_DOCUMENT
                  MYX_XML_NO_VALID_DOCUMENT \endlink 
                  if the root of the document wasn't equal to
                  "messages"
             - initial value (value that was before calling the function)
                  on success
    @param kind enum said wich textgroups in the MYX_TRANS struct 
            should be filled
           possible values are:
           - \link FILE_KIND::APPLICATION_SPECIFIC_FILE 
                                             APPLICATION_SPECIFIC_FILE \endlink
               to fill MYX_TRANS::textgroups
           - \link FILE_KIND::GLOBAL_FILE GLOBAL_FILE \endlink
               to fill MYX_TRANS::textgroups_general

    @return -1 on success 0 on error
*//////////////////////////////////////////////////////////////////////////////
static int parse_translation_file(MYX_TRANS *trans, const char *filename,
                                  const char *lang, MYX_LIB_ERROR *error_code,
                                  FILE_KIND kind)
{
  xmlDocPtr doc;
  xmlNodePtr root, cur;
  MYX_TRANS_TEXTGROUP * group= 0;
  unsigned int groups_num;

  if (!file_exists(filename))
  {
    *error_code= MYX_ERROR_CANT_OPEN_FILE;
    return -1;
  }

  doc= myx_xmlParseFile(filename);
  if (doc == NULL )
  {
    *error_code= MYX_XML_PARSE_ERROR;
    return -1;
  }
    
  root= xmlDocGetRootElement(doc);
  if (root == NULL)
  {
    *error_code= MYX_XML_EMPTY_DOCUMENT;
    xmlFreeDoc(doc);
    return -1;
  }
    
  if (xmlStrcmp(root->name, (const xmlChar *) "messages"))
  {
    *error_code= MYX_XML_NO_VALID_DOCUMENT;
    xmlFreeDoc(doc);
    return -1;
  }

  groups_num= get_child_count(root, (xmlChar*)"textgroup");
  if (groups_num > 0)
    group= calloc(groups_num, sizeof(MYX_TRANS_TEXTGROUP));

  switch (kind)
  {
  case APPLICATION_SPECIFIC_FILE:
    trans->textgroups_num= groups_num;
    trans->textgroups= group;
    break;
  case GLOBAL_FILE:
    trans->textgroups_general_num= groups_num;
    trans->textgroups_general= group;
    break;
  default:
    assert(0);
  }

  for (cur= root->children; cur; cur= cur->next)
  {
    if (!xmlStrcmp(cur->name, (xmlChar*)"textgroup") )
    {
      read_in_textgroup(cur, group, lang);
      group++;
    }
  }

  return 0;
}

///////////////////////////////////////////////////////////////////////////////
/** @brief read translation group for given language from xml node
    @param textgroup_node xml-node to read from
    @param group          MYX_TRANS_TEXTGROUP struct to read in
    @param lang           given language
*//////////////////////////////////////////////////////////////////////////////
static void read_in_textgroup(const xmlNodePtr textgroup_node,
                              MYX_TRANS_TEXTGROUP *group, const char *lang)
{
  MYX_TRANS_TEXT * text;
  xmlNodePtr cur;

  group->id= xmlGetProp(textgroup_node, (xmlChar*)"id");

  group->texts_num= get_child_count(textgroup_node, (xmlChar*)"text");
  if (group->texts_num > 0)
    group->texts= calloc(group->texts_num, sizeof(MYX_TRANS_TEXT) );

  text= group->texts;
  for (cur= textgroup_node->children; cur; cur= cur->next)
  {
    if ( !xmlStrcmp(cur->name, (xmlChar*)"text") &&  cur->type == XML_ELEMENT_NODE)
    {
      read_in_text(cur, text, lang);
      text++;
    }
  }
}

///////////////////////////////////////////////////////////////////////////////
/** @brief read translation for given language from xml node
    @param text_node  xml-node to read from
    @param text       MYX_TRANS_TEXT struct to read in (id, text)
    @param lang       given language
*//////////////////////////////////////////////////////////////////////////////
static void read_in_text(const xmlNodePtr text_node,
                         MYX_TRANS_TEXT *text, const char *lang)
{
  xmlDocPtr doc= text_node->doc;
  xmlNodePtr cur;
  xmlChar *tmp;

  text->id= xmlGetProp(text_node, (xmlChar*)"id");

  for (cur= text_node->children; cur; cur = cur->next)
  {
    if (! xmlStrcmp(cur->name, (xmlChar*)"trans") )
    {
      tmp= xmlGetProp(cur, (xmlChar*)"lang");
      if (! xmlStrcmp(tmp, (xmlChar*)lang) )
      {
        text->trans = (char*)xmlNodeListGetString(doc,cur->children,1);
        xmlFree(tmp);
        break;
      }
      xmlFree(tmp);
    }
  }
}

///////////////////////////////////////////////////////////////////////////////
/** @brief free memory for the MYX_TRANS_TEXTGROUP struct
    @param group MYX_TRANS_TEXTGROUP struct to free
*//////////////////////////////////////////////////////////////////////////////
static void free_textgroup_content(MYX_TRANS_TEXTGROUP *group)
{
  unsigned int i;
  xmlFree(group->id);
  for (i= 0; i< group->texts_num; i++)
    free_text_content(group->texts+i);
  free(group->texts);
}

///////////////////////////////////////////////////////////////////////////////
/** @brief free memory for the MYX_TRANS_TEXT struct
    @param text MYX_TRANS_TEXT struct to free
*//////////////////////////////////////////////////////////////////////////////
static void free_text_content(MYX_TRANS_TEXT *text)
{
  xmlFree(text->id);
  xmlFree(text->trans);
}
