/* Copyright (C) 2005 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "MGCanvas.h"

#include "myx_gc_figure.h"

#include <gdkmm.h>
#include <gdk/gdkx.h>
#include <sys/time.h>
#include <gtkmm/adjustment.h>
#include "MGCanvasScroller.h"

static float ZoomSteps[]= {
  0.1,
  0.2,
  0.3,
  0.4,
  0.5,
  0.6,
  0.65,
  0.7,
  0.75,
  0.8,
  0.85,
  0.9,
  1.0,
  1.1,
  1.2,
  1.5,
  2.0
};

#define ZOOM_STEP(s) (ZoomSteps[s]/1.45)

#define ZOOM_LEVEL_1 12

#define MAX_ZOOM_STEPS (sizeof(ZoomSteps)/sizeof(float))


// scroller support
// correct zooming
// font and sys-color support


static int getModifiers(int mods)
{
  int res= GC_MODIFIER_NONE;
  
  if (mods & GDK_SHIFT_MASK)
    res|= GC_MODIFIER_ADD;
  if (mods & GDK_CONTROL_MASK)
    res|= GC_MODIFIER_TOGGLE;
  if (mods & GDK_MOD1_MASK)
    res|= GC_MODIFIER_ALTERNATIVE;

  return res;
}



static TGCViewport viewportFromRect(Gdk::Rectangle rect)
{
  TGCViewport viewport;

  viewport.top= rect.get_y();
  viewport.left= rect.get_x();
  viewport.width= rect.get_width();
  viewport.height= rect.get_height();

  return viewport;
}


MGCanvas::MGCanvas()
  : superclass(), _context(0), _canvas(0)
{
  add_events(Gdk::BUTTON_PRESS_MASK|Gdk::BUTTON_RELEASE_MASK|Gdk::BUTTON_MOTION_MASK|
             Gdk::KEY_PRESS_MASK|Gdk::KEY_RELEASE_MASK|Gdk::POINTER_MOTION_MASK);

  set_double_buffered(false);

  _zoom= 1.0;
  _grabPanning= false;
  _state.panning= 0;
  _changingScrollers= false;
  
  _refresh.connect(sigc::mem_fun(*this, &Gtk::Widget::queue_draw));
  _switchView.connect(sigc::mem_fun(*this, &MGCanvas::handle_switch_view));
}


MGCanvas::~MGCanvas()
{ 
  if (_context)
    glXDestroyContext(gdk_display, _context);
  
  delete _canvas;
}


void MGCanvas::handle_switch_view()
{
  _canvas->beginUpdate();
  if (_canvas->currentViewGet())
  {
    _canvas->currentViewGet()->viewportSet(viewportFromRect(_viewport));
    _offset.x= _canvas->currentViewGet()->offsetXGet();
    _offset.y= _canvas->currentViewGet()->offsetYGet();
    _zoom= _canvas->currentViewGet()->zoomGet();
  }
  _canvas->endUpdate();
  update_scrollers();
}


void MGCanvas::set_mouse_down_handler(sigc::slot<bool,MGCanvas*,int,Point> handler)
{
  _handleMouseDown= handler;
}


void MGCanvas::set_mouse_up_handler(sigc::slot<bool,MGCanvas*,int,Point> handler)
{
  _handleMouseUp= handler;
}


void MGCanvas::set_mouse_move_handler(sigc::slot<bool,MGCanvas*,Point> handler)
{
  _handleMouseDrag= handler;
}



void MGCanvas::force_reconfigure()
{
  _canvas->beginUpdate();
  _canvas->currentViewGet()->viewportSet(viewportFromRect(_viewport));
  _canvas->endUpdate();
}


bool MGCanvas::on_configure_event(GdkEventConfigure* event)
{
  superclass::on_configure_event(event);

  if (_canvas && _canvas->currentViewGet())
  {
    CGCView *view= _canvas->currentViewGet();
    
    _viewport.set_width(event->width);
    _viewport.set_height(event->height);
    
    _canvas->beginUpdate();
    view->viewportSet(viewportFromRect(_viewport));
    _canvas->endUpdate();    

    update_scrollers();
  }
  
  return true;
}


void MGCanvas::on_realize()
{  
  superclass::on_realize();
 
  XVisualInfo *visinfo;
  int attrib[] = { 
    GLX_RGBA,
      GLX_RED_SIZE, 1,
      GLX_GREEN_SIZE, 1,
      GLX_BLUE_SIZE, 1,
      GLX_DOUBLEBUFFER,
      GLX_DEPTH_SIZE, 1,
      None 
  };
  visinfo = glXChooseVisual(gdk_display,
                            gdk_x11_screen_get_screen_number(Gdk::Screen::get_default()->gobj()),
                            attrib);
  if (!visinfo) {
    g_warning("could not get visual for OpenGL");
    return;
  }

  _context= glXCreateContext(gdk_display, visinfo, NULL, GL_TRUE);
  XSync(gdk_display, False);
  if (!_context) 
  {
    g_warning("Could not initialize GLX context");
    XFree(visinfo);
  }
  XFree(visinfo);
  
  glXMakeCurrent(gdk_display, gdk_x11_drawable_get_xid(get_window()->gobj()), _context);
  
  _canvas= new CGenericCanvas(_context, L"");
  _canvas->addListener(this);
  
  CGCView *view= _canvas->createView("main");
  _canvas->currentViewSet(view);

  view->viewportSet(TGCViewport(0, 0, 
                             get_allocation().get_width(),
                             get_allocation().get_height()));
  view->color(1.0, 1.0, 1.0, 1.0);
  
  update_scrollers();
}


bool MGCanvas::on_expose_event(GdkEventExpose* event)
{
  if (event->count == 0)
  {
    glXMakeCurrent(gdk_display, gdk_x11_drawable_get_xid(get_window()->gobj()), _context);

    render_scene();
    glXSwapBuffers(gdk_display, gdk_x11_drawable_get_xid(get_window()->gobj()));
  }
  return true;
}


bool MGCanvas::on_button_press_event(GdkEventButton* event)
{
  Gdk::Point point((int)event->x, (int)event->y);

  switch (event->button)
  {
  case 1:
    if (_grabPanning)
    {
      _lastClickPoint= point;
      _state.panning= 1;
      get_window()->set_cursor(_hand_closed_cursor);
    }
    else if (_handleMouseDown.empty() || !_handleMouseDown(this, event->button, convert_to_canvas_point(point)))
    {
      _canvas->currentViewGet()->handleMouseInput(GC_MOUSE_DOWN,
                                               GC_MOUSE_BUTTON_LEFT,
                                               getModifiers(event->state),
                                               point.get_x(),
                                               point.get_y());
    }
    break;
    
  case 2:
    _lastClickPoint= point;
    //if (_grabPanning)
    {
      _state.panning= 2;
      get_window()->set_cursor(_hand_closed_cursor);
    }
    /*
    else if (_handleMouseDown.empty() || !_handleMouseDown(this, event->button, convert_to_canvas_point(point)))
      _canvas->currentViewGet()->handleMouseInput(GC_MOUSE_DOWN,
                                               GC_MOUSE_BUTTON_MIDDLE,
                                               getModifiers(event->state),
                                               point.get_x(),
                                               point.get_y());
     */
    break;
    
  case 3:
    if (_handleMouseDown.empty() || !_handleMouseDown(this, event->button, convert_to_canvas_point(point)))
      _canvas->currentViewGet()->handleMouseInput(GC_MOUSE_DOWN,
                                               GC_MOUSE_BUTTON_RIGHT,
                                               getModifiers(event->state),
                                               point.get_x(),
                                               point.get_y());
    break;
  }
  
  return true;
}

  
bool MGCanvas::on_button_release_event(GdkEventButton* event)
{
  Gdk::Point point((int)event->x, (int)event->y);
  
  switch (event->button)
  {
  case 1:
    if (_state.panning==1)
    {
      _state.panning= 0;
      get_window()->set_cursor(_hand_cursor);
    }
    else if (_handleMouseUp.empty() || !_handleMouseUp(this, event->button, convert_to_canvas_point(point)))
      _canvas->currentViewGet()->handleMouseInput(GC_MOUSE_UP,
                                               GC_MOUSE_BUTTON_LEFT,
                                               getModifiers(event->state),
                                               point.get_x(),
                                               point.get_y());
    break;
  case 2:
    if (_state.panning==2)
    {
      _state.panning= 0;
      if (_grabPanning)
        get_window()->set_cursor(_hand_cursor);
      else
        get_window()->set_cursor();
    }
    else
    {
      if (_handleMouseUp.empty() || !_handleMouseUp(this, event->button, convert_to_canvas_point(point)))
        _canvas->currentViewGet()->handleMouseInput(GC_MOUSE_UP,
                                                 GC_MOUSE_BUTTON_MIDDLE,
                                                 getModifiers(event->state),
                                                 point.get_x(),
                                                 point.get_y());
    }
    break;
  case 3:
    if (_handleMouseUp.empty() || !_handleMouseUp(this, event->button, convert_to_canvas_point(point)))
      _canvas->currentViewGet()->handleMouseInput(GC_MOUSE_UP,
                                               GC_MOUSE_BUTTON_RIGHT,
                                               getModifiers(event->state),
                                               point.get_x(),
                                               point.get_y());
    break;
  }
  
  return true;
}


bool MGCanvas::on_motion_notify_event(GdkEventMotion* event)
{
  Gdk::Point pos((int)event->x, (int)event->y);
  
  if (_state.panning)
  {
    int dx, dy;
    
    dx= _lastClickPoint.get_x() - pos.get_x();
    dy= _lastClickPoint.get_y() - pos.get_y();
    
    _lastClickPoint= pos;
    
    _offset.x= _offset.x+dx;
    _offset.y= _offset.y+dy;

    update_scrollers();
  }
  else
  {
    if (_handleMouseDrag.empty() || !_handleMouseDrag(this, convert_to_canvas_point(pos)))
    {
      _canvas->currentViewGet()->handleMouseInput(GC_MOUSE_MOVE,
                                               GC_MOUSE_BUTTON_NONE,
                                               getModifiers(event->state),
                                               pos.get_x(),
                                               pos.get_y());
    }
  }
  
  return true;
}


bool MGCanvas::on_scroll_event(GdkEventScroll* event)
{
  bool moved= false;
  
  if (event->state == 0)
  {
    switch (event->direction)
    {
    case GDK_SCROLL_UP:
      _offset.y-= 10.0*_zoom;
      moved= true;
      break;
    case GDK_SCROLL_DOWN:
      _offset.y+= 10.0*_zoom;
      moved= true;
      break;
    case GDK_SCROLL_LEFT:
      _offset.x-= 10.0*_zoom;
      moved= true;
      break;
    case GDK_SCROLL_RIGHT:
      _offset.x+= 10.0*_zoom;
      moved= true;
      break;
    }
  }
  else if ((event->state & GDK_CONTROL_MASK) == GDK_CONTROL_MASK)
  {
    if (event->direction == GDK_SCROLL_UP)
    {
      _offset.x-= 10.0*_zoom;
      moved= true;
    }
    else if (event->direction == GDK_SCROLL_DOWN)
    {
      _offset.x+= 10.0*_zoom;
      moved= true;
    }
  }
  else if ((event->state & GDK_SHIFT_MASK) == GDK_SHIFT_MASK)
  {
    double zoom_increment= 0.2;
    double swidth= get_allocation().get_width();
    double sheight= get_allocation().get_height();

    //calc stuff and do here

    if (event->direction == GDK_SCROLL_UP)
    {
      _zoom+= zoom_increment;

      moved= true;
    }
    else if (event->direction == GDK_SCROLL_DOWN)
    {
      _zoom-= zoom_increment;

      moved= true;
    }
    
    if (moved)
    {
      _offset.x += (event->x/swidth)*_zoom;
      _offset.y += (event->y/sheight)*_zoom;
    }
  }

  if (moved)
    update_scrollers();

  return true;
}


void MGCanvas::onAction(CGCBase *sender, CGCBase *origin, TAction **action)
{
}


void MGCanvas::onChange(CGCBase *sender, CGCBase *origin, TGCChangeReason Reason)
{
  switch (Reason)
  {
  case GC_CHANGE_CANVAS_REFRESH:
  case GC_CHANGE_VIEW_PROPERTY:
    _refresh.emit();
    break;
  case GC_CHANGE_CANVAS_SWITCH_VIEW:
    _switchView.emit();
    break;
    
  default:
    break;
  }
}


void MGCanvas::onError(CGCBase *sender, CGCBase *origin, const char* Message)
{
  g_warning("CANVAS ERROR: %s", Message);
}


void MGCanvas::onDestroy(CGCBase *sender)
{
}


std::vector<float> MGCanvas::get_zoom_steps()
{
  std::vector<float> steps;
  for (unsigned int i= 0; i < MAX_ZOOM_STEPS; i++)
  {
    steps.push_back(ZoomSteps[i]);
  }
  return steps;
}


void MGCanvas::set_zoom_level(int level)
{
  set_zoom_level(level < 0 ? ZOOM_LEVEL_1 : level, get_center_for_zoom());
}


void MGCanvas::set_zoom_level(int level, Point center)
{
  if (level < 0)
    level= 0;
  else if (level >= (int)MAX_ZOOM_STEPS)
    level= MAX_ZOOM_STEPS-1;
  
  _currentZoomStep= level;
  
  set_zoom(ZOOM_STEP(_currentZoomStep), center);
}


void MGCanvas::set_zoom(float zoom, Point center)
{
  int width, height;
  
  get_window()->get_size(width, height);
  
  _offset.x= center.x * zoom / _zoom + width * _zoom/2 - width * zoom / 2;
  _offset.y= center.y * zoom / _zoom + height * _zoom/2 - height * zoom / 2;
  
  _offset.x= MAX(_offset.x, 0);
  _offset.y= MAX(_offset.y, 0);

  get_enclosing_scroller()->get_hadjustment()->set_page_size(width * zoom);
  get_enclosing_scroller()->get_vadjustment()->set_page_size(height * zoom);
  
  _zoom= zoom;
  _canvas->beginUpdate();
  if (_canvas->currentViewGet())
    _canvas->currentViewGet()->zoomSet(zoom);
  _canvas->endUpdate();
  
  update_scrollers();
}


void MGCanvas::toggle_overview()
{
  if (_canvas->currentViewGet()->overviewActive())
    _canvas->currentViewGet()->overviewStop(true);
  else
    _canvas->currentViewGet()->overviewStart();
  
  // reflectContentRect();
}


void MGCanvas::set_base_size(Size size)
{
  if (size.width != _baseSize.width && size.height != _baseSize.height)
  {
    _baseSize= size;
    
    update_scrollers();
  }
}


MGCanvas::Point MGCanvas::convert_to_canvas_point(Gdk::Point point)
{
  TVertex vert;
  
  _canvas->currentViewGet()->windowToView(point.get_x(), point.get_y(), vert);
  
  return Point(vert.x, vert.y);
}


bool MGCanvas::load_layouts(const Glib::ustring &file)
{
  TGCError error= _canvas->addLayoutsFromFile(file.c_str());
  
  return error == GC_NO_ERROR;
}


bool MGCanvas::load_styles(const Glib::ustring &file, 
                           std::map<std::string,std::string> &variables)
{
  TGCError error= _canvas->addStylesFromFile(file.c_str(), variables);
  
  return error == GC_NO_ERROR;
}


void MGCanvas::render_scene()
{
  struct timeval tm1, tm2;
  double fps;
  
  gettimeofday(&tm1, NULL);
  _canvas->render();
  gettimeofday(&tm2, NULL);

  fps= 1 / ((tm2.tv_sec - tm1.tv_sec) + (tm2.tv_usec - tm1.tv_usec) / 1000000.0);
}


void MGCanvas::set_grab_panning(bool flag)
{
  _grabPanning= flag;
}
  


void MGCanvas::set_grid_enabled(bool flag)
{
  _canvas->currentViewGet()->grid()->visibleSet(flag);
  queue_draw();
}


MGCanvas::Size MGCanvas::actual_size()
{
  return MGCanvas::Size(_zoom * _baseSize.width,
                        _zoom * _baseSize.height);
}


MGCanvas::Rectangle MGCanvas::actual_visible_rect()
{
  int w, h;
  get_window()->get_size(w, h);
  
  return MGCanvas::Rectangle(_offset.x, _offset.y,
                             w * _zoom, h * _zoom);
}


void MGCanvas::set_offset(Point point)
{
  _offset= point;
  update_scrollers();
}


void MGCanvas::update_from_scrollers()
{
  if (get_enclosing_scroller() && !_changingScrollers)
  {
    Gtk::Adjustment *hadj= get_enclosing_scroller()->get_hadjustment();
    Gtk::Adjustment *vadj= get_enclosing_scroller()->get_vadjustment();
    
    if (hadj && vadj)
    {
      float hv= hadj->get_value();
      float vv= vadj->get_value();
      Size actual= actual_size();
      
      _offset.x= hv * (actual.width - get_width()) / hadj->get_upper();
      _offset.y= vv * (actual.height - get_height()) / vadj->get_upper();

      if (_canvas && _canvas->currentViewGet())
      {        
        _canvas->beginUpdate();
        _canvas->currentViewGet()->offsetXSet(-_offset.x);
        _canvas->currentViewGet()->offsetYSet(-_offset.y);
        _canvas->endUpdate();
      }
    }
  }
}



MGCanvasScroller *MGCanvas::get_enclosing_scroller()
{
  return (MGCanvasScroller*)get_parent();
}


void MGCanvas::update_scrollers()
{
  _changingScrollers= true;
  if (get_enclosing_scroller())
  {
    Gtk::Adjustment *hadj= get_enclosing_scroller()->get_hadjustment();
    Gtk::Adjustment *vadj= get_enclosing_scroller()->get_vadjustment();
  
    if (hadj && vadj)
    {
      Size asize= actual_size();
      Size size;
      int w, h;
      get_window()->get_size(w, h);
      size.width= w;
      size.height= h;
      
      _canvas->beginUpdate();
      
      if (asize.width > size.width)
        _offset.x= MAX(MIN(_offset.x, asize.width - _zoom * size.width), 0);
      else
        _offset.x= 0;
      
      if (asize.height > size.height)
        _offset.y= MAX(MIN(_offset.y, asize.height - _zoom * size.height), 0);
      else
        _offset.y= 0;
      if (_canvas->currentViewGet())
      {
        _canvas->currentViewGet()->offsetXSet(-_offset.x);
        _canvas->currentViewGet()->offsetYSet(-_offset.y);

        if (hadj->get_upper() != _baseSize.width * _zoom)
        {
          hadj->set_upper(_baseSize.width * _zoom);
        }
        if (vadj->get_upper() != _baseSize.height * _zoom)
        {
          vadj->set_upper(_baseSize.height * _zoom);
        }
        hadj->set_value(_offset.x);
        vadj->set_value(_offset.y);
      }
      _canvas->endUpdate();
    }
  }
  _changingScrollers= false;
}


void MGCanvas::create_view(const Glib::ustring &name)
{
  CGCView *view;
  
  _canvas->beginUpdate();

  view= _canvas->createView(name.c_str());
//  _baseSize= frame.size;

  view->color(1.0, 1.0, 1.0, 1.0);

  int w, h;
  get_window()->get_size(w, h);
  _viewport.set_x(0);
  _viewport.set_y(0);
  _viewport.set_width(w);
  _viewport.set_height(h);
  view->viewportSet(viewportFromRect(_viewport));

  view->setWorkspace(_baseSize.width, _baseSize.height);

  _canvas->endUpdate();

  set_zoom_level(ZOOM_LEVEL_1, get_center_for_zoom());
}


MGCanvas::Point MGCanvas::get_center_for_zoom()
{
  return Point(0,0);
}


void MGCanvas::create_view(CGCView *view)
{  
  _canvas->beginUpdate();

  view->color(1.0, 1.0, 1.0, 1.0);

  _viewport.set_x(0);
  _viewport.set_y(0);
  int w, h;
  get_window()->get_size(w,h);
  _viewport.set_width(w);
  _viewport.set_height(h);
  view->viewportSet(viewportFromRect(_viewport));

  view->setWorkspace(_baseSize.width, _baseSize.height);

  _canvas->endUpdate();

  set_zoom_level(ZOOM_LEVEL_1, get_center_for_zoom());
}


void MGCanvas::switch_to_view(CGCView *view)
{
  _canvas->currentViewSet(view);
}


void MGCanvas::start_area_selection_at(MGCanvas::Point point)
{
  TVertex v;

  v.x= point.x;
  v.y= point.y;
  v.z= 0;
  v.w= 0;
  _canvas->currentViewGet()->rubberRectStart(GC_RRSTYLE_SOLID_THIN,
                                          v, false);
}


MGCanvas::Rectangle MGCanvas::finish_area_selection()
{
  TBoundingBox bounds;
  _canvas->currentViewGet()->getLastRubberBounds(bounds);
  _canvas->currentViewGet()->rubberRectStop();

  return Rectangle(bounds.upper.x, bounds.upper.y,
                   bounds.lower.x - bounds.upper.x, bounds.lower.y - bounds.upper.y);
}




void MGCanvas::set_hand_cursor(const Gdk::Cursor &hand, const Gdk::Cursor &closed_hand)
{
  _hand_cursor= hand;
  _hand_closed_cursor= closed_hand;
}


