/* Copyright (C) 2003,2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <assert.h>

#include "myx_library.h"
#include "myx_shared_aux_functions.h"

// Windows includes
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
#include <windows.h>
#include <direct.h>
#else
// unix/linux includes

#include <sys/time.h>
#include <sys/select.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/utsname.h> // uname()
#include <fcntl.h>

#define SIZE_T size_t

# include <sys/types.h>
# include <sys/stat.h>
# include <unistd.h>
#endif

#include "myx_aux_functions.h"



/* Transforms a string consisting of
 * 2byte hexadecimal characters into
 * a binary string.
 *
 * If ret_str_len is not NULL it will be set
 * to the length of the resulting string.
 *
 * The (newly allocated) returned string
 * is exactly strlen(hex_str)/2 bytes long.
 * The returned string should be considered
 * binary. It may contain embedded 0's.
 */
char* hex_decode(const char *hex_str, int *ret_str_len)
{
  int i;
  int hex_str_len, result_len;
  unsigned char *result;

  g_return_val_if_fail(hex_str, NULL);

  hex_str_len= (int) strlen(hex_str);
  g_return_val_if_fail(hex_str_len%2 == 0, NULL);

  result_len= hex_str_len/2;  
  result= g_malloc(result_len+1);
  if (ret_str_len) *ret_str_len= result_len;

  for (i= 0; i < result_len; i++)
  {
    unsigned int num;
    sscanf(hex_str+i*2, "%2x", &num);
    result[i]= num;
  }
  result[i]= 0;

  return result;
}

/* This functions transforms an arbitrary sequence of bytes
 * into a string consisting of 0-9 and A-F characters.
 * Each byte of the input string is mapped to two bytes 
 * in the resulting-string.
 *
 * len is the length of the binary_string or -1 if the
 * string is a normal null-terminated string.
 *
 * The returned string is a valid ASCII string and thus also
 * a valid UTF-8 string.
 */
char *hex_encode(const char *binary_string, int len)
{
  //This array contains the characters 0-9 and A-F. Although it is not very readable I find this
  //more secure than using '1' etc since the value of '1' is determined at execution time and 
  //depends on the user's system.
  static char hex_char[]= {0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x41,0x42,0x43,0x44,0x45,0x46};
  int binary_string_len;
  char *hex_str;
  int j,k;
  const char *from;

  g_return_val_if_fail(binary_string,NULL);
  g_return_val_if_fail(len, NULL);

  if (len == -1)
    binary_string_len= (int) strlen(binary_string);
  else
    binary_string_len= len;

  
  hex_str= g_malloc(binary_string_len*2+1);

  from= binary_string;
  for (j=0,k=0; k < binary_string_len; j+=2, k++)
  {
    unsigned int data= (unsigned int) (unsigned char) from[k];
    hex_str[j]= hex_char[data >> 4];
    hex_str[j+1]= hex_char[data & 15];
  }
  hex_str[j]= 0;

  return hex_str;
}


static char *find_name_with_value(const MYX_NAME_VALUE_PAIR *list, const char *value)
{
  for (; list->name; list++)
  {
    if ( !g_utf8_casecollate(list->value,value) )
      return list->name;
  }

  return NULL;
}

/*
 * Translates a mysql character-set name to a
 * character-set name that is understood by iconv
 *
 * Return value: The corresponding iconv char name
 *               or "-" if there is no appropriate
 *               iconv character-set
 *               Don't free the result of this function!
 */
char *iconv_char_name(const char *mysql_character_set_name)
{
  //attribute "name": iconv-name
  //attribute "value": mysql-name
  static MYX_NAME_VALUE_PAIR iconv_mysql[]=
  {
    /* different names */
    {"HP-ROMAN8","hp8"},
    {"KOI8-R", "koi8r"},
    {"US-ASCII", "ascii"},
    {"EUC-JP", "ujis"},
    {"KOI8-U", "koi8u"},
    {"ARMSCII-8", "armscii8"},
    {"UCS-2", "ucs2"},
    {"MACCENTRALEUROPE", "macce"},

    /* same names */
    {"big5", "big5"},
    {"sjis", "sjis"},
    {"cp850","cp850"},
    {"cp866", "cp866"},
    {"latin1", "latin1"},
    {"latin2", "latin2"},
    {"latin5", "latin5"},
    {"latin7", "latin7"},
    {"hebrew", "hebrew"},
    {"tis620", "tis620"},
    {"euckr", "euckr"},
    {"gb2312", "gb2312"},
    {"greek", "greek"},
    {"cp1250", "cp1250"},
    {"gbk", "gbk"},
    {"utf8", "utf8"},
    {"macroman", "macroman"},
    {"cp1251", "cp1251"},
    {"cp1256", "cp1256"},
    {"cp1257", "cp1257"},
    {NULL, NULL}
  };
  char *iconv_name;

  iconv_name= find_name_with_value(iconv_mysql,mysql_character_set_name);
  g_return_val_if_fail(iconv_name, "-"); //this will trigger a conversion error
  return iconv_name;
}


/*
 * Normally a mysql identifier is quoted with ` (backtick).
 * In Ansi Mode " is also a valid identifier.
 * This function handles both.
 */
char* unquote_identifier(char *table_name)
{
  char quote_char;
  int write_back= -1;
  char previous_symbol= 0;
  char *beginning= table_name;

  if (*table_name == '"' || *table_name == '`')
    quote_char= *table_name++;
  else
    return table_name; /* the string is not quoted */

  for(;*table_name; table_name++)
  {
    if (table_name[0] == quote_char || table_name[0] == '\\')
    {
      if (table_name[0]!=previous_symbol)
      {
        previous_symbol= table_name[0];
      }
      else
      {
        write_back--;
        previous_symbol= 0;
      }
    }

    table_name[write_back]= *table_name;
  }

  table_name[write_back-1]= '\0';

  return beginning;
}

/* This is a helper function
 * that is only used by 
 * quote_identifier
 */
static char *quote_name(const char *name, char *buff, char quote_char)
{
  char *to= buff;

  *to++= quote_char;
  while (*name)
  {
    if (*name == quote_char)
      *to++= quote_char;
    *to++= *name++;
  }
  to[0]= quote_char;
  to[1]= 0;
  return buff;
}

/*
 * Returns a string allocated with gmalloc.
 * Quotes a string with ` (backticks). This 
 * also works in Ansi-mode. (ie it always 
 * works)
 */
char *quote_identifier(const char *identifier)
{
  char *buff;

  /* worst case: identifier consists of backticks only */
  buff= g_malloc((gulong)(strlen(identifier) * 2 + 3));

  return quote_name(identifier,buff, '`'); 
}


/*
 * Returns a string allocated with gmalloc.
 * Quotes a string with quote_char.
 */
char *quote_identifier_with(const char *identifier, char quote_char)
{
  char *buff;

  /* worst case: identifier consists of backticks only */
  buff= g_malloc((gulong)(strlen(identifier) * 2 + 3));

  return quote_name(identifier,buff, quote_char); 
}


char *mask_out_string_re(char *str, const char *open_re,
                         const char open_trigger, const char close_trigger,
                         const char mask)
{
  int maskout= 0;
  unsigned int i;
  pcre *pcre_exp;
  const char *error_str;
  int erroffset;
  int o_vector[10];
  int rc;
  int start= -1;
  size_t len= strlen(str);
  
  pcre_exp= pcre_compile(open_re, PCRE_CASELESS, &error_str, &erroffset, NULL);
  g_return_val_if_fail(pcre_exp!=NULL, str);

  if ((rc= pcre_exec(pcre_exp, NULL, str, (int)len, 0, 0, o_vector, 3) ) > 0)
    start= o_vector[0];
  pcre_free(pcre_exp);

  if (start < 0)
    return str;

  for (i= start; str[i]!=0; i++)
  {
    int c= str[i];
    if((c==close_trigger)&&(maskout>0))
    {
      maskout--;
      if(!maskout)
        continue;
    }
    if(maskout>0)
    {
      str[i]= mask;
    }
    if(c==open_trigger)
    {
      maskout++;
    }
  }

  return str;
}


char *mask_out_string(char *str, const char open_trigger,
                      const char close_trigger, const char mask)
{
  int maskout= 0;
  unsigned int i;

  for(i=0;str[i]!=0;i++)
  {
    int c= str[i];
    if((c==close_trigger)&&(maskout>0))
    {
      maskout--;

      if(!maskout)
        continue;
    }

    if(maskout>0)
    {
      str[i]= mask;
    }

    if(c==open_trigger)
    {
      maskout++;
    }
  }

  return str;
}

/** 
 * Replaces the search string with the replace string
 * in the given str
 *
 * @str becomes invalid after calling this
 * function!
 *
 * Return value: Newly allocated result string
 **/
char *str_g_replace(char *str, const char *search, const char *replace)
{
  char **splited= g_strsplit(str, search, -1);
  char *replaced= g_strjoinv(replace, splited);

  g_strfreev(splited);
  g_free(str);

  return replaced;
}

/**
 * Appends the @addon string to the end of
 * @base_str. @base_str is expected to have
 * been allocated with a glib-memory allocatio
 * function.
 *
 * @base_str becomes invalid after calling this
 * function!
 *
 * Return value: The joined string
 **/
char *str_g_append(char *base_str, const char *addon)
{
  unsigned int addon_len;
  char *tmp;

  if(!base_str)
  {
    if(!addon)
      base_str= g_strdup("");
    else
      base_str= g_strdup(addon);
  }
  else
  {
    if(addon)
    {
      addon_len= (unsigned int)strlen(addon);

      tmp= g_realloc(base_str, (unsigned int)strlen(base_str)+addon_len+1);

      base_str= strncat(tmp, addon,addon_len);
    }
  }
  
  return base_str;
}

char *str_g_append_and_free(char *base_str, char *addon)
{
  char *tmp= str_g_append(base_str, addon);

  g_free(addon);

  return tmp;
}

char *str_g_insert(const char *base_str, const char *addon, unsigned int index)
{
  char *tmp= g_malloc(sizeof(char *)*(unsigned int)(strlen(base_str)+strlen(addon)+1));

  g_stpcpy(tmp, base_str);
  g_stpcpy(tmp+index, addon);
  g_stpcpy(tmp+(index+(unsigned int)(strlen(addon))), base_str+index);

  return tmp;
}


//from libmysql/strmov.c
char *strmov(register char *dst, register const char *src)
{
  while ((*dst++ = *src++)) ;
  return dst-1;
}

int strcmp2(const char *str1, const char *str2)
{
  if((!str1)&&(!str2))
    return 0;
  if((!str1)||(!str2))
    return -1;
  else
    return strcmp(str1, str2);
}

int strcmp3(const char *str1, const char *str2)
{
  if((!str1)&&(!str2))
    return 0;
  else if(!str1)
  {
    if(strcmp(str2, "") == 0)
      return 0;
    else
      return -1;
  }
  else if(!str2)
  {
    if(strcmp(str1, "") == 0)
      return 0;
    else
      return -1;
  }
  else
    return strcmp(str1, str2);
}

unsigned int strlen2(const char *str)
{
  if(!str)
    return 0;
  else
    return (unsigned int)strlen(str);
}

char *strcpy2(char *dst, const char *src)
{
  if(!src)
  {
    *dst= 0;
  }
  else
  {
    strcpy(dst, src);
  }

  return dst;
}

char *str_trim(char *str)
{
  SIZE_T i, j;
  SIZE_T length;

  length= strlen(str);

  //Get length of leading spaces
  for (i=0; (i<length)&&isspace(str[i]); i++) 
  {}

  //Get length of trailing spaces
  for (j=length-1; (j>i)&&isspace(str[j]); j--)
  {}

  j++;
  
  //shift chars to the begining of the string
  memmove(str, str+i, j-i);

  //terminate string
  str[j-i]=0;

  return str;
}

char *str_left(char *dest, char *src, unsigned int len)
{
    unsigned int i;
    unsigned int s_len= (int)strlen(src);

    if(s_len<=len)
    {
        strncpy(dest, src, s_len);
        dest[len]= 0;
    }
    else
    {
        for(i=0;i<len;i++)
        {
            dest[i]= src[i];
        }
        dest[len]= 0;
    }

    return dest;
}

char *str_right(char *dest, char *src, unsigned int len)
{
    unsigned int i;
    unsigned int s_len= (unsigned int)strlen(src);

    if(s_len<=len)
    {
        strncpy(dest, src, s_len);
    }
    else
    {
        for(i=0;i<=len;i++)
        {
            dest[i]= src[i+s_len-len];
        }
    }

    return dest;
}

int sub_str_count(char *search_str, const char *src)
{
    unsigned int i;
    char *pdest;
    char *psrc = (char *)src;
    unsigned int ss_len= (unsigned int)strlen(search_str);

    i= 0;
    while((pdest= strstr(psrc, search_str)))
    {
        i++;
        psrc= pdest+ss_len;
    }

    return i;
}


int str_endswith(const char *str, const char *substr)
{
  int slen= (int)strlen(str);
  int sslen= (int)strlen(substr);
  
  if (slen >= sslen && strcmp(str+slen-sslen, substr)==0)
      return 1;
  else
      return 0;
}


int str_beginswith(const char *str, const char *substr)
{
  return strncmp(str, substr, strlen(substr))==0;
}


MYX_PUBLIC_FUNC int g_utf8_casecollate(const char *str1, const char *str2)
{
  char *s1= g_utf8_casefold(str1, (gssize)strlen(str1));
  char *s2= g_utf8_casefold(str2, (gssize)strlen(str2));
  int res;

  res= g_utf8_collate(s1, s2);
  g_free(s1);
  g_free(s2);

  return res;
}


char *str_toupper(char *str)
{
  char *s= str;

  while (*s) 
  {
    *s= toupper(*s);
    s++;
  }
  return str;
}


char *name_of_str(char *dst, const char *src)
{
  char *dst_start= dst;

  while (*src && *src!='=')
    *dst++= *src++;

  *dst++= 0;

  return dst_start;
}

char *value_of_str(char *dst, const char *src)
{
  char *psrc= (char *)src;

  psrc = strchr(psrc, '=');

  if (psrc)
  {
    if (*psrc)
      strcpy(dst, psrc+1);
    else
      *dst= 0;
  }
  else
    *dst= 0;

  return dst;
}

int set_value(char **string_list, unsigned int string_list_num, char *name, char *new_value)
{
  unsigned int i;
  unsigned int name_len= (unsigned int)strlen(name);
  int rc= -1;

  for(i=0; i<string_list_num; i++)
  {
    if(strncmp(string_list[i], name, name_len)==0 && 
       (string_list[i][name_len]=='\0' || string_list[i][name_len]=='='))
    {
      string_list[i]= g_realloc(string_list[i], (name_len+strlen(new_value)+2));

      if (string_list[i][name_len]=='=') 
      {
        strcpy(string_list[i]+name_len+1, new_value);
      }
      else
      {
        sprintf(string_list[i], "%s=%s", string_list[i], new_value);
      }

      rc= 0;
      break;
    }
  }

  return rc;
}

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)

int get_value_from_registry(HKEY root_key, const char *sub_key, const char *key, const char *def, char *value)
{
  HKEY hSubKey;
  DWORD dwType;
  DWORD dwSize;
  LONG retval;
  unsigned char Buffer[512];

  if((retval=RegOpenKeyEx(root_key, sub_key, 0, KEY_READ, &hSubKey))==ERROR_SUCCESS)
  {
    dwSize = 512;
    if((RegQueryValueEx(hSubKey, key, NULL, &dwType, Buffer, &dwSize))==ERROR_SUCCESS)
    {
      if(dwType==REG_DWORD)
      {
        sprintf(value, "%d", Buffer[0]+Buffer[1]*256+Buffer[2]*65536+Buffer[3]*4294967296);
      }
      else
      {
        strcpy(value, (const char *)Buffer);
      }
    }
    else
    {
      strcpy(value, def);
    }

    return 0;
  }
  else
  {
    strcpy(value, "");
    return retval;
  }
}

int set_value_to_registry(HKEY root_key, const char *sub_key, const char *key, const char *value)
{
  HKEY hSubKey;
  LONG retval;
  DWORD dwDispo;

  if((retval= RegCreateKeyEx(root_key, sub_key, 0, "",
    REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hSubKey, &dwDispo))==ERROR_SUCCESS)
  {
    retval = RegSetValueEx(hSubKey, key, 0, REG_SZ, value, (DWORD)strlen(value)+1);

    if(retval != ERROR_SUCCESS)
    {
       return GetLastError();
    } 
    else 
    {
      return 0;
    }
  }
  else
  {
    return -1;
  }
}

char *get_local_os_name()
{
  char *os_name = g_malloc(32);

  OSVERSIONINFO info;
  info.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
  GetVersionEx(&info);

  if (!os_name)
      return NULL;
  
  strcpy(os_name, "unknown");

  switch (info.dwPlatformId)
  {
    case VER_PLATFORM_WIN32_WINDOWS:
      if((info.dwMajorVersion>=4)&&(info.dwMinorVersion==0))
      {
        strcpy(os_name, "Windows 95");
      }
      else if((info.dwMajorVersion>=4)&&(info.dwMinorVersion>0)&&(info.dwMinorVersion>90))
      {
        strcpy(os_name, "Windows 98");
      }
      else if((info.dwMajorVersion>=4)&&(info.dwMinorVersion>0)&&(info.dwMinorVersion>=90))
      {
        strcpy(os_name, "Windows ME");
      }
      break;
    case VER_PLATFORM_WIN32_NT:
      if((info.dwMajorVersion<=4)&&(info.dwMinorVersion==0))
      {
        strcpy(os_name, "Windows NT");
      }
      else if((info.dwMajorVersion==5)&&(info.dwMinorVersion==0))
      {
        strcpy(os_name, "Windows 2k");
      }
      else if((info.dwMajorVersion==5)&&(info.dwMinorVersion==1))
      {
        strcpy(os_name, "Windows XP");
      }
      else if((info.dwMajorVersion==5)&&(info.dwMinorVersion==2))
      {
        strcpy(os_name, "Windows Server 2003");
      }
      break;
  }

  return os_name;
}


char *get_local_hardware_info()
{
  char *hardware_string;
  SYSTEM_INFO sysinfo;
  MEMORYSTATUS memstat;
  int retval;
  char processor_name[100];
  char processor_mhz[10];
  char total_phys_ram[16];
  SIZE_T total_phys_ram_val;


  GetSystemInfo(&sysinfo);
  
  GlobalMemoryStatus(&memstat);

  strcpy(processor_mhz, "");
  strcpy(processor_name, "");

  retval= get_value_from_registry(HKEY_LOCAL_MACHINE, 
    "HARDWARE\\DESCRIPTION\\System\\CentralProcessor\\0", "~MHz", "", 
    processor_mhz);
  
  retval= get_value_from_registry(HKEY_LOCAL_MACHINE, 
    "HARDWARE\\DESCRIPTION\\System\\CentralProcessor\\0", "ProcessorNameString", "", 
    processor_name);

  if(!processor_name[0])
  {
    retval= get_value_from_registry(HKEY_LOCAL_MACHINE, 
      "HARDWARE\\DESCRIPTION\\System\\CentralProcessor\\0", "Identifier", "", 
      processor_name);

    sprintf(processor_name, "%s %s", processor_name, processor_mhz);
  }

  str_trim(processor_name);


  total_phys_ram_val= memstat.dwTotalPhys;
  if(total_phys_ram_val>=1072410624/1.9)
  {
    sprintf(total_phys_ram, "%1.1f GB RAM", (double)total_phys_ram_val/1072410624.0);
  }
  else if(total_phys_ram_val>=1047276/1.9)
  {
    sprintf(total_phys_ram, "%1.0f MB RAM", (double)total_phys_ram_val/1047276.0);
  }
  else
  {
    sprintf(total_phys_ram, "%d B RAM", total_phys_ram_val);
  }
   
  hardware_string = g_malloc(16 + (gulong)strlen(processor_name)
                           + (gulong)strlen(total_phys_ram));

  if(sysinfo.dwNumberOfProcessors>1) 
  {
    sprintf(hardware_string, "%dx %s, %s", sysinfo.dwNumberOfProcessors, processor_name, total_phys_ram);
  }
  else
  {
    sprintf(hardware_string, "%s, %s", processor_name, total_phys_ram);
  }

  return hardware_string;
}
#else /* !__WIN__ */

// Linux/Unix version

char *get_local_os_name()
{
  char *buf;
  struct utsname info;

  if (uname(&info) < 0)
  {
    return NULL;
  }
  
  buf = g_malloc(strlen(info.sysname)+strlen(info.release)+8);
  if (!buf)
  {
      return NULL;
  }

  sprintf(buf, "%s %s", info.sysname, info.release);

  return buf;
}


char *get_local_hardware_info()
{
  char *hardware_string;
  int processor_count = 0;
  char processor_name[256];
  char processor_mhz[128];
  char total_phys_ram[64];
  long total_phys_ram_val = 0;
  FILE *proc;
  char line[256];

  
 
  // fetch processor info from /proc/cpuinfo

  proc= fopen("/proc/cpuinfo", "r");
  if (!proc) 
  {
    return NULL;
  }
    
  while (!feof(proc)) 
  {
    if (!fgets(line, sizeof(line), proc))
        break;

    if (str_beginswith(line,"model name")) 
    {
      processor_count++;
      str_trim(strcpy(processor_name, strchr(line, ':')+1));
    } 
    else if (str_beginswith(line,"cpu MHz"))
    {
      str_trim(strcpy(processor_mhz, strchr(line, ':')+1));
    }
  }
  fclose(proc);
  

  // fetch physical memory info from /proc/meminfo
  proc= fopen("/proc/meminfo", "r");
  if (proc) 
  {
    char *ptr, *end;
    
    fgets(line, sizeof(line), proc); // header
    fgets(line, sizeof(line), proc); // meminfo
    fclose(proc);
    
    ptr= strchr(line, ':')+1;
    while (*ptr && *ptr==' ') ptr++;
    end= strchr(ptr, ' ');
    if (end)
        *end= 0;
    total_phys_ram_val = atoi(str_trim(ptr));
  }

  if(total_phys_ram_val>=1072410624/1.9)
  {
    sprintf(total_phys_ram, "%1.1f GB RAM", (double)total_phys_ram_val/1072410624.0);
  }
  else if(total_phys_ram_val>=1047276/1.9)
  {
    sprintf(total_phys_ram, "%1.0f MB RAM", (double)total_phys_ram_val/1047276.0);
  }
  else
  {
    sprintf(total_phys_ram, "%ld B RAM", total_phys_ram_val);
  }

  hardware_string= g_malloc(16+strlen(processor_name)+strlen(processor_mhz)+strlen(total_phys_ram));

  if(processor_count>1)
  {
    sprintf(hardware_string, "%dx %s %s MHz, %s", processor_count, processor_name, processor_mhz, total_phys_ram);
  }
  else
  {
    sprintf(hardware_string, "%s %s MHz, %s", processor_name, processor_mhz, total_phys_ram);
  }

  return hardware_string;
}



FILE *myx_popen(char *const args[], pid_t *pid_ret)
{
  FILE *f;
  int fd[2];
  
  if (pipe(fd) >= 0)
  {
    *pid_ret = fork();
    if (*pid_ret == 0) 
    {
      close(1);
      close(2);
      dup2(fd[1], 1);
      dup2(fd[1], 2);

      close(fd[0]);

      execvp(args[0], args);
      exit(-1);
    }
    else if (*pid_ret > 0)
    {
      close(fd[1]);

      f = fdopen(fd[0], "r");
      if (!f)
      {
        int ret;
        kill(*pid_ret, 9);
        waitpid(*pid_ret, &ret, 0);
        close(fd[0]);
      }
      return f;
    }
  }

  return NULL;
}


int myx_read_timeout(FILE *f, int timeout, char *result, size_t result_len)
{
  fd_set fds;
  struct timeval tv;
  int ret;
  
  FD_ZERO(&fds);
  FD_SET(fileno(f), &fds);

  tv.tv_sec = timeout / 1000;
  tv.tv_usec = (timeout % 1000)*1000;
  
  do {
    ret= select(fileno(f)+1, &fds, NULL, NULL, timeout<0?NULL:&tv);
  } while (ret < 0 && errno == EINTR);
  
  if (ret > 0)
  {
    if (fgets(result, result_len, f))
      return strlen(result);
    else
      return 0;
  }
  return -1;
}


int myx_pclose(FILE *f, pid_t pid)
{
  int status = -1;

  if (kill(pid, SIGKILL) == 0)
      waitpid(pid, &status, 0);
  fclose(f);

  return status;
}

int copy_file(const char *orig_file, const char *new_file)
{
  char buffer[4*1024];
  int from_fd, to_fd;
  int count;
  
  from_fd= open(orig_file, O_RDONLY, 0666);
  if (from_fd < 0)
    return -1;
  to_fd= open(new_file, O_WRONLY|O_CREAT, 0666);
  if (to_fd < 0)
  {
    int err= errno;
    close(from_fd);
    errno= err;
    return -1;
  }
  
  do {
    count= read(from_fd, buffer, sizeof(buffer));
    if (count < 0)
    {
      int err= errno;
      close(from_fd);
      close(to_fd);
      errno= err;
      return -1;
    }
    if (count > 0 && write(to_fd, buffer, count) < 0)
    {
      int err= errno;
      close(from_fd);
      close(to_fd);
      errno= err;
      return -1;
    }
  } while (count > 0);

  close(from_fd);
  close(to_fd);

  return 0;
}
#endif


bigint get_file_size(const char *filename)
{
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  DWORD dwSizeLow;
  DWORD dwSizeHigh = 0;
  HANDLE hfile;
  char *local_filename;

  if (! (local_filename= g_filename_from_utf8(filename,-1,NULL,NULL,NULL)))
    return -1;
  hfile = CreateFile(local_filename, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
                     OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

  if (hfile != INVALID_HANDLE_VALUE) 
  {
    dwSizeLow= GetFileSize(hfile, &dwSizeHigh);

    CloseHandle(hfile);
    g_free(local_filename);

    if((dwSizeLow==INVALID_FILE_SIZE)&&(GetLastError()) != NO_ERROR )
    { 
      return -1;
    }
    else
    {
      return ((dwSizeHigh << 32) + dwSizeLow);
    }
  }
  else
  {
    g_free(local_filename);
    return -1;
  }
#else //!WINDOWS
  struct stat buf;
  char *local_filename;

  if (! (local_filename= g_filename_from_utf8(filename,-1,NULL,NULL,NULL)))
    return -1;

  if (stat(local_filename, &buf) < 0) {
    g_free(local_filename);
    return -1;
  }
  g_free(local_filename);
  return buf.st_size;
#endif //!WINDOWS
}

// note, needle has to be ascii!
char *strcasestr_len(const char *haystack, int haystack_len, const char *needle)
{
  gssize needle_len= (gssize)strlen(needle);
  int i;

  if (needle_len > haystack_len)
    return NULL;

  i= 0;
  while (i <= haystack_len - needle_len)
  {
    if (g_ascii_strncasecmp(needle, haystack+i, needle_len)==0)
      return (char *)haystack+i;
    i++;
  }
  return NULL;
}

#define O_VECTOR_COUNT 64 // max # of ()*2+2

char * get_value_from_text_ex_opt(const char *txt, int txt_length,
                                  const char *regexpr,
                                  unsigned int substring_nr,
                                  int options_for_exec)
{
  pcre *pcre_exp;
  const char *error_str;
  int erroffset;
  int o_vector[O_VECTOR_COUNT];
  int rc;
  const char *ret_val;
  char *value= NULL;

  if(txt && *txt)
  {
    pcre_exp= pcre_compile(regexpr, PCRE_CASELESS, &error_str, &erroffset, NULL);
    if (pcre_exp)
    {
      if ( (rc= pcre_exec(pcre_exp, NULL, txt, txt_length, 0, 
                          options_for_exec, o_vector, O_VECTOR_COUNT) ) > 0)
      {
        pcre_get_substring(txt, o_vector, rc, substring_nr, &ret_val);

        value= g_strdup(ret_val);

        pcre_free_substring((char*)ret_val);
      }

      pcre_free(pcre_exp);
    }
  }

  return value;
}

char * get_value_from_text_ex(const char *txt, int txt_length,
                              const char *regexpr, unsigned int substring_nr)
{
  return get_value_from_text_ex_opt(txt,txt_length,regexpr,substring_nr,0);
}

char * get_value_from_text(const char *txt, int txt_length, const char *regexpr)
{
  return get_value_from_text_ex(txt, txt_length, regexpr, 1);
}

void strlist_g_append(char ***list, char *value)
{
  unsigned int i= 0;

  if (*list)
  {
    for (i= 0; (*list)[i]; i++);
    *list= g_realloc(*list, sizeof(char*)*(i+2));
    (*list)[i]= value;
    (*list)[i+1]= 0;
  }
  else
  {
    *list= g_malloc(sizeof(char*)*2);
    (*list)[0]= value;
    (*list)[1]= 0;
  }
}


static void __sappend(char **str, int *ressize, int *reslen, const char *sbegin, int count)
{
  if (*reslen + count + 1 >= *ressize)
  {
    *ressize+= count + 100;
    *str= g_realloc(*str, *ressize);
  }
  strncpy(*str+*reslen, sbegin, count);
  *reslen+= count;
  (*str)[*reslen]= 0;
}


const char *strfindword(const char *str, const char *word)
{
  const char *ptr;
  size_t wordlen= strlen(word);
  
  ptr= str;
  for (;;)
  {
    // find match
    ptr= strcasestr_len(ptr, (int)strlen(ptr), word);
    if (!ptr)
      return NULL;

    // check if its acceptable
    if ((ptr == str || isspace(*(ptr-1))) && // space before
        (isspace(*(ptr+wordlen)) || *(ptr+wordlen)==0)) // space after
      return ptr;

    ptr+= wordlen;
  }

  return NULL;
}


char *subst_pcre(const char *pattern, const char *repl, 
                 int flags, int max_matches, 
                 const char *string)
{
  pcre *pcre_exp;
  const char *error_str;
  int erroffset;
  char *res= NULL;

  pcre_exp= pcre_compile(pattern, flags, &error_str, &erroffset, NULL);
  if (pcre_exp)
  {
    int rc;
    int *matches= (int*)g_malloc(sizeof(int)*(max_matches*3));
    rc= pcre_exec(pcre_exp, NULL, string, (int)strlen(string), 0, 0, matches, max_matches);
    if (rc > 0)
    {
      res= subst_pcre_matches(string, matches, rc, repl);
    }
    pcre_free(pcre_exp);
    g_free(matches);
  }
  else
  {
    g_message("error compiling PCRE pattern: %s", error_str);
  }
  return res;
}


char *subst_pcre_matches(const char *src, int *matches, int matchcount,
                         const char *repl)
{
  const char *p0, *pf;
  int ressize= (int)strlen(repl);
  int reslen= 0;
  char *res= g_malloc(ressize);
  char number[4];
  int index;

  p0= repl;
  while (p0)
  {
    pf= strchr(p0, '\\');
    if (pf)
    {
      __sappend(&res, &ressize, &reslen, p0, (int)(pf-p0));
      pf++;
      // no more than 99 groups!
      if (isdigit(*pf) && isdigit(*(pf+1)))
      {
        number[0]=*(pf++);
        number[1]=*(pf++);
        number[2]=0;
        index= atoi(number);
      }
      else if (isdigit(*pf))
      {
        number[0]=*(pf++);
        number[1]=0;
        index= atoi(number);
      }
      else
        index= -1;
      if (index > 0 && index <= matchcount)
        __sappend(&res, &ressize, &reslen, src+matches[index*2],
                  matches[index*2+1]-matches[index*2]);
    }
    else
    {
      __sappend(&res, &ressize, &reslen, p0, (int)strlen(p0));
      pf= NULL;
    }
    p0= pf;
  }

  return g_realloc(res, reslen+1);
}

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)

void timer_start(MYX_TIMER_VALUE *tval)
{
  QueryPerformanceCounter(tval);
}


double timer_stop(MYX_TIMER_VALUE *tval)
{
  MYX_TIMER_VALUE end;
  LARGE_INTEGER freq;
  
  QueryPerformanceCounter(&end);
  
  QueryPerformanceFrequency(&freq);
  
  return ((double)end.QuadPart - (double)tval->QuadPart)/((double)freq.QuadPart);
}

#else


void timer_start(MYX_TIMER_VALUE *tval)
{
  gettimeofday(tval, NULL);
}


double timer_stop(MYX_TIMER_VALUE *tval)
{
  MYX_TIMER_VALUE end;
  
  gettimeofday(&end, NULL);
  
  return (end.tv_sec - tval->tv_sec) + (end.tv_usec - tval->tv_usec)/1000000;
}

#endif


void *vec_insert_resize(void *vec, gulong size, gulong *vecsize, guint pos, void *data)
{
  vec= g_realloc(vec, size*(*vecsize+1));

  if (*vecsize > 0 && ((size_t)pos) < (*vecsize-1))
    memmove((char*)vec+(pos+1)*size, (char*)vec+pos*size, size*(*vecsize-pos));

  memcpy((char*)vec+pos*size, data, size);

  (*vecsize)++;

  return vec;
}


void *vec_remove(void *vec, gulong size, gulong *vecsize, guint pos)
{
  if (*vecsize > 0 && ((size_t)pos) < (*vecsize-1))
    memmove((char*)vec+pos*size, (char*)vec+(pos+1)*size, size*(*vecsize-pos-1));

  (*vecsize)--;

  return vec;
}


int split_schema_table(const char *ident, char **schema, char **table)
{
  pcre *pcre_exp;
  const char *error_str;
  int erroffset;
  int o_vector[32];
  int rc;

  pcre_exp= pcre_compile("(\\w+|`.+?`|\".+?\")(?:\\.(\\w+|`.+?`|\".+?\"))?",
                         PCRE_CASELESS|PCRE_UTF8|PCRE_DOTALL, &error_str, &erroffset, NULL);

  *schema= NULL;
  *table= NULL;
  
  if (pcre_exp)
  {
    if ((rc= pcre_exec(pcre_exp, NULL, ident, (int)strlen(ident),
                       0, 0, o_vector, sizeof(o_vector)/sizeof(int))) > 0)
    {
      const char *a= NULL, *b= NULL;

      pcre_get_substring(ident, o_vector, rc, 1, &a);
      if (a)
      {
        *schema= unquote_identifier(g_strdup(a));
        pcre_free_substring(a);
      }

      pcre_get_substring(ident, o_vector, rc, 2, &b);
      if (b)
      {
        *table= unquote_identifier(g_strdup(b));
        pcre_free_substring(b);
      }
      
      pcre_free(pcre_exp);
      
      if (*schema && *table)
        return 2;
      else if (*schema && !*table)
      {
        *table= *schema;
        *schema= NULL;
        return 1;
      }
    }
    pcre_free(pcre_exp);
  }
  return -1;
}


#include "entities.h"

static int cmp_entity(const void *a, const void *b)
{
  HTMLEntity *aa= (HTMLEntity*)a;
  HTMLEntity *bb= (HTMLEntity*)b;
  
  if (aa->code < bb->code)
    return -1;
  else if (aa->code > bb->code)
    return 1;
  else
    return 0;
}


static char *escape_entities(const char *str, HTMLEntity *entities)
{
  char *outstr= NULL;
  int alloced= 0;
  int length= 0;
  gunichar uc;
  int entity_num= 0;
  while (entities[entity_num].name) entity_num++;
  
  while (*str)
  {
    HTMLEntity key;
    HTMLEntity *p;
    
    uc= g_utf8_get_char(str);

    key.code= uc;
    p= (HTMLEntity*)bsearch(&key, entities, entity_num, sizeof(entities[0]), cmp_entity);

    if (p)
    {
      char buffer[100];
      
      buffer[0]= '&';
      strcpy(buffer+1, p->name);
      strcat(buffer, ";");

      __sappend(&outstr, &alloced, &length, buffer, (int)strlen(buffer));
    }
    else
    {
      __sappend(&outstr, &alloced, &length, str, (int)(g_utf8_next_char(str)-str));
    }

    str= g_utf8_next_char(str);
  }

  return outstr;
}

char *escape_xml_entities(const char *str)
{
  return escape_entities(str, XMLEntities);
}


char *escape_html_entities(const char *str)
{
  return escape_entities(str, HTMLEntities);
}


char *unescape_html_entities(const char *str)
{
  char *outstr= NULL;
  int alloced= 0;
  int length= 0;
  int i;
  const char *b, *e;
  gchar utf8char[10];
  
  b= str;
  while (*b)
  {
    e= strchr(b, '&');
    if (e)
    {
      const char *p;

      if (e-b > 0) // copy until &
        __sappend(&outstr, &alloced, &length, b, (int)(e-b));

      // find end of &thing;
      p= e+1;
      while (*p && *p!='&' && *p!=';') p++; 

      if (*p == 0) // no ; found
        __sappend(&outstr, &alloced, &length, e, (int)(p-e));
      else if (*p == '&') // another & found, copy what we have and start over
        __sappend(&outstr, &alloced, &length, e, (int)(p-e));
      else if (*p == ';') // &ewq;
      {
        p++;

        i= -1;
        if (p-e-2 > 0)
        {
          for (i= 0; HTMLEntities[i].name; i++)
          {
            if (strncmp(HTMLEntities[i].name, e+1, p-e-2)==0)
              break;
          }
        }

        if (i >= 0 && HTMLEntities[i].name && p-e-2>0)
        {
          i= g_unichar_to_utf8(HTMLEntities[i].code, utf8char);
          __sappend(&outstr, &alloced, &length, utf8char, i);
        }
        else // invalid &thing; just copy it over
          __sappend(&outstr, &alloced, &length, e, (int)(p-e));
      }

      b= p;
    }
    else
    {
      /* no &things; in the rest of the string */
      __sappend(&outstr, &alloced, &length, b, (int)strlen(b));
      break;
    }
  }

  return outstr;
}


//taken from glib
int myx_mkdir(const char *filename, int mode)
{
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  wchar_t *wfilename = g_utf8_to_utf16 (filename, -1, NULL, NULL, NULL);
  int retval = _wmkdir (wfilename);
  int save_errno = errno;

  g_free (wfilename);
  
  errno = save_errno;
  return retval;
#else
  return mkdir (filename, mode);
#endif
}

