## -*- coding: utf-8 -*-
#
# «proprietary_codecs» - MCC Proprietary Codecs enablement plugin
#
# Copyright (C) 2009, Mario Limonciello, for Mythbuntu
#
#
# Mythbuntu is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this application; if not, write to the Free Software Foundation, Inc., 51
# Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##################################################################################

from MythbuntuControlCentre.plugin import MCCPlugin
import commands
import re
import os
import logging
from mythbuntu_common.dictionaries import *

MEDIBUNTU_LIST = '/etc/apt/sources.list.d/medibuntu.list'

class ProprietaryCodecsPlugin(MCCPlugin):
    """Proprietary Codec Enablement"""

    def __init__(self):
        #Initialize parent class
        information = {}
        information["name"] = "Proprietary Codecs"
        information["icon"] = "gtk-dialog-warning"
        information["ui"] = "tab_proprietary_codecs"
        MCCPlugin.__init__(self,information)

    def captureState(self):
        """Determines the state of the items on managed by this plugin
           and stores it into the plugin's own internal structures"""
        #Dictionaries
        self.dictionary_state={}
        list=get_nonfree_dictionary(self)
        for item in list:
            self.dictionary_state[list[item]]=self.query_installed(item)

        self.medibuntu=self.find_medibuntu()

    def applyStateToGUI(self):
        """Takes the current state information and sets the GUI
           for this plugin"""
        #Load the detected dictionary
        for item in self.dictionary_state:
            item.set_active(self.dictionary_state[item])

        #Only set appropriate items for the architecture
        #if and when multiarch is enabled, we'll have to adjust this
        arch = commands.getstatusoutput('uname -m')[1]
        if arch != "x86_64":
            self.enable_w64codecs.set_sensitive(False)
        if arch != "i686":
            self.enable_w32codecs.set_sensitive(False)

        self.enable_medibuntu.set_active(self.medibuntu)

        #This is done here on purpose so that you need to enable medibuntu
        #before you install these packages
        self.medibuntu_keyring.set_sensitive(self.medibuntu)
        self.proprietary_codecs.set_sensitive(self.medibuntu and self.medibuntu_keyring.get_active())


    def compareState(self):
        """Determines what items have been modified on this plugin"""
        #Prepare for state capturing
        MCCPlugin.clearParentState(self)

        #Installable items
        list=get_nonfree_dictionary(self)
        for item in list:
            if list[item].get_active() != self.dictionary_state[list[item]]:
                if list[item].get_active():
                    if item == "medibuntu-keyring":
                        self._markUnauthenticatedPackages(True)
                        self._markUpdatePackageList()
                    self._markInstall(item)
                else:
                    self._markRemove(item)

        if self.enable_medibuntu.get_active() != self.medibuntu:
            self._markReconfigureRoot("medibuntu",self.enable_medibuntu.get_active())
            self._markUpdatePackageList()

    def find_medibuntu(self):
        """Determines if the medibuntu repo is active"""
        in_f=open("/etc/apt/sources.list")
        pattern=re.compile("^deb http://packages.medibuntu.org/")
        found=False
        for line in in_f:
            if pattern.search(line):
                found=True
        if not found and os.path.exists(MEDIBUNTU_LIST):
            found=True
        return found

    def update_source(self, line, remove):
        from aptsources import distro
        from aptsources.sourceslist import SourcesList, SourceEntry
        from softwareproperties.SoftwareProperties import SoftwareProperties
        SoftwareProperties = SoftwareProperties()
        try:
            if remove == False:
                SoftwareProperties.sourceslist.list.append(SourceEntry(line))
            else:
                SoftwareProperties.sourceslist.list.remove(SourceEntry(line))
        except ValueError:
            logging.warning("Unable to remove '%s' from /etc/apt/sources.list. Please manually remove." % line)
        SoftwareProperties.save_sourceslist()
        SoftwareProperties.reload_sourceslist()

    def root_scripted_changes(self,reconfigure):
        """System-wide changes that need root access to be applied.
           This function is ran by the dbus backend"""
        #TODO
        # - add the gpg key (nicely)
        # - do an apt update (nicely)
        for item in reconfigure:
            if item == 'medibuntu':
                try:
                    import lsb_release
                    codename = lsb_release.get_distro_information()['CODENAME']
                except ImportError:
                    logging.warning("Unable to import lsb_release")
                    import subprocess
                    p1 = subprocess.Popen(['lsb_release','-cs'], stdout=subprocess.PIPE)
                    codename = p1.communicate()[0].split()[0]
                line = 'deb http://packages.medibuntu.org/ ' + codename + ' non-free free'
                #enabling
                if reconfigure[item]:
                    file = open(MEDIBUNTU_LIST,'w')
                    print >> file, line
                    file.close()
                else:
                    if os.path.exists(MEDIBUNTU_LIST):
                        logging.debug("Removing %s" % MEDIBUNTU_LIST)
                        try:
                            os.remove(MEDIBUNTU_LIST)
                        except OSError, msg:
                            logging.warning("Error removing medibuntu.list, %s" % msg)
                    else:
                        logging.debug("Updating main /etc/apt/sources.list")
                        self.update_source(line,remove=True)
