/*
 * Mausezahn - A fast versatile traffic generator
 * Copyright (C) 2008 Herbert Haas
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 as published by the 
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more 
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with 
 * this program; if not, see http://www.gnu.org/licenses/gpl-2.0.html
 * 
*/


#include "mz.h"


// Check if current system supports the nanosecond timer functions.
// Additionally, measure the precision.
// This function should be called upon program start.
// 
int check_timer()
{
   
   struct timespec res;
   int r;


   
   // Check if the glibc is recent enough:
   if (_POSIX_C_SOURCE >= 199309L)
     {
	r = clock_getres(CLOCK_MONOTONIC, &res);


	
	if (r!=0) 
	  {
	     perror(" mz/check_timer:");
	  }

	
	if (verbose)
	  {
	     fprintf(stderr, " System supports high resolution clock.\n");
	     fprintf(stderr, "  The clock resolution is %li nanoseconds.\n",
		     res.tv_nsec);
	  }
	
	
	if (res.tv_nsec > 1)
	  {
	     if (verbose)
	       {
		  fprintf(stderr, "  Your system does not support nanosecond precision; Mausezahn will\n"
		         	  "  use the traditional clock instead. The main consequence is that if\n"
			          "  somebody (or some process) updates your system clock during a\n"
			          "  long-term jitter measurement, then your data will include a spike.\n"
			          "\n");
	       }
	     
	     monotonic_clock = 0;  // use gettimeofday()
	  }
	else
	  {
	     
	     monotonic_clock = 1; // use clock_gettime(CLOCK_MONOTONIC)
	  }
	
     }
   else
     {
	fprintf(stderr, " WARNING: Your system does NOT support the newer high resolution clock\n"
		        "          Please inform the author: herbert@perihel.at\n");
	exit(1);
     }

   return 0;
}




// This is the replacement for gettimeofday() which would result in 'jumps' if
// the system clock is adjusted (e. g. via a NTP process) and finally the jitter
// measurement would include wrong datapoints.
// 
// Furthermore the function below utilizes the newer hi-res nanosecond timers.
// However it is rounded to microseconds so all other functions may use the
// traditional struct timeval (which consists of sec and usec).
// 
int getcurtime (struct timeval *tv)
{

   struct timespec ct;
   int r;

   
   if (monotonic_clock)
     {
	r = clock_gettime(CLOCK_MONOTONIC, &ct);
	
	tv->tv_sec = ct.tv_sec;
  // TODO: make faster division, consider shifting by 10 bits (/=1024)
  // also consider different timer precision than 1 nsec...
  // What is the content of ct.tv_nsec on a system with lower precision?
  // Would this be also with nsec units?
	tv->tv_usec = ct.tv_nsec / 1000;  
	
	return 0;
     }
   else // use traditional clock
     {
	gettimeofday(tv, NULL);
	return 0;
     }
   
}







//////////////////////////////////////////////////////////////////////////////////////
// Purpose: Calculate time deltas of two timestamps stored in struct timeval.
// 
// This function is a slightly modified version of an example found in the glibc book
// Subtract the "struct timeval" values X and Y, storing the result in RESULT.
// Return 1 if the difference is negative, otherwise 0.
int timeval_subtract (struct timeval *x,
		      struct timeval *y,
		      struct timeval *result)
{
   
   /* Perform the carry for the later subtraction by updating x. */
   if (y->tv_usec < x->tv_usec)
     {
	
	int nsec = (x->tv_usec - y->tv_usec) / 1000000 + 1;
	x->tv_usec -= 1000000 * nsec;
	x->tv_sec += nsec;
     }
   
   
   if (y->tv_usec - x->tv_usec > 1000000)
     {
	
	int nsec = (y->tv_usec - x->tv_usec) / 1000000;
	x->tv_usec += 1000000 * nsec;
	x->tv_sec -= nsec;
     }
   
   
   /* Compute the time remaining to wait.
    * tv_usec is certainly positive. */
   result->tv_sec = y->tv_sec - x->tv_sec;
   result->tv_usec = y->tv_usec - x->tv_usec;
   
   /* Return 1 if result is negative. */
   return y->tv_sec < x->tv_sec;
}



// Returns a human readable timestamp in the string result.
// Optionally a prefix can be specified, for example if the
// timestamp is part of a filename.
// 
// Example: 
//    char myTimeStamp[128];
//    
//    timestamp_human(myTimeStamp, NULL);
//    
//    => "20080718_155521"
//    
//    /* or with prefix */
//    
//    timestamp_human(myTimeStamp, "MZ_RTP_jitter_");
// 
//    => "MZ_RTP_jitter_20080718_155521"
// 
int timestamp_human(char* result, const char* prefix)
{
   time_t curtime;
   struct tm curtime_broken;
   char curtime_str[32];
   
   time(&curtime);
   localtime_r (&curtime, &curtime_broken);

   sprintf(curtime_str, "%4i%02i%02i-%02i%02i%02i",
	   curtime_broken.tm_year+1900,
	   curtime_broken.tm_mon+1,
	   curtime_broken.tm_mday,
	   curtime_broken.tm_hour,
	   curtime_broken.tm_min,
	   curtime_broken.tm_sec);
   
   if (prefix==NULL)
     {
	strncpy(result, curtime_str, 32);
     }
   else
     {
	strncpy(result, prefix, 32);
	strncat(result, curtime_str, 32);
     }
   
   return 0;
}


// Returns a human readable timestamp in the string result.
// Optionally a prefix can be specified, for example if the
// timestamp is part of a filename.
// 
// Example: 
//    char myTimeStamp[8];
//    
//    timestamp_hms (myTimeStamp);
//    
//    => "15:55:21"
int timestamp_hms(char* result)
{
   time_t curtime;
   struct tm curtime_broken;
   char curtime_str[32];
   
   time(&curtime);
   localtime_r (&curtime, &curtime_broken);

   sprintf(curtime_str, "%02i:%02i:%02i",
	   curtime_broken.tm_hour,
	   curtime_broken.tm_min,
	   curtime_broken.tm_sec);
   
   strncpy(result, curtime_str, 8);
   
   return 0;
}





