#include <assert.h>
extern "C" {
#include <lauxlib.h>
#include <lua.hpp>
}

#ifdef HAVE_CEF
#include "cef_capture.h"
#endif
#include "ffmpeg_capture.h"
#include "flags.h"
#include "image_input.h"
#include "input_state.h"
#include "lua_utils.h"
#include "scene.h"
#include "theme.h"

using namespace movit;
using namespace std;

bool display(Block *block, lua_State *L, int idx);

EffectType current_type(const Block *block)
{
	return block->alternatives[block->currently_chosen_alternative]->effect_type;
}

int find_index_of(const Block *block, EffectType val)
{
	for (size_t idx = 0; idx < block->alternatives.size(); ++idx) {
		if (block->alternatives[idx]->effect_type == val) {
			return idx;
		}
	}
	return -1;
}

string get_declaration_point(lua_State *L)
{
	lua_Debug ar;
	lua_getstack(L, 1, &ar);
	lua_getinfo(L, "nSl", &ar);
	char buf[256];
	snprintf(buf, sizeof(buf), "%s:%d", ar.source, ar.currentline);
	return buf;
}

Scene::Scene(Theme *theme, float aspect_nom, float aspect_denom)
	: theme(theme), aspect_nom(aspect_nom), aspect_denom(aspect_denom), resource_pool(theme->get_resource_pool()) {}

size_t Scene::compute_chain_number(bool is_main_chain) const
{
	assert(chains.size() > 0);
	assert(chains.size() % 2 == 0);
	bitset<256> disabled = find_disabled_blocks(size_t(-1));

	size_t chain_number = compute_chain_number_for_block(blocks.size() - 1, disabled);
	assert(chain_number < chains.size() / 2);
	if (is_main_chain) {
		chain_number += chains.size() / 2;
	}
	return chain_number;
}

size_t Scene::compute_chain_number_for_block(size_t block_idx, const bitset<256> &disabled) const
{
	Block *block = blocks[block_idx];
	size_t chain_number;

	size_t currently_chosen_alternative;
	if (disabled.test(block_idx)) {
		// It doesn't matter, so pick the canonical choice
		// (this is the only one that is actually instantiated).
		currently_chosen_alternative = block->canonical_alternative;
	} else {
		currently_chosen_alternative = block->currently_chosen_alternative;
	}
	assert(currently_chosen_alternative < block->alternatives.size());

	if (block_idx == 0) {
		assert(block->cardinality_base == 1);
		chain_number = currently_chosen_alternative;
	} else {
		chain_number = compute_chain_number_for_block(block_idx - 1, disabled) + block->cardinality_base * currently_chosen_alternative;
	}
	return chain_number;
}

bitset<256> Scene::find_disabled_blocks(size_t chain_idx) const
{
	assert(blocks.size() < 256);

	// The find_disabled_blocks() recursion logic needs only one pass by itself,
	// but the disabler logic is not so smart, so we just run multiple times
	// until it converges.
	bitset<256> prev, ret;
	do {
		find_disabled_blocks(chain_idx, blocks.size() - 1, /*currently_disabled=*/false, &ret);
		prev = ret;
		for (Block *block : blocks) {
			EffectType chosen_type = block->alternatives[block->chosen_alternative(chain_idx)]->effect_type;
			if (ret.test(block->idx) || chosen_type == IDENTITY_EFFECT) continue;  // Already disabled.

			for (Block::Index disabler_idx : block->disablers) {
				Block *disabler = blocks[disabler_idx];
				EffectType chosen_type = disabler->alternatives[disabler->chosen_alternative(chain_idx)]->effect_type;
				if (ret.test(disabler->idx) || chosen_type == IDENTITY_EFFECT) {
					ret.set(block->idx);
					break;
				}
			}
		}
	} while (prev != ret);
	return ret;
}

void Scene::find_disabled_blocks(size_t chain_idx, size_t block_idx, bool currently_disabled, bitset<256> *disabled) const
{
	if (currently_disabled) {
		disabled->set(block_idx);
	}
	Block *block = blocks[block_idx];
	EffectType chosen_type = block->alternatives[block->chosen_alternative(chain_idx)]->effect_type;
	for (size_t input_idx = 0; input_idx < block->inputs.size(); ++input_idx) {
		if (chosen_type == IDENTITY_EFFECT && input_idx > 0) {
			// Multi-input effect that has been replaced by
			// IdentityEffect, so every effect but the first are
			// disabled and will not participate in the chain.
			find_disabled_blocks(chain_idx, block->inputs[input_idx], /*currently_disabled=*/true, disabled);
		} else {
			// Just keep on recursing down.
			find_disabled_blocks(chain_idx, block->inputs[input_idx], currently_disabled, disabled);
		}
	}
}

bool Scene::is_noncanonical_chain(size_t chain_idx) const
{
	bitset<256> disabled = find_disabled_blocks(chain_idx);
	if (disabled.none()) {
		return false;
	}
	assert(blocks.size() < 256);
	for (size_t block_idx = 0; block_idx < blocks.size(); ++block_idx) {
		Block *block = blocks[block_idx];
		if (disabled.test(block_idx) && block->chosen_alternative(chain_idx) != block->canonical_alternative) {
			return true;
		}
	}
	return false;
}

int Scene::add_input(lua_State* L)
{
	assert(lua_gettop(L) == 1 || lua_gettop(L) == 2);
	Scene *scene = (Scene *)luaL_checkudata(L, 1, "Scene");

	Block *block = new Block;
	block->declaration_point = get_declaration_point(L);
	block->idx = scene->blocks.size();
	if (lua_gettop(L) == 1) {
		// No parameter given, so a flexible input.
		block->alternatives.emplace_back(new EffectBlueprint(LIVE_INPUT_YCBCR));
		block->alternatives.emplace_back(new EffectBlueprint(LIVE_INPUT_YCBCR_WITH_DEINTERLACE));
		block->alternatives.emplace_back(new EffectBlueprint(LIVE_INPUT_YCBCR_PLANAR));
		block->alternatives.emplace_back(new EffectBlueprint(LIVE_INPUT_BGRA));
		block->alternatives.emplace_back(new EffectBlueprint(IMAGE_INPUT));
	} else {
		// Input of a given type. We'll specialize it here, plus connect the input as given.
		if (lua_isnumber(L, 2)) {
			block->alternatives.emplace_back(new EffectBlueprint(LIVE_INPUT_YCBCR));
			block->alternatives.emplace_back(new EffectBlueprint(LIVE_INPUT_YCBCR_WITH_DEINTERLACE));
#ifdef HAVE_CEF
		} else if (luaL_testudata(L, 2, "HTMLInput")) {
			block->alternatives.emplace_back(new EffectBlueprint(LIVE_INPUT_BGRA));
#endif
		} else if (luaL_testudata(L, 2, "VideoInput")) {
			FFmpegCapture *capture = *(FFmpegCapture **)luaL_checkudata(L, 2, "VideoInput");
			if (capture->get_current_pixel_format() == bmusb::PixelFormat_8BitYCbCrPlanar) {
				block->alternatives.emplace_back(new EffectBlueprint(LIVE_INPUT_YCBCR_PLANAR));
			} else {
				assert(capture->get_current_pixel_format() == bmusb::PixelFormat_8BitBGRA);
				block->alternatives.emplace_back(new EffectBlueprint(LIVE_INPUT_BGRA));
			}
		} else if (luaL_testudata(L, 2, "ImageInput")) {
			block->alternatives.emplace_back(new EffectBlueprint(IMAGE_INPUT));
		} else {
			luaL_error(L, "add_input() called with something that's not a signal (a signal number, a HTML input, or a VideoInput)");
		}
		bool ok = display(block, L, 2);
		assert(ok);
	}
	block->is_input = true;
	scene->blocks.push_back(block);

	return wrap_lua_existing_object_nonowned<Block>(L, "Block", block);
}

void Scene::find_inputs_for_block(lua_State *L, Scene *scene, Block *block)
{
	if (lua_gettop(L) == 2) {
		// Implicitly the last added effect.
		assert(!scene->blocks.empty());
		block->inputs.push_back(scene->blocks.size() - 1);
		return;
	}

	for (int idx = 3; idx <= lua_gettop(L); ++idx) {
		Block *input_block = nullptr;
		if (luaL_testudata(L, idx, "Block")) {
			input_block = *(Block **)luaL_checkudata(L, idx, "Block");
		} else {
			EffectBlueprint *blueprint = *(EffectBlueprint **)luaL_checkudata(L, idx, "EffectBlueprint");

			// Search through all the blocks to figure out which one contains this effect.
			for (Block *block : scene->blocks) {
				if (find(block->alternatives.begin(), block->alternatives.end(), blueprint) != block->alternatives.end()) {
					input_block = block;
					break;
				}
			}
			if (input_block == nullptr) {
				luaL_error(L, "Input effect in parameter #%d has not been added to this scene", idx - 1);
			}
		}
		block->inputs.push_back(input_block->idx);
	}
}

int Scene::add_effect(lua_State* L)
{
	assert(lua_gettop(L) >= 2);
	Scene *scene = (Scene *)luaL_checkudata(L, 1, "Scene");

	Block *block = new Block;
	block->declaration_point = get_declaration_point(L);
	block->idx = scene->blocks.size();

	if (lua_istable(L, 2)) {
		size_t len = lua_objlen(L, 2);
		for (size_t i = 0; i < len; ++i) {
			lua_rawgeti(L, 2, i + 1);
			EffectBlueprint *blueprint = *(EffectBlueprint **)luaL_checkudata(L, -1, "EffectBlueprint");
			block->alternatives.push_back(blueprint);
			lua_settop(L, -2);
		}
	} else {
		EffectBlueprint *blueprint = *(EffectBlueprint **)luaL_checkudata(L, 2, "EffectBlueprint");
		block->alternatives.push_back(blueprint);
	}

	int identity_index = find_index_of(block, IDENTITY_EFFECT);
	if (identity_index == -1) {
		block->canonical_alternative = 0;
	} else {
		// Pick the IdentityEffect as the canonical alternative, in case it
		// helps us disable more stuff.
		block->canonical_alternative = identity_index;
	}

	find_inputs_for_block(L, scene, block);
	scene->blocks.push_back(block);

	return wrap_lua_existing_object_nonowned<Block>(L, "Block", block);
}

int Scene::add_optional_effect(lua_State* L)
{
	assert(lua_gettop(L) >= 2);
	Scene *scene = (Scene *)luaL_checkudata(L, 1, "Scene");

	Block *block = new Block;
	block->declaration_point = get_declaration_point(L);
	block->idx = scene->blocks.size();

	EffectBlueprint *blueprint = *(EffectBlueprint **)luaL_checkudata(L, 2, "EffectBlueprint");
	block->alternatives.push_back(blueprint);

	// An IdentityEffect will be the alternative for when the effect is disabled.
	block->alternatives.push_back(new EffectBlueprint(IDENTITY_EFFECT));

	block->canonical_alternative = 1;

	find_inputs_for_block(L, scene, block);
	scene->blocks.push_back(block);

	return wrap_lua_existing_object_nonowned<Block>(L, "Block", block);
}

Effect *Scene::instantiate_effects(const Block *block, size_t chain_idx, Scene::Instantiation *instantiation)
{
	// Find the chosen alternative for this block in this instance.
	EffectType chosen_type = block->alternatives[block->chosen_alternative(chain_idx)]->effect_type;

	vector<Effect *> inputs;
	for (size_t input_idx : block->inputs) {
		inputs.push_back(instantiate_effects(blocks[input_idx], chain_idx, instantiation));

		// As a special case, we allow IdentityEffect to take only one input
		// even if the other alternative (or alternatives) is multi-input.
		// Thus, even if there are more than one inputs, instantiate only
		// the first one.
		if (chosen_type == IDENTITY_EFFECT) {
			break;
		}
	}

	Effect *effect;
	switch (chosen_type) {
	case LIVE_INPUT_YCBCR:
	case LIVE_INPUT_YCBCR_WITH_DEINTERLACE:
	case LIVE_INPUT_YCBCR_PLANAR:
	case LIVE_INPUT_BGRA: {
		bool deinterlace = (chosen_type == LIVE_INPUT_YCBCR_WITH_DEINTERLACE);
		bool override_bounce = !deinterlace;  // For most chains, this will be fine. Reconsider if we see real problems somewhere; it's better than having the user try to understand it.
		bmusb::PixelFormat pixel_format;
		if (chosen_type == LIVE_INPUT_BGRA) {
			pixel_format = bmusb::PixelFormat_8BitBGRA;
		} else if (chosen_type == LIVE_INPUT_YCBCR_PLANAR) {
			pixel_format = bmusb::PixelFormat_8BitYCbCrPlanar;
		} else if (global_flags.ten_bit_input) {
			pixel_format = bmusb::PixelFormat_10BitYCbCr;
		} else {
			pixel_format = bmusb::PixelFormat_8BitYCbCr;
		}
		LiveInputWrapper *input = new LiveInputWrapper(theme, instantiation->chain.get(), pixel_format, override_bounce, deinterlace, /*user_connectable=*/true);
		effect = input->get_effect();  // Adds itself to the chain, so no need to call add_effect().
		instantiation->inputs.emplace(block->idx, input);
		break;
	}
	case IMAGE_INPUT: {
		ImageInput *input = new ImageInput;
		instantiation->chain->add_input(input);
		instantiation->image_inputs.emplace(block->idx, input);
		effect = input;
		break;
	}
	default:
		effect = instantiate_effect(instantiation->chain.get(), chosen_type);
		instantiation->chain->add_effect(effect, inputs);
		break;
	}
	instantiation->effects.emplace(block->idx, effect);
	return effect;
}

int Scene::finalize(lua_State* L)
{
	bool only_one_mode = false;
	bool chosen_mode = false;
	if (lua_gettop(L) == 2) {
		only_one_mode = true;
		chosen_mode = checkbool(L, 2);
	} else {
		assert(lua_gettop(L) == 1);
	}
	Scene *scene = (Scene *)luaL_checkudata(L, 1, "Scene");
	Theme *theme = get_theme_updata(L);

	size_t base = 1;
	for (Block *block : scene->blocks) {
		block->cardinality_base = base;
		base *= block->alternatives.size();
	}

	const size_t cardinality = base;
	size_t real_cardinality = 0;
	for (size_t chain_idx = 0; chain_idx < cardinality; ++chain_idx) {
		if (!scene->is_noncanonical_chain(chain_idx)) {
			++real_cardinality;
		}
	}
	const size_t total_cardinality = real_cardinality * (only_one_mode ? 1 : 2);
	if (total_cardinality > 200) {
		print_warning(L, "The given Scene will instantiate %zu different versions. This will take a lot of time and RAM to compile; see if you could limit some options by e.g. locking the input type in some cases (by giving a fixed input to add_input()).\n",
			total_cardinality);
	}

	Block *output_block = scene->blocks.back();
	for (bool is_main_chain : { false, true }) {
		for (size_t chain_idx = 0; chain_idx < cardinality; ++chain_idx) {
			if ((only_one_mode && is_main_chain != chosen_mode) ||
			    scene->is_noncanonical_chain(chain_idx)) {
				scene->chains.emplace_back();
				continue;
			}

			Scene::Instantiation instantiation;
			instantiation.chain.reset(new EffectChain(scene->aspect_nom, scene->aspect_denom, theme->get_resource_pool()));
			scene->instantiate_effects(output_block, chain_idx, &instantiation);

			add_outputs_and_finalize(instantiation.chain.get(), is_main_chain);
			scene->chains.emplace_back(move(instantiation));
		}
	}
	return 0;
}

std::pair<movit::EffectChain *, std::function<void()>>
Scene::get_chain(Theme *theme, lua_State *L, unsigned num, const InputState &input_state)
{
	// For video inputs, pick the right interlaced/progressive version
	// based on the current state of the signals.
	InputStateInfo info(input_state);
	for (Block *block : blocks) {
		if (block->is_input && block->signal_type_to_connect == Block::CONNECT_SIGNAL) {
			EffectType chosen_type = current_type(block);
			assert(chosen_type == LIVE_INPUT_YCBCR || chosen_type == LIVE_INPUT_YCBCR_WITH_DEINTERLACE);
			if (info.last_interlaced[block->signal_to_connect]) {
				block->currently_chosen_alternative = find_index_of(block, LIVE_INPUT_YCBCR_WITH_DEINTERLACE);
			} else {
				block->currently_chosen_alternative = find_index_of(block, LIVE_INPUT_YCBCR);
			}
		}
	}

	// Pick out the right chain based on the current selections,
	// and snapshot all the set variables so that we can set them
	// in the prepare function even if they're being changed by
	// the Lua code later.
	bool is_main_chain = (num == 0);
	size_t chain_idx = compute_chain_number(is_main_chain);
	const Scene::Instantiation &instantiation = chains[chain_idx];
	EffectChain *effect_chain = instantiation.chain.get();

	map<LiveInputWrapper *, int> signals_to_connect;
	map<ImageInput *, string> images_to_select;
	map<pair<Effect *, string>, int> int_to_set;
	map<pair<Effect *, string>, float> float_to_set;
	map<pair<Effect *, string>, array<float, 3>> vec3_to_set;
	map<pair<Effect *, string>, array<float, 4>> vec4_to_set;
	for (const auto &index_and_input : instantiation.inputs) {
		Block *block = blocks[index_and_input.first];
		EffectType chosen_type = current_type(block);
		LiveInputWrapper *input = index_and_input.second;
		if (chosen_type == LIVE_INPUT_YCBCR ||
		    chosen_type == LIVE_INPUT_YCBCR_WITH_DEINTERLACE ||
		    chosen_type == LIVE_INPUT_YCBCR_PLANAR ||
		    chosen_type == LIVE_INPUT_BGRA) {
			if (block->signal_type_to_connect == Block::CONNECT_SIGNAL) {
				signals_to_connect.emplace(input, block->signal_to_connect);
#ifdef HAVE_CEF
			} else if (block->signal_type_to_connect == Block::CONNECT_CEF) {
				signals_to_connect.emplace(input, block->cef_to_connect->get_card_index());
#endif
			} else if (block->signal_type_to_connect == Block::CONNECT_VIDEO) {
				signals_to_connect.emplace(input, block->video_to_connect->get_card_index());
			} else if (block->signal_type_to_connect == Block::CONNECT_NONE) {
				luaL_error(L, "An input in a scene was not connected to anything (forgot to call display())");
			} else {
				assert(false);
			}
		}
	}
	for (const auto &index_and_input : instantiation.image_inputs) {
		Block *block = blocks[index_and_input.first];
		ImageInput *input = index_and_input.second;
		if (current_type(block) == IMAGE_INPUT) {
			images_to_select.emplace(input, block->pathname);
		}
	}
	for (const auto &index_and_effect : instantiation.effects) {
		Block *block = blocks[index_and_effect.first];
		Effect *effect = index_and_effect.second;

		bool missing_width = (current_type(block) == RESIZE_EFFECT ||
			current_type(block) == RESAMPLE_EFFECT ||
			current_type(block) == PADDING_EFFECT);
		bool missing_height = missing_width;

		// Get the effects currently set on the block.
		if (current_type(block) != IDENTITY_EFFECT) {  // Ignore settings on optional effects.
			if (block->int_parameters.count("width") && block->int_parameters["width"] > 0) {
				missing_width = false;
			}
			if (block->int_parameters.count("height") && block->int_parameters["height"] > 0) {
				missing_height = false;
			}
			for (const auto &key_and_tuple : block->int_parameters) {
				int_to_set.emplace(make_pair(effect, key_and_tuple.first), key_and_tuple.second);
			}
			for (const auto &key_and_tuple : block->float_parameters) {
				float_to_set.emplace(make_pair(effect, key_and_tuple.first), key_and_tuple.second);
			}
			for (const auto &key_and_tuple : block->vec3_parameters) {
				vec3_to_set.emplace(make_pair(effect, key_and_tuple.first), key_and_tuple.second);
			}
			for (const auto &key_and_tuple : block->vec4_parameters) {
				vec4_to_set.emplace(make_pair(effect, key_and_tuple.first), key_and_tuple.second);
			}
		}

		// Parameters set on the blueprint itself override those that are set for the block,
		// so they are set afterwards.
		if (!block->alternatives.empty()) {
			EffectBlueprint *blueprint = block->alternatives[block->currently_chosen_alternative];
			if (blueprint->int_parameters.count("width") && blueprint->int_parameters["width"] > 0) {
				missing_width = false;
			}
			if (blueprint->int_parameters.count("height") && blueprint->int_parameters["height"] > 0) {
				missing_height = false;
			}
			for (const auto &key_and_tuple : blueprint->int_parameters) {
				int_to_set[make_pair(effect, key_and_tuple.first)] = key_and_tuple.second;
			}
			for (const auto &key_and_tuple : blueprint->float_parameters) {
				float_to_set[make_pair(effect, key_and_tuple.first)] = key_and_tuple.second;
			}
			for (const auto &key_and_tuple : blueprint->vec3_parameters) {
				vec3_to_set[make_pair(effect, key_and_tuple.first)] = key_and_tuple.second;
			}
			for (const auto &key_and_tuple : blueprint->vec4_parameters) {
				vec4_to_set[make_pair(effect, key_and_tuple.first)] = key_and_tuple.second;
			}
		}

		if (missing_width || missing_height) {
			fprintf(stderr, "WARNING: Unset or nonpositive width/height for effect declared at %s "
				"when getting scene for signal %u; setting to 1x1 to avoid crash.\n",
				block->declaration_point.c_str(), num);
			int_to_set[make_pair(effect, "width")] = 1;
			int_to_set[make_pair(effect, "height")] = 1;
		}
	}

	lua_pop(L, 1);

	auto setup_chain = [L, theme, signals_to_connect, images_to_select, int_to_set, float_to_set, vec3_to_set, vec4_to_set, input_state]{
		lock_guard<mutex> lock(theme->m);

		// Set up state, including connecting signals.
		for (const auto &input_and_signal : signals_to_connect) {
			LiveInputWrapper *input = input_and_signal.first;
			input->connect_signal_raw(input_and_signal.second, input_state);
		}
		for (const auto &input_and_filename : images_to_select) {
			input_and_filename.first->switch_image(input_and_filename.second);
		}
		for (const auto &effect_and_key_and_value : int_to_set) {
			Effect *effect = effect_and_key_and_value.first.first;
			const string &key = effect_and_key_and_value.first.second;
			const int value = effect_and_key_and_value.second;
			if (!effect->set_int(key, value)) {
				luaL_error(L, "Effect refused set_int(\"%s\", %d) (invalid key?)", key.c_str(), value);
			}
		}
		for (const auto &effect_and_key_and_value : float_to_set) {
			Effect *effect = effect_and_key_and_value.first.first;
			const string &key = effect_and_key_and_value.first.second;
			const float value = effect_and_key_and_value.second;
			if (!effect->set_float(key, value)) {
				luaL_error(L, "Effect refused set_float(\"%s\", %f) (invalid key?)", key.c_str(), value);
			}
		}
		for (const auto &effect_and_key_and_value : vec3_to_set) {
			Effect *effect = effect_and_key_and_value.first.first;
			const string &key = effect_and_key_and_value.first.second;
			const float *value = effect_and_key_and_value.second.data();
			if (!effect->set_vec3(key, value)) {
				luaL_error(L, "Effect refused set_vec3(\"%s\", %f, %f, %f) (invalid key?)", key.c_str(),
						value[0], value[1], value[2]);
			}
		}
		for (const auto &effect_and_key_and_value : vec4_to_set) {
			Effect *effect = effect_and_key_and_value.first.first;
			const string &key = effect_and_key_and_value.first.second;
			const float *value = effect_and_key_and_value.second.data();
			if (!effect->set_vec4(key, value)) {
				luaL_error(L, "Effect refused set_vec4(\"%s\", %f, %f, %f, %f) (invalid key?)", key.c_str(),
						value[0], value[1], value[2], value[3]);
			}
		}
	};
	return make_pair(effect_chain, move(setup_chain));
}

bool display(Block *block, lua_State *L, int idx)
{
	if (lua_isnumber(L, idx)) {
		Theme *theme = get_theme_updata(L);
		int signal_idx = luaL_checknumber(L, idx);
		block->signal_type_to_connect = Block::CONNECT_SIGNAL;
		block->signal_to_connect = theme->map_signal(signal_idx);
		block->currently_chosen_alternative = find_index_of(block, LIVE_INPUT_YCBCR);  // Will be changed to deinterlaced at get_chain() time if needed.
		return true;
#ifdef HAVE_CEF
	} else if (luaL_testudata(L, idx, "HTMLInput")) {
		CEFCapture *capture = *(CEFCapture **)luaL_checkudata(L, idx, "HTMLInput");
		block->signal_type_to_connect = Block::CONNECT_CEF;
		block->cef_to_connect = capture;
		block->currently_chosen_alternative = find_index_of(block, LIVE_INPUT_BGRA);
		assert(capture->get_current_pixel_format() == bmusb::PixelFormat_8BitBGRA);
		return true;
#endif
	} else if (luaL_testudata(L, idx, "VideoInput")) {
		FFmpegCapture *capture = *(FFmpegCapture **)luaL_checkudata(L, idx, "VideoInput");
		block->signal_type_to_connect = Block::CONNECT_VIDEO;
		block->video_to_connect = capture;
		if (capture->get_current_pixel_format() == bmusb::PixelFormat_8BitYCbCrPlanar) {
			block->currently_chosen_alternative = find_index_of(block, LIVE_INPUT_YCBCR_PLANAR);
		} else {
			assert(capture->get_current_pixel_format() == bmusb::PixelFormat_8BitBGRA);
			block->currently_chosen_alternative = find_index_of(block, LIVE_INPUT_BGRA);
		}
		return true;
	} else if (luaL_testudata(L, idx, "ImageInput")) {
		ImageInput *image = *(ImageInput **)luaL_checkudata(L, idx, "ImageInput");
		block->signal_type_to_connect = Block::CONNECT_NONE;
		block->currently_chosen_alternative = find_index_of(block, IMAGE_INPUT);
		block->pathname = image->get_pathname();
		return true;
	} else {
		return false;
	}
}

int Block_display(lua_State* L)
{
	assert(lua_gettop(L) == 2);
	Block *block = *(Block **)luaL_checkudata(L, 1, "Block");
	if (!block->is_input) {
		luaL_error(L, "display() called on something that isn't an input");
	}

	bool ok = display(block, L, 2);
	if (!ok) {
		luaL_error(L, "display() called with something that's not a signal (a signal number, a HTML input, or a VideoInput)");
	}

	if (block->currently_chosen_alternative == -1) {
		luaL_error(L, "display() called on an input whose type was fixed at construction time, with a signal of different type");
	}

	return 0;
}

int Block_choose(lua_State* L)
{
	assert(lua_gettop(L) == 2);
	Block *block = *(Block **)luaL_checkudata(L, 1, "Block");
	int alternative_idx = -1;
	if (lua_isnumber(L, 2)) {
		alternative_idx = luaL_checknumber(L, 2);
	} else if (lua_istable(L, 2)) {
		// See if it's an Effect metatable (e.g. foo:choose(ResampleEffect))
		lua_getfield(L, 2, "__effect_type_id");
		if (lua_isnumber(L, -1)) {
			EffectType effect_type = EffectType(luaL_checknumber(L, -1));
			alternative_idx = find_index_of(block, effect_type);
		}
		lua_pop(L, 1);
	}

	if (alternative_idx == -1) {
		luaL_error(L, "choose() called with something that was not an index or an effect type (e.g. ResampleEffect) that was part of the alternatives");
	}

	assert(alternative_idx >= 0);
	assert(size_t(alternative_idx) < block->alternatives.size());
	block->currently_chosen_alternative = alternative_idx;

	return wrap_lua_existing_object_nonowned<EffectBlueprint>(L, "EffectBlueprint", block->alternatives[alternative_idx]);
}

int Block_enable(lua_State *L)
{
	assert(lua_gettop(L) == 1);
	Block *block = *(Block **)luaL_checkudata(L, 1, "Block");

	if (block->alternatives.size() != 2 ||
	    block->alternatives[1]->effect_type != IDENTITY_EFFECT) {
		luaL_error(L, "enable() called on something that wasn't added with add_optional_effect()");
	}
	block->currently_chosen_alternative = 0;  // The actual effect.
	return 0;
}

int Block_enable_if(lua_State *L)
{
	assert(lua_gettop(L) == 2);
	Block *block = *(Block **)luaL_checkudata(L, 1, "Block");

	if (block->alternatives.size() != 2 ||
	    block->alternatives[1]->effect_type != IDENTITY_EFFECT) {
		luaL_error(L, "enable_if() called on something that wasn't added with add_optional_effect()");
	}
	bool enabled = checkbool(L, 2);
	block->currently_chosen_alternative = enabled ? 0 : 1;
	return 0;
}

int Block_disable(lua_State *L)
{
	assert(lua_gettop(L) == 1);
	Block *block = *(Block **)luaL_checkudata(L, 1, "Block");

	block->currently_chosen_alternative = find_index_of(block, IDENTITY_EFFECT);
	if (block->currently_chosen_alternative == -1) {
		luaL_error(L, "disable() called on something that didn't have an IdentityEffect fallback (try add_optional_effect())");
	}
	assert(block->currently_chosen_alternative != -1);
	return 0;
}

int Block_always_disable_if_disabled(lua_State *L)
{
	assert(lua_gettop(L) == 2);
	Block *block = *(Block **)luaL_checkudata(L, 1, "Block");
	Block *disabler_block = *(Block **)luaL_checkudata(L, 2, "Block");

	int my_alternative = find_index_of(block, IDENTITY_EFFECT);
	int their_alternative = find_index_of(disabler_block, IDENTITY_EFFECT);
	if (my_alternative == -1) {
		luaL_error(L, "always_disable_if_disabled() called on something that didn't have an IdentityEffect fallback (try add_optional_effect())");
	}
	if (their_alternative == -1) {
		luaL_error(L, "always_disable_if_disabled() with an argument that didn't have an IdentityEffect fallback (try add_optional_effect())");
	}

	block->disablers.push_back(disabler_block->idx);

	lua_pop(L, 2);
	return 0;
}

int Block_set_int(lua_State *L)
{
	assert(lua_gettop(L) == 3);
	Block *block = *(Block **)luaL_checkudata(L, 1, "Block");
	string key = checkstdstring(L, 2);
	float value = luaL_checknumber(L, 3);

	// TODO: check validity already here, if possible?
	block->int_parameters[key] = value;

	return 0;
}

int Block_set_float(lua_State *L)
{
	assert(lua_gettop(L) == 3);
	Block *block = *(Block **)luaL_checkudata(L, 1, "Block");
	string key = checkstdstring(L, 2);
	float value = luaL_checknumber(L, 3);

	// TODO: check validity already here, if possible?
	block->float_parameters[key] = value;

	return 0;
}

int Block_set_vec3(lua_State *L)
{
	assert(lua_gettop(L) == 5);
	Block *block = *(Block **)luaL_checkudata(L, 1, "Block");
	string key = checkstdstring(L, 2);
	array<float, 3> v;
	v[0] = luaL_checknumber(L, 3);
	v[1] = luaL_checknumber(L, 4);
	v[2] = luaL_checknumber(L, 5);

	// TODO: check validity already here, if possible?
	block->vec3_parameters[key] = v;

	return 0;
}

int Block_set_vec4(lua_State *L)
{
	assert(lua_gettop(L) == 6);
	Block *block = *(Block **)luaL_checkudata(L, 1, "Block");
	string key = checkstdstring(L, 2);
	array<float, 4> v;
	v[0] = luaL_checknumber(L, 3);
	v[1] = luaL_checknumber(L, 4);
	v[2] = luaL_checknumber(L, 5);
	v[3] = luaL_checknumber(L, 6);

	// TODO: check validity already here, if possible?
	block->vec4_parameters[key] = v;

	return 0;
}

