/*
 * NASPRO - The NASPRO Architecture for Sound Processing
 * LV2 bridging helper library
 *
 * Copyright (C) 2007-2011 NASPRO Bridge it development team
 *
 * See the COPYING file for license conditions.
 */

#include "internal.h"

_NABRIT_DEF nabrit_bridge
nabrit_bridge_new(const char *binary)
{
	nabrit_bridge ret;
	int errsv;

	ret = malloc(sizeof(struct _nabrit_bridge));
	if (ret == NULL)
	  {
		errno = ENOMEM;
		goto alloc_err;
	  }

	ret->pluglibs = nacore_list_new(_nabrit_pluglib_get_size);
	if (ret == NULL)
	  {
		errsv = errno;
		goto pluglibs_err;
	  }

	ret->plugins = nacore_avl_tree_new(_nabrit_plugin_cmp, NULL);
	if (ret->plugins == NULL)
	  {
		errsv = errno;
		goto plugins_err;
	  }

	ret->mutex = nacore_mutex_new();
	if (ret->mutex == NULL)
	  {
		errsv = errno;
		goto mutex_err;
	  }

	ret->binary = binary;

	return ret;

mutex_err:
	nacore_avl_tree_free(ret->plugins, NULL, NULL);
plugins_err:
	nacore_list_free(ret->pluglibs, NULL, NULL);
pluglibs_err:
	free(ret);
	errno = errsv;
alloc_err:
	return NULL;
}

typedef struct
  {
	nacore_op_with_msg_cb	 free_cb;
	void			*free_opaque;
	nacore_msg_context	 msg_context;
	void			*msg_opaque;
  } pluglib_free_data_t;

static void
pluglib_free_cb(void *value, void *opaque)
{
	nabrit_pluglib p;
	pluglib_free_data_t *o;
	nacore_msg_context ctx;

	p = (nabrit_pluglib)value;
	o = (pluglib_free_data_t *)opaque;

	ctx = nacore_msg_status_begin(o->msg_context, o->msg_opaque,
				      "Unloading plugin library %s",
				      p->filename);

	o->free_cb(p, ctx, o->msg_opaque, o->free_opaque);

	nacore_msg_status_end(ctx, nacore_msg_result_ok);
}

_NABRIT_DEF void
nabrit_bridge_free(nabrit_bridge bridge, nacore_op_with_msg_cb free_cb,
		   void *free_opaque, nacore_msg_context msg_context,
		   void *msg_opaque)
{
	pluglib_free_data_t opaque;
	nacore_msg_context ctx;

	opaque.free_cb		= free_cb;
	opaque.free_opaque	= free_opaque;
	opaque.msg_opaque	= msg_opaque;

	ctx = nacore_msg_status_begin(msg_context , msg_opaque ,
				      "Unloading plugin libraries");
	opaque.msg_context = ctx;

	nacore_list_free(bridge->pluglibs,
			 (free_cb != NULL) ? pluglib_free_cb : NULL, &opaque);
	nacore_avl_tree_free(bridge->plugins, NULL, NULL);

	nacore_mutex_free(bridge->mutex);

	free(bridge);

	nacore_msg_status_end(ctx, nacore_msg_result_ok);
}

_NABRIT_DEF const LV2_Descriptor *
nabrit_bridge_get_descriptor(nabrit_bridge bridge, uint32_t index)
{
	const LV2_Descriptor *ret;
	nacore_avl_tree_elem elem;
	char unlock;

	unlock = nacore_mutex_lock(bridge->mutex) == 0;

	for (elem = nacore_avl_tree_get_first(bridge->plugins);
	     (index != 0) && (elem != NULL);
	     elem = nacore_avl_tree_elem_get_next(bridge->plugins, elem),
	     index--) ;

	if ((elem == NULL) || (index != 0))
		ret = NULL;
	else
		ret = &((nabrit_plugin)nacore_avl_tree_elem_get_value(
					bridge->plugins, elem))->descriptor;

	if (unlock)
		nacore_mutex_unlock(bridge->mutex);

	return ret;
}

_NABRIT_DEF int
nabrit_bridge_begin_op(nabrit_bridge bridge)
{
	return nacore_mutex_lock(bridge->mutex);
}

_NABRIT_DEF void
nabrit_bridge_end_op(nabrit_bridge bridge)
{
	nacore_mutex_unlock(bridge->mutex);
}
