/*
 * NASPRO - NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#ifndef _NACORE_AVL_H
#define _NACORE_AVL_H

#include <stdlib.h>

#ifdef __cplusplus
extern "C" {
#endif

/* WARNING: this API is not thread-safe */

/* AVL tree handle */
typedef struct _nacore_avl_tree * nacore_avl_tree_t;

/*
 * Creates a new AVL tree, returns NULL in case of faliure.
 *
 * content_cmp is a pointer to a function which compares the content of two
 *   nodes (strcmp()-like)
 * key_cmp is a pointer to a function which compares the content of a node with
     a key value (strcmp()-like)
 */
nacore_avl_tree_t nacore_avl_tree_new(int (*content_cmp)(void *c1, void *c2),
				      int (*key_cmp)(void *content, void *key));

/*
 * Adds a node to the tree and balances it, in case of failure the node is
 * simply not added.
 *
 * tree is the AVL tree handle
 * content is the node content
 */
void nacore_avl_tree_add(nacore_avl_tree_t tree, void *content);

/*
 * Calls a callback function for each node of the tree (the order of such calls
 * is not defined)
 *
 * tree is the AVL tree handle
 * callback is the callback function
 * data is the data pointer which gets passed to the callback
 */
void nacore_avl_tree_for_each(nacore_avl_tree_t tree,
			      void (*callback)(void *content, void *data),
			      void *data);

/*
 * Finds a node in the tree by comparing node contents with a key, in case none
 * is found NULL is returned.
 *
 * tree is the AVL tree handle
 * key is the key value to look for
 */
void *nacore_avl_tree_find(nacore_avl_tree_t tree, void *key);

/*
 * Returns the number of nodes in a tree
 * 
 * tree is the AVL tree handle
 */
size_t nacore_avl_tree_get_nodes_count(nacore_avl_tree_t tree);

/*
 * Destroys a tree (frees the tree structure only, not nodes' content)
 *
 * tree is the AVL tree handle
 */
void nacore_avl_tree_free(nacore_avl_tree_t tree);

/*
 * Convenience function for comparing nodes in trees of NASPRO core plugin
 * descriptors where the URI is the ordering field.
 *
 * c1 and c2 are plugin descriptors
 */
int nacore_content_cmp_descriptor_by_uri(void *c1, void *c2);

/*
 * Convenience function for comparing a node's content with a key value where
 * nodes are NASPRO core plugin descriptors and the key is a URI.
 *
 * content is a plugin descriptor
 * data is the URI string
 */
int nacore_key_cmp_descriptor_by_uri(void *content, void *data);

#ifdef __cplusplus
}
#endif

#endif /* !_NACORE_AVL_H */
