/*
 * NASPRO - NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#ifndef _NACORE_BRIDGE_H
#define _NACORE_BRIDGE_H

#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>

#include <lv2.h>

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Plugin descriptor class values
 *
 * Plugins beloging to a subclass should have the superclass disabled:
 * e.g. if NACORE_DESCRIPTOR_CLASS_INSTRUMENT is on, then
 * NACORE_DESCRIPTOR_CLASS_GENERATOR must be off.
 */

/* Values for the classes_1 field */
#define NACORE_DESCRIPTOR_CLASS_GENERATOR	(1)
#define NACORE_DESCRIPTOR_CLASS_INSTRUMENT	(1 << 1)
#define NACORE_DESCRIPTOR_CLASS_OSCILLATOR	(1 << 2)
#define NACORE_DESCRIPTOR_CLASS_UTILITY		(1 << 3)
#define NACORE_DESCRIPTOR_CLASS_CONVERTER	(1 << 4)
#define NACORE_DESCRIPTOR_CLASS_ANALYSER	(1 << 5)
#define NACORE_DESCRIPTOR_CLASS_MIXER		(1 << 6)
#define NACORE_DESCRIPTOR_CLASS_SIMULATOR	(1 << 7)
#define NACORE_DESCRIPTOR_CLASS_DELAY		(1 << 8)
#define NACORE_DESCRIPTOR_CLASS_MODULATOR	(1 << 9)
#define NACORE_DESCRIPTOR_CLASS_REVERB		(1 << 10)
#define NACORE_DESCRIPTOR_CLASS_PHASER		(1 << 11)
#define NACORE_DESCRIPTOR_CLASS_FLANGER		(1 << 12)
#define NACORE_DESCRIPTOR_CLASS_CHORUS		(1 << 13)
#define NACORE_DESCRIPTOR_CLASS_FILTER		(1 << 14)
#define NACORE_DESCRIPTOR_CLASS_LOWPASS		(1 << 15)
#define NACORE_DESCRIPTOR_CLASS_BANDPASS	(1 << 16)
#define NACORE_DESCRIPTOR_CLASS_HIGHPASS	(1 << 17)
#define NACORE_DESCRIPTOR_CLASS_COMB		(1 << 18)
#define NACORE_DESCRIPTOR_CLASS_ALLPASS		(1 << 19)
#define NACORE_DESCRIPTOR_CLASS_EQ		(1 << 20)
#define NACORE_DESCRIPTOR_CLASS_PARAEQ		(1 << 21)
#define NACORE_DESCRIPTOR_CLASS_MULTIEQ		(1 << 22)
#define NACORE_DESCRIPTOR_CLASS_SPECTRAL	(1 << 23)
#define NACORE_DESCRIPTOR_CLASS_PITCH		(1 << 24)
#define NACORE_DESCRIPTOR_CLASS_AMPLIFIER	(1 << 25)
#define NACORE_DESCRIPTOR_CLASS_DISTORTION	(1 << 26)
#define NACORE_DESCRIPTOR_CLASS_WAVESHAPER	(1 << 27)

/* Values for the classes_2 field */
#define NACORE_DESCRIPTOR_CLASS_DYNAMICS	(1)
#define NACORE_DESCRIPTOR_CLASS_COMPRESSOR	(1 << 1)
#define NACORE_DESCRIPTOR_CLASS_EXPANDER	(1 << 2)
#define NACORE_DESCRIPTOR_CLASS_LIMITER		(1 << 3)
#define NACORE_DESCRIPTOR_CLASS_GATE		(1 << 4)

/*
 * Plugin descriptor property values
 */

/* Values for the properties field */
#define NACORE_DESCRIPTOR_HAS_LIVE_DEP		(1)
#define NACORE_DESCRIPTOR_INPLACE_BROKEN	(1 << 1)
#define NACORE_DESCRIPTOR_HARD_RT_CAPABLE	(1 << 2)

/*
 * Port descriptor property values
 */

/* Values for the properties field */
#define NACORE_PORT_IS_OUTPUT			(1)
#define NACORE_PORT_IS_AUDIO			(1 << 1)
#define NACORE_PORT_IS_MIDI			(1 << 2)
#define NACORE_PORT_CONNECTION_OPTIONAL		(1 << 3)
#define NACORE_PORT_REPORTS_LATENCY		(1 << 4)
#define NACORE_PORT_TOGGLED			(1 << 5)
#define NACORE_PORT_SAMPLE_RATE			(1 << 6)
#define NACORE_PORT_INTEGER			(1 << 7)
#define NACORE_PORT_LOGARITHMIC			(1 << 8)

/*
 * Port descriptor scale values
 */

/* Values for the properties field */
#define NACORE_SCALE_HAS_MIN			(1)
#define NACORE_SCALE_HAS_MAX			(1 << 1)
#define NACORE_SCALE_HAS_DEFAULT		(1 << 2)

/* Scale units */
enum nacore_scale_unit
  {
	nacore_scale_unit_none = 0,
	nacore_scale_unit_bar,
	nacore_scale_unit_beat,
	nacore_scale_unit_bpm,
	nacore_scale_unit_cent,
	nacore_scale_unit_cm,
	nacore_scale_unit_coeff,
	nacore_scale_unit_db,
	nacore_scale_unit_deg,
	nacore_scale_unit_hz,
	nacore_scale_unit_inch,
	nacore_scale_unit_khz,
	nacore_scale_unit_km,
	nacore_scale_unit_m,
	nacore_scale_unit_mhz,
	nacore_scale_unit_midi_note,
	nacore_scale_unit_mi,
	nacore_scale_unit_min,
	nacore_scale_unit_mm,
	nacore_scale_unit_ms,
	nacore_scale_unit_oct,
	nacore_scale_unit_pc,
	nacore_scale_unit_s,
	nacore_scale_unit_semitone
  };

/* Scale points */
struct nacore_scale_point
  {
	char		*label;
	float		 value;
  };

/* Scale values for a port descriptor */
struct nacore_port_scale
  {
	uint32_t			 properties;
	float				 min;
	float				 max;
	float				 defaultv;
	enum nacore_scale_unit		 unit;
	struct nacore_scale_point	*points;
	size_t				 points_count;
  };

/* Port descriptor */
struct nacore_port_descriptor
  {
	char				*name;
	uint32_t			 properties;
	struct nacore_port_scale	 scale;
	void				*data;
  };

/* Plugin descriptor */
struct nacore_descriptor
  {
	char				*name;
	char				*uri;
	char				*creator;
	char				*rights;
	struct nacore_port_descriptor	*port_descs;
	size_t				 port_descs_count;
	uint32_t			 properties;
	uint32_t			 classes_1;
	uint32_t			 classes_2;
	void				*data;

	LV2_Handle	(*instantiate)	(const LV2_Descriptor *descriptor,
					 double sample_rate,
					 const char *bundle_path,
					 const LV2_Feature * const *features);
	void		(*connect_port)	(LV2_Handle instance, uint32_t port,
					 void *data_location);
	void		(*activate)	(LV2_Handle instance);
	void		(*run)		(LV2_Handle instance,
					 uint32_t sample_count);
	void		(*deactivate)	(LV2_Handle instance);
	void		(*cleanup)	(LV2_Handle instance);
	const void *	(*extension_data)(const char *uri);
  };

#ifdef __cplusplus
}
#endif

#endif /* !_NACORE_BRIDGE_H */
