/*
 * NASPRO - The NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2012 NASPRO core development team
 *
 * See the COPYING file for license conditions.
 */

/*
   Title: Compiler support

   Compiler-level abstraction.
 */

/*
   Group: Global Declaration Macros

   These macros are meant to be used to delimit declarations in public header
   files.

   Example:

     myheader.h

     > #ifndef MY_HEADER_H
     > #define MY_HEADER_H
     >
     > NACORE_BEGIN_C_DECLS
     >
     > ... blah blah blah ...
     >
     > NACORE_END_C_DECLS
     >
     > #endif
 */

/*
   Macro: NACORE_BEGIN_C_DECLS

   Delimits the beginning of public declarations.

   So that C++ compilers don't mangle their names.
 */

/*
   Macro: NACORE_END_C_DECLS

   Delimits the end of public declarations.

   So that C++ compilers don't mangle their names.
 */

/*
   Group: Symbol Declaration Macros

   These macros are meant to be used for *SINGLE* symbol declarations (variables
   or functions) by prepending them to such declarations.

   They can be combined together, unless otherwise told.

   Example:

     myfunc.h

     > NACORE_PUBLIC void myfunc();

     myfunc.c

     > NACORE_PUBLIC void
     > myfunc()
     > {
     >         ... blah blah blah ...
     > }
 */

/*
   Macro: NACORE_IMPORT

   Specifies that a symbol is imported from a library.

   Cannot be combined with <NACORE_EXPORT>.
 */

/*
   Macro: NACORE_EXPORT

   Specifies that a symbol is to be exported.

   Cannot be combined with <NACORE_IMPORT>.
 */

/*
   Macro: NACORE_PUBLIC

   Specifies that a symbol is publicly visible.

   Cannot be combined with <NACORE_PRIVATE>.
 */

/*
   Macro: NACORE_PRIVATE

   Specifies that a symbol has hidden visibility.

   Cannot be combined with <NACORE_PUBLIC>.
 */

/*
   Macro: NACORE_FORMAT_PRINTF(string_index, first_to_check)
 
   Specifies that a function takes printf()-style arguments.

   Cannot be combined with <NACORE_FORMAT_VPRINTF(string_index)>.

   Paramters:

    string_index	- Index of the format string argument (starting from 1).
    first_to_check	- Index of the first argument to be checked against the
			  format string (starting from 1).
 */

/*
   Macro: NACORE_FORMAT_VPRINTF(string_index)

   Specifies that a function takes vprintf()-style arguments.

   Cannot be combined with <NACORE_FORMAT_PRINTF(string_index, first_to_check)>.

   Parameters:

     string_index	- Index of the format string argument (starting from 1).
 */

#ifndef _NASPRO_CORE_CC_H
#define _NASPRO_CORE_CC_H

#ifndef _NASPRO_CORE_LIB_H
# error Only <NASPRO/core/lib.h> can be included directly.
#endif

#ifdef __cplusplus
# define NACORE_BEGIN_C_DECLS	extern "C" {
# define NACORE_END_C_DECLS	}
#else
# define NACORE_BEGIN_C_DECLS
# define NACORE_END_C_DECLS
#endif

#ifdef __GNUC__		/* assuming gcc >= 2.95 */

# ifdef _WIN32
#  define NACORE_IMPORT __attribute__((dllimport))
#  define NACORE_EXPORT __attribute__((dllexport))
# else
#  define NACORE_IMPORT
#  define NACORE_EXPORT
# endif

# if (__GNUC__ > 3 || (__GNUC__ == 3 && __GNUC_MINOR__ >= 3)) \
     && !defined(_WIN32)
#  define NACORE_PUBLIC		__attribute__((visibility("default")))
#  define NACORE_PRIVATE 	__attribute__((visibility("hidden")))
# else
#  define NACORE_PUBLIC
#  define NACORE_PRIVATE
# endif

# define NACORE_FORMAT_PRINTF(string_index, first_to_check) \
	__attribute__((format(printf, string_index, first_to_check)))
# define NACORE_FORMAT_VPRINTF(string_index) \
	NACORE_FORMAT_PRINTF(string_index, 0)

#else

# define NACORE_IMPORT
# define NACORE_EXPORT
# define NACORE_PUBLIC
# define NACORE_PRIVATE
# define NACORE_FORMAT_PRINTF(string_index, first_to_check)
# define NACORE_FORMAT_VPRINTF(string_index)

#endif

#endif
